# -*- coding: utf-8 -*-
"""
/***************************************************************************
 D2SBrowser
                                 A QGIS plugin
 This plugin allows you to browse your data on a D2S instance.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-06-10
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Geospatial Data Science Lab
        email                : jinha@purdue.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.core import Qgis, QgsProject, QgsRasterLayer, QgsVectorLayer, QgsJsonExporter, QgsWkbTypes

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QThread
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QListWidgetItem, QApplication, QFileDialog

from datetime import date
import json

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the dialog
from .d2s_browser_dialog import D2SBrowserDialog

# Import worker classes for threaded API calls
from .d2s_browser_workers import ProjectsWorker, FlightsWorker, DataProductsWorker, VectorLayersWorker, DataProductUploadWorker
import os.path
import sys

# Add bundled libraries to sys.path
plugin_dir = os.path.dirname(__file__)
libs_dir = os.path.join(plugin_dir, "libs")
if libs_dir not in sys.path:
    sys.path.insert(0, libs_dir)

# Import d2spy from bundled libraries
from d2spy.auth import Auth
from d2spy.workspace import Workspace


NON_RASTER_TYPES = frozenset({"panoramic", "point_cloud", "3dgs"})


class D2SBrowser:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value("locale/userLocale")[0:2]
        locale_path = os.path.join(
            self.plugin_dir, "i18n", "D2SBrowser_{}.qm".format(locale)
        )

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr("&D2S Browser")

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # D2S workspace for authorized user
        self.workspace = None

        # API key for authorized user
        self.api_key = ""

        # Store projects returned from API
        self.projects = []

        # Store flights returned from API
        self.flights = []

        # Store data products returned from API
        self.data_products = []

        # Store vector layers returned from API
        self.vector_layers = []

        # Store active threads to prevent garbage collection
        self.projects_thread = None
        self.flights_thread = None
        self.data_products_thread = None
        self.vector_layers_thread = None
        self.upload_thread = None

        # Cache API responses to avoid redundant requests
        self.projects_cache = None
        self.flights_cache = {}  # Key: project_id, Value: list of flights
        self.data_products_cache = {}  # Key: flight_id, Value: list of data products
        self.vector_layers_cache = {}  # Key: project_id, Value: list of vector layers

        # State management for Create tab workflow
        self.selected_project = None  # Currently selected project for Create tab
        self.selected_flight = None  # Currently selected flight for Create tab

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("D2SBrowser", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ":/plugins/d2s_browser/icon.png"
        self.add_action(
            icon_path,
            text=self.tr("Browse D2S data"),
            callback=self.run,
            parent=self.iface.mainWindow(),
        )

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        # Clean up any running threads
        if self.projects_thread is not None:
            self.projects_thread.quit()
            self.projects_thread.wait()
        if self.flights_thread is not None:
            self.flights_thread.quit()
            self.flights_thread.wait()
        if self.data_products_thread is not None:
            self.data_products_thread.quit()
            self.data_products_thread.wait()
        if self.vector_layers_thread is not None:
            self.vector_layers_thread.quit()
            self.vector_layers_thread.wait()
        if self.upload_thread is not None:
            self.upload_thread.quit()
            self.upload_thread.wait()

        # Remove UI elements
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&D2S Browser"), action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = D2SBrowserDialog()

            # EVENTS

            # Event when login button clicked
            self.dlg.loginPushButton.clicked.connect(self.login)
            # Event when project combobox changed
            self.dlg.projectsComboBox.currentIndexChanged.connect(self.update_flights)
            # Event when flight combobox changed
            self.dlg.flightsComboBox.currentIndexChanged.connect(
                self.update_data_products
            )
            # Event when project refresh button clicked
            self.dlg.projectsRefreshPushButton.clicked.connect(self.refresh_projects)
            # Event when add to map button clicked
            self.dlg.dataProductsPushButton.clicked.connect(
                self.add_data_products_to_map
            )
            # Event when map layers add to map button clicked
            self.dlg.mapLayersPushButton.clicked.connect(
                self.add_map_layers_to_map
            )

            # CREATE TAB EVENTS

            # Project column events
            self.dlg.selectProjectPushButton.clicked.connect(self.on_select_project)
            self.dlg.createProjectPushButton.clicked.connect(self.on_create_project)
            self.dlg.projectBoundaryFileBrowseButton.clicked.connect(self.browse_project_boundary_file)
            self.dlg.projectBoundaryFileRadio.toggled.connect(self.toggle_project_boundary_source)
            self.dlg.projectBoundaryCanvasRadio.toggled.connect(self.toggle_project_boundary_source)
            self.dlg.startDateCheckBox.toggled.connect(
                lambda checked: self.dlg.startDateEdit.setEnabled(checked)
            )
            self.dlg.endDateCheckBox.toggled.connect(
                lambda checked: self.dlg.endDateEdit.setEnabled(checked)
            )

            # Flight column events
            self.dlg.selectFlightPushButton.clicked.connect(self.on_select_flight)
            self.dlg.createFlightPushButton.clicked.connect(self.on_create_flight)
            self.dlg.platformComboBox.currentTextChanged.connect(self.on_platform_changed)

            # Data Product column events
            self.dlg.rasterFileBrowseButton.clicked.connect(self.browse_raster_file)
            self.dlg.rasterFileRadio.toggled.connect(self.toggle_raster_source)
            self.dlg.rasterCanvasRadio.toggled.connect(self.toggle_raster_source)
            self.dlg.rasterLayerComboBox.currentIndexChanged.connect(
                lambda: self.dlg.rasterUploadProgressBar.setValue(0)
            )
            self.dlg.dataTypeComboBox.currentTextChanged.connect(self.on_data_type_changed)
            self.dlg.uploadRasterPushButton.clicked.connect(self.on_upload_raster)

            # Map Layer column events
            self.dlg.vectorFileBrowseButton.clicked.connect(self.browse_vector_file)
            self.dlg.vectorFileRadio.toggled.connect(self.toggle_vector_source)
            self.dlg.vectorCanvasRadio.toggled.connect(self.toggle_vector_source)
            self.dlg.uploadVectorPushButton.clicked.connect(self.on_upload_vector)

            # Set default dates to today for Create tab
            from qgis.PyQt.QtCore import QDate
            today = QDate.currentDate()
            self.dlg.acquisitionDateEdit.setDate(today)
            self.dlg.startDateEdit.setDate(today)
            self.dlg.endDateEdit.setDate(today)

            # Initially hide map layers section until data is loaded
            if hasattr(self.dlg, "mapLayersLabel"):
                self.dlg.mapLayersLabel.setVisible(False)
            if hasattr(self.dlg, "mapLayersListWidget"):
                self.dlg.mapLayersListWidget.setVisible(False)
            if hasattr(self.dlg, "mapLayersPushButton"):
                self.dlg.mapLayersPushButton.setVisible(False)

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass

    def set_status(self, message):
        """Set status bar message."""
        if hasattr(self.dlg, "statusBar"):
            self.dlg.statusBar.setText(message)
            # Force the UI to update immediately
            QApplication.processEvents()

    def clear_status(self):
        """Clear status bar message."""
        if hasattr(self.dlg, "statusBar"):
            self.dlg.statusBar.setText("")
            QApplication.processEvents()

    def set_ui_enabled(self, enabled):
        """Enable or disable UI controls during API calls."""
        if hasattr(self.dlg, "projectsComboBox"):
            self.dlg.projectsComboBox.setEnabled(enabled)
        if hasattr(self.dlg, "flightsComboBox"):
            self.dlg.flightsComboBox.setEnabled(enabled)
        if hasattr(self.dlg, "dataProductsListWidget"):
            self.dlg.dataProductsListWidget.setEnabled(enabled)
        if hasattr(self.dlg, "projectsRefreshPushButton"):
            self.dlg.projectsRefreshPushButton.setEnabled(enabled)
        if hasattr(self.dlg, "dataProductsPushButton"):
            self.dlg.dataProductsPushButton.setEnabled(enabled)
        if hasattr(self.dlg, "mapLayersListWidget"):
            self.dlg.mapLayersListWidget.setEnabled(enabled)
        if hasattr(self.dlg, "mapLayersPushButton"):
            self.dlg.mapLayersPushButton.setEnabled(enabled)

    def clear_cache(self):
        """Clear all cached API responses."""
        self.projects_cache = None
        self.flights_cache = {}
        self.data_products_cache = {}
        self.vector_layers_cache = {}

    def login(self):
        """Login to D2S instance using server, email, and password collected from UI."""
        # Clear cache from any previous session
        self.clear_cache()

        # Show status
        self.set_status("Logging in...")

        # Clear existing projects
        self.dlg.projectsComboBox.clear()

        # Get credentials from form fields
        server = self.dlg.serverLineEdit.text()
        email = self.dlg.emailLineEdit.text()
        password = self.dlg.passwordLineEdit.text()

        # Login to D2S instance using provided credentials
        auth = Auth(server)
        try:
            session = auth.login(email, password)
        except EOFError:
            # This occurs when d2spy tries to use getpass.getpass() for password re-entry
            # after authentication failure, but there's no interactive terminal in QGIS
            self.clear_status()
            self.iface.messageBar().pushMessage(
                "Error",
                "Authentication failed. Please check your email and password.",
                level=Qgis.Critical,
                duration=10,
            )
            return
        except Exception as e:
            # Catch any other authentication errors
            self.clear_status()
            self.iface.messageBar().pushMessage(
                "Error",
                f"Authentication error: {str(e)}",
                level=Qgis.Critical,
                duration=10,
            )
            return

        if not session:
            self.clear_status()
            self.iface.messageBar().pushMessage(
                "Error",
                "Unable to sign in with provided credentials",
                level=Qgis.Critical,
                duration=10,
            )
            return

        # Get user model
        user = auth.get_current_user()

        # Check for API key
        if (
            not user
            or not hasattr(user, "api_access_token")
            or user.api_access_token is None
        ):
            self.iface.messageBar().pushMessage(
                "Warning",
                "Please request an API key from the D2S profile page.",
                level=Qgis.Warning,
                duration=10,
            )
        else:
            self.api_key = user.api_access_token

        # Create a workspace
        workspace = Workspace(server, session)
        self.workspace = workspace

        # Get user projects
        self.update_projects()

    def refresh_projects(self):
        """Refresh projects by clearing cache and fetching fresh data."""
        # Clear cache
        self.clear_cache()

        # Block signals to prevent cascading updates during clear
        self.dlg.projectsComboBox.blockSignals(True)
        self.dlg.flightsComboBox.blockSignals(True)

        # Clear all UI elements immediately
        self.dlg.projectsComboBox.clear()
        self.dlg.flightsComboBox.clear()
        self.dlg.dataProductsListWidget.clear()
        self.dlg.mapLayersListWidget.clear()

        # Unblock signals
        self.dlg.projectsComboBox.blockSignals(False)
        self.dlg.flightsComboBox.blockSignals(False)

        # Disable UI during refresh
        self.set_ui_enabled(False)

        # Show status
        self.set_status("Refreshing projects...")

        # Fetch fresh data (which will cascade to flights and data products)
        self.update_projects(use_cache=False)

    def update_projects(self, use_cache=True):
        """Fetch user's projects from D2S instance and update projects UI combobox.

        Args:
            use_cache (bool): If True, use cached data if available. Defaults to True.
        """
        # Check cache first
        if use_cache and self.projects_cache is not None:
            # Clear current projects before loading from cache
            self.dlg.projectsComboBox.clear()
            # Use cached data
            self.on_projects_loaded(self.projects_cache)
            return

        # Clear current projects (if not already cleared)
        if self.dlg.projectsComboBox.count() > 0:
            self.dlg.projectsComboBox.clear()

        # Show status and disable UI
        self.set_status("Loading projects...")
        self.set_ui_enabled(False)

        # Clean up existing thread if any
        if self.projects_thread is not None:
            self.projects_thread.quit()
            self.projects_thread.wait()

        # Create worker and thread
        self.projects_thread = QThread()
        self.projects_worker = ProjectsWorker(self.workspace)
        self.projects_worker.moveToThread(self.projects_thread)

        # Connect signals
        self.projects_thread.started.connect(self.projects_worker.run)
        self.projects_worker.finished.connect(self.on_projects_loaded)
        self.projects_worker.error.connect(self.on_projects_error)
        self.projects_worker.finished.connect(self.projects_thread.quit)
        self.projects_worker.error.connect(self.projects_thread.quit)

        # Start thread
        self.projects_thread.start()

    def on_projects_loaded(self, projects):
        """Handle successful projects load."""
        self.projects = projects
        # Cache the projects for future use
        self.projects_cache = projects
        if len(self.projects) > 0:
            # Sort projects by title a - z
            self.projects = sorted(
                self.projects, key=lambda project: project.title.lower()
            )
            # Block signals to prevent duplicate update_flights() calls
            self.dlg.projectsComboBox.blockSignals(True)
            # Add projects to combobox
            self.dlg.projectsComboBox.addItems(
                [project.title for project in self.projects]
            )
            # Unblock signals
            self.dlg.projectsComboBox.blockSignals(False)
            # Populate Create tab projects combobox
            self.populate_create_projects()
            # Get user flights for first project
            self.update_flights()
        else:
            # No projects, clear current projects, flights, data products, and map layers
            self.dlg.projectsComboBox.clear()
            self.dlg.flightsComboBox.clear()
            self.dlg.dataProductsListWidget.clear()
            self.dlg.mapLayersListWidget.clear()
            # Hide map layers section
            if hasattr(self.dlg, "mapLayersLabel"):
                self.dlg.mapLayersLabel.setVisible(False)
            if hasattr(self.dlg, "mapLayersListWidget"):
                self.dlg.mapLayersListWidget.setVisible(False)
            if hasattr(self.dlg, "mapLayersPushButton"):
                self.dlg.mapLayersPushButton.setVisible(False)
            self.set_status("No projects found")
            # Re-enable UI
            self.set_ui_enabled(True)

    def on_projects_error(self, error_message):
        """Handle projects load error."""
        self.clear_status()
        self.iface.messageBar().pushMessage(
            "Error",
            f"Failed to load projects: {error_message}",
            level=Qgis.Critical,
            duration=10,
        )
        # Re-enable UI
        self.set_ui_enabled(True)

    def update_flights(self, use_cache=True):
        """Fetch flights from selected project and update flights UI combobox.

        Args:
            use_cache (bool): If True, use cached data if available. Defaults to True.
        """
        # Clear current flights
        self.dlg.flightsComboBox.clear()

        # Currently selected project
        selected_project = self.projects[self.dlg.projectsComboBox.currentIndex()]

        # Also update map layers for this project
        self.update_map_layers(use_cache=use_cache)

        # Check cache first
        if use_cache and selected_project.id in self.flights_cache:
            # Use cached data
            self.on_flights_loaded(self.flights_cache[selected_project.id])
            return

        # Show status
        self.set_status("Loading flights...")

        # Disable UI during API call
        self.set_ui_enabled(False)

        # Clean up existing thread if any
        if self.flights_thread is not None:
            self.flights_thread.quit()
            self.flights_thread.wait()

        # Create worker and thread
        self.flights_thread = QThread()
        self.flights_worker = FlightsWorker(selected_project)
        self.flights_worker.moveToThread(self.flights_thread)

        # Connect signals
        self.flights_thread.started.connect(self.flights_worker.run)
        self.flights_worker.finished.connect(self.on_flights_loaded)
        self.flights_worker.error.connect(self.on_flights_error)
        self.flights_worker.finished.connect(self.flights_thread.quit)
        self.flights_worker.error.connect(self.flights_thread.quit)

        # Start thread
        self.flights_thread.start()

    def on_flights_loaded(self, flights):
        """Handle successful flights load."""
        self.flights = flights

        # Cache the flights for this project
        selected_project = self.projects[self.dlg.projectsComboBox.currentIndex()]
        self.flights_cache[selected_project.id] = flights

        # Sort by acquisition date
        self.flights = sorted(
            self.flights, key=lambda flight: flight.acquisition_date, reverse=True
        )

        # Add flights (if any) to combobox
        if len(self.flights) > 0:
            # Add flight acquisition dates to combobox
            flight_items = []
            for flight in self.flights:
                if flight.name:
                    if len(flight.name) > 20:
                        name = flight.name[:17] + "..."
                    else:
                        name = flight.name
                    flight_items.append(f"{flight.acquisition_date} ({name})")
                else:
                    flight_items.append(flight.acquisition_date)
            # Block signals to prevent duplicate update_data_products() calls
            self.dlg.flightsComboBox.blockSignals(True)
            self.dlg.flightsComboBox.addItems(flight_items)
            # Unblock signals
            self.dlg.flightsComboBox.blockSignals(False)
            # Update data products
            self.update_data_products()
        else:
            # No flights, clear current flights and data products
            self.dlg.flightsComboBox.clear()
            self.dlg.dataProductsListWidget.clear()
            self.set_status("No flights found")
            # Re-enable UI
            self.set_ui_enabled(True)

    def on_flights_error(self, error_message):
        """Handle flights load error."""
        self.clear_status()
        self.iface.messageBar().pushMessage(
            "Error",
            f"Failed to load flights: {error_message}",
            level=Qgis.Critical,
            duration=10,
        )
        # Re-enable UI
        self.set_ui_enabled(True)

    def update_data_products(self, use_cache=True):
        """Fetch data products from selected flight and update data products UI list.

        Args:
            use_cache (bool): If True, use cached data if available. Defaults to True.
        """
        # Clear current data products
        self.dlg.dataProductsListWidget.clear()

        # Currently selected flight
        selected_flight = self.flights[self.dlg.flightsComboBox.currentIndex()]

        # Check cache first
        if use_cache and selected_flight.id in self.data_products_cache:
            # Use cached data
            self.on_data_products_loaded(self.data_products_cache[selected_flight.id])
            return

        # Show status
        self.set_status("Loading data products...")

        # Disable UI during API call
        self.set_ui_enabled(False)

        # Clean up existing thread if any
        if self.data_products_thread is not None:
            self.data_products_thread.quit()
            self.data_products_thread.wait()

        # Create worker and thread
        self.data_products_thread = QThread()
        self.data_products_worker = DataProductsWorker(selected_flight)
        self.data_products_worker.moveToThread(self.data_products_thread)

        # Connect signals
        self.data_products_thread.started.connect(self.data_products_worker.run)
        self.data_products_worker.finished.connect(self.on_data_products_loaded)
        self.data_products_worker.error.connect(self.on_data_products_error)
        self.data_products_worker.finished.connect(self.data_products_thread.quit)
        self.data_products_worker.error.connect(self.data_products_thread.quit)

        # Start thread
        self.data_products_thread.start()

    def on_data_products_loaded(self, all_data_products):
        """Handle successful data products load."""
        # Cache the data products for this flight
        selected_flight = self.flights[self.dlg.flightsComboBox.currentIndex()]
        self.data_products_cache[selected_flight.id] = all_data_products

        # Filter out any non-raster data products (e.g., point clouds)
        self.data_products = [
            data_product
            for data_product in all_data_products
            if data_product.data_type not in NON_RASTER_TYPES
        ]
        # Sort by data type
        self.data_products = sorted(
            self.data_products, key=lambda data_product: data_product.data_type
        )

        if len(self.data_products) > 0:
            # Create list item for each data product
            for index, data_product in enumerate(self.data_products):
                # Add data product to list with unchecked checkbox
                item = QListWidgetItem(data_product.data_type)
                item.setFlags(item.flags() | Qt.ItemIsUserCheckable)
                item.setCheckState(Qt.Unchecked)

                # Add data product list item to list widget
                self.dlg.dataProductsListWidget.addItem(item)
            self.set_status("Ready")
        else:
            # No data products, clear current data products
            self.dlg.dataProductsListWidget.clear()
            self.set_status("No data products found")

        # Re-enable UI
        self.set_ui_enabled(True)

    def on_data_products_error(self, error_message):
        """Handle data products load error."""
        self.clear_status()
        self.iface.messageBar().pushMessage(
            "Error",
            f"Failed to load data products: {error_message}",
            level=Qgis.Critical,
            duration=10,
        )
        # Re-enable UI
        self.set_ui_enabled(True)

    def add_data_products_to_map(self):
        """Add data products selected in data products UI list to map."""
        # Show status
        self.set_status("Adding layers to map...")

        # Get flight for constructing data product layer names
        selected_flight = self.flights[self.dlg.flightsComboBox.currentIndex()]
        flight_name = selected_flight.name if selected_flight.name else "Flight"
        flight_date = selected_flight.acquisition_date
        flight_sensor = selected_flight.sensor
        flight_prefix = f"{flight_name}_{flight_date}_{flight_sensor}"

        # Track number of layers added
        layers_added = 0

        # Iterate over each data product and add urls for checked data products in list
        for index in range(self.dlg.dataProductsListWidget.count()):
            item = self.dlg.dataProductsListWidget.item(index)
            if item.checkState() == Qt.Checked:
                # Get url for data product
                data_type = self.data_products[index].data_type
                layer_name = f"{flight_prefix}_{data_type}"
                url = self.data_products[index].url
                # Create raster layer and add to map
                raster_layer = QgsRasterLayer(
                    f"/vsicurl/{url}?API_KEY={self.api_key}", layer_name
                )
                if raster_layer.isValid():
                    QgsProject().instance().addMapLayer(raster_layer)
                    layers_added += 1
                else:
                    self.iface.messageBar().pushMessage(
                        "Warning",
                        f"Invalid url: {url}",
                        level=Qgis.Warning,
                        duration=5,
                    )

        # Zoom to last added layer
        self.iface.zoomToActiveLayer()
        self.iface.mapCanvas().refresh()

        # Update status
        if layers_added > 0:
            self.set_status(
                f"Added {layers_added} layer{'s' if layers_added > 1 else ''} to map"
            )
        else:
            self.set_status("No layers to add")

    def update_map_layers(self, use_cache=True):
        """Fetch vector layers from selected project and update map layers UI list.

        Args:
            use_cache (bool): If True, use cached data if available. Defaults to True.
        """
        # Clear current map layers
        self.dlg.mapLayersListWidget.clear()

        # Currently selected project
        selected_project = self.projects[self.dlg.projectsComboBox.currentIndex()]

        # Check cache first
        if use_cache and selected_project.id in self.vector_layers_cache:
            # Use cached data
            self.on_map_layers_loaded(self.vector_layers_cache[selected_project.id])
            return

        # Show status
        self.set_status("Loading map layers...")

        # Disable UI during API call
        self.set_ui_enabled(False)

        # Clean up existing thread if any
        if self.vector_layers_thread is not None:
            self.vector_layers_thread.quit()
            self.vector_layers_thread.wait()

        # Create worker and thread
        self.vector_layers_thread = QThread()
        self.vector_layers_worker = VectorLayersWorker(self.workspace, selected_project.id)
        self.vector_layers_worker.moveToThread(self.vector_layers_thread)

        # Connect signals
        self.vector_layers_thread.started.connect(self.vector_layers_worker.run)
        self.vector_layers_worker.finished.connect(self.on_map_layers_loaded)
        self.vector_layers_worker.error.connect(self.on_map_layers_error)
        self.vector_layers_worker.finished.connect(self.vector_layers_thread.quit)
        self.vector_layers_worker.error.connect(self.vector_layers_thread.quit)

        # Start thread
        self.vector_layers_thread.start()

    def on_map_layers_loaded(self, layers):
        """Handle successful vector layers load."""
        # Cache the vector layers for this project
        selected_project = self.projects[self.dlg.projectsComboBox.currentIndex()]
        self.vector_layers_cache[selected_project.id] = layers

        self.vector_layers = layers

        # Check if there are any layers
        if len(self.vector_layers) > 0:
            # Show map layers section
            if hasattr(self.dlg, "mapLayersLabel"):
                self.dlg.mapLayersLabel.setVisible(True)
            if hasattr(self.dlg, "mapLayersListWidget"):
                self.dlg.mapLayersListWidget.setVisible(True)
            if hasattr(self.dlg, "mapLayersPushButton"):
                self.dlg.mapLayersPushButton.setVisible(True)

            # Create list item for each vector layer
            for layer in self.vector_layers:
                # Add layer to list with unchecked checkbox
                item = QListWidgetItem(layer.get("layer_name", "Unknown"))
                item.setFlags(item.flags() | Qt.ItemIsUserCheckable)
                item.setCheckState(Qt.Unchecked)

                # Add layer list item to list widget
                self.dlg.mapLayersListWidget.addItem(item)

            self.set_status("Ready")
        else:
            # No map layers, hide the section
            if hasattr(self.dlg, "mapLayersLabel"):
                self.dlg.mapLayersLabel.setVisible(False)
            if hasattr(self.dlg, "mapLayersListWidget"):
                self.dlg.mapLayersListWidget.setVisible(False)
            if hasattr(self.dlg, "mapLayersPushButton"):
                self.dlg.mapLayersPushButton.setVisible(False)

        # Re-enable UI
        self.set_ui_enabled(True)

    def on_map_layers_error(self, error_message):
        """Handle vector layers load error."""
        self.clear_status()
        self.iface.messageBar().pushMessage(
            "Error",
            f"Failed to load map layers: {error_message}",
            level=Qgis.Critical,
            duration=10,
        )
        # Hide map layers section on error
        if hasattr(self.dlg, "mapLayersLabel"):
            self.dlg.mapLayersLabel.setVisible(False)
        if hasattr(self.dlg, "mapLayersListWidget"):
            self.dlg.mapLayersListWidget.setVisible(False)
        if hasattr(self.dlg, "mapLayersPushButton"):
            self.dlg.mapLayersPushButton.setVisible(False)
        # Re-enable UI
        self.set_ui_enabled(True)

    def add_map_layers_to_map(self):
        """Add vector layers selected in map layers UI list to map."""
        # Show status
        self.set_status("Adding vector layers to map...")

        # Get project for constructing layer URLs
        selected_project = self.projects[self.dlg.projectsComboBox.currentIndex()]
        project_title = selected_project.title
        project_id = selected_project.id

        # Track number of layers added
        layers_added = 0

        # Iterate over each vector layer and add URLs for checked layers in list
        for index in range(self.dlg.mapLayersListWidget.count()):
            item = self.dlg.mapLayersListWidget.item(index)
            if item.checkState() == Qt.Checked:
                # Get layer info
                layer_data = self.vector_layers[index]
                layer_name = layer_data.get("layer_name", "Unknown")
                layer_id = layer_data.get("layer_id")

                # Construct FlatGeobuf URL with API key for authentication
                fgb_url = f"{self.workspace.base_url}/static/projects/{project_id}/vector/{layer_id}/{layer_id}.fgb?API_KEY={self.api_key}"

                # Create layer name for QGIS
                qgis_layer_name = f"{project_title}_{layer_name}"

                # Create vector layer and add to map
                vector_layer = QgsVectorLayer(
                    f"/vsicurl/{fgb_url}", qgis_layer_name, "ogr"
                )
                if vector_layer.isValid():
                    QgsProject().instance().addMapLayer(vector_layer)
                    layers_added += 1
                else:
                    self.iface.messageBar().pushMessage(
                        "Warning",
                        f"Failed to load layer: {layer_name}",
                        level=Qgis.Warning,
                        duration=5,
                    )

        # Zoom to last added layer
        if layers_added > 0:
            self.iface.zoomToActiveLayer()
            self.iface.mapCanvas().refresh()

        # Update status
        if layers_added > 0:
            self.set_status(
                f"Added {layers_added} vector layer{'s' if layers_added > 1 else ''} to map"
            )
        else:
            self.set_status("No vector layers to add")

    # ========== CREATE TAB METHODS ==========

    # ----- Helper Methods -----

    def populate_create_projects(self):
        """Populate the create tab projects combobox with ALL available projects (not just those with rasters)."""
        if not self.workspace:
            return

        try:
            # Fetch ALL projects (has_raster=False gets all projects)
            all_projects = self.workspace.get_projects(has_raster=False)
            self.dlg.createProjectsComboBox.clear()

            # Sort and populate
            sorted_projects = sorted(all_projects.collection, key=lambda p: p.title.lower())
            for project in sorted_projects:
                self.dlg.createProjectsComboBox.addItem(project.title, project)
        except Exception as e:
            # Silently fail - this is not critical
            pass

    def populate_create_flights(self, project):
        """Populate the create tab flights combobox for a given project."""
        self.dlg.createFlightsComboBox.clear()
        if project:
            flights = project.get_flights()
            for flight in flights.collection:
                display_name = flight.name if flight.name else f"Flight {flight.acquisition_date}"
                self.dlg.createFlightsComboBox.addItem(display_name, flight)

    def populate_polygon_layers(self):
        """Populate combobox with polygon vector layers from map canvas."""
        self.dlg.projectBoundaryLayerComboBox.clear()
        layers = QgsProject.instance().mapLayers().values()
        for layer in layers:
            if isinstance(layer, QgsVectorLayer) and layer.geometryType() == QgsWkbTypes.PolygonGeometry:
                self.dlg.projectBoundaryLayerComboBox.addItem(layer.name(), layer)

    def populate_vector_layers(self):
        """Populate combobox with vector layers from map canvas."""
        self.dlg.vectorLayerComboBox.clear()
        layers = QgsProject.instance().mapLayers().values()
        for layer in layers:
            if isinstance(layer, QgsVectorLayer):
                self.dlg.vectorLayerComboBox.addItem(layer.name(), layer)

    def populate_raster_layers(self):
        """Populate combobox with raster layers from map canvas."""
        self.dlg.rasterLayerComboBox.clear()
        layers = QgsProject.instance().mapLayers().values()
        for layer in layers:
            if isinstance(layer, QgsRasterLayer):
                self.dlg.rasterLayerComboBox.addItem(layer.name(), layer)

    def vector_layer_to_geojson_feature(self, layer):
        """Convert first feature of a QGIS vector layer to GeoJSON Feature (for project boundary).

        Args:
            layer: QgsVectorLayer to convert

        Returns:
            dict: GeoJSON Feature dict or None on error
        """
        try:
            # Get first feature
            features = list(layer.getFeatures())
            if not features:
                return None

            feature = features[0]

            # Export to GeoJSON
            exporter = QgsJsonExporter(layer)
            geojson_str = exporter.exportFeature(feature)
            geojson_dict = json.loads(geojson_str)

            # D2S API expects Polygon geometry only, not MultiPolygon
            # Convert MultiPolygon to Polygon by taking the first polygon
            if geojson_dict.get("geometry", {}).get("type") == "MultiPolygon":
                coordinates = geojson_dict["geometry"]["coordinates"]
                if coordinates and len(coordinates) > 0:
                    # Take the first polygon from the MultiPolygon
                    geojson_dict["geometry"]["type"] = "Polygon"
                    geojson_dict["geometry"]["coordinates"] = coordinates[0]

                    self.iface.messageBar().pushMessage(
                        "Info",
                        "MultiPolygon detected. Using first polygon as project boundary.",
                        level=Qgis.Info,
                        duration=5,
                    )

            return geojson_dict
        except Exception as e:
            self.iface.messageBar().pushMessage(
                "Error",
                f"Failed to convert layer to GeoJSON: {str(e)}",
                level=Qgis.Critical,
                duration=10,
            )
            return None

    def vector_layer_to_geojson_feature_collection(self, layer):
        """Convert entire QGIS vector layer to GeoJSON FeatureCollection (for map layers).

        Args:
            layer: QgsVectorLayer to convert

        Returns:
            dict: GeoJSON FeatureCollection dict or None on error
        """
        try:
            # Export entire layer to GeoJSON
            exporter = QgsJsonExporter(layer)
            geojson_str = exporter.exportFeatures(layer.getFeatures())
            geojson_dict = json.loads(geojson_str)

            return geojson_dict
        except Exception as e:
            self.iface.messageBar().pushMessage(
                "Error",
                f"Failed to convert layer to GeoJSON: {str(e)}",
                level=Qgis.Critical,
                duration=10,
            )
            return None

    def vector_file_to_geojson(self, filepath, as_feature_collection=True):
        """Load vector file and convert to GeoJSON.

        Args:
            filepath: Path to vector file
            as_feature_collection: If True, return FeatureCollection; if False, return first Feature

        Returns:
            dict: GeoJSON dict or None on error
        """
        try:
            # Load vector file as temporary layer
            layer = QgsVectorLayer(filepath, "temp", "ogr")
            if not layer.isValid():
                self.iface.messageBar().pushMessage(
                    "Error",
                    f"Failed to load vector file: {filepath}",
                    level=Qgis.Critical,
                    duration=10,
                )
                return None

            if as_feature_collection:
                return self.vector_layer_to_geojson_feature_collection(layer)
            else:
                return self.vector_layer_to_geojson_feature(layer)
        except Exception as e:
            self.iface.messageBar().pushMessage(
                "Error",
                f"Failed to process vector file: {str(e)}",
                level=Qgis.Critical,
                duration=10,
            )
            return None

    # ----- UI Toggle Methods -----

    def on_data_type_changed(self, text):
        """Enable/disable custom data type input based on combobox selection."""
        self.dlg.dataTypeCustomLineEdit.setEnabled(text == "other")
        if text != "other":
            # Clear the custom input when switching away from "other"
            self.dlg.dataTypeCustomLineEdit.clear()

    def on_platform_changed(self, text):
        """Enable/disable custom platform input based on combobox selection."""
        self.dlg.platformCustomLineEdit.setEnabled(text == "Other")
        if text != "Other":
            # Clear the custom input when switching away from "Other"
            self.dlg.platformCustomLineEdit.clear()

    def toggle_project_boundary_source(self):
        """Toggle between file and canvas source for project boundary."""
        if self.dlg.projectBoundaryFileRadio.isChecked():
            self.dlg.projectBoundaryFileLineEdit.setEnabled(True)
            self.dlg.projectBoundaryFileBrowseButton.setEnabled(True)
            self.dlg.projectBoundaryLayerComboBox.setEnabled(False)
        else:
            self.dlg.projectBoundaryFileLineEdit.setEnabled(False)
            self.dlg.projectBoundaryFileBrowseButton.setEnabled(False)
            self.dlg.projectBoundaryLayerComboBox.setEnabled(True)
            self.populate_polygon_layers()

    def toggle_raster_source(self):
        """Toggle between file and canvas source for raster."""
        # Reset progress bar when source changes
        self.dlg.rasterUploadProgressBar.setValue(0)

        if self.dlg.rasterFileRadio.isChecked():
            self.dlg.rasterFileLineEdit.setEnabled(True)
            self.dlg.rasterFileBrowseButton.setEnabled(True)
            self.dlg.rasterLayerComboBox.setEnabled(False)
        else:
            self.dlg.rasterFileLineEdit.setEnabled(False)
            self.dlg.rasterFileBrowseButton.setEnabled(False)
            self.dlg.rasterLayerComboBox.setEnabled(True)
            self.populate_raster_layers()

    def toggle_vector_source(self):
        """Toggle between file and canvas source for vector."""
        if self.dlg.vectorFileRadio.isChecked():
            self.dlg.vectorFileLineEdit.setEnabled(True)
            self.dlg.vectorFileBrowseButton.setEnabled(True)
            self.dlg.vectorLayerComboBox.setEnabled(False)
        else:
            self.dlg.vectorFileLineEdit.setEnabled(False)
            self.dlg.vectorFileBrowseButton.setEnabled(False)
            self.dlg.vectorLayerComboBox.setEnabled(True)
            self.populate_vector_layers()

    # ----- File Browser Methods -----

    def browse_project_boundary_file(self):
        """Open file dialog to select project boundary file."""
        filepath, _ = QFileDialog.getOpenFileName(
            self.dlg,
            "Select Project Boundary File",
            "",
            "Vector Files (*.shp *.geojson *.gpkg *.kml *.zip);;All Files (*)",
        )
        if filepath:
            self.dlg.projectBoundaryFileLineEdit.setText(filepath)

    def browse_raster_file(self):
        """Open file dialog to select raster file."""
        filepath, _ = QFileDialog.getOpenFileName(
            self.dlg,
            "Select Raster File",
            "",
            "Raster Files (*.tif *.tiff *.las *.laz);;All Files (*)",
        )
        if filepath:
            self.dlg.rasterFileLineEdit.setText(filepath)
            # Reset progress bar when new file is selected
            self.dlg.rasterUploadProgressBar.setValue(0)

    def browse_vector_file(self):
        """Open file dialog to select vector file."""
        filepath, _ = QFileDialog.getOpenFileName(
            self.dlg,
            "Select Vector File",
            "",
            "Vector Files (*.shp *.geojson *.gpkg *.kml);;All Files (*)",
        )
        if filepath:
            self.dlg.vectorFileLineEdit.setText(filepath)

    # ----- Project Methods -----

    def on_select_project(self):
        """Handle project selection from dropdown."""
        if not self.workspace:
            self.iface.messageBar().pushMessage(
                "Warning",
                "Please log in first",
                level=Qgis.Warning,
                duration=5,
            )
            return

        index = self.dlg.createProjectsComboBox.currentIndex()
        if index < 0:
            return

        self.selected_project = self.dlg.createProjectsComboBox.itemData(index)
        self.dlg.projectStatusLabel.setText(f"Selected: {self.selected_project.title}")

        # Enable Flight and Map Layer columns
        self.dlg.flightGroupBox.setEnabled(True)
        self.dlg.mapLayerGroupBox.setEnabled(True)

        # Populate flights for selected project
        self.populate_create_flights(self.selected_project)

        # Reset flight selection and forms
        self.selected_flight = None
        self.dlg.flightStatusLabel.setText("No flight selected")
        self.dlg.dataProductGroupBox.setEnabled(False)

        # Clear Create Flight form
        self.dlg.flightNameLineEdit.clear()
        self.dlg.altitudeSpinBox.setValue(0)
        self.dlg.sideOverlapSpinBox.setValue(0)
        self.dlg.forwardOverlapSpinBox.setValue(0)
        from qgis.PyQt.QtCore import QDate
        self.dlg.acquisitionDateEdit.setDate(QDate.currentDate())

        # Clear Data Product form
        self.dlg.rasterFileLineEdit.clear()
        self.dlg.rasterUploadProgressBar.setValue(0)
        self.dlg.dataTypeComboBox.setCurrentIndex(0)

        # Clear Map Layer form
        self.dlg.vectorLayerNameLineEdit.clear()
        self.dlg.vectorFileLineEdit.clear()

        self.set_status(f"Project selected: {self.selected_project.title}")

    def on_create_project(self):
        """Handle project creation."""
        if not self.workspace:
            self.iface.messageBar().pushMessage(
                "Warning",
                "Please log in first",
                level=Qgis.Warning,
                duration=5,
            )
            return

        # Validate inputs
        title = self.dlg.projectTitleLineEdit.text().strip()
        description = self.dlg.projectDescriptionTextEdit.toPlainText().strip()

        if not title:
            self.iface.messageBar().pushMessage(
                "Warning",
                "Please enter a project title",
                level=Qgis.Warning,
                duration=5,
            )
            return

        if not description:
            self.iface.messageBar().pushMessage(
                "Warning",
                "Please enter a project description",
                level=Qgis.Warning,
                duration=5,
            )
            return

        # Get project boundary GeoJSON
        location = None
        if self.dlg.projectBoundaryFileRadio.isChecked():
            filepath = self.dlg.projectBoundaryFileLineEdit.text().strip()
            if not filepath:
                self.iface.messageBar().pushMessage(
                    "Warning",
                    "Please select a project boundary file",
                    level=Qgis.Warning,
                    duration=5,
                )
                return
            location = self.vector_file_to_geojson(filepath, as_feature_collection=False)
        else:
            index = self.dlg.projectBoundaryLayerComboBox.currentIndex()
            if index < 0:
                self.iface.messageBar().pushMessage(
                    "Warning",
                    "Please select a project boundary layer",
                    level=Qgis.Warning,
                    duration=5,
                )
                return
            layer = self.dlg.projectBoundaryLayerComboBox.itemData(index)
            location = self.vector_layer_to_geojson_feature(layer)

        if not location:
            return

        # Get optional dates
        start_date = None
        end_date = None
        if self.dlg.startDateCheckBox.isChecked():
            qt_date = self.dlg.startDateEdit.date()
            start_date = date(qt_date.year(), qt_date.month(), qt_date.day())
        if self.dlg.endDateCheckBox.isChecked():
            qt_date = self.dlg.endDateEdit.date()
            end_date = date(qt_date.year(), qt_date.month(), qt_date.day())

        # Create project
        self.set_status("Creating project...")
        try:
            new_project = self.workspace.add_project(
                title=title,
                description=description,
                location=location,
                start_date=start_date,
                end_date=end_date,
            )

            # Update projects list
            self.projects.append(new_project)
            self.selected_project = new_project

            # Update UI
            self.dlg.projectStatusLabel.setText(f"Created: {new_project.title}")
            self.populate_create_projects()

            # Set the combobox to show the newly created project
            for i in range(self.dlg.createProjectsComboBox.count()):
                if self.dlg.createProjectsComboBox.itemData(i) == new_project:
                    self.dlg.createProjectsComboBox.setCurrentIndex(i)
                    break

            # Enable Flight and Map Layer columns
            self.dlg.flightGroupBox.setEnabled(True)
            self.dlg.mapLayerGroupBox.setEnabled(True)

            # Populate flights (empty for new project)
            self.populate_create_flights(self.selected_project)

            # Reset flight selection
            self.selected_flight = None
            self.dlg.flightStatusLabel.setText("No flight selected")
            self.dlg.dataProductGroupBox.setEnabled(False)

            # Clear form
            self.dlg.projectTitleLineEdit.clear()
            self.dlg.projectDescriptionTextEdit.clear()
            self.dlg.projectBoundaryFileLineEdit.clear()

            self.iface.messageBar().pushMessage(
                "Success",
                f"Project '{title}' created successfully",
                level=Qgis.Success,
                duration=5,
            )
            self.set_status(f"Project created: {new_project.title}")

        except Exception as e:
            self.iface.messageBar().pushMessage(
                "Error",
                f"Failed to create project: {str(e)}",
                level=Qgis.Critical,
                duration=10,
            )
            self.clear_status()

    # ----- Flight Methods -----

    def on_select_flight(self):
        """Handle flight selection from dropdown."""
        if not self.selected_project:
            self.iface.messageBar().pushMessage(
                "Warning",
                "Please select or create a project first",
                level=Qgis.Warning,
                duration=5,
            )
            return

        index = self.dlg.createFlightsComboBox.currentIndex()
        if index < 0:
            return

        self.selected_flight = self.dlg.createFlightsComboBox.itemData(index)
        display_name = self.selected_flight.name if self.selected_flight.name else f"Flight {self.selected_flight.acquisition_date}"
        self.dlg.flightStatusLabel.setText(f"Selected: {display_name}")

        # Enable Data Product column
        self.dlg.dataProductGroupBox.setEnabled(True)

        # Clear Data Product form
        self.dlg.rasterFileLineEdit.clear()
        self.dlg.rasterUploadProgressBar.setValue(0)
        self.dlg.dataTypeComboBox.setCurrentIndex(0)

        self.set_status(f"Flight selected: {display_name}")

    def on_create_flight(self):
        """Handle flight creation."""
        if not self.selected_project:
            self.iface.messageBar().pushMessage(
                "Warning",
                "Please select or create a project first",
                level=Qgis.Warning,
                duration=5,
            )
            return

        # Get inputs
        name = self.dlg.flightNameLineEdit.text().strip() or None
        qt_date = self.dlg.acquisitionDateEdit.date()
        acquisition_date = date(qt_date.year(), qt_date.month(), qt_date.day())
        altitude = self.dlg.altitudeSpinBox.value()
        side_overlap = self.dlg.sideOverlapSpinBox.value()
        forward_overlap = self.dlg.forwardOverlapSpinBox.value()
        sensor = self.dlg.sensorComboBox.currentText()

        # Get platform (use custom value if "Other" is selected)
        platform_selection = self.dlg.platformComboBox.currentText()
        if platform_selection == "Other":
            platform = self.dlg.platformCustomLineEdit.text().strip()
            if not platform:
                self.iface.messageBar().pushMessage(
                    "Warning",
                    "Please enter a custom platform",
                    level=Qgis.Warning,
                    duration=5,
                )
                return
        else:
            platform = platform_selection

        # Validate
        if altitude <= 0:
            self.iface.messageBar().pushMessage(
                "Warning",
                "Please enter a valid altitude",
                level=Qgis.Warning,
                duration=5,
            )
            return

        # Create flight
        self.set_status("Creating flight...")
        try:
            new_flight = self.selected_project.add_flight(
                name=name,
                acquisition_date=acquisition_date,
                altitude=altitude,
                side_overlap=side_overlap,
                forward_overlap=forward_overlap,
                sensor=sensor,
                platform=platform,
            )

            self.selected_flight = new_flight
            display_name = new_flight.name if new_flight.name else f"Flight {new_flight.acquisition_date}"

            # Update UI
            self.dlg.flightStatusLabel.setText(f"Created: {display_name}")
            self.populate_create_flights(self.selected_project)

            # Set the combobox to show the newly created flight
            for i in range(self.dlg.createFlightsComboBox.count()):
                if self.dlg.createFlightsComboBox.itemData(i) == new_flight:
                    self.dlg.createFlightsComboBox.setCurrentIndex(i)
                    break

            # Enable Data Product column
            self.dlg.dataProductGroupBox.setEnabled(True)

            # Clear form
            self.dlg.flightNameLineEdit.clear()
            self.dlg.altitudeSpinBox.setValue(0)
            self.dlg.sideOverlapSpinBox.setValue(0)
            self.dlg.forwardOverlapSpinBox.setValue(0)

            self.iface.messageBar().pushMessage(
                "Success",
                f"Flight '{display_name}' created successfully",
                level=Qgis.Success,
                duration=5,
            )
            self.set_status(f"Flight created: {display_name}")

        except Exception as e:
            self.iface.messageBar().pushMessage(
                "Error",
                f"Failed to create flight: {str(e)}",
                level=Qgis.Critical,
                duration=10,
            )
            self.clear_status()

    # ----- Raster Upload Methods -----

    def on_upload_raster(self):
        """Handle raster upload."""
        if not self.selected_flight:
            self.iface.messageBar().pushMessage(
                "Warning",
                "Please select or create a flight first",
                level=Qgis.Warning,
                duration=5,
            )
            return

        # Get raster file path
        filepath = None
        if self.dlg.rasterFileRadio.isChecked():
            filepath = self.dlg.rasterFileLineEdit.text().strip()
            if not filepath:
                self.iface.messageBar().pushMessage(
                    "Warning",
                    "Please select a raster file",
                    level=Qgis.Warning,
                    duration=5,
                )
                return
        else:
            index = self.dlg.rasterLayerComboBox.currentIndex()
            if index < 0:
                self.iface.messageBar().pushMessage(
                    "Warning",
                    "Please select a raster layer",
                    level=Qgis.Warning,
                    duration=5,
                )
                return
            layer = self.dlg.rasterLayerComboBox.itemData(index)
            # Get file path from layer source
            filepath = layer.source()
            # Check if it's a file path (not a WMS or other remote source)
            if not os.path.isfile(filepath):
                self.iface.messageBar().pushMessage(
                    "Error",
                    "Selected layer is not a local file. Please use a file-based raster layer.",
                    level=Qgis.Critical,
                    duration=10,
                )
                return

        # Get data type
        data_type_selection = self.dlg.dataTypeComboBox.currentText().strip()
        if data_type_selection == "other":
            # Use custom data type from line edit
            data_type = self.dlg.dataTypeCustomLineEdit.text().strip()
            if not data_type:
                self.iface.messageBar().pushMessage(
                    "Warning",
                    "Please enter a custom data type",
                    level=Qgis.Warning,
                    duration=5,
                )
                return
        else:
            # Use selected data type from combobox
            data_type = data_type_selection
            if not data_type:
                self.iface.messageBar().pushMessage(
                    "Warning",
                    "Please select a data type",
                    level=Qgis.Warning,
                    duration=5,
                )
                return

        # Upload raster with progress tracking
        self.set_status(f"Uploading raster from {os.path.basename(filepath)}...")
        self.dlg.rasterUploadProgressBar.setValue(0)

        # Disable UI during upload
        self.set_ui_enabled(False)

        # Clean up existing thread if any
        if self.upload_thread is not None:
            self.upload_thread.quit()
            self.upload_thread.wait()

        # Create worker and thread
        self.upload_thread = QThread()
        self.upload_worker = DataProductUploadWorker(
            self.selected_flight, filepath, data_type
        )
        self.upload_worker.moveToThread(self.upload_thread)

        # Connect signals
        self.upload_thread.started.connect(self.upload_worker.run)
        self.upload_worker.progress.connect(self.on_upload_progress)
        self.upload_worker.finished.connect(self.on_upload_finished)
        self.upload_worker.error.connect(self.on_upload_error)
        self.upload_worker.finished.connect(self.upload_thread.quit)
        self.upload_worker.error.connect(self.upload_thread.quit)

        # Start thread
        self.upload_thread.start()

    def on_upload_progress(self, progress):
        """Handle upload progress update."""
        self.dlg.rasterUploadProgressBar.setValue(progress)

    def on_upload_finished(self):
        """Handle successful upload completion."""
        # Clear form
        self.dlg.rasterFileLineEdit.clear()

        self.iface.messageBar().pushMessage(
            "Success",
            "Raster uploaded successfully. It may take a few minutes to process on the server.",
            level=Qgis.Success,
            duration=10,
        )
        self.set_status("Raster upload complete")

        # Re-enable UI
        self.set_ui_enabled(True)

    def on_upload_error(self, error_msg):
        """Handle upload error."""
        self.dlg.rasterUploadProgressBar.setValue(0)
        self.iface.messageBar().pushMessage(
            "Error",
            f"Failed to upload raster: {error_msg}",
            level=Qgis.Critical,
            duration=10,
        )
        self.clear_status()

        # Re-enable UI
        self.set_ui_enabled(True)

    # ----- Vector Upload Methods -----

    def on_upload_vector(self):
        """Handle vector layer upload."""
        if not self.selected_project:
            self.iface.messageBar().pushMessage(
                "Warning",
                "Please select or create a project first",
                level=Qgis.Warning,
                duration=5,
            )
            return

        # Get layer name
        layer_name = self.dlg.vectorLayerNameLineEdit.text().strip()
        if not layer_name:
            self.iface.messageBar().pushMessage(
                "Warning",
                "Please enter a layer name",
                level=Qgis.Warning,
                duration=5,
            )
            return

        # Get vector GeoJSON
        feature_collection = None
        if self.dlg.vectorFileRadio.isChecked():
            filepath = self.dlg.vectorFileLineEdit.text().strip()
            if not filepath:
                self.iface.messageBar().pushMessage(
                    "Warning",
                    "Please select a vector file",
                    level=Qgis.Warning,
                    duration=5,
                )
                return
            feature_collection = self.vector_file_to_geojson(filepath, as_feature_collection=True)
        else:
            index = self.dlg.vectorLayerComboBox.currentIndex()
            if index < 0:
                self.iface.messageBar().pushMessage(
                    "Warning",
                    "Please select a vector layer",
                    level=Qgis.Warning,
                    duration=5,
                )
                return
            layer = self.dlg.vectorLayerComboBox.itemData(index)
            feature_collection = self.vector_layer_to_geojson_feature_collection(layer)

        if not feature_collection:
            return

        # Upload vector layer
        self.set_status(f"Uploading vector layer '{layer_name}'...")
        try:
            self.selected_project.add_map_layer(
                layer_name=layer_name,
                feature_collection=feature_collection,
            )

            # Clear form
            self.dlg.vectorLayerNameLineEdit.clear()
            self.dlg.vectorFileLineEdit.clear()

            self.iface.messageBar().pushMessage(
                "Success",
                f"Vector layer '{layer_name}' uploaded successfully",
                level=Qgis.Success,
                duration=5,
            )
            self.set_status(f"Vector layer '{layer_name}' uploaded")

        except Exception as e:
            self.iface.messageBar().pushMessage(
                "Error",
                f"Failed to upload vector layer: {str(e)}",
                level=Qgis.Critical,
                duration=10,
            )
            self.clear_status()
