# -*- coding: utf-8 -*-
"""
/***************************************************************************
 D2SBrowser
                                 A QGIS plugin
 This plugin allows you to browse your data on a D2S instance.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-06-10
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Geospatial Data Science Lab
        email                : jinha@purdue.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.core import Qgis, QgsProject, QgsRasterLayer

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QThread
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QListWidgetItem, QApplication

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the dialog
from .d2s_browser_dialog import D2SBrowserDialog

# Import worker classes for threaded API calls
from .d2s_browser_workers import ProjectsWorker, FlightsWorker, DataProductsWorker
import os.path
import sys

# Add bundled libraries to sys.path
plugin_dir = os.path.dirname(__file__)
libs_dir = os.path.join(plugin_dir, "libs")
if libs_dir not in sys.path:
    sys.path.insert(0, libs_dir)

# Import d2spy from bundled libraries
from d2spy.auth import Auth
from d2spy.workspace import Workspace


NON_RASTER_TYPES = frozenset({"panoramic", "point_cloud", "3dgs"})


class D2SBrowser:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value("locale/userLocale")[0:2]
        locale_path = os.path.join(
            self.plugin_dir, "i18n", "D2SBrowser_{}.qm".format(locale)
        )

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr("&D2S Browser")

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # D2S workspace for authorized user
        self.workspace = None

        # API key for authorized user
        self.api_key = ""

        # Store projects returned from API
        self.projects = []

        # Store flights returned from API
        self.flights = []

        # Store data products returned from API
        self.data_products = []

        # Store active threads to prevent garbage collection
        self.projects_thread = None
        self.flights_thread = None
        self.data_products_thread = None

        # Cache API responses to avoid redundant requests
        self.projects_cache = None
        self.flights_cache = {}  # Key: project_id, Value: list of flights
        self.data_products_cache = {}  # Key: flight_id, Value: list of data products

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate("D2SBrowser", message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None,
    ):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(self.menu, action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ":/plugins/d2s_browser/icon.png"
        self.add_action(
            icon_path,
            text=self.tr("Browse D2S data"),
            callback=self.run,
            parent=self.iface.mainWindow(),
        )

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        # Clean up any running threads
        if self.projects_thread is not None:
            self.projects_thread.quit()
            self.projects_thread.wait()
        if self.flights_thread is not None:
            self.flights_thread.quit()
            self.flights_thread.wait()
        if self.data_products_thread is not None:
            self.data_products_thread.quit()
            self.data_products_thread.wait()

        # Remove UI elements
        for action in self.actions:
            self.iface.removePluginMenu(self.tr("&D2S Browser"), action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = D2SBrowserDialog()

            # EVENTS

            # Event when login button clicked
            self.dlg.loginPushButton.clicked.connect(self.login)
            # Event when project combobox changed
            self.dlg.projectsComboBox.currentIndexChanged.connect(self.update_flights)
            # Event when flight combobox changed
            self.dlg.flightsComboBox.currentIndexChanged.connect(
                self.update_data_products
            )
            # Event when project refresh button clicked
            self.dlg.projectsRefreshPushButton.clicked.connect(self.refresh_projects)
            # Event when add to map button clicked
            self.dlg.dataProductsPushButton.clicked.connect(
                self.add_data_products_to_map
            )

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass

    def set_status(self, message):
        """Set status bar message."""
        if hasattr(self.dlg, "statusBar"):
            self.dlg.statusBar.setText(message)
            # Force the UI to update immediately
            QApplication.processEvents()

    def clear_status(self):
        """Clear status bar message."""
        if hasattr(self.dlg, "statusBar"):
            self.dlg.statusBar.setText("")
            QApplication.processEvents()

    def set_ui_enabled(self, enabled):
        """Enable or disable UI controls during API calls."""
        if hasattr(self.dlg, "projectsComboBox"):
            self.dlg.projectsComboBox.setEnabled(enabled)
        if hasattr(self.dlg, "flightsComboBox"):
            self.dlg.flightsComboBox.setEnabled(enabled)
        if hasattr(self.dlg, "dataProductsListWidget"):
            self.dlg.dataProductsListWidget.setEnabled(enabled)
        if hasattr(self.dlg, "projectsRefreshPushButton"):
            self.dlg.projectsRefreshPushButton.setEnabled(enabled)
        if hasattr(self.dlg, "dataProductsPushButton"):
            self.dlg.dataProductsPushButton.setEnabled(enabled)

    def clear_cache(self):
        """Clear all cached API responses."""
        self.projects_cache = None
        self.flights_cache = {}
        self.data_products_cache = {}

    def login(self):
        """Login to D2S instance using server, email, and password collected from UI."""
        # Clear cache from any previous session
        self.clear_cache()

        # Show status
        self.set_status("Logging in...")

        # Clear existing projects
        self.dlg.projectsComboBox.clear()

        # Get credentials from form fields
        server = self.dlg.serverLineEdit.text()
        email = self.dlg.emailLineEdit.text()
        password = self.dlg.passwordLineEdit.text()

        # Login to D2S instance using provided credentials
        auth = Auth(server)
        try:
            session = auth.login(email, password)
        except EOFError:
            # This occurs when d2spy tries to use getpass.getpass() for password re-entry
            # after authentication failure, but there's no interactive terminal in QGIS
            self.clear_status()
            self.iface.messageBar().pushMessage(
                "Error",
                "Authentication failed. Please check your email and password.",
                level=Qgis.Critical,
                duration=10,
            )
            return
        except Exception as e:
            # Catch any other authentication errors
            self.clear_status()
            self.iface.messageBar().pushMessage(
                "Error",
                f"Authentication error: {str(e)}",
                level=Qgis.Critical,
                duration=10,
            )
            return

        if not session:
            self.clear_status()
            self.iface.messageBar().pushMessage(
                "Error",
                "Unable to sign in with provided credentials",
                level=Qgis.Critical,
                duration=10,
            )
            return

        # Get user model
        user = auth.get_current_user()

        # Check for API key
        if (
            not user
            or not hasattr(user, "api_access_token")
            or user.api_access_token is None
        ):
            self.iface.messageBar().pushMessage(
                "Warning",
                "Please request an API key from the D2S profile page.",
                level=Qgis.Warning,
                duration=10,
            )
        else:
            self.api_key = user.api_access_token

        # Create a workspace
        workspace = Workspace(server, session)
        self.workspace = workspace

        # Get user projects
        self.update_projects()

    def refresh_projects(self):
        """Refresh projects by clearing cache and fetching fresh data."""
        # Clear cache
        self.clear_cache()

        # Block signals to prevent cascading updates during clear
        self.dlg.projectsComboBox.blockSignals(True)
        self.dlg.flightsComboBox.blockSignals(True)

        # Clear all UI elements immediately
        self.dlg.projectsComboBox.clear()
        self.dlg.flightsComboBox.clear()
        self.dlg.dataProductsListWidget.clear()

        # Unblock signals
        self.dlg.projectsComboBox.blockSignals(False)
        self.dlg.flightsComboBox.blockSignals(False)

        # Disable UI during refresh
        self.set_ui_enabled(False)

        # Show status
        self.set_status("Refreshing projects...")

        # Fetch fresh data (which will cascade to flights and data products)
        self.update_projects(use_cache=False)

    def update_projects(self, use_cache=True):
        """Fetch user's projects from D2S instance and update projects UI combobox.

        Args:
            use_cache (bool): If True, use cached data if available. Defaults to True.
        """
        # Check cache first
        if use_cache and self.projects_cache is not None:
            # Clear current projects before loading from cache
            self.dlg.projectsComboBox.clear()
            # Use cached data
            self.on_projects_loaded(self.projects_cache)
            return

        # Clear current projects (if not already cleared)
        if self.dlg.projectsComboBox.count() > 0:
            self.dlg.projectsComboBox.clear()

        # Show status and disable UI
        self.set_status("Loading projects...")
        self.set_ui_enabled(False)

        # Clean up existing thread if any
        if self.projects_thread is not None:
            self.projects_thread.quit()
            self.projects_thread.wait()

        # Create worker and thread
        self.projects_thread = QThread()
        self.projects_worker = ProjectsWorker(self.workspace)
        self.projects_worker.moveToThread(self.projects_thread)

        # Connect signals
        self.projects_thread.started.connect(self.projects_worker.run)
        self.projects_worker.finished.connect(self.on_projects_loaded)
        self.projects_worker.error.connect(self.on_projects_error)
        self.projects_worker.finished.connect(self.projects_thread.quit)
        self.projects_worker.error.connect(self.projects_thread.quit)

        # Start thread
        self.projects_thread.start()

    def on_projects_loaded(self, projects):
        """Handle successful projects load."""
        self.projects = projects
        # Cache the projects for future use
        self.projects_cache = projects
        if len(self.projects) > 0:
            # Sort projects by title a - z
            self.projects = sorted(
                self.projects, key=lambda project: project.title.lower()
            )
            # Block signals to prevent duplicate update_flights() calls
            self.dlg.projectsComboBox.blockSignals(True)
            # Add projects to combobox
            self.dlg.projectsComboBox.addItems(
                [project.title for project in self.projects]
            )
            # Unblock signals
            self.dlg.projectsComboBox.blockSignals(False)
            # Get user flights for first project
            self.update_flights()
        else:
            # No projects, clear current projects, flights, and data products
            self.dlg.projectsComboBox.clear()
            self.dlg.flightsComboBox.clear()
            self.dlg.dataProductsListWidget.clear()
            self.set_status("No projects found")
            # Re-enable UI
            self.set_ui_enabled(True)

    def on_projects_error(self, error_message):
        """Handle projects load error."""
        self.clear_status()
        self.iface.messageBar().pushMessage(
            "Error",
            f"Failed to load projects: {error_message}",
            level=Qgis.Critical,
            duration=10,
        )
        # Re-enable UI
        self.set_ui_enabled(True)

    def update_flights(self, use_cache=True):
        """Fetch flights from selected project and update flights UI combobox.

        Args:
            use_cache (bool): If True, use cached data if available. Defaults to True.
        """
        # Clear current flights
        self.dlg.flightsComboBox.clear()

        # Currently selected project
        selected_project = self.projects[self.dlg.projectsComboBox.currentIndex()]

        # Check cache first
        if use_cache and selected_project.id in self.flights_cache:
            # Use cached data
            self.on_flights_loaded(self.flights_cache[selected_project.id])
            return

        # Show status
        self.set_status("Loading flights...")

        # Disable UI during API call
        self.set_ui_enabled(False)

        # Clean up existing thread if any
        if self.flights_thread is not None:
            self.flights_thread.quit()
            self.flights_thread.wait()

        # Create worker and thread
        self.flights_thread = QThread()
        self.flights_worker = FlightsWorker(selected_project)
        self.flights_worker.moveToThread(self.flights_thread)

        # Connect signals
        self.flights_thread.started.connect(self.flights_worker.run)
        self.flights_worker.finished.connect(self.on_flights_loaded)
        self.flights_worker.error.connect(self.on_flights_error)
        self.flights_worker.finished.connect(self.flights_thread.quit)
        self.flights_worker.error.connect(self.flights_thread.quit)

        # Start thread
        self.flights_thread.start()

    def on_flights_loaded(self, flights):
        """Handle successful flights load."""
        self.flights = flights

        # Cache the flights for this project
        selected_project = self.projects[self.dlg.projectsComboBox.currentIndex()]
        self.flights_cache[selected_project.id] = flights

        # Sort by acquisition date
        self.flights = sorted(
            self.flights, key=lambda flight: flight.acquisition_date, reverse=True
        )

        # Add flights (if any) to combobox
        if len(self.flights) > 0:
            # Add flight acquisition dates to combobox
            flight_items = []
            for flight in self.flights:
                if flight.name:
                    if len(flight.name) > 20:
                        name = flight.name[:17] + "..."
                    else:
                        name = flight.name
                    flight_items.append(f"{flight.acquisition_date} ({name})")
                else:
                    flight_items.append(flight.acquisition_date)
            # Block signals to prevent duplicate update_data_products() calls
            self.dlg.flightsComboBox.blockSignals(True)
            self.dlg.flightsComboBox.addItems(flight_items)
            # Unblock signals
            self.dlg.flightsComboBox.blockSignals(False)
            # Update data products
            self.update_data_products()
        else:
            # No flights, clear current flights and data products
            self.dlg.flightsComboBox.clear()
            self.dlg.dataProductsListWidget.clear()
            self.set_status("No flights found")
            # Re-enable UI
            self.set_ui_enabled(True)

    def on_flights_error(self, error_message):
        """Handle flights load error."""
        self.clear_status()
        self.iface.messageBar().pushMessage(
            "Error",
            f"Failed to load flights: {error_message}",
            level=Qgis.Critical,
            duration=10,
        )
        # Re-enable UI
        self.set_ui_enabled(True)

    def update_data_products(self, use_cache=True):
        """Fetch data products from selected flight and update data products UI list.

        Args:
            use_cache (bool): If True, use cached data if available. Defaults to True.
        """
        # Clear current data products
        self.dlg.dataProductsListWidget.clear()

        # Currently selected flight
        selected_flight = self.flights[self.dlg.flightsComboBox.currentIndex()]

        # Check cache first
        if use_cache and selected_flight.id in self.data_products_cache:
            # Use cached data
            self.on_data_products_loaded(self.data_products_cache[selected_flight.id])
            return

        # Show status
        self.set_status("Loading data products...")

        # Disable UI during API call
        self.set_ui_enabled(False)

        # Clean up existing thread if any
        if self.data_products_thread is not None:
            self.data_products_thread.quit()
            self.data_products_thread.wait()

        # Create worker and thread
        self.data_products_thread = QThread()
        self.data_products_worker = DataProductsWorker(selected_flight)
        self.data_products_worker.moveToThread(self.data_products_thread)

        # Connect signals
        self.data_products_thread.started.connect(self.data_products_worker.run)
        self.data_products_worker.finished.connect(self.on_data_products_loaded)
        self.data_products_worker.error.connect(self.on_data_products_error)
        self.data_products_worker.finished.connect(self.data_products_thread.quit)
        self.data_products_worker.error.connect(self.data_products_thread.quit)

        # Start thread
        self.data_products_thread.start()

    def on_data_products_loaded(self, all_data_products):
        """Handle successful data products load."""
        # Cache the data products for this flight
        selected_flight = self.flights[self.dlg.flightsComboBox.currentIndex()]
        self.data_products_cache[selected_flight.id] = all_data_products

        # Filter out any non-raster data products (e.g., point clouds)
        self.data_products = [
            data_product
            for data_product in all_data_products
            if data_product.data_type not in NON_RASTER_TYPES
        ]
        # Sort by data type
        self.data_products = sorted(
            self.data_products, key=lambda data_product: data_product.data_type
        )

        if len(self.data_products) > 0:
            # Create list item for each data product
            for index, data_product in enumerate(self.data_products):
                # Add data product to list with unchecked checkbox
                item = QListWidgetItem(data_product.data_type)
                item.setFlags(item.flags() | Qt.ItemIsUserCheckable)
                item.setCheckState(Qt.Unchecked)

                # Add data product list item to list widget
                self.dlg.dataProductsListWidget.addItem(item)
            self.set_status("Ready")
        else:
            # No data products, clear current data products
            self.dlg.dataProductsListWidget.clear()
            self.set_status("No data products found")

        # Re-enable UI
        self.set_ui_enabled(True)

    def on_data_products_error(self, error_message):
        """Handle data products load error."""
        self.clear_status()
        self.iface.messageBar().pushMessage(
            "Error",
            f"Failed to load data products: {error_message}",
            level=Qgis.Critical,
            duration=10,
        )
        # Re-enable UI
        self.set_ui_enabled(True)

    def add_data_products_to_map(self):
        """Add data products selected in data products UI list to map."""
        # Show status
        self.set_status("Adding layers to map...")

        # Get flight for constructing data product layer names
        selected_flight = self.flights[self.dlg.flightsComboBox.currentIndex()]
        flight_name = selected_flight.name if selected_flight.name else "Flight"
        flight_date = selected_flight.acquisition_date
        flight_sensor = selected_flight.sensor
        flight_prefix = f"{flight_name}_{flight_date}_{flight_sensor}"

        # Track number of layers added
        layers_added = 0

        # Iterate over each data product and add urls for checked data products in list
        for index in range(self.dlg.dataProductsListWidget.count()):
            item = self.dlg.dataProductsListWidget.item(index)
            if item.checkState() == Qt.Checked:
                # Get url for data product
                data_type = self.data_products[index].data_type
                layer_name = f"{flight_prefix}_{data_type}"
                url = self.data_products[index].url
                # Create raster layer and add to map
                raster_layer = QgsRasterLayer(
                    f"/vsicurl/{url}?API_KEY={self.api_key}", layer_name
                )
                if raster_layer.isValid():
                    QgsProject().instance().addMapLayer(raster_layer)
                    layers_added += 1
                else:
                    self.iface.messageBar().pushMessage(
                        "Warning",
                        f"Invalid url: {url}",
                        level=Qgis.Warning,
                        duration=5,
                    )

        # Zoom to last added layer
        self.iface.zoomToActiveLayer()
        self.iface.mapCanvas().refresh()

        # Update status
        if layers_added > 0:
            self.set_status(
                f"Added {layers_added} layer{'s' if layers_added > 1 else ''} to map"
            )
        else:
            self.set_status("No layers to add")
