# -*- coding: utf-8 -*-
"""
/***************************************************************************
 CustomMapDownloader
                                 A QGIS plugin
 Download georeferenced maps from any layer
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-11-18
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Abhinav Jayswal
        email                : abhinavjayaswal10@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QSize
from qgis.PyQt.QtGui import QIcon, QImage, QPainter
from qgis.PyQt.QtWidgets import QAction, QMessageBox, QProgressDialog
from qgis.core import (
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform,
    QgsProject,
    QgsPointXY,
    QgsRectangle,
    QgsMapSettings,
    QgsMapRendererParallelJob,
    QgsRasterLayer
)

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .CustomMapDownloader_dialog import CustomMapDownloaderDialog
import os
import os.path
from osgeo import gdal, osr
import numpy as np


class CustomMapDownloader:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'CustomMapDownloader_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&MapDownloader')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('CustomMapDownloader', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/CustomMapDownloader/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Download GeoTIFF from Map'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&MapDownloader'),
                action)
            self.iface.removeToolBarIcon(action)


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = CustomMapDownloaderDialog()

        # Refresh layers every time the dialog is opened
        self.dlg.populate_layers()
        
        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Get parameters from dialog
            params = self.dlg.get_parameters()
            
            if params is None:
                QMessageBox.critical(
                    self.iface.mainWindow(),
                    "Error",
                    "Invalid input parameters. Please check latitude and longitude values."
                )
                return
            
            # Validate parameters
            if not params['layer']:
                QMessageBox.critical(
                    self.iface.mainWindow(),
                    "Error",
                    "Please select a layer."
                )
                return
            
            if not params['output_path']:
                QMessageBox.critical(
                    self.iface.mainWindow(),
                    "Error",
                    "Please specify an output file path."
                )
                return
            
            # Execute the export with progress dialog
            progress = QProgressDialog("Exporting GeoTIFF...", "Cancel", 0, 0, self.iface.mainWindow())
            progress.setWindowTitle("Custom Map Downloader")
            progress.setWindowModality(2)  # Qt.WindowModal
            progress.setMinimumDuration(0)
            progress.setValue(0)
            progress.show()
            
            try:
                self.export_geotiff(params)
                progress.close()
                
                # Load as layer if checkbox is selected
                if params['load_as_layer']:
                    layer_name = os.path.basename(params['output_path'])
                    raster_layer = QgsRasterLayer(params['output_path'], layer_name)
                    if raster_layer.isValid():
                        QgsProject.instance().addMapLayer(raster_layer)
                        QMessageBox.information(
                            self.iface.mainWindow(),
                            "Success",
                            f"GeoTIFF successfully saved and loaded as layer:\n{params['output_path']}"
                        )
                    else:
                        QMessageBox.warning(
                            self.iface.mainWindow(),
                            "Partial Success",
                            f"GeoTIFF saved but failed to load as layer:\n{params['output_path']}"
                        )
                else:
                    QMessageBox.information(
                        self.iface.mainWindow(),
                        "Success",
                        f"GeoTIFF successfully saved to:\n{params['output_path']}"
                    )
            except Exception as e:
                progress.close()
                QMessageBox.critical(
                    self.iface.mainWindow(),
                    "Error",
                    f"Failed to export GeoTIFF:\n{str(e)}"
                )
    
    def export_geotiff(self, params):
        """Export the map to a GeoTIFF file (with or without georeferencing)."""
        lat = params['lat']
        lon = params['lon']
        gsd = params['gsd']
        width = params['width']
        height = params['height']
        layer = params['layer']
        output_path = params['output_path']
        add_georeferencing = params['add_georeferencing']
        
        # 1. Coordinate transformation to EPSG:3857 (Web Mercator)
        crs_src = QgsCoordinateReferenceSystem("EPSG:4326")
        crs_dest = QgsCoordinateReferenceSystem("EPSG:3857")
        transformer = QgsCoordinateTransform(crs_src, crs_dest, QgsProject.instance())
        center_3857 = transformer.transform(QgsPointXY(lon, lat))
        
        # 2. Calculate output extent in meters
        half_width = (width * gsd) / 2
        half_height = (height * gsd) / 2
        extent = QgsRectangle(
            center_3857.x() - half_width,
            center_3857.y() - half_height,
            center_3857.x() + half_width,
            center_3857.y() + half_height
        )
        
        # 3. Set up map settings for exporting
        map_settings = QgsMapSettings()
        map_settings.setLayers([layer])
        map_settings.setExtent(extent)
        map_settings.setOutputSize(QSize(width, height))
        map_settings.setDestinationCrs(crs_dest)
        
        # 4. Render the map
        image = QImage(width, height, QImage.Format_ARGB32_Premultiplied)
        image.fill(0)
        
        render = QgsMapRendererParallelJob(map_settings)
        render.start()
        render.waitForFinished()
        
        # Get the rendered image
        rendered_image = render.renderedImage()
        
        # 5. Check if georeferencing is requested
        if not add_georeferencing:
            # Simple save without georeferencing
            rendered_image.save(output_path, "tif")
            return
        
        # 6. Convert QImage to numpy array for georeferenced export
        rendered_image = rendered_image.convertToFormat(QImage.Format_RGBA8888)
        ptr = rendered_image.bits()
        ptr.setsize(rendered_image.byteCount())
        arr = np.array(ptr).reshape(height, width, 4)
        
        # 7. Create georeferenced GeoTIFF using GDAL
        driver = gdal.GetDriverByName('GTiff')
        dataset = driver.Create(output_path, width, height, 4, gdal.GDT_Byte, 
                               options=['COMPRESS=LZW', 'TILED=YES'])
        
        if dataset is None:
            raise Exception("Failed to create GeoTIFF file")
        
        # 8. Set geotransform (defines pixel size and upper-left corner)
        # GeoTransform: [top_left_x, pixel_width, 0, top_left_y, 0, -pixel_height]
        geotransform = [
            extent.xMinimum(),  # top-left x
            gsd,                # pixel width (west-east)
            0,                  # rotation (0 for north-up)
            extent.yMaximum(),  # top-left y
            0,                  # rotation (0 for north-up)
            -gsd                # pixel height (north-south, negative because y decreases)
        ]
        dataset.SetGeoTransform(geotransform)
        
        # 9. Set projection (EPSG:3857 - Web Mercator)
        srs = osr.SpatialReference()
        srs.ImportFromEPSG(3857)
        dataset.SetProjection(srs.ExportToWkt())
        
        # 10. Write the image data (RGBA bands)
        for i in range(4):
            band = dataset.GetRasterBand(i + 1)
            band.WriteArray(arr[:, :, i])
            band.FlushCache()
        
        # 11. Close the dataset to write to disk
        dataset = None
