# -*- coding: utf-8 -*-
"""
/***************************************************************************
 CurvedLabelFixer
                                 A QGIS plugin
 Fix rendering of RTL labels on curved lines
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2026-01-28
        git sha              : $Format:%H$
        copyright            : (C) 2026 by Dror Bogin
        email                : dror.bogin@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QToolBar
from qgis.core import (
    Qgis,
    QgsExpression,
    QgsProject, 
    QgsVectorLayer, 
    QgsVectorTileLayer,
    QgsPalLayerSettings, 
    QgsVectorLayerSimpleLabeling,
    QgsVectorTileLabeling  # noqa: F401
)
from .normalize_rtl_label import normalize_rtl_label
import re

# Initialize Qt resources from file resources.py
#from .resources import *
import os.path


class CurvedLabelFixer:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        mainWindow = self.iface.mainWindow()
        self.label_toolbar = mainWindow.findChild(QToolBar, 'mLabelToolBar')
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'CurvedLabelFixer_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Curved RTL Label Fixer')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('CurvedLabelFixer', message)

    def add_action_to_label_toolbar(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add an action to the Label toolbar.
        """
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        
        if self.label_toolbar is not None:
            self.label_toolbar.addAction(action)
            self.actions.append(action)

            return action
        else:
            self.iface.messageBar().pushMessage(
                "Error", "Label toolbar not found.", level=Qgis.Critical, duration=5)
            return None


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        # Check if already registered to avoid duplicates or errors
        if "normalize_rtl_label" not in [f.name() for f in QgsExpression.Functions()]:
            QgsExpression.registerFunction(normalize_rtl_label)
        else:
            self.unregister_custom_function()
            QgsExpression.registerFunction(normalize_rtl_label)

        # Create action that will start plugin configuration
        icon_path = os.path.join(self.plugin_dir, 'label_fixer_icon.png')

        self.add_action_to_label_toolbar(
            icon_path,
            text=self.tr(u'Fix Curved RTL Labels'),
            callback=self.run,
            status_tip=self.tr(u'Fix rendering of RTL labels on curved lines'),
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Curved RTL Label Fixer'),
                action)
            self.label_toolbar.removeAction(action)

    def unregister_custom_function(self):
        """Removes the function from the QGIS Expression Engine."""
        try:
            if "normalize_rtl_label" in [f.name() for f in QgsExpression.Functions()]:
                    QgsExpression.unregisterFunction("normalize_rtl_label")
        except Exception as e:
            self.iface.messageBar().pushMessage("Error", f"Failed to unregister function: {e}",level=Qgis.Critical, duration=5)


    def reset_rtl_normalization(self):
        layers = QgsProject.instance().mapLayers().values()
        func_name = "normalize_rtl_label"
        # Regex to find the function and grab everything inside the first parentheses
        pattern = rf"{func_name}\((.*)\)"
        project_changed = False

        for layer in layers:
            # Handle Vector Layers
            if isinstance(layer, QgsVectorLayer):
                if not layer.labelsEnabled():
                    continue
                labeling = layer.labeling()
                if not labeling: 
                    continue
                
                new_labeling = labeling.clone()
                if isinstance(new_labeling, QgsVectorLayerSimpleLabeling):
                    settings = new_labeling.settings()
                    match = re.search(pattern, settings.fieldName)
                    if match:
                        settings.fieldName = match.group(1)
                        # Note: We keep isExpression=True because the inner 
                        # part might still be an expression (e.g. "Field" + 'str')
                        new_labeling.setSettings(settings)
                        layer.setLabeling(new_labeling)
                        project_changed = True

                elif new_labeling.type() == 'rule-based':
                    if self._recursive_strip_rules(new_labeling.rootRule(), pattern):
                        layer.setLabeling(new_labeling)
                        project_changed = True
                
                if project_changed: 
                    layer.triggerRepaint()

            # Handle Vector Tile Layers
            elif isinstance(layer, QgsVectorTileLayer):
                if not layer.labelsEnabled():
                    continue
                labeling = layer.labeling()
                if not labeling: 
                    continue
                new_labeling = labeling.clone()
                styles = new_labeling.styles()
                layer_changed = False
                for style in styles:
                    settings = style.labelSettings()
                    match = re.search(pattern, settings.fieldName)
                    if match:
                        settings.fieldName = match.group(1)
                        style.setLabelSettings(settings)
                        layer_changed = True
                
                if layer_changed:
                    new_labeling.setStyles(styles)
                    layer.setLabeling(new_labeling)
                    layer.triggerRepaint()
                    project_changed = True

        if project_changed:
            QgsProject.instance().setDirty(True)

    def _recursive_strip_rules(self, parent_rule, pattern):
        changed = False
        for rule in parent_rule.children():
            settings = rule.settings()
            if settings:
                match = re.search(pattern, settings.fieldName)
                if match:
                    settings.fieldName = match.group(1)
                    rule.setSettings(settings)
                    changed = True
            if self._recursive_strip_rules(rule, pattern):
                changed = True
        return changed

    def run_normalization_action(self):
        layers = QgsProject.instance().mapLayers().values()
        func_name = "normalize_rtl_label"

        for layer in layers:
            # --- Vector Layers ---
            if isinstance(layer, QgsVectorLayer):
                if not layer.labelsEnabled():
                    continue
                labeling = layer.labeling()
                if not labeling: 
                    continue
                
                new_labeling = labeling.clone()
                if isinstance(new_labeling, QgsVectorLayerSimpleLabeling):
                    settings = new_labeling.settings()
                    if self._apply_wrapper_to_settings(settings, func_name):
                        new_labeling.setSettings(settings)
                        layer.setLabeling(new_labeling)
                elif new_labeling.type() == 'rule-based':
                    if self._process_rules(new_labeling.rootRule(), func_name):
                        layer.setLabeling(new_labeling)
                        # Notify QGIS that the project has changed
                        QgsProject.instance().setDirty(True)
                
                layer.triggerRepaint()

            # --- Vector Tile Layers ---
            elif isinstance(layer, QgsVectorTileLayer):
                if not layer.labelsEnabled():
                    continue
                labeling = layer.labeling()
                if not labeling: 
                    continue

                new_labeling = labeling.clone()
                styles = new_labeling.styles()
                changed = False

                for style in styles:
                    pal_settings = style.labelSettings()
                    if self._apply_wrapper_to_settings(pal_settings, func_name):
                        style.setLabelSettings(pal_settings)
                        changed = True
                
                if changed:
                    new_labeling.setStyles(styles)
                    layer.setLabeling(new_labeling)
                    # Vector tiles often need a full notify to reload the cache
                    layer.triggerRepaint()

    def _apply_wrapper_to_settings(self, settings, func_name):
        """Wraps fieldName and ensures the IsExpression flag is True."""
        current_expr = settings.fieldName
        
        # If it's already wrapped, skip
        if func_name in current_expr:
            if settings.placement == QgsPalLayerSettings.Curved:
                return False
            else:
                # If placement changed from Curved to something else, we should unwrap
                match = re.search(rf"{func_name}\((.*)\)", current_expr)
                if match:
                    settings.fieldName = match.group(1)
                    settings.isExpression = True  # Keep as expression
                    return True
                return False
            
        # Only target Curved placement
        if settings.placement == QgsPalLayerSettings.Curved:
            # Wrap the expression
            settings.fieldName = f"{func_name}({current_expr})"
            # CRITICAL: Tell QGIS this is an expression, not a field name
            settings.isExpression = True 
            return True
            
        return False

    def _process_rules(self, parent_rule, func_name):
        """Recursive helper for rule-based labeling. Returns True if any change made."""
        any_changed = False
        for rule in parent_rule.children():
            settings = rule.settings()
            if settings and self._apply_wrapper_to_settings(settings, func_name):
                rule.setSettings(settings)
                any_changed = True
            # Check children
            if self._process_rules(rule, func_name):
                any_changed = True
        return any_changed

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start is True:
            self.first_start = False

        self.run_normalization_action()
        
        # Optional: feedback to the user
        self.iface.messageBar().pushMessage(
            "Success", "RTL Curved labels normalized and project marked as changed.", 
            level=0, duration=3
        )
        
