# -*- coding: utf-8 -*-

"""
/***************************************************************************
 CurveNumberGenerator
                                 A QGIS plugin
 This plugin generates a Curve Number layer for the given Area of Interest within the contiguous United States. It can also download Soil, Land Cover, and Impervious Surface datasets for the same area.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-07-22
        copyright            : (C) 2022 by Abdul Raheem Siddiqui
        email                : mailto:ar-siddiqui@outlook.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""


import inspect
import os
import sys

import processing
from qgis.core import (
    QgsProcessing,
    QgsProcessingMultiStepFeedback,
    QgsProcessingParameterField,
    QgsProcessingParameterRasterLayer,
    QgsProcessingParameterVectorDestination,
    QgsProcessingParameterVectorLayer,
)
from qgis.PyQt.QtGui import QIcon

from curve_number_generator.processing.config import PLUGIN_VERSION
from curve_number_generator.processing.curve_number_generator_algorithm import (
    CurveNumberGeneratorAlgorithm,
)
from curve_number_generator.processing.tools.curve_numper import CurveNumber
from curve_number_generator.processing.tools.utils import (
    clip,
    fixGeometries,
    gdalPolygonize,
    getAndUpdateMessage,
    getExtentWKTIn3857,
)

cmd_folder = os.path.split(inspect.getfile(inspect.currentframe()))[0]
sys.path.append(cmd_folder)

__author__ = "Abdul Raheem Siddiqui"
__date__ = "2022-07-22"
__copyright__ = "(C) 2022 by Abdul Raheem Siddiqui"

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = "$Format:%H$"


class Custom(CurveNumberGeneratorAlgorithm):

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = "OUTPUT"
    INPUT = "INPUT"

    def initAlgorithm(self, config=None):

        self.addParameter(
            QgsProcessingParameterVectorLayer(
                "aoi",
                "Area of Interest",
                types=[QgsProcessing.TypeVectorPolygon],
                defaultValue=None,
            )
        )
        self.addParameter(QgsProcessingParameterRasterLayer("LandCover", "Land Cover Raster", defaultValue=None))
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                "Soils",
                "Soils Layer",
                types=[QgsProcessing.TypeVectorPolygon],
                defaultValue=None,
            )
        )
        self.addParameter(
            QgsProcessingParameterField(
                "SoilLookupField",
                "Soil Lookup Field",
                parentLayerParameterName="Soils",
                allowMultiple=False,
                defaultValue=None,
            )
        )
        self.addParameter(
            QgsProcessingParameterVectorLayer(
                "CnLookup",
                "Lookup Table",
                types=[QgsProcessing.TypeVector],
                defaultValue="",
            )
        )
        self.addParameter(QgsProcessingParameterVectorDestination("CurveNumber", "Curve Number", defaultValue=None))

    def processAlgorithm(self, parameters, context, model_feedback):
        # Use a multi-step feedback, so that individual child algorithm progress reports are adjusted for the
        # overall progress through the model
        feedback = QgsProcessingMultiStepFeedback(8, model_feedback)
        results = {}
        outputs = {}

        aoi_layer = self.parameterAsVectorLayer(parameters, "aoi", context)
        self.aoi_wkt_3857 = getExtentWKTIn3857(aoi_layer)

        # Prepare Land Cover for Curve Number Calculation
        # Polygonize (raster to vector)
        outputs["LandCoverPolygonize"] = gdalPolygonize(
            parameters["LandCover"], "land_cover", context=context, feedback=feedback
        )

        step = 1
        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}

        # Fix geometries
        outputs["LandCoverVector"] = fixGeometries(outputs["LandCoverPolygonize"], context=context, feedback=feedback)

        step += 1
        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}

        # Clip to AOI
        outputs["LandCoverClipped"] = clip(
            outputs["LandCoverVector"],
            parameters["aoi"],
            context=context,
            feedback=feedback,
        )

        step += 1
        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}

        curve_number = CurveNumber(
            outputs["LandCoverClipped"],
            parameters["Soils"],
            parameters["CnLookup"],
            context=context,
            feedback=feedback,
        )

        try:
            parameters["CurveNumber"].destinationName = "Curve Number"
        except AttributeError:
            pass

        results["CurveNumber"], step = curve_number.generateCurveNumber(
            [f"{parameters['SoilLookupField']}"],
            [],
            f'''"land_cover" || \'_\' || "{parameters['SoilLookupField']}"''',
            start_step=step + 1,
            output=parameters["CurveNumber"],
        )

        step += 1
        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}

        cn_style_path = os.path.join(os.path.dirname(cmd_folder), "curve_number.qml")
        self.handle_post_processing(results["CurveNumber"], cn_style_path, context)

        return results

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "custom"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("Curve Number Generator (Custom)")

    def icon(self):
        icon = QIcon(os.path.join(cmd_folder, "icon.png"))
        return icon

    def shortHelpString(self):
        try:
            msg = getAndUpdateMessage()
        except Exception as e:
            print(e)

        return (
            msg
            + f"""<html><body>
<h2>Algorithm description</h2>
<p>This algorithm generates Curve Number layer for the given Area of Interest given a Land Cover Raster, Soil Layer, and a Lookup Table.</p>
<h2>Input parameters</h2>
<h3>Area of Interest</h3>
<p>Polygon layer representing area of interest.</p>
<h3>Land Cover Raster</h3>
<p>Raster representing land cover values of the AOI</p>
<h3>Soils Layer</h3>
<p>Vector layer representing soils</p>
<h3>Soil Lookup Field</h3>
<p>Field in the Soils Layer that describe soil properties and should be used in relating Curve Number to soils</p>
<h3>Lookup Table</h3>
<p>Table to relate Land Cover Value and Soils Lookup Field value to a particular curve number. The table must have two columns 'grid_code' and 'cn'. grid_code is concatenation of land cover and soil lookup field. <a href="https://raw.githubusercontent.com/ar-siddiqui/curve_number_generator/v{PLUGIN_VERSION}/curve_number_generator/processing/algorithms/conus_nlcd_ssurgo/default_lookup.csv">Example table.</a></p>

<h2>Outputs</h2>
<h3>Curve Number</h3>
<p>Generated Curve Number Layer based on Land Cover and Soils.</p>
<br><p align="right">Algorithm author: Abdul Raheem Siddiqui</p><p align="right">Help author: Abdul Raheem Siddiqui</p><p align="right">Algorithm version: {PLUGIN_VERSION}</p><p align="right">Contact email: ar-siddiqui@outlook.com</p></body></html>"""
        )

    def createInstance(self):
        return Custom()
