# -*- coding: utf-8 -*-

"""
/***************************************************************************
 CurveNumberGenerator
                                 A QGIS plugin
 This plugin generates a Curve Number layer for the given Area of Interest within the contiguous United States. It can also download Soil, Land Cover, and Impervious Surface datasets for the same area.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-07-07
        copyright            : (C) 2024 by Abdul Raheem Siddiqui
        email                : ar-siddiqui@outlook.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import inspect
import os
import sys

import processing
from qgis.core import (
    QgsCoordinateReferenceSystem,
    QgsProcessing,
    QgsProcessingMultiStepFeedback,
    QgsProcessingParameterDefinition,
    QgsProcessingParameterEnum,
    QgsProcessingParameterRasterDestination,
    QgsProcessingParameterVectorDestination,
    QgsProcessingParameterVectorLayer,
)
from qgis.PyQt.QtGui import QIcon

from curve_number_generator.processing.config import GLOBAL_ESA_ORNL, PLUGIN_VERSION
from curve_number_generator.processing.curve_number_generator_algorithm import (
    CurveNumberGeneratorAlgorithm,
)
from curve_number_generator.processing.tools.curve_numper import CurveNumber
from curve_number_generator.processing.tools.utils import (
    createDefaultLookup,
    createRequestBBOXDim,
    downloadFile,
    gdalPolygonize,
    gdalWarp,
    generate_cn_exprs,
    getAndUpdateMessage,
    getExtentInEPSG4326,
    perform_raster_math,
)

cmd_folder = os.path.split(inspect.getfile(inspect.currentframe()))[0]
if cmd_folder not in sys.path:
    sys.path.insert(0, cmd_folder)

__author__ = "Abdul Raheem Siddiqui"
__date__ = "2024-03-12"
__copyright__ = "(C) 2024 by Abdul Raheem Siddiqui"

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = "$Format:%H$"


class GlobalEsaORNL(CurveNumberGeneratorAlgorithm):
    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = "OUTPUT"
    INPUT = "INPUT"

    def initAlgorithm(self, config=None):

        self.hc = ["Poor", "Fair", "Good"]
        self.arc = ["I", "II", "III"]
        self.lc_pixel_size = 0.000083333333333
        self.soils_pixel_size = 0.00208333

        self.addParameter(
            QgsProcessingParameterVectorLayer(
                "aoi",
                "Area of Interest",
                types=[QgsProcessing.TypeVectorPolygon],
                defaultValue=None,
            )
        )

        param = QgsProcessingParameterVectorLayer(
            "CnLookup",
            "Lookup Table",
            optional=True,
            types=[QgsProcessing.TypeVector],
            defaultValue="",
        )
        param.setFlags(param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(param)

        param = QgsProcessingParameterEnum(
            "HC",
            "Hydrologic Condition [Ignored when custom lookup table is provided]",
            options=self.hc,
            allowMultiple=False,
            defaultValue=1,
        )
        param.setFlags(param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(param)

        param = QgsProcessingParameterEnum(
            "ARC",
            "Antecedent Runoff Condition [Ignored when custom lookup table is provided]",
            options=self.arc,
            allowMultiple=False,
            defaultValue=1,
        )

        param.setFlags(param.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(param)

        self.addParameter(
            QgsProcessingParameterRasterDestination(
                "ESALandCover",
                "ESA Land Cover",
                optional=True,
                createByDefault=False,
                defaultValue=None,
            )
        )
        self.addParameter(
            QgsProcessingParameterRasterDestination(
                "Soils",
                "HSG",
                optional=True,
                createByDefault=False,
                defaultValue=None,
            )
        )
        self.addParameter(
            QgsProcessingParameterRasterDestination(
                "CurveNumber",
                "Curve Number",
                optional=True,
                createByDefault=True,
                defaultValue=None,
            )
        )
        self.addParameter(
            QgsProcessingParameterVectorDestination(
                "CurveNumberVector",
                "Curve Number (Vectorized)",
                optional=True,
                createByDefault=False,
                defaultValue=None,
            )
        )

    def processAlgorithm(self, parameters, context, model_feedback):
        # Use a multi-step feedback, so that individual child algorithm progress reports are adjusted for the
        # overall progress through the model
        feedback = QgsProcessingMultiStepFeedback(7, model_feedback)
        results = {}
        outputs = {}

        # Assiging Default CN_Lookup Table
        if not parameters.get("CnLookup", None):
            index_hc = self.parameterAsInt(parameters, "HC", context)
            index_arc = self.parameterAsInt(parameters, "ARC", context)
            parameters["CnLookup"] = createDefaultLookup(
                os.path.join(cmd_folder, "lookups"),
                f"default_lookup_{(self.hc[index_hc][:1].lower())}_{(self.arc[index_arc].lower())}.csv",
            )

        aoi_layer = self.parameterAsVectorLayer(parameters, "aoi", context)

        extent = getExtentInEPSG4326(aoi_layer)
        # add a buffer cell on each side, refer to #49 for reasoning
        extent_esa = (
            extent[0] - 2 * self.lc_pixel_size,
            extent[1] - 2 * self.lc_pixel_size,
            extent[2] + 2 * self.lc_pixel_size,
            extent[3] + 2 * self.lc_pixel_size,
        )
        extent_ornl = (
            extent[0] - 2 * self.soils_pixel_size,
            extent[1] - 2 * self.soils_pixel_size,
            extent[2] + 2 * self.soils_pixel_size,
            extent[3] + 2 * self.soils_pixel_size,
        )
        bbox_dim_soil = createRequestBBOXDim(extent_ornl, self.soils_pixel_size)

        step = 1
        feedback.setCurrentStep(step)
        if feedback.isCanceled():
            return {}

        if any(
            [
                parameters.get("ESALandCover", None),
                parameters.get("CurveNumber", None),
                parameters.get("CurveNumberVector", None),
            ]
        ):
            # ESA Land Cover Data
            if parameters.get("ESALandCover", None):
                try:
                    parameters["ESALandCover"].destinationName = "ESA Land Cover"
                except AttributeError:
                    pass

                lc_output = parameters["ESALandCover"]
            else:
                lc_output = QgsProcessing.TEMPORARY_OUTPUT

            alg_params = {
                "DATA_TYPE": 0,
                "EXTRA": "",
                "INPUT": os.path.join(cmd_folder, "esa_worldcover_2021.vrt"),
                "NODATA": None,
                "OPTIONS": "",
                "PROJWIN": f"{extent_esa[0]},{extent_esa[2]},{extent_esa[1]},{extent_esa[3]} [EPSG:4326]",
                "OUTPUT": lc_output,
            }
            outputs["ESALandCover"] = processing.run(
                "gdal:cliprasterbyextent",
                alg_params,
                context=context,
                feedback=feedback,
                is_child_algorithm=True,
            )["OUTPUT"]

            step += 1
            feedback.setCurrentStep(step)
            if feedback.isCanceled():
                return {}

            if parameters.get("ESALandCover", None):
                lc_style_path = os.path.join(cmd_folder, "esa_land_cover.qml")
                results["ESALandCover"] = outputs["ESALandCover"]
                self.handle_post_processing(results["ESALandCover"], lc_style_path, context)

        # Soil Layer
        if any(
            [
                parameters.get("Soils", None),
                parameters.get("CurveNumber", None),
                parameters.get("CurveNumberVector", None),
            ]
        ):

            if parameters.get("Soils", None):
                try:
                    parameters["Soils"].destinationName = "HSG"
                except AttributeError:
                    pass

                soils_output = parameters["Soils"]
            else:
                soils_output = QgsProcessing.TEMPORARY_OUTPUT

            outputs["DownloadedSoils"] = downloadFile(
                GLOBAL_ESA_ORNL["ORNL_HYSOG"].format(
                    bbox_dim_soil[0],
                    bbox_dim_soil[1],
                    ",".join([str(item) for item in extent_ornl]),
                ),
                "https://webmap.ornl.gov/ogcbroker/wcs",
                "Error getting Hydorologic Soil Group data from 'https://webmap.ornl.gov/'. Most probably because either their server is down or there is a certification issue.\nThis should be temporary. Try again later.\n",
                context=context,
                feedback=feedback,
            )

            alg_params = {
                "INPUT": outputs["DownloadedSoils"],
                "TARGET_CRS": None,
                "NODATA": None,
                "COPY_SUBDATASETS": False,
                "OPTIONS": "",
                "EXTRA": "",
                "DATA_TYPE": 0,
                "OUTPUT": soils_output,
            }

            step += 1
            feedback.setCurrentStep(step)
            if feedback.isCanceled():
                return {}

            outputs["Soils"] = processing.run(
                "gdal:translate",
                alg_params,
                context=context,
                feedback=feedback,
                is_child_algorithm=True,
            )["OUTPUT"]

            step += 1
            feedback.setCurrentStep(step)
            if feedback.isCanceled():
                return {}

            if parameters.get("Soils", None):
                soils_style_path = os.path.join(os.path.dirname(cmd_folder), "hsg_raster.qml")
                results["Soils"] = outputs["Soils"]
                self.handle_post_processing(results["Soils"], soils_style_path, context)

        if any(
            [
                parameters.get("CurveNumber", None),
                parameters.get("CurveNumberVector", None),
            ]
        ):
            outputs["SoilsAligned"] = gdalWarp(
                outputs["Soils"],
                QgsCoordinateReferenceSystem("EPSG:4326"),
                context=context,
                feedback=feedback,
                extent_layer=outputs["ESALandCover"],
                target_resolution=self.lc_pixel_size,
            )
            step += 1
            feedback.setCurrentStep(step)
            if feedback.isCanceled():
                return {}

            cn_exprs = generate_cn_exprs(parameters["CnLookup"], nodata=255)
            input_dict = {
                "input_a": outputs["ESALandCover"],
                "band_a": 1,
                "input_b": outputs["SoilsAligned"],
                "band_b": 1,
            }

            if parameters.get("CurveNumber", None):
                try:
                    parameters["CurveNumber"].destinationName = "Curve Number"
                except AttributeError:
                    pass

                soils_output = parameters["CurveNumber"]
            else:
                soils_output = QgsProcessing.TEMPORARY_OUTPUT

            outputs["CurveNumber"] = perform_raster_math(
                cn_exprs,
                input_dict,
                context,
                feedback,
                output=soils_output,
                no_data=255,
                out_data_type=0,
                hide_no_data=True,
            )

            step += 1
            feedback.setCurrentStep(step)
            if feedback.isCanceled():
                return {}

            if parameters.get("CurveNumber", None):
                cn_style_path = os.path.join(os.path.dirname(cmd_folder), "curve_number_raster.qml")
                results["CurveNumber"] = outputs["CurveNumber"]
                self.handle_post_processing(results["CurveNumber"], cn_style_path, context)

        if parameters.get("CurveNumberVector", None):
            try:
                parameters["CurveNumberVector"].destinationName = "Curve Number"
            except AttributeError:
                pass

            # Polygonize (raster to vector)
            outputs["CurveNumberVector"] = gdalPolygonize(
                outputs["CurveNumber"],
                "cn",
                output=parameters["CurveNumberVector"],
                context=context,
                feedback=feedback,
            )

            step += 1
            feedback.setCurrentStep(step)
            if feedback.isCanceled():
                return {}

            cn_style_path = os.path.join(os.path.dirname(cmd_folder), "curve_number.qml")
            results["CurveNumberVector"] = outputs["CurveNumberVector"]
            self.handle_post_processing(results["CurveNumberVector"], cn_style_path, context)

        return results

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return "globalesaornl"

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr("Curve Number Generator (Global) (ESA & ORNL)")

    def icon(self):
        icon = QIcon(os.path.join(cmd_folder, "icon.png"))
        return icon

    def shortHelpString(self):
        msg = ""
        try:
            msg = getAndUpdateMessage()
        except Exception as e:
            print(e)

        return (
            msg
            + f"""<html><body>
<h2>Algorithm description</h2>
<p>This algorithm generates a Curve Number layer for the given Area of Interest. It can also generate Hydrologic Soil Groups (HSG) raster from ORNL Hydrologic Soil Group dataset (HYSOGs250m) and download ESA Land Cover 2021 raster for the same area.

For areas where the ORNL dataset has no data such as permanent water bodies, HSG D is assumed.</p>
<h2>Input parameters</h2>
<h3>Area of Interest</h3>
<p>Polygon layer representing an area of interest.</p>
<h3>Lookup Table [optional</h3>
<p>Optional Table to relate ESA Land Cover Value and HSG Value to a particular curve number. By default the algorithm uses pre defined tables. The table must have two columns 'grid_code' and 'cn'. grid_code is a concatenation of ESA Land Cover code and Hydrologic Soil Group. <a href="https://raw.githubusercontent.com/ar-siddiqui/curve_number_generator/v{PLUGIN_VERSION}/curve_number_generator/processing/algorithms/global_esa_ornl/lookups/default_lookup_f_ii.csv">Template csv file to create custom table</a>.</p>
<h3>Hydrologic Condition [Ignored when custom lookup table is provided]</h3>
<p> Hydrologic condition is a parameter that represents combinations of factors that affect infiltration and runof. It can be Poor, Fair, or Good. (see <a href="https://directives.sc.egov.usda.gov/17758.wba">Table 9-1</a> for further description).

If unsure, use the default fair hydrologic conidtion which is the most common case in hydrologic studies.
<h3>Antecedent Runoff Condition [Ignored when custom lookup table is provided]</h3>
<p> Antecedent Runoff Condition (ARC) is the relative wetness or dryness index for the soil. I for dry, II for average, and III for wet conditions. (see <a href="https://directives.sc.egov.usda.gov/17752.wba">Table 10-1</a> for further understanding)</p>

If unsure, use the default ARC II which is the most common case in hydrologic studies.
<h2>Outputs</h2>
<h3>ESA World Cover</h3>
<p>ESA Land Cover 2021 raster.</p>
<h3>HSG</h3>
<p>ORNL Hydrologic Soil Groups (HYSOGs250m).</p>
<h3>Curve Number</h3>
<p>Generated Curve Number layer based on Land Cover and HSG values.</p>
<h3>Curve Number (Vectorized)</h3>
<p>Vector form of the generated Curve Number layer.</p>
<br>
<p align="right">Algorithm science author: Abdullah Azzam</p><p align="right">Code author: Abdul Raheem Siddiqui</p><p align="right">Help author: Abdullah Azzam</p><p align="right">Algorithm version: {PLUGIN_VERSION}</p><p align="right">Contact email: ar-siddiqui@outlook.com</p><p>Disclaimer: The curve numbers generated with this algorithm are high level estimates and should be reviewed in detail before being used for detailed modeling or construction projects.</p></body></html>"""
        )

    def createInstance(self):
        return GlobalEsaORNL()
