# -*- coding: utf-8 -*-
"""
/***************************************************************************
 CsvToPointsPlugin
                                 A QGIS plugin
 Drag & drop a CSV to create point layer; auto-detect encoding/CRS; optional geocoding.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-08-19
        git sha              : $Format:%H$
        copyright            : (C) 2025 by TechTeam Solutions LLC
        email                : info@techteam.co.jp
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtWidgets import QAction, QDialog, QVBoxLayout

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .csv_to_points_dialog import CsvToPointsPluginDialog
from .csv_dock_widget import CsvDropDockWidget
from .csv_inspector import CsvInspector
from .field_detector import FieldDetector
from .point_layer_builder import PointLayerBuilder
from .settings_dialog import SettingsDialog
from .nominatim_geocoder import NominatimGeocoder
from .geocode_task import GeocodeTask
from .google_geocoder import GoogleGeocoder
from .mapbox_geocoder import MapboxGeocoder
from .opencage_geocoder import OpenCageGeocoder
from .here_geocoder import HereGeocoder
from .yahoojp_geocoder import YahooJapanGeocoder
from .settings_store import SettingsStore
from qgis.core import (
    QgsProject,
    QgsVectorLayer,
    QgsFields,
    QgsField,
    QgsGeometry,
    QgsPointXY,
    QgsFeature,
    QgsApplication,
)
from qgis.PyQt.QtCore import QVariant
import csv
import time
import os.path


class CsvToPointsPlugin:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale', 'en')[:2]
        if locale == 'ja':
            qm_file = 'CsvToPointsPlugin_ja.qm'
        else:
            qm_file = 'CsvToPointsPlugin_en.qm'

        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            qm_file
        )

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&CSV to Points')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None
        self.window = None  # 将来 Dock/Window 切替を想定し、今回はウインドウ運用
        self.inspector = CsvInspector()
        self.field_detector = FieldDetector()
        self.layer_builder = PointLayerBuilder()
        self._last_meta = None  # 最新解析結果を保持
        self._active_task = None
        self._pending_layer = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('CsvToPointsPlugin', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/csv_to_points/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'CSV to Points'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # Plugin メニューに「設定」を追加（ツールバーには追加しない）
        setting_icon_path = ':/plugins/csv_to_points/setting.png'
        self.add_action(
            setting_icon_path,
            text=self.tr(u'設定'),
            callback=self.on_open_settings,
            add_to_toolbar=False,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

        # ウインドウ作成 (常駐)。将来 Dock との切替に備え、ラッパを分離
        self.init_window()

    def init_window(self):
        if self.window is not None:
            return
        # コンテンツ本体は既存の CsvDropDockWidget(QWidget)
        self.drop_panel = CsvDropDockWidget()
        self.drop_panel.fileDropped.connect(self.on_file_dropped)
        self.drop_panel.build_btn.clicked.connect(self.on_build_points)
        self.drop_panel.openSettings.connect(self.on_open_settings)
        # ラッパとして QDialog を用意（将来 Dock にも差し替え可能な構成）
        dlg = QDialog(self.iface.mainWindow())
        dlg.setObjectName('CsvToPointsWindow')
        dlg.setWindowTitle(self.tr('CSV to Points'))
        layout = QVBoxLayout(dlg)
        layout.setContentsMargins(6, 6, 6, 6)
        layout.addWidget(self.drop_panel)
        dlg.setLayout(layout)
        #dlg.resize(400, 600)
        self.window = dlg

    def on_file_dropped(self, path: str):
        meta = {}
        try:
            meta = self.inspector.inspect(path)
            header = meta.get('header') or []
            fd = self.field_detector.detect(header)
            meta.update({
                'lat_field_auto': fd.get('chosen_lat'),
                'lon_field_auto': fd.get('chosen_lon'),
                'address_field_auto': fd.get('chosen_address'),
                'lat_candidates': fd.get('lat_candidates'),
                'lon_candidates': fd.get('lon_candidates'),
                'address_candidates': fd.get('address_candidates'),
            })
            # 追加: 行数カウント (ヘッダ除く) & ファイル名
            try:
                row_count = 0
                with open(path, 'r', encoding=meta.get('encoding','utf-8'), errors='replace', newline='') as fcsv:
                    reader = csv.reader(fcsv, delimiter=meta.get('delimiter',','))
                    next(reader, None)  # skip header
                    for _ in reader:
                        row_count += 1
                meta['record_count'] = row_count
            except Exception:
                pass
            meta['file_name'] = os.path.basename(path)
        except Exception as e:  # collect error only
            meta = {'error': str(e)}
        self.drop_panel.showResult(meta)
        self._last_meta = {'path': path, **meta}

    def on_build_points(self):
        if not self._last_meta:
            return
        m = self._last_meta
        header = m.get('header') or []
        lat = self.drop_panel.selected_lat_field() or m.get('lat_field_auto')
        lon = self.drop_panel.selected_lon_field() or m.get('lon_field_auto')
        addr_field = self.drop_panel.selected_address_field() or m.get('address_field_auto')
        want_geocode = self.drop_panel.selected_mode_geocode()
        ui_provider = self.drop_panel.selected_provider()
        store = SettingsStore()
        provider = ui_provider or store.get_provider()
        # ユーザがジオコーディングモードを選択している場合は
        # 緯度経度列が存在しても geocoding を実行する (従来は lat/lon 優先でスキップしていた)
        creating_from_latlon = bool(lat and lon) and not want_geocode

        if creating_from_latlon:
            try:
                layer = self.layer_builder.build_memory_layer(
                    csv_path=m['path'],
                    encoding=m.get('encoding','utf-8'),
                    delimiter=m.get('delimiter',','),
                    header=header,
                    lat_field=lat,
                    lon_field=lon,
                    layer_name=os.path.basename(m['path'])+' (pts)'
                )
                QgsProject.instance().addMapLayer(layer)
                self.iface.messageBar().pushSuccess('CSV to Points', self.tr('レイヤ追加: {count} 件 (lat/lon)').format(count=layer.featureCount()))
                # ここに来るのは want_geocode=False の場合のみ (上で creating_from_latlon 定義変更済)
                self._clear_after_build(async_in_progress=False)
            except Exception as e:
                self.iface.messageBar().pushCritical('CSV to Points', self.tr('レイヤ生成失敗: {err}').format(err=e))
            return
    # 依存インポートはモジュール先頭へ移動済み
        # Qt5 向け: 非推奨コンストラクタを避け、QVariant 型を明示
        def _mk_field(name: str, qvar_type):
            """QgsField を型付きで生成（QGIS 3.34/3.40 両対応）。
            可能ならコンストラクタで型を指定し、古い環境で失敗した場合のみ setType にフォールバック。
            """
            try:
                # QGIS 3 系ではコンストラクタで QVariant 型指定が可能
                return QgsField(name, qvar_type)
            except Exception:
                f = QgsField(name)
                # 旧 API 環境向けフォールバック（3.40 では非推奨だがここには到達しない想定）
                try:
                    f.setType(qvar_type)
                except Exception:
                    pass
                return f
        sync_threshold = store.get_sync_threshold()
        force_sync = store.get_sync_all()
        rows = []
        try:
            with open(m['path'], 'r', encoding=m.get('encoding','utf-8'), errors='replace', newline='') as fcsv:
                reader = csv.reader(fcsv, delimiter=m.get('delimiter', ','))
                next(reader, None)
                for row in reader:
                    if len(row) < len(header):
                        row = row + [''] * (len(header)-len(row))
                    rows.append(row)
        except Exception as e:
            self.iface.messageBar().pushCritical('CSV to Points', self.tr('CSV読込失敗: {err}').format(err=e))
            return
        if not addr_field or addr_field not in header:
            layer = QgsVectorLayer('Point?crs=EPSG:4326', os.path.basename(m['path'])+' (attr)', 'memory')
            pr = layer.dataProvider()
            flds = QgsFields()
            for col in header:
                flds.append(_mk_field(col, QVariant.String))
            pr.addAttributes(flds)
            layer.updateFields()
            for row in rows:
                feat = QgsFeature(layer.fields())
                feat.setAttributes(row)
                pr.addFeatures([feat])
            QgsProject.instance().addMapLayer(layer)
            self.iface.messageBar().pushWarning('CSV to Points', self.tr('住所フィールドが無いためポイント空ジオメトリ (属性のみ) を追加しました'))
            self._clear_after_build(async_in_progress=False)
            return

        do_sync = want_geocode and (force_sync or len(rows) <= sync_threshold)
        layer_name = os.path.basename(m['path']) + (' (addr-sync)' if do_sync else ' (addr)')
        layer = QgsVectorLayer('Point?crs=EPSG:4326', layer_name, 'memory')
        pr = layer.dataProvider()
        flds = QgsFields()
        for col in header:
            flds.append(_mk_field(col, QVariant.String))
        if want_geocode:
            # プロバイダ別の型付きフィールドを追加（QVariant 型指定）
            def _add(name: str, qvar_type):
                if layer.fields().indexFromName(name) < 0:
                    flds.append(_mk_field(name, qvar_type))
            if provider == 'google':
                # lat/lng は Double, partial_match は Bool
                for cname, mtype in [
                    ('status', QVariant.String),
                    ('error', QVariant.String),
                    ('location_type', QVariant.String),
                    ('formatted_address', QVariant.String),
                    ('place_id', QVariant.String),
                    ('types', QVariant.String),
                    ('postal_code', QVariant.String),
                    ('lat', QVariant.Double),
                    ('lng', QVariant.Double),
                    ('partial_match', QVariant.Bool),
                ]:
                    _add(cname, mtype)
            elif provider == 'nominatim':
                for cname, mtype in [
                    ('status', QVariant.String),
                    ('error', QVariant.String),
                    ('place_id', QVariant.String),  # 文字列に変更
                    ('display_name', QVariant.String),
                    ('lat', QVariant.Double),
                    ('lng', QVariant.Double),
                    ('place_rank', QVariant.Int),
                ]:
                    _add(cname, mtype)
            elif provider == 'mapbox':
                for cname, mtype in [
                    ('status', QVariant.String),
                    ('error', QVariant.String),
                    ('id', QVariant.String),
                    ('place_name', QVariant.String),
                    ('postcode', QVariant.String),
                    ('lat', QVariant.Double),
                    ('lng', QVariant.Double),
                    ('accuracy', QVariant.String),
                ]:
                    _add(cname, mtype)
            elif provider == 'opencage':
                for cname, mtype in [
                    ('status', QVariant.String),
                    ('error', QVariant.String),
                    ('formatted', QVariant.String),
                    ('postcode', QVariant.String),
                    ('lat', QVariant.Double),
                    ('lng', QVariant.Double),
                    ('confidence', QVariant.Int),
                ]:
                    _add(cname, mtype)
            elif provider == 'here':
                for cname, mtype in [
                    ('status', QVariant.String),
                    ('error', QVariant.String),
                    ('id', QVariant.String),
                    ('title', QVariant.String),
                    ('postalCode', QVariant.String),
                    ('lat', QVariant.Double),
                    ('lng', QVariant.Double),
                ]:
                    _add(cname, mtype)
            elif provider == 'yahoojp':
                for cname, mtype in [
                    ('status', QVariant.String),
                    ('error', QVariant.String),
                    ('Uid', QVariant.String),
                    ('Name', QVariant.String),
                    ('lat', QVariant.Double),
                    ('lng', QVariant.Double),
                    ('AddressMatchingLevel', QVariant.Int),  # 整数に変更
                ]:
                    _add(cname, mtype)
        pr.addAttributes(flds)
        layer.updateFields()
        addr_idx = header.index(addr_field)

        if do_sync and want_geocode:
            if provider == 'google':
                geocoder = GoogleGeocoder()
                rate_interval = 0.0
            elif provider == 'nominatim':
                geocoder = NominatimGeocoder()
                rate_interval = 1.0 / max(getattr(geocoder,'rate',1.0),1.0)
            elif provider == 'mapbox':
                geocoder = MapboxGeocoder()
                rate_interval = 0.0
            elif provider == 'opencage':
                geocoder = OpenCageGeocoder()
                rate_interval = 0.0
            elif provider == 'here':
                geocoder = HereGeocoder()
                rate_interval = 0.0
            elif provider == 'yahoojp':
                geocoder = YahooJapanGeocoder()
                rate_interval = 0.0
            else:
                geocoder = NominatimGeocoder()
                rate_interval = 1.0
            added = failed = 0
            google_extra = provider == 'google'
            for r in rows:
                feat = QgsFeature(layer.fields())
                status = error = precision = ''
                geom = None
                gdict = { 'status':'', 'error':'', 'location_type':'', 'formatted_address':'', 'place_id':'', 'types':'', 'postal_code':'','lat':'','lng':'','partial_match':'', }
                addr_val = (r[addr_idx] or '').strip()
                if do_sync and addr_val:
                    res = geocoder.geocode(addr_val)
                    if res.status == 'OK' and res.lat is not None and res.lon is not None:
                        geom = QgsGeometry.fromPointXY(QgsPointXY(res.lon, res.lat))
                        status = 'OK'
                        precision = res.precision or ''
                        if google_extra and isinstance(getattr(res,'raw',None), dict):
                            raw = res.raw
                            gdict['formatted_address'] = raw.get('formatted_address','')
                            gdict['place_id'] = raw.get('place_id','')
                            gdict['location_type'] = (raw.get('geometry') or {}).get('location_type','')
                            tlist = raw.get('types') or []
                            if isinstance(tlist,list):
                                gdict['types'] = '|'.join(tlist)
                            comps = raw.get('address_components') or []
                            if isinstance(comps,list):
                                type_map = {}
                                for c in comps:
                                    try:
                                        for t in c.get('types',[]):
                                            if t not in type_map:
                                                type_map[t]=c.get('long_name') or ''
                                    except Exception:
                                        pass
                                gdict['postal_code'] = type_map.get('postal_code','')
                            loc = (raw.get('geometry') or {}).get('location') or {}
                            try:
                                gdict['lat'] = float(loc.get('lat')) if loc.get('lat') is not None else None
                            except Exception:
                                gdict['lat'] = None
                            try:
                                gdict['lng'] = float(loc.get('lng')) if loc.get('lng') is not None else None
                            except Exception:
                                gdict['lng'] = None
                            gdict['partial_match'] = bool(raw.get('partial_match'))
                        added += 1
                    else:
                        status = 'FAIL'
                        error = getattr(res,'error','') or ''
                        failed += 1
                    if rate_interval>0:
                        time.sleep(rate_interval)
                attrs = list(r)
                if want_geocode:
                    if google_extra:
                        gdict['status'] = status
                        gdict['error'] = error
                        if gdict['location_type'] == '' and precision:
                            gdict['location_type'] = precision
                        ordered = [
                            gdict['status'], gdict['error'], gdict['location_type'], gdict['formatted_address'],
                            gdict['place_id'], gdict['types'], gdict['postal_code'],
                            gdict['lat'], gdict['lng'], gdict['partial_match']
                        ]
                        attrs += ordered
                    elif provider == 'nominatim':
                        if 'res' in locals() and getattr(res,'status','') == 'OK':
                            raw = getattr(res,'raw',{}) or {}
                            pid = str(raw.get('place_id')) if raw.get('place_id') is not None else None
                            try:
                                prk = int(raw.get('place_rank')) if raw.get('place_rank') is not None else None
                            except Exception:
                                prk = None
                            attrs += [status, error, pid, raw.get('display_name','') or '', float(res.lat), float(res.lon), prk]
                        else:
                            attrs += [status, error, None, None, None, None, None]
                    elif provider == 'mapbox':
                        if 'res' in locals() and getattr(res,'status','') == 'OK':
                            raw = getattr(res,'raw',{}) or {}
                            postcode = ''
                            ctx = raw.get('context') or []
                            if isinstance(ctx,list):
                                for c in ctx:
                                    try:
                                        cid = c.get('id','')
                                        if cid.startswith('postcode.'):
                                            postcode = c.get('text','') or ''
                                            break
                                    except Exception:
                                        pass
                            accuracy = (raw.get('properties') or {}).get('accuracy','')
                            attrs += [status, error, raw.get('id',''), raw.get('place_name',''), postcode, float(res.lat), float(res.lon), accuracy]
                        else:
                            attrs += [status, error, None, None, None, None, None, None]
                    elif provider == 'opencage':
                        if 'res' in locals() and getattr(res,'status','') == 'OK':
                            raw = getattr(res,'raw',{}) or {}
                            comps = raw.get('components') or {}
                            try:
                                conf = int(raw.get('confidence')) if raw.get('confidence') is not None else None
                            except Exception:
                                conf = None
                            attrs += [status, error, raw.get('formatted',''), comps.get('postcode',''), float(res.lat), float(res.lon), conf]
                        else:
                            attrs += [status, error, None, None, None, None, None]
                    elif provider == 'here':
                        if 'res' in locals() and getattr(res,'status','') == 'OK':
                            raw = getattr(res,'raw',{}) or {}
                            addr = raw.get('address') or {}
                            attrs += [status, error, raw.get('id',''), raw.get('title',''), addr.get('postalCode',''), float(res.lat), float(res.lon)]
                        else:
                            attrs += [status, error, None, None, None, None, None]
                    elif provider == 'yahoojp':
                        if 'res' in locals() and getattr(res,'status','') == 'OK':
                            raw = getattr(res,'raw',{}) or {}
                            prop = raw.get('Property') or {}
                            try:
                                lvl = int(prop.get('AddressMatchingLevel')) if prop.get('AddressMatchingLevel') is not None else None
                            except Exception:
                                lvl = None
                            attrs += [status, error, raw.get('Id',''), raw.get('Name',''), float(res.lat), float(res.lon), lvl]
                        else:
                            attrs += [status, error, None, None, None, None, None]
                feat.setAttributes(attrs)
                if geom:
                    feat.setGeometry(geom)
                pr.addFeatures([feat])
            layer.updateExtents()
            if do_sync:
                self.iface.messageBar().pushSuccess('CSV to Points', self.tr('同期ジオコーディング完了({provider}): 成功 {added} / 失敗 {failed}').format(provider=provider, added=added, failed=failed))
            QgsProject.instance().addMapLayer(layer)
            self.iface.messageBar().pushSuccess('CSV to Points', self.tr('レイヤ追加: {count} 件 (住所同期)').format(count=layer.featureCount()))
            self._clear_after_build(async_in_progress=False)
            return

        for r in rows:
            feat = QgsFeature(layer.fields())
            attrs = list(r)
            if want_geocode:
                if provider == 'google':
                    attrs += [None]*10
                elif provider == 'nominatim':
                    attrs += [None]*7
                elif provider == 'mapbox':
                    attrs += [None]*8
                elif provider == 'opencage':
                    attrs += [None]*7
                elif provider == 'here':
                    attrs += [None]*7
                elif provider == 'yahoojp':
                    attrs += [None]*7
            feat.setAttributes(attrs)
            pr.addFeatures([feat])
        layer.updateExtents()
        if want_geocode:
            if provider == 'google':
                geocoder = GoogleGeocoder()
            elif provider == 'nominatim':
                geocoder = NominatimGeocoder()
            elif provider == 'mapbox':
                geocoder = MapboxGeocoder()
            elif provider == 'opencage':
                geocoder = OpenCageGeocoder()
            elif provider == 'here':
                geocoder = HereGeocoder()
            elif provider == 'yahoojp':
                geocoder = YahooJapanGeocoder()
            else:
                geocoder = NominatimGeocoder()
            self.drop_panel.start_progress()
            task = GeocodeTask('Geocoding addresses', layer, addr_field, geocoder.geocode, lambda ok,a,f,p: self._on_geocode_finished(ok,a,f,p,layer))
            self._active_task = task
            self._pending_layer = layer
            try:
                task.progressChanged.connect(lambda val: self.drop_panel.update_progress(val))
                self.drop_panel.cancel_btn.clicked.connect(task.cancel)
                QgsApplication.taskManager().addTask(task)
            except Exception:
                QgsProject.instance().addMapLayer(layer)
                self.iface.messageBar().pushWarning('CSV to Points', self.tr('タスク開始失敗: レイヤを直接追加しました'))
                self.drop_panel.finish_progress()
        else:
            QgsProject.instance().addMapLayer(layer)
            self.iface.messageBar().pushSuccess('CSV to Points', self.tr('レイヤ追加: {count} 件 (住所のみ)').format(count=layer.featureCount()))
            self._clear_after_build(async_in_progress=False)

    def _on_geocode_finished(self, ok: bool, added: int, failed: int, processed: int, layer):
        try:
            if ok and layer and layer.id() not in QgsProject.instance().mapLayers():
                QgsProject.instance().addMapLayer(layer)
                self.iface.messageBar().pushSuccess('CSV to Points', self.tr('ジオコーディング完了: 成功 {added} / 失敗 {failed}').format(added=added, failed=failed))
            elif not ok:
                self.iface.messageBar().pushWarning('CSV to Points', self.tr('ジオコーディング中断: 成功 {added} / 失敗 {failed}').format(added=added, failed=failed))
        finally:
            self.drop_panel.finish_progress()
            self._active_task = None
            self._pending_layer = None
            # 非同期完了後にリセット
            self._clear_after_build(async_in_progress=False)

    def _clear_after_build(self, async_in_progress: bool):
        """ポイント生成後に UI / 内部状態を初期化する。非同期進行中なら何もしない。"""
        if async_in_progress:
            return
        self._last_meta = None
        try:
            self.drop_panel.reset()
        except Exception:
            pass

    def on_open_settings(self):
        """設定ダイアログを開く。保存後の再初期化が必要な処理があればここで反映。"""
        try:
            dlg = SettingsDialog(self.iface.mainWindow())
            if dlg.exec_():
                # 現状は保存＝QSettingsへ書き込み済。必要なら UI リフレッシュなど追加可能。
                self.iface.messageBar().pushInfo('CSV to Points', '設定を保存しました')
                # プロバイダ一覧を即時更新
                try:
                    self.drop_panel._populate_providers()
                    # 解析結果が残っていれば Build ボタン有効性を再評価
                    if self._last_meta:
                        meta = self._last_meta
                        has_latlon = bool(meta.get('lat_field_auto') and meta.get('lon_field_auto'))
                        has_address = bool(meta.get('address_field_auto'))
                        self.drop_panel._update_mode_enable(has_latlon, has_address)
                except Exception:
                    pass
        except Exception as e:
            self.iface.messageBar().pushCritical('CSV to Points', f'設定ダイアログエラー: {e}')


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&CSV to Points'),
                action)
            self.iface.removeToolBarIcon(action)
        if self.window:
            try:
                self.window.close()
            except Exception:
                pass
            self.window = None


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = CsvToPointsPluginDialog()
        # Phase1: ウインドウを前面に出す
        if self.window:
            self.window.show()
            self.window.raise_()
        # 将来的に設定ダイアログなどで self.dlg を使う
        # return silently
        return

        # 既存ダイアログは必要時のみ表示（今は利用しない）
        # self.dlg.show()
        # result = self.dlg.exec_()
        # if result:
        #     pass
