# -*- coding: utf-8 -*-

"""
/***************************************************************************
 CSVGeometryExportDialog
 A QGIS plugin
 Export layers as CSV with custom geometry column names
 -------------------
 begin    : 2025-07-13
 copyright: (C) 2025 by Mirjan Ali Sha
 email    : mastools.help@gmail.com
 ***************************************************************************/
 /***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import encodings
import pkgutil
import urllib.parse
from PyQt5.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QLabel, QLineEdit, QCheckBox,
    QPushButton, QTableWidget, QTableWidgetItem, QHeaderView, QFileDialog,
    QMessageBox, QGroupBox, QComboBox, QSpinBox)
from PyQt5.QtCore import Qt, QSettings
from PyQt5.QtGui import QFont
from qgis.core import (
    QgsVectorFileWriter, QgsProject, QgsWkbTypes,
    QgsCoordinateReferenceSystem
)
from qgis.gui import QgsProjectionSelectionWidget
from qgis.utils import iface

class CSVGeometryExportDialog(QDialog):
    """
    Dialog for exporting vector layers as CSV with custom geometry column names
    """
    SEPARATOR_OPTIONS = {
        'Comma (,)': ',',
        'Semicolon (;)': ';',
        'Tab': '\t',
        'Space': ' '
    }

    def __init__(self, layer, parent=None):
        super().__init__(parent)
        self.layer = layer
        self.setup_ui()
        self.populate_fields()
        self.load_settings()
        self.update_selected_features_text()

    def get_all_available_encodings(self):
        try:
            available_encodings = []
            for _, modname, ispkg in pkgutil.iter_modules(encodings.__path__):
                if not ispkg and modname != 'aliases':
                    encoding_name = modname.replace('_', '-')
                    available_encodings.append(encoding_name)
            common_encodings = [
                'utf-8', 'utf-16', 'utf-32', 'ascii', 'latin1', 'iso-8859-1',
                'cp1252', 'windows-1252', 'big5', 'gb2312', 'gbk', 'gb18030',
                'shift-jis', 'euc-jp', 'euc-kr', 'koi8-r', 'koi8-u'
            ]
            all_encodings = sorted(set(available_encodings + common_encodings))
            return [enc.upper() for enc in all_encodings]
        except Exception as e:
            print(f"Warning: Could not get all encodings: {e}")
            return [
                'UTF-8', 'UTF-16', 'UTF-32', 'ASCII', 'LATIN1', 'ISO-8859-1',
                'CP1252', 'WINDOWS-1252', 'BIG5', 'GB2312', 'SHIFT-JIS'
            ]

    def setup_ui(self):
        self.setWindowTitle("Export Layer as CSV with Geometry")
        self.setModal(True)
        self.resize(700, 700)
        main_layout = QVBoxLayout()

        # Output File group
        file_group = QGroupBox("Output File")
        file_layout = QVBoxLayout()
        file_path_layout = QHBoxLayout()
        file_path_layout.addWidget(QLabel("File name:"))
        self.file_path_edit = QLineEdit()
        self.file_path_edit.setPlaceholderText("Select output CSV file...")
        self.browse_button = QPushButton("...")
        self.browse_button.setMaximumWidth(30)
        file_path_layout.addWidget(self.file_path_edit)
        file_path_layout.addWidget(self.browse_button)
        file_layout.addLayout(file_path_layout)

        separator_layout = QHBoxLayout()
        separator_layout.addWidget(QLabel("Separator:"))
        self.separator_combo = QComboBox()
        self.separator_combo.addItems(list(self.SEPARATOR_OPTIONS.keys()))
        self.separator_combo.setCurrentText('Comma (,)')
        separator_layout.addWidget(self.separator_combo)
        separator_layout.addStretch()
        file_layout.addLayout(separator_layout)

        encoding_layout = QHBoxLayout()
        encoding_layout.addWidget(QLabel("Encoding:"))
        self.encoding_combo = QComboBox()
        self.encoding_combo.setEditable(True)
        self.encoding_combo.addItems(self.get_all_available_encodings())
        encoding_layout.addWidget(self.encoding_combo)
        encoding_layout.addStretch()
        file_layout.addLayout(encoding_layout)

        file_group.setLayout(file_layout)
        main_layout.addWidget(file_group)

        # Geometry group
        geom_group = QGroupBox("Geometry Options")
        geom_layout = QVBoxLayout()
        geom_name_layout = QHBoxLayout()
        geom_name_layout.addWidget(QLabel("Geometry column name:"))
        self.geometry_field_edit = QLineEdit("geometry")
        geom_name_layout.addWidget(self.geometry_field_edit)
        geom_layout.addLayout(geom_name_layout)
        crs_layout = QHBoxLayout()
        crs_layout.addWidget(QLabel("CRS:"))
        self.crs_selector = QgsProjectionSelectionWidget()
        self.crs_selector.setCrs(self.layer.crs())
        self.crs_selector.setShowAccuracyWarnings(True)
        crs_layout.addWidget(self.crs_selector)
        geom_layout.addLayout(crs_layout)
        geom_group.setLayout(geom_layout)
        main_layout.addWidget(geom_group)

        # Fields group
        fields_group = QGroupBox("Select Fields to Export")
        fields_layout = QVBoxLayout()
        select_buttons_layout = QHBoxLayout()
        self.select_all_button = QPushButton("Select All")
        self.deselect_all_button = QPushButton("Deselect All")
        self.reset_names_button = QPushButton("Reset Export Names")
        select_buttons_layout.addWidget(self.select_all_button)
        select_buttons_layout.addWidget(self.deselect_all_button)
        select_buttons_layout.addWidget(self.reset_names_button)
        select_buttons_layout.addStretch()
        fields_layout.addLayout(select_buttons_layout)
        self.fields_table = QTableWidget()
        self.fields_table.setColumnCount(4)
        self.fields_table.setHorizontalHeaderLabels([
            "Export", "Field Name", "Export Name", "Type"
        ])
        header = self.fields_table.horizontalHeader()
        header.setSectionResizeMode(0, QHeaderView.ResizeToContents)
        header.setSectionResizeMode(1, QHeaderView.Stretch)
        header.setSectionResizeMode(2, QHeaderView.Stretch)
        header.setSectionResizeMode(3, QHeaderView.ResizeToContents)
        fields_layout.addWidget(self.fields_table)
        fields_group.setLayout(fields_layout)
        main_layout.addWidget(fields_group)

        # Export options
        options_group = QGroupBox("Export Options")
        options_layout = QVBoxLayout()
        self.selected_only_checkbox = QCheckBox("Export only selected features")
        self.selected_only_checkbox.setChecked(True)
        options_layout.addWidget(self.selected_only_checkbox)
        self.add_geometry_checkbox = QCheckBox("Add geometry column")
        self.add_geometry_checkbox.setChecked(True)
        options_layout.addWidget(self.add_geometry_checkbox)
        self.open_layer_checkbox = QCheckBox("Open result in QGIS Layer")
        self.open_layer_checkbox.setChecked(True)
        options_layout.addWidget(self.open_layer_checkbox)
        options_group.setLayout(options_layout)
        main_layout.addWidget(options_group)

        button_layout = QHBoxLayout()
        button_layout.addStretch()
        self.export_button = QPushButton("OK")
        self.export_button.setDefault(True)
        self.cancel_button = QPushButton("Cancel")
        button_layout.addWidget(self.export_button)
        button_layout.addWidget(self.cancel_button)
        main_layout.addLayout(button_layout)

        self.setLayout(main_layout)

        # Connect signals
        self.export_button.clicked.connect(self.export_layer)
        self.cancel_button.clicked.connect(self.reject)
        self.browse_button.clicked.connect(self.browse_file)
        self.select_all_button.clicked.connect(self.select_all_fields)
        self.deselect_all_button.clicked.connect(self.deselect_all_fields)
        self.reset_names_button.clicked.connect(self.reset_export_names)
        self.add_geometry_checkbox.toggled.connect(self.toggle_geometry_options)
        self.crs_selector.crsChanged.connect(self.validate_crs_selection)
        self.layer.selectionChanged.connect(self.update_selected_features_text)

    def get_selected_separator(self):
        selected_text = self.separator_combo.currentText()
        return self.SEPARATOR_OPTIONS.get(selected_text, ',')

    def get_separator_name_for_gdal(self):
        separator = self.get_selected_separator()
        separator_map = {
            ',': 'COMMA',
            ';': 'SEMICOLON',
            '\t': 'TAB',
            ' ': 'SPACE'
        }
        return separator_map.get(separator, 'COMMA')

    def update_selected_features_text(self):
        selected_count = self.layer.selectedFeatureCount()
        if selected_count > 0:
            self.selected_only_checkbox.setText(
                f"Export only selected features ({selected_count})"
            )
            self.selected_only_checkbox.setEnabled(True)
        else:
            self.selected_only_checkbox.setText("Export only selected features")
            self.selected_only_checkbox.setEnabled(False)
            self.selected_only_checkbox.setChecked(False)

    def populate_fields(self):
        fields = self.layer.fields()
        self.fields_table.setRowCount(len(fields))
        for i, field in enumerate(fields):
            checkbox = QCheckBox()
            checkbox.setChecked(True)
            self.fields_table.setCellWidget(i, 0, checkbox)
            name_item = QTableWidgetItem(field.name())
            name_item.setFlags(name_item.flags() & ~Qt.ItemIsEditable)
            self.fields_table.setItem(i, 1, name_item)
            export_name_item = QTableWidgetItem(field.name())
            self.fields_table.setItem(i, 2, export_name_item)
            type_item = QTableWidgetItem(field.typeName())
            type_item.setFlags(type_item.flags() & ~Qt.ItemIsEditable)
            self.fields_table.setItem(i, 3, type_item)

    def select_all_fields(self):
        for i in range(self.fields_table.rowCount()):
            checkbox = self.fields_table.cellWidget(i, 0)
            checkbox.setChecked(True)

    def deselect_all_fields(self):
        for i in range(self.fields_table.rowCount()):
            checkbox = self.fields_table.cellWidget(i, 0)
            checkbox.setChecked(False)

    def reset_export_names(self):
        fields = self.layer.fields()
        for i, field in enumerate(fields):
            export_name_item = QTableWidgetItem(field.name())
            self.fields_table.setItem(i, 2, export_name_item)

    def toggle_geometry_options(self, checked):
        self.geometry_field_edit.setEnabled(checked)
        self.crs_selector.setEnabled(checked)

    def validate_crs_selection(self):
        selected_crs = self.crs_selector.crs()
        layer_crs = self.layer.crs()
        if selected_crs != layer_crs:
            if not selected_crs.isValid():
                QMessageBox.warning(
                    self,
                    "Invalid CRS",
                    "The selected CRS is not valid. Please choose a valid coordinate reference system."
                )
                self.crs_selector.setCrs(layer_crs)

    def browse_file(self):
        settings = QSettings()
        last_dir = settings.value("csv_geometry_export/last_dir", os.path.expanduser("~"))
        file_path, _ = QFileDialog.getSaveFileName(
            self,
            "Save CSV File",
            os.path.join(last_dir, f"{self.layer.name()}.csv"),
            "CSV files (*.csv);;All files (*.*)"
        )
        if file_path:
            self.file_path_edit.setText(file_path)
            settings.setValue("csv_geometry_export/last_dir", os.path.dirname(file_path))

    def get_selected_fields_with_export_names(self):
        selected = []
        for i in range(self.fields_table.rowCount()):
            checkbox = self.fields_table.cellWidget(i, 0)
            if checkbox.isChecked():
                export_name_item = self.fields_table.item(i, 2)
                export_name = export_name_item.text() if export_name_item else ""
                selected.append((i, export_name))
        return selected

    def export_layer(self):
        if not self.file_path_edit.text():
            QMessageBox.warning(self, "Warning", "Please select an output file.")
            return
        selected_fields = self.get_selected_fields_with_export_names()
        if not selected_fields and not self.add_geometry_checkbox.isChecked():
            QMessageBox.warning(self, "Warning", "Please select at least one field to export or include geometry.")
            return
        geometry_field_name = self.geometry_field_edit.text().strip()
        if self.add_geometry_checkbox.isChecked() and not geometry_field_name:
            QMessageBox.warning(self, "Warning", "Please provide a geometry column name.")
            return
        encoding = self.encoding_combo.currentText()
        if not encoding:
            QMessageBox.warning(self, "Warning", "Please select or enter an encoding.")
            return
        try:
            success, err_msg = self.perform_export(
                self.file_path_edit.text(),
                selected_fields,
                geometry_field_name if self.add_geometry_checkbox.isChecked() else None,
                encoding
            )
            if success:
                self.save_settings()
                file_location = self.file_path_edit.text()
                file_url = f"file:///{file_location.replace(os.sep, '/')}"
                from qgis.utils import iface
                iface.messageBar().pushMessage(
                    "Layer Exported",
                    f'Successfully saved vector layer to: <a href="{file_url}">{file_location}</a>',
                    level=0,
                    duration=0
                )
                self.accept()
            else:
                QMessageBox.critical(self, "Error", err_msg)
        except Exception as e:
            QMessageBox.critical(self, "Error", f"Export failed: {str(e)}")

    def perform_export(self, output_path, selected_fields, geometry_field_name=None, encoding="UTF-8"):
        import traceback
        from qgis.core import QgsVectorFileWriter, QgsProject
        try:
            layer = self.layer
            if not output_path or not layer or not layer.isValid():
                return False, "Invalid export parameters"
            separator = self.get_selected_separator()
            separator_name = self.get_separator_name_for_gdal()
            field_mapping = {}
            selected_field_indices = []
            if selected_fields:
                for field_index, export_name in selected_fields:
                    original_name = layer.fields()[field_index].name()
                    field_mapping[original_name] = export_name
                    selected_field_indices.append(field_index)
            save_options = QgsVectorFileWriter.SaveVectorOptions()
            save_options.driverName = "CSV"
            save_options.fileEncoding = encoding
            if selected_field_indices:
                save_options.attributes = selected_field_indices
            # Layer option for geometry field name
            layer_options = [
                "GEOMETRY=AS_WKT",
                f"SEPARATOR={separator_name}",
                "STRING_QUOTING=NEVER"
            ]
            wkt_name = geometry_field_name if geometry_field_name and geometry_field_name.strip() else "geometry"
            layer_options.append(f"WKT_FIELD={wkt_name}")
            save_options.layerOptions = layer_options
            # CRS
            crs = self.crs_selector.crs()
            if crs and crs != layer.crs():
                save_options.destCRS = crs
            save_options.onlySelectedFeatures = self.selected_only_checkbox.isChecked()
            if field_mapping:
                save_options.fieldNameMap = field_mapping
            error = QgsVectorFileWriter.writeAsVectorFormatV3(
                layer,
                output_path,
                QgsProject.instance().transformContext(),
                save_options
            )
            result_code = error[0] if isinstance(error, (tuple, list)) else error
            if result_code != QgsVectorFileWriter.NoError:
                return False, f"Export failed with error code: {error}\nPath: {output_path}"
            if not os.path.exists(output_path):
                return False, f"The exported CSV file was not found at: {output_path}"
            # Double-check column rename fallback (optional)
            with open(output_path, "r", encoding=encoding) as f:
                header = f.readline()
                if geometry_field_name and geometry_field_name not in header:
                    self.rename_wkt_column_in_csv(output_path, geometry_field_name, separator, encoding)
            return True, ""
        except Exception as e:
            return False, f"Export error: {str(e)}\n{traceback.format_exc()}"


    def rename_wkt_column_in_csv(self, file_path, new_column_name, separator=',', encoding="UTF-8"):
        # No limit for csv fields (dangerous only for malicious CSV, never for spatial data)
        import csv
        import sys
        try:
            # Set to CSV module maximum possible value on this platform (for all practical files!)
            max_csv_size = sys.maxsize
            csv.field_size_limit(max_csv_size)
        except Exception:
            # Fallback to a very high value
            csv.field_size_limit(2 ** 31 - 1)
        with open(file_path, "r", encoding=encoding, newline='') as infile:
            reader = csv.reader(infile, delimiter=separator)
            rows = list(reader)
        if not rows:
            raise Exception("CSV seems empty.")
        header = rows[0]
        for i, col in enumerate(header):
            if col.strip().upper() == "WKT":
                header[i] = new_column_name
                break
        rows[0] = header
        with open(file_path, "w", encoding=encoding, newline='') as outfile:
            writer = csv.writer(outfile, delimiter=separator)
            writer.writerows(rows)


    # def perform_export(self, output_path, selected_fields, geometry_field_name=None, encoding="UTF-8"):
    #     import traceback
    #     try:
    #         layer = self.layer
    #         if not output_path or not layer or not layer.isValid():
    #             return False, "Invalid export parameters"

    #         separator = self.get_selected_separator()
    #         separator_name = self.get_separator_name_for_gdal()

    #         field_mapping = {}
    #         selected_field_indices = []
    #         if selected_fields:
    #             for field_index, export_name in selected_fields:
    #                 original_name = layer.fields()[field_index].name()
    #                 field_mapping[original_name] = export_name
    #                 selected_field_indices.append(field_index)

    #         save_options = QgsVectorFileWriter.SaveVectorOptions()
    #         save_options.driverName = "CSV"
    #         save_options.fileEncoding = encoding
    #         if selected_field_indices:
    #             save_options.attributes = selected_field_indices

    #         # Layer options setup
    #         layer_options = [
    #             "GEOMETRY=AS_WKT",
    #             f"SEPARATOR={separator_name}",
    #             "STRING_QUOTING=NEVER"
    #         ]
    #         # Geometry column name for WKT (QGIS API >= 3.14)
    #         wkt_name = geometry_field_name if geometry_field_name and geometry_field_name.strip() else "geometry"
    #         if wkt_name.lower() != "wkt":
    #             layer_options.append(f"WKT_FIELD={wkt_name}")
    #         save_options.layerOptions = layer_options

    #         crs = self.crs_selector.crs()
    #         if crs and crs != layer.crs():
    #             save_options.destCRS = crs
    #         save_options.onlySelectedFeatures = self.selected_only_checkbox.isChecked()
    #         if field_mapping:
    #             save_options.fieldNameMap = field_mapping

    #         error = QgsVectorFileWriter.writeAsVectorFormatV3(
    #             layer,
    #             output_path,
    #             QgsProject.instance().transformContext(),
    #             save_options
    #         )
    #         # Handle return value
    #         if isinstance(error, (tuple, list)):
    #             result_code = error[0]
    #         else:
    #             result_code = error

    #         if result_code != QgsVectorFileWriter.NoError:
    #             return False, f"Export failed with error code: {error}\nPath: {output_path}"

    #         if not os.path.exists(output_path):
    #             return False, f"The exported CSV file was not found at: {output_path}"

    #         # Post-process geometry column name if WKT column is not renamed
    #         try:
    #             with open(output_path, "r", encoding=encoding) as f:
    #                 header = f.readline()
    #                 if geometry_field_name and geometry_field_name.strip() and geometry_field_name not in header:
    #                     self.rename_wkt_column_in_csv(output_path, geometry_field_name, separator, encoding)
    #         except Exception:
    #             pass

    #         return True, ""
    #     except Exception as e:
    #         return False, f"Export error: {str(e)}\n{traceback.format_exc()}"

    # def rename_wkt_column_in_csv(self, file_path, new_column_name, separator=',', encoding="UTF-8"):
    #     import csv
    #     with open(file_path, "r", encoding=encoding, newline='') as infile:
    #         reader = csv.reader(infile, delimiter=separator)
    #         rows = list(reader)
    #     if not rows:
    #         raise Exception("CSV seems empty.")
    #     header = rows[0]
    #     for i, col in enumerate(header):
    #         if col.strip().upper() == "WKT":
    #             header[i] = new_column_name
    #             break
    #     rows[0] = header
    #     with open(file_path, "w", encoding=encoding, newline='') as outfile:
    #         writer = csv.writer(outfile, delimiter=separator)
    #         writer.writerows(rows)

    def open_exported_layer(self, file_path):
        try:
            import os
            from qgis.core import QgsVectorLayer, QgsProject
            from qgis.utils import iface
            if not os.path.exists(file_path):
                QMessageBox.critical(
                    self,
                    "File Not Found",
                    f"The exported CSV file was not found:\n{file_path}"
                )
                return
            crs_authid = self.crs_selector.crs().authid()
            normalized_path = file_path.replace('\\', '/')
            separator = self.get_selected_separator()
            encoded_separator = urllib.parse.quote(separator)
            if self.add_geometry_checkbox.isChecked():
                geometry_field = self.geometry_field_edit.text().strip()
                uri = f"file:///{normalized_path}?type=csv&delimiter={encoded_separator}&geomType=none&wktField={geometry_field}&crs={crs_authid}&spatialIndex=no&subsetIndex=no&watchFile=no&detectTypes=yes"
            else:
                uri = f"file:///{normalized_path}?type=csv&delimiter={encoded_separator}&geomType=none&crs={crs_authid}&spatialIndex=no&subsetIndex=no&watchFile=no&detectTypes=yes"
            layer_name = os.path.splitext(os.path.basename(file_path))[0]
            csv_layer = QgsVectorLayer(uri, layer_name, "delimitedtext")
            if csv_layer.isValid():
                QgsProject.instance().addMapLayer(csv_layer)
                iface.messageBar().pushMessage(
                    "Success",
                    f"CSV layer '{layer_name}' loaded successfully with {csv_layer.featureCount()} features",
                    level=0, duration=5
                )
            else:
                error_msg = csv_layer.error().message() if csv_layer.error() else "Unknown error"
                QMessageBox.warning(
                    self,
                    "Layer Loading Error",
                    f"Failed to load CSV as layer.\n\nFile: {file_path}\n\nPossible solutions:\n"
                    "1. Check if file exists and is readable\n"
                    "2. Verify geometry column contains valid WKT\n"
                    "3. Check file encoding matches selection\n"
                    "4. Verify separator matches the exported file\n\n"
                    f"Technical details:\nURI: {uri}\nError: {error_msg}"
                )
        except Exception as e:
            QMessageBox.critical(
                self,
                "Error",
                f"Failed to open CSV as layer.\n\nError: {str(e)}\n\nFile path: {file_path}"
            )

    def load_settings(self):
        settings = QSettings()
        geometry_name = settings.value("csv_geometry_export/geometry_field_name", "geometry")
        self.geometry_field_edit.setText(geometry_name)
        encoding = settings.value("csv_geometry_export/encoding", "UTF-8")
        self.encoding_combo.setCurrentText(encoding.upper())
        separator_text = settings.value("csv_geometry_export/separator", "Comma (,)")
        if separator_text in self.SEPARATOR_OPTIONS.keys():
            self.separator_combo.setCurrentText(separator_text)
        add_geometry = settings.value("csv_geometry_export/add_geometry", True, type=bool)
        self.add_geometry_checkbox.setChecked(add_geometry)
        selected_only = settings.value("csv_geometry_export/selected_only", True, type=bool)
        self.selected_only_checkbox.setChecked(selected_only)
        open_layer = settings.value("csv_geometry_export/open_layer", True, type=bool)
        self.open_layer_checkbox.setChecked(open_layer)
        last_crs_authid = settings.value("csv_geometry_export/last_crs", self.layer.crs().authid())
        if last_crs_authid:
            crs = QgsCoordinateReferenceSystem(last_crs_authid)
            if crs.isValid():
                self.crs_selector.setCrs(crs)

    def save_settings(self):
        settings = QSettings()
        settings.setValue("csv_geometry_export/geometry_field_name", self.geometry_field_edit.text())
        settings.setValue("csv_geometry_export/encoding", self.encoding_combo.currentText())
        settings.setValue("csv_geometry_export/separator", self.separator_combo.currentText())
        settings.setValue("csv_geometry_export/add_geometry", self.add_geometry_checkbox.isChecked())
        settings.setValue("csv_geometry_export/selected_only", self.selected_only_checkbox.isChecked())
        settings.setValue("csv_geometry_export/open_layer", self.open_layer_checkbox.isChecked())
        settings.setValue("csv_geometry_export/last_crs", self.crs_selector.crs().authid())

# END OF FILE

