import os

from qgis.core import QgsGeometry
from qgis.core import QgsFeature

from ..vector import Vector
from .. import config


class Planning(Vector):
    """Base class for planning modules."""

    def __init__(self):
        """Initialize Planning."""
        self.module = 'PLANNING'
        self.config = config.CruiseToolsConfig()
        self.plugin_dir = f'{os.path.dirname(__file__)}/..'

    def lines_to_vertices(self, features, fields):
        """Create Vertex features from Line features.

        Parameters
        ----------
        features : QgsFeature list/iterator
            input line features
        fields : QgsFields
            fields to be created for the vertices

        Returns
        -------
        vertices : QgsFeature list
            output vertex features

        """
        # create empty list for vertices
        vertices = []

        # fid increment
        i = 1

        # loop over input features
        for feature in features:
            # get attributes from input feature
            attributes = feature.attributes()

            # get geometry from input feature
            geom_f = feature.geometry()

            # empty list for geometry parts
            geoms = []

            # check if geometry is multipart
            # and convert from line to vertex geometry
            if geom_f.isMultipart():
                for part in geom_f.asMultiPolyline():
                    geoms.append(part)
            else:
                geoms = [geom_f.asPolyline()]

            # iterate through geometry parts
            for geom in geoms:
                # loop over vertices
                for idx, pnt in enumerate(geom):
                    # create vertex feature
                    vertex = QgsFeature(fields)

                    # set vertex geometry
                    vertex.setGeometry(QgsGeometry.fromPointXY(pnt))

                    # loop over feature fields
                    for field in feature.fields():
                        # if field exists in input fields
                        if field.name() in fields.names():
                            # set attribute from line to vertex
                            if self.add_vertex_suffix and (
                                    field.name() == 'name'):  # add order numbering (for default field "name")
                                vertex.setAttribute(field.name(), feature.attribute(field.name()) + f'_{idx + 1:03d}')
                            else:
                                vertex.setAttribute(field.name(), feature.attribute(field.name()))

                    # set fid
                    vertex.setAttribute('fid', i)
                    i += 1

                    # append vertex to vertices list
                    vertices.append(vertex)

        return vertices
