# -*- coding: utf-8 -*-
"""
/***************************************************************************
 CreateRelation
                                 A QGIS plugin
 Crea relazioni trale tabelle degli attributi dei layer
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-05-08
        git sha              : $Format:%H$
        copyright            : (C) 2025 by dr. Geol. Faustino Cetraro
        email                : geol-faustino@libero.it
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .create_relation_dialog import CreateRelationDialog
import os.path

from qgis.core import QgsProject, QgsRelation, QgsVectorLayer, QgsVectorFileWriter
from datetime import datetime
from qgis.core import QgsVectorLayerJoinInfo
from qgis.core import QgsField
from PyQt5.QtCore import QVariant
from qgis.core import QgsFeature
from qgis.PyQt.QtWidgets import QMessageBox
from PyQt5.QtWidgets import QDialog, QFileDialog, QMessageBox

class CreateRelation:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = str(QSettings().value('locale/userLocale'))[0:2]
        locale_path = os.path.join(self.plugin_dir, 'i18n', f'create_relation_{locale}.qm')

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Create Relation')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('CreateRelation', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToDatabaseMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = os.path.join(self.plugin_dir, 'icon.png')
        self.add_action(
            icon_path,
            text=self.tr(u'Create Relation'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginDatabaseMenu(
                self.tr(u'&Create Relation'),
                action)
            self.iface.removeToolBarIcon(action)
        
    def run(self):
        """Run method that performs all the real work"""
        if self.first_start:
            self.first_start = False
            self.dlg = CreateRelationDialog()

            # Popola i comboBox con i layer vettoriali caricati
            layers = [layer for layer in self.iface.mapCanvas().layers() if isinstance(layer, QgsVectorLayer)]
            self.dlg.layer1Combo.clear()
            self.dlg.layer2Combo.clear()
            for layer in layers:
                self.dlg.layer1Combo.addItem(layer.name(), layer)
                self.dlg.layer2Combo.addItem(layer.name(), layer)

            # Connessione ai campi quando si seleziona un layer
            self.dlg.layer1Combo.currentIndexChanged.connect(self.populate_fields_layer1)
            self.dlg.layer2Combo.currentIndexChanged.connect(self.populate_fields_layer2)

            # Azione bottone "Crea relazione"
            self.dlg.createRelationButton.clicked.connect(self.create_relation)

            # Bottone di chiusura
            self.dlg.closeButton.clicked.connect(self.on_close_clicked)

            # Popola i field comboBox con il primo layer selezionato (se presente)
            self.populate_fields_layer1()
            self.populate_fields_layer2()

        self.dlg.show()

    def populate_fields_layer1(self):
        layer = self.dlg.layer1Combo.currentData()
        self.dlg.field1Combo.clear()
        if layer:
            self.dlg.field1Combo.addItems([f.name() for f in layer.fields()])

    def populate_fields_layer2(self):
        layer = self.dlg.layer2Combo.currentData()
        self.dlg.field2Combo.clear()
        if layer:
            self.dlg.field2Combo.addItems([f.name() for f in layer.fields()])

    def on_close_clicked(self):
        self.dlg.processLogTextEdit.clear()
        self.dlg.layer1Combo.setCurrentIndex(0)
        self.dlg.layer2Combo.setCurrentIndex(0)
        self.dlg.field1Combo.setCurrentIndex(0)
        self.dlg.field2Combo.setCurrentIndex(0)
        self.dlg.relationTypeCombo.setCurrentIndex(0)
        self.dlg.ponteTableNameLineEdit.clear()
        self.dlg.saveToDiskCheckBox.setChecked(False)
        self.dlg.close()
    
    def create_relation(self):
        layer1 = self.dlg.layer1Combo.currentData()
        layer2 = self.dlg.layer2Combo.currentData()
        field1 = self.dlg.field1Combo.currentText()
        field2 = self.dlg.field2Combo.currentText()
        relation_type = self.dlg.relationTypeCombo.currentText()

        ponte_table_name = self.dlg.ponteTableNameLineEdit.text()
        save_to_disk = self.dlg.saveToDiskCheckBox.isChecked()

        log = self.dlg.processLogTextEdit
        log.clear()

        if not (layer1 and layer2 and field1 and field2):
            log.append(self.tr("❌️ Errore: selezionare entrambi i layer e i campi."))
            return

        if not isinstance(layer1, QgsVectorLayer) or not isinstance(layer2, QgsVectorLayer):
            log.append(self.tr("❌ Errore: i layer selezionati devono essere di tipo vettoriale."))
            return

        relation_manager = QgsProject.instance().relationManager()
        if not relation_manager:
            log.append(self.tr("❌ Errore: il manager delle relazioni non è disponibile."))
            return

        if relation_type == self.tr("uno-a-uno"):
            for join in layer1.vectorJoins():
                if (join.joinLayerId() == layer2.id() and
                    join.joinFieldName() == field2 and
                    join.targetFieldName() == field1):
                    risposta = QMessageBox.question(
                        self.dlg,
                        self.tr("Join già esistente"),
                        self.tr("⚠️ Esiste già un join tra questi layer con gli stessi campi.\nVuoi procedere comunque?"),
                        QMessageBox.Yes | QMessageBox.No,
                        QMessageBox.No
                    )
                    if risposta == QMessageBox.No:
                        log.append(self.tr("ℹ️ Operazione annullata: il join esisteva già."))
                        return
                    else:
                        log.append(self.tr("ℹ️ Procedo nonostante il join esistente."))
                        break

            join_info = QgsVectorLayerJoinInfo()
            join_info.setJoinLayer(layer2)
            join_info.setJoinFieldName(field2)
            join_info.setTargetFieldName(field1)
            join_info.setUsingMemoryCache(True)
            layer1.addJoin(join_info)
            log.append(self.tr(f"✔️ Join creato con successo tra {layer1.name()} e {layer2.name()} usando i campi {field1} → {field2}."))
            return

        if relation_type == self.tr("molti-a-molti"):
            for rel in relation_manager.relations().values():
                if ((rel.referencingLayerId() in [layer1.id(), layer2.id()]) or
                    (rel.referencedLayerId() in [layer1.id(), layer2.id()])):
                    risposta = QMessageBox.question(
                        self.dlg,
                        self.tr("Relazione già esistente"),
                        self.tr("⚠️ Esistono già relazioni tra questi layer.\nVuoi procedere comunque?"),
                        QMessageBox.Yes | QMessageBox.No,
                        QMessageBox.No
                    )
                    if risposta == QMessageBox.No:
                        log.append(self.tr("ℹ️ Operazione annullata: relazione già esistente."))
                        return
                    else:
                        log.append(self.tr("ℹ️ Procedo nonostante la relazione esistente."))
                        break

            ponte_layer = QgsVectorLayer("None", ponte_table_name or "ponte_n_m", "memory")
            pr = ponte_layer.dataProvider()
            pr.addAttributes([
                QgsField(field1, QVariant.Int),
                QgsField(field2, QVariant.Int)
            ])
            ponte_layer.updateFields()

            features = []
            layer1_ids = set(f[field1] for f in layer1.getFeatures())
            layer2_ids = set(f[field2] for f in layer2.getFeatures())

            for val1 in layer1_ids:
                for val2 in layer2_ids:
                    feat = QgsFeature()
                    feat.setFields(ponte_layer.fields())
                    feat.setAttribute(field1, val1)
                    feat.setAttribute(field2, val2)
                    features.append(feat)

            pr.addFeatures(features)
            ponte_layer.updateExtents()
            QgsProject.instance().addMapLayer(ponte_layer)

            rel1 = QgsRelation()
            rel1.setId("rel_ponte_" + layer1.name().lower())
            rel1.setName(layer1.name())
            rel1.setReferencingLayer(ponte_layer.id())
            rel1.setReferencedLayer(layer1.id())
            rel1.addFieldPair(field1, field1)

            rel2 = QgsRelation()
            rel2.setId("rel_ponte_" + layer2.name().lower())
            rel2.setName(layer2.name())
            rel2.setReferencingLayer(ponte_layer.id())
            rel2.setReferencedLayer(layer2.id())
            rel2.addFieldPair(field2, field2)

            success1 = relation_manager.addRelation(rel1)
            success2 = relation_manager.addRelation(rel2)

            if save_to_disk:
                file_path, selected_filter = QFileDialog.getSaveFileName(
                    self.dlg, self.tr("Salva la tabella ponte"), "", "File Shapefile (*.shp);;GeoJSON (*.geojson)"
                )
                if file_path:
                    if "Shapefile" in selected_filter and not file_path.endswith(".shp"):
                        file_path += ".shp"
                        driver = "ESRI Shapefile"
                    elif "GeoJSON" in selected_filter and not file_path.endswith(".geojson"):
                        file_path += ".geojson"
                        driver = "GeoJSON"
                    else:
                        driver = "GeoJSON"

                    error = QgsVectorFileWriter.writeAsVectorFormat(
                        ponte_layer, file_path, "UTF-8", ponte_layer.crs(), driverName=driver
                    )
                    if error == QgsVectorFileWriter.NoError:
                        log.append(self.tr(f"✔️ Tabella ponte salvata su disco: {file_path}"))
                    else:
                        log.append(self.tr("❌ Errore nel salvataggio della tabella ponte su disco."))

            if success1 and success2:
                log.append(self.tr(f"✔ Relazioni molti-a-molti create correttamente tra '{layer1.name()}' e '{layer2.name()}' tramite la tabella '{ponte_layer.name()}'."))
            else:
                log.append(self.tr("✅ Le relazioni molti-a-molti sembrano funzionare, ma si è verificato un problema nell'aggiunta formale delle relazioni al progetto."))
            return

        # Default: uno-a-molti
        for rel in relation_manager.relations().values():
            if (rel.referencingLayerId() == layer2.id() and
                rel.referencedLayerId() == layer1.id() and
                rel.fieldPairs() == [(field2, field1)]):
                risposta = QMessageBox.question(
                    self.dlg,
                    self.tr("Relazione già esistente"),
                    self.tr("⚠️ Esiste già una relazione con gli stessi parametri.\nVuoi procedere comunque?"),
                    QMessageBox.Yes | QMessageBox.No,
                    QMessageBox.No
                )
                if risposta == QMessageBox.No:
                    log.append(self.tr("ℹ️ Operazione annullata: relazione già presente."))
                    return
                else:
                    log.append(self.tr("ℹ️ Procedo nonostante la relazione già presente."))
                    break

        timestamp = datetime.now().strftime("%Y%m%d%H%M%S")
        relation_id = f"{layer1.id()}_{layer2.id()}_{field1}_{field2}_{timestamp}"

        rel = QgsRelation()
        rel.setId(relation_id)
        rel.setName(layer2.name())
        rel.setReferencingLayer(layer2.id())
        rel.setReferencedLayer(layer1.id())

        if layer2.fields().indexOf(field2) == -1 or layer1.fields().indexOf(field1) == -1:
            log.append(self.tr("❌ Errore: i campi specificati non esistono nei layer selezionati."))
            return

        rel.addFieldPair(field2, field1)

        if not rel.isValid():
            log.append(self.tr("❌ Errore: la relazione creata non è valida. Controlla i layer e i campi."))
            return

        if relation_manager.addRelation(rel):
            log.append(self.tr(f"✔ Relazione {relation_type} creata con successo."))
        else:
            if relation_manager.relation(relation_id):
                log.append(self.tr("✅ Relazione creata correttamente, ma addRelation() ha restituito False."))
            else:
                log.append(self.tr("❌ Errore nella creazione della relazione."))
