import os
from qgis.core import (QgsProject, QgsVectorLayer, QgsFeature, QgsGeometry,
                       QgsPointXY, QgsFields, QgsField, QgsCoordinateReferenceSystem)
from qgis.gui import QgsProjectionSelectionWidget
from PyQt5.QtWidgets import (QAction, QDialog, QVBoxLayout, QHBoxLayout, QLabel,
                             QLineEdit, QPushButton, QTabWidget, QWidget,
                             QListWidget, QMessageBox)
from PyQt5.QtGui import QIcon
from PyQt5.QtCore import QVariant, Qt

class PolygonCreatorDialog(QDialog):
    def __init__(self, iface):
        super().__init__()
        self.iface = iface
        self.setWindowTitle("Polygon generator")
        self.resize(400, 550)

        self.layout = QVBoxLayout()
        self.setLayout(self.layout)

        self.layout.addWidget(QLabel("<b>1. Select Coordinate Reference System (CRS):</b>"))
        self.crs_selector = QgsProjectionSelectionWidget()
        self.crs_selector.setCrs(QgsCoordinateReferenceSystem("EPSG:4326"))
        self.layout.addWidget(self.crs_selector)

        self.layout.addSpacing(10)
        self.layout.addWidget(QLabel("<b>2. Select method:</b>"))
        self.tabs = QTabWidget()
        self.tab_rect = QWidget()
        self.tab_poly = QWidget()

        self.tabs.addTab(self.tab_rect, "Rectangle from min/max")
        self.tabs.addTab(self.tab_poly, "Manual vertex input")
        self.layout.addWidget(self.tabs)

        self.setup_rect_tab()
        self.setup_poly_tab()

    def setup_rect_tab(self):
        layout = QVBoxLayout()
        self.min_lon = self.create_input(layout, "Min Longitude (X Min):")
        self.max_lon = self.create_input(layout, "Max Longitude (X Max):")
        self.min_lat = self.create_input(layout, "Min Latitude (Y Min):")
        self.max_lat = self.create_input(layout, "Max Latitude (Y Max):")

        btn_create = QPushButton("Build Rectangle")
        btn_create.clicked.connect(self.create_rectangle)
        layout.addWidget(btn_create)
        layout.addStretch()
        self.tab_rect.setLayout(layout)

    def setup_poly_tab(self):
        layout = QVBoxLayout()
        self.poly_lon = self.create_input(layout, "Longitude (X):")
        self.poly_lat = self.create_input(layout, "Latitude (Y):")

        btn_add = QPushButton("Add point to list")
        btn_add.clicked.connect(self.add_point_to_list)
        layout.addWidget(btn_add)

        layout.addWidget(QLabel("Vertex list:"))
        self.points_list_widget = QListWidget()
        layout.addWidget(self.points_list_widget)

        self.custom_points = []

        hbox = QHBoxLayout()
        btn_clear = QPushButton("Clear")
        btn_clear.clicked.connect(self.clear_points)
        btn_create = QPushButton("Build Polygon")
        btn_create.clicked.connect(self.create_custom_polygon)

        hbox.addWidget(btn_clear)
        hbox.addWidget(btn_create)
        layout.addLayout(hbox)
        self.tab_poly.setLayout(layout)

    def create_input(self, layout, label_text):
        layout.addWidget(QLabel(label_text))
        le = QLineEdit()
        layout.addWidget(le)
        return le

    def get_float(self, line_edit):
        try:
            val = float(line_edit.text().replace(',', '.'))
            return val
        except ValueError:
            return None

    def create_rectangle(self):
        x_min = self.get_float(self.min_lon)
        x_max = self.get_float(self.max_lon)
        y_min = self.get_float(self.min_lat)
        y_max = self.get_float(self.max_lat)

        if None in [x_min, x_max, y_min, y_max]:
            QMessageBox.warning(self, "Error", "Invalid numerical input.")
            return

        p1 = QgsPointXY(x_min, y_min)
        p2 = QgsPointXY(x_max, y_min)
        p3 = QgsPointXY(x_max, y_max)
        p4 = QgsPointXY(x_min, y_max)

        points = [p1, p2, p3, p4, p1]
        self.generate_layer_and_feature(points, "Rectangle")

    def add_point_to_list(self):
        x = self.get_float(self.poly_lon)
        y = self.get_float(self.poly_lat)
        if x is None or y is None:
            QMessageBox.warning(self, "Error", "Invalid coordinate.")
            return
        self.custom_points.append(QgsPointXY(x, y))
        self.points_list_widget.addItem(f"Point {len(self.custom_points)}: {x}, {y}")
        self.poly_lon.clear()
        self.poly_lat.clear()
        self.poly_lon.setFocus()

    def clear_points(self):
        self.custom_points = []
        self.points_list_widget.clear()

    def create_custom_polygon(self):
        if len(self.custom_points) < 3:
            QMessageBox.warning(self, "Error", "At least 3 points are required.")
            return
        points = self.custom_points[:]
        if points[0] != points[-1]:
            points.append(points[0])
        self.generate_layer_and_feature(points, "PolygonByVertex")

    def generate_layer_and_feature(self, points, layer_name_prefix):
        crs = self.crs_selector.crs()
        uri = f"Polygon?crs={crs.authid()}"
        layer = QgsVectorLayer(uri, f"{layer_name_prefix}_Output", "memory")

        if not layer.isValid():
            return

        pr = layer.dataProvider()
        pr.addAttributes([
            QgsField("id", QVariant.Int),
            QgsField("Description", QVariant.String),
            QgsField("VertexCoordinates", QVariant.String)
        ])
        layer.updateFields()

        coords_str = "; ".join([f"({p.x():.5f}, {p.y():.5f})" for p in points])

        feature = QgsFeature()
        feature.setGeometry(QgsGeometry.fromPolygonXY([points]))
        feature.setAttributes([1, layer_name_prefix, coords_str])

        pr.addFeatures([feature])
        layer.updateExtents()
        QgsProject.instance().addMapLayer(layer)
        QMessageBox.information(self, "Done", "Polygon created!")


class PolyCoordsPlugin:
    def __init__(self, iface):
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)
        self.action = None
        self.dlg = None

    def initGui(self):
        icon_path = os.path.join(self.plugin_dir, 'icon.png')

        self.action = QAction(QIcon(icon_path), "Build Polygon from Coords.", self.iface.mainWindow())
        self.action.triggered.connect(self.run)

        self.iface.addToolBarIcon(self.action)
        self.iface.addPluginToMenu("&Polygon Generator", self.action)

    def unload(self):
        self.iface.removeToolBarIcon(self.action)
        self.iface.removePluginMenu("&Polygon Generator", self.action)

    def run(self):
        if not self.dlg:
            self.dlg = PolygonCreatorDialog(self.iface)
        self.dlg.show()
        self.dlg.raise_()
        self.dlg.activateWindow()
