# -*- coding: utf-8 -*-
"""
Consulta_tm
=================
QGIS Plugin: Consulta Lotes em Terrenos de Marinha e seus Acrescidos.

- Data de início: 2024-12-06
- Autor: Letícia Estephane de Brito Ferreira
- Email: leticiaestephane2@gmail.com
"""

import os
import webbrowser
from qgis.PyQt.QtCore import QVariant
from PyQt5.QtGui import QColor
from qgis.PyQt import uic, QtWidgets
from qgis.core import (
    QgsGeometry,
    QgsFeature,
    QgsVectorLayer,
    QgsProject,
    QgsField,
    QgsSymbol,
    QgsSingleSymbolRenderer,
    QgsCategorizedSymbolRenderer,
    QgsGraduatedSymbolRenderer,
)

# Carregar a UI
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'consulta_tm_dialog_base.ui'
))

class Consulta_tmDialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, iface, parent=None):
        super(Consulta_tmDialog, self).__init__(parent)
        self.iface = iface  # Armazena a interface do QGIS
        self.setupUi(self)
        self.bConsulta.clicked.connect(self.consulta)
        self.bResete.clicked.connect(self.apagar)

    def consulta(self):
        self.iface.setActiveLayer(lotes)  # Usando self.iface

    def criar_camada_intersecao(self, nome_camada, geometria_intersecoes, cor):
        """Cria uma camada temporária contendo as geometrias de interseção e aplica a cor especificada."""
        camada = QgsVectorLayer("Polygon?crs=EPSG:32985", nome_camada, "memory")
        provedor = camada.dataProvider()
        provedor.addAttributes([QgsField("Área (m²)", QVariant.Double)])
        camada.updateFields()

        for geom in geometria_intersecoes:
            feature = QgsFeature()
            feature.setGeometry(geom)
            feature.setAttributes([round(geom.area(), 3)])
            provedor.addFeature(feature)

        # Definir a simbologia com a cor especificada
        symbol = QgsSymbol.defaultSymbol(camada.geometryType())
        symbol.setColor(cor)  # A cor será definida aqui
        renderer = QgsSingleSymbolRenderer(symbol)
        camada.setRenderer(renderer)

        camada.updateExtents()
        QgsProject.instance().addMapLayer(camada)
        self.iface.setActiveLayer(camada)
        return camada

    def consulta(self):
        """Realiza a consulta do lote baseado no número do imóvel."""
        projeto = QgsProject.instance()
        camProjeto = "G:/TCC_14042025/MAPA_BESSA/ExMapa.qgz"

        if projeto.fileName() != camProjeto:
            projeto.read(camProjeto)
            print('O projeto foi carregado com sucesso.')

        lotes = QgsProject.instance().mapLayersByName('LOTES_JP')
        if not lotes:
            self.resultado.setText("A camada LOTES_JP não foi encontrada.")
            return
        lotes = lotes[0]

        TM = QgsProject.instance().mapLayersByName('TERRENOS_MARINHA')
        if not TM:
            self.resultado.setText("A camada TERRENOS_MARINHA não foi encontrada.")
            return
        TM = TM[0]

        AM = QgsProject.instance().mapLayersByName('ACRESCIDOS_MARINHA')
        if not AM:
            self.resultado.setText("A camada ACRESCIDOS_MARINHA não foi encontrada.")
            return
        AM = AM[0]

        num_lote = self.numLote.text()
        logradouro = self.logradouroInput.text()
        numero = self.numeroImovelInput.text()

        if num_lote:
            expression = f'"CODI_CART" = {num_lote}'
        elif logradouro and numero:
            expression = f'"Desc_Logr" in \'%{logradouro}%\' AND "Nume_Imov" = {numero}'
        else:
            self.resultado.setText("Informe o CODI_CART ou o logradouro e número do imóvel.")
            return

        print(f"Expressão de consulta: {expression}")

        lotes.selectByExpression(expression)
        self.iface.setActiveLayer(lotes)

        mc = self.iface.mapCanvas()
        mc.zoomToSelected()
        mc.zoomOut()

        if lotes.selectedFeatureCount() == 1:
            lote = next(lotes.getSelectedFeatures())
            lote_geometry = lote.geometry()

            geometrias_tm = [
                feature.geometry().intersection(lote_geometry)
                for feature in TM.getFeatures()
                if feature.geometry().intersects(lote_geometry)
            ]
            geometrias_am = [
                feature.geometry().intersection(lote_geometry)
                for feature in AM.getFeatures()
                if feature.geometry().intersects(lote_geometry)
            ]

            if geometrias_tm:
                self.criar_camada_intersecao("Terreno de Marinha do Lote", geometrias_tm, QColor(128, 0, 128))  # Roxo

            if geometrias_am:
                self.criar_camada_intersecao("Acrescido de Marinha do Lote", geometrias_am, QColor(255, 20, 147))  # Pink

            classificacao = "Terreno Alodial"
            mensagens = []

            total_area_tm = sum(geom.area() for geom in geometrias_tm) if geometrias_tm else 0
            total_area_am = sum(geom.area() for geom in geometrias_am) if geometrias_am else 0
            total_area_uniao = total_area_tm + total_area_am

            if geometrias_tm and geometrias_am:
                classificacao = "Dentro de Terreno de Marinha e Acrescido de Marinha"
                mensagens.append(f"Terreno de Marinha: {str(round(total_area_tm, 3)).replace('.', ',')} m²")
                mensagens.append(f"Acrescido de Marinha: {str(round(total_area_am, 3)).replace('.', ',')} m²")
            elif geometrias_tm:
                if total_area_tm == lote_geometry.area():
                    classificacao = "Totalmente dentro de Terreno de Marinha"
                else:
                    classificacao = "Parcialmente dentro de Terreno de Marinha"
                mensagens.append(f"Terreno de Marinha: {str(round(total_area_tm, 3)).replace('.', ',')} m²")
            elif geometrias_am:
                if total_area_am == lote_geometry.area():
                    classificacao = "Totalmente dentro de Acrescido de Marinha"
                else:
                    classificacao = "Parcialmente dentro de Acrescido de Marinha"
                mensagens.append(f"Acrescido de Marinha: {str(round(total_area_am, 3)).replace('.', ',')} m²")

            if total_area_uniao > 0:
                mensagens.append(f"Área total da União: {str(round(total_area_uniao, 3)).replace('.', ',')} m²")

            self.resultado.setText(f"{classificacao}\n" + "\n".join(mensagens))

            if self.gerar_html.isChecked():
                self.gerar_html_funcao(classificacao, "\n".join(mensagens))
        else:
            self.resultado.setText("Lote não encontrado ou dados inválidos.")


    def gerar_html_funcao(self, classificacao, mensagem):
        """Gera um arquivo HTML com a imagem do mapa, a escala fixa (1:900)"""
        caminho_html = os.path.join(os.path.expanduser("~"), "Desktop", "resultado_consulta.html")
        caminho_imagem = os.path.join(os.path.expanduser("~"), "Desktop", "mapa_qgis.png")

        self.iface.mapCanvas().zoomScale(900)
        self.iface.mapCanvas().saveAsImage(caminho_imagem)

        explicacao = f"O lote consultado está classificado como <b>{classificacao}</b>."

        projeto = QgsProject.instance()
        root = projeto.layerTreeRoot()
        

        for layer in projeto.mapLayers().values():
            layer_node = root.findLayer(layer.id())

            if layer.isValid() and layer_node and layer_node.isVisible():
                renderer = layer.renderer()
                cor = "#000000"

                
        html_content = f"""
        <html>
            <head>
                <title>Resultado da Consulta</title>
                <style>
                    body {{
                        font-family: Arial, sans-serif;
                        font-size: 14px;
                    }}
                    .cor {{
                        width: 20px;
                        height: 20px;
                        display: inline-block;
                        margin-right: 10px;
                    }}
                </style>
            </head>
            <body>
                <img src="G:/TCC_14042025/MAPA_BESSA/SPU.png" width="600" height="400">
                <h1>Ministério da Gestão e Inovação em Serviços Públicos</h1>
                <h1>Consulta de possíveis áreas da União</h1>
                <p>{explicacao}</p>
                <p>{mensagem}</p>
                <h3>Ilustração:</h3>
                <img src="file:///{caminho_imagem}" alt="Mapa do QGIS" width="900"/>
                <h3>Escala: 1:900</h3>
                
                
            </body>
        </html>
        """

        with open(caminho_html, "w", encoding="utf-8") as f:
            f.write(html_content)

        webbrowser.open(f"file://{caminho_html}")

    def apagar(self):
        """Limpa todos os campos de entrada e atualiza o mapa."""
        self.numLote.clear()
        self.logradouroInput.clear()
        self.numeroImovelInput.clear()
        self.resultado.clear()
        self.iface.mapCanvas().refreshAllLayers()

# Classe que inicializa o plugin
class Consulta_tm:
    def __init__(self, iface):
        print("Inicializando plugin com iface:", iface)
        self.iface = iface
        self.dlg = None

    def initGui(self):
        """Inicializa o plugin na interface do QGIS"""
        pass  # Aqui você pode adicionar ícones, menus, ações ou qualquer integração com a interface do QGIS.

    def run(self):
        """Exibe o diálogo do plugin"""
        print("Running plugin with iface:", self.iface)
        self.dlg = Consulta_tmDialog(self.iface)
        self.dlg.show()

    def unload(self):
        """Descarrega o plugin quando o QGIS for fechado"""
        pass
