# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ConcaveHull
                                 A QGIS plugin
 Computes a concave hull containing a set of features
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2018-12-27
        git sha              : $Format:%H$
        copyright            : (C) 2018 by Detlev Neumann, Geospatial Services
        email                : dneumann@geospatial-services.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt5.QtCore import Qt, QSettings, QTranslator, qVersion, QCoreApplication, QItemSelectionModel, QItemSelection, QVariant, pyqtSlot
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QApplication, QAction, QDialogButtonBox, QMessageBox
from qgis.core import Qgis, QgsProject, QgsMapLayer, QgsFeature, QgsGeometry, QgsVectorLayer, QgsVectorFileWriter, QgsPoint, QgsPointXY, QgsField
from qgis.gui import QgsMessageBar

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .concave_hull_dialog import ConcaveHullDialog
import os.path
import math
from .shared_nearest_neighbor_clustering import SSNClusters


def write_segments(linestring):
    """
    nur zum Debuggen: Ausgabe der Hülle für die reachable-Webseite
    """
    outfile = open("boundData.csv", "w")
    for p in range(len(linestring)-1):
        outfile.write('hull\t%s\t%s\t%s\t%s\n' % (linestring[p][0], linestring[p][1],
                                                  linestring[p+1][0], linestring[p+1][1]))
    outfile.close()


def clean_list(list_of_points):
    """Deletes duplicate points in list_of_points

    :param list_of_points: list of points
    :type list_of_points: list
    :return: list of points without duplicates
    :rtype: list
    """
    return list(set(list_of_points))


def length(vector):
    """Returns the number of elements in vector

    :param vector: list of elements
    :type vector: list
    :return: number of elements in vector
    :rtype: int
    """
    return len(vector)


def find_min_y_point(list_of_points):
    """Returns that point of *list_of_points* having minimal y-coordinate

    :param list_of_points: list of points
    :type list_of_points: list
    :return: point of list_of_points having minimal y-coordinate
    :rtype: tuple
    """
    min_y_pt = list_of_points[0]
    for point in list_of_points[1:]:
        if point[1] < min_y_pt[1] or (point[1] == min_y_pt[1] and point[0] < min_y_pt[0]):
            min_y_pt = point
    return min_y_pt


def add_point(vector, element):
    """Returns vector with the given element append to the right
    """
    vector.append(element)
    return vector


def remove_point(vector, element):
    """Returns a copy of vector without the given element
    """
    vector.pop(vector.index(element))
    return vector


def euclidian_distance(point1, point2):
    """Returns the euclidian distance of the 2 given points.

    :param point1: first point
    :type point1: tuple
    :param point2: second point
    :type point2: tuple
    :return: euclidian distance
    :rtype: float
    """
    return math.sqrt(math.pow(point1[0] - point2[0], 2) + math.pow(point1[1] - point2[1], 2))


def nearest_points(list_of_points, point, k):
    """
    gibt eine Liste mit den Indizes der k nächsten Nachbarn aus list_of_points zum angegebenen Punkt zurück.
    Das Maß für die Nähe ist die euklidische Distanz. Intern wird k auf das Minimum zwischen dem gegebenen Wert
    für k und der Anzahl der Punkte in list_of_points gesetzt

    :param list_of_points: list of points to search k nearest neighbors from
    :type list_of_points: list
    :param point: x,y-coordinate of point
    :type point: tuple
    :param k: count of nearest neighbors to search
    :type k: int
    :return: k nearest neighbors
    :rtype: list of tuple
    """
    # build a list of tuples of distances between point *point* and every point in *list_of_points*, and
    # their respective index of list *list_of_distances*
    list_of_distances = []
    for index in range(len(list_of_points)):
        list_of_distances.append((euclidian_distance(list_of_points[index], point), index))

    # sort distances in ascending order
    list_of_distances.sort()

    # get the k nearest neighbors of point
    nearest_list = []
    for index in range(min(k, len(list_of_points))):
        nearest_list.append((list_of_points[list_of_distances[index][1]]))
    return nearest_list


def angle(from_point, to_point):
    """
    Returns the angle of the directed line segment, going from *from_point* to *to_point*, in radians. The angle is
    positive for segments with upward direction (north), otherwise negative (south). Values ranges from 0 at the
    right (east) to pi at the left side (west).

    :param from_point: start point of line segment
    :type from_point: tuple
    :param to_point: end point of line segment
    :type to_point: tuple
    :return: angle of the line segment
    :rtype: float
    """
    return math.atan2(to_point[1] - from_point[1], to_point[0] - from_point[0])


def angle_difference(angle1, angle2):
    """Calculates the difference between the given angles in clockwise direction as radians.

    :param angle1: first angle, given in radians
    :type angle1: float
    :param angle2: second angle, given in radians
    :type angle2: float
    :return: difference of two angles in clockwise direction between 0 and 2*Pi
    :rtype: float
    """
    if (angle1 > 0 and angle2 >= 0) and angle1 > angle2:
        return abs(angle1 - angle2)
    elif (angle1 >= 0 and angle2 > 0) and angle1 < angle2:
        return 2 * math.pi + angle1 - angle2
    elif (angle1 < 0 and angle2 <= 0) and angle1 < angle2:
        return 2 * math.pi + angle1 + abs(angle2)
    elif (angle1 <= 0 and angle2 < 0) and angle1 > angle2:
        return abs(angle1 - angle2)
    elif angle1 <= 0 < angle2:
        return 2 * math.pi + angle1 - angle2
    elif angle1 >= 0 >= angle2:
        return angle1 + abs(angle2)
    else:
        return 0


def intersect(line1, line2):
    """Checks if the two given line segments intersect each other.

    :param line1: first line given by start point and end point, as tuples of (x, y) each
    :type line1: tuple
    :param line2: second line given by start point and end point, as tuples of (x, y) each
    :type line2: tuple
    :return: True, if lines intersect each other, False otherwise
    :rtype: bool
    """
    a1 = line1[1][1] - line1[0][1]
    b1 = line1[0][0] - line1[1][0]
    c1 = a1 * line1[0][0] + b1 * line1[0][1]
    a2 = line2[1][1] - line2[0][1]
    b2 = line2[0][0] - line2[1][0]
    c2 = a2 * line2[0][0] + b2 * line2[0][1]
    tmp = (a1 * b2 - a2 * b1)
    if tmp == 0:
        return False
    sx = (c1 * b2 - c2 * b1) / tmp
    if (sx > line1[0][0] and sx > line1[1][0]) or (sx > line2[0][0] and sx > line2[1][0]) or\
            (sx < line1[0][0] and sx < line1[1][0]) or (sx < line2[0][0] and sx < line2[1][0]):
        return False
    sy = (a1 * c2 - a2 * c1) / tmp
    if (sy > line1[0][1] and sy > line1[1][1]) or (sy > line2[0][1] and sy > line2[1][1]) or\
            (sy < line1[0][1] and sy < line1[1][1]) or (sy < line2[0][1] and sy < line2[1][1]):
        return False
    return True


def point_in_polygon_q(point, list_of_points):
    """Checks if given point *point* is laying in the polygon described by the vertices *list_of_points*

    Based on the "Ray Casting Method" described by Joel Lawhead in this blog article:
    http://geospatialpython.com/2011/01/point-in-polygon.html

    :param point: x,y-coordinate of point
    :type point: tuple
    :param list_of_points: polygon as list of vertices
    :type list_of_points: list of tuple
    :return: True, if given point is laying in the polygon described by the vertices *list_of_points*,
        otherwise False
    :rtype: bool
    """
    x = point[0]
    y = point[1]
    poly = [(pt[0], pt[1]) for pt in list_of_points]
    n = len(poly)
    inside = False

    p1x, p1y = poly[0]
    for i in range(n + 1):
        p2x, p2y = poly[i % n]
        if y > min(p1y, p2y):
            if y <= max(p1y, p2y):
                if x <= max(p1x, p2x):
                    if p1y != p2y:
                        xints = (y - p1y) * (p2x - p1x) / (p2y - p1y) + p1x
                    if p1x == p2x or x <= xints:
                        inside = not inside
        p1x, p1y = p2x, p2y

    return inside


def write_wkt(list_of_points, file_name):
    """Writes the geometry described by *list_of_points* in Well Known Text format to file

    :param list_of_points: polygon as list of vertices
    :type list_of_points: list of tuple
    :return: True, if geometry was successfully exported, otherwise False
    :rtype: bool
    """
    if file_name is None:
        file_name = 'hull2.wkt'

    try:
        if os.path.isfile(file_name):
            outfile = open(file_name, 'a')
        else:
            outfile = open(file_name, 'w')
            outfile.write('%s\n' % 'WKT')

        wkt = 'POLYGON((' + str(list_of_points[0][0]) + ' ' + str(list_of_points[0][1])
        for p in list_of_points[1:]:
            wkt += ', ' + str(p[0]) + ' ' + str(p[1])
        wkt += '))'
        outfile.write('%s\n' % wkt)
        success = True
    except:
        success = False
    finally:
        outfile.close()
        return success


def as_wkt(point_list):
    """Returns the geometry described by *point_list* in Well Known Text format

    Example: hull = self.as_wkt(the_hull)
             feature.setGeometry(QgsGeometry.fromWkt(hull))

    :param point_list: list of points, as tuple (x, y) each
    :type point_list: list
    :return: polygon geometry as WTK
    :rtype: str
    """
    wkt = 'POLYGON((' + str(point_list[0][0]) + ' ' + str(point_list[0][1])
    for p in point_list[1:]:
        wkt += ', ' + str(p[0]) + ' ' + str(p[1])
    wkt += '))'
    return wkt


def as_polygon(point_list):
    """Returns the geometry described by *point_list* in as QgsGeometry

    :param point_list: list of points, as tuple (x, y) each
    :type point_list: list
    :return: geometry of closed connection of given list of points
    :rtype: QgsGeometry
    """
    # create a list of QgsPoint() from list of point coordinate strings in *point_list*
    points = [QgsPointXY(point[0], point[1]) for point in point_list]

    # create the polygon geometry from list of point geometries
    poly = QgsGeometry.fromPolygonXY([points])

    return poly


def enable_use_of_global_CRS():
    """Set new layers to use the project CRS.

    Code snipped taken from http://pyqgis.blogspot.co.nz/2012/10/basics-automatic-use-of-crs-for-new.html

    Example: old_behaviour = enable_use_of_global_CRS()

    :return: old setting for Projections/defaultBehaviour
    :rtype: str
    """
    settings = QSettings()
    old_behaviour = settings.value('/Projections/defaultBehaviour')
    settings.setValue('/Projections/defaultBehaviour', 'useProject')
    return old_behaviour


def disable_use_of_global_CRS(default_behaviour='prompt'):
    """Enables old settings again. If argument is missing then set behaviour to prompt.

    Example: disable_use_of_global_CRS(old_behaviour)

    :param default_behaviour: original setting for Projections/defaultBehaviour
    :type default_behaviour: str
    :return: None
    """
    settings = QSettings()
    settings.setValue('/Projections/defaultBehaviour', default_behaviour)
    return None


def extract_points(geom):
    """
    Generates list of QgsPoints from QgsGeometry *geom* (can be point, line, or polygon)
    Code taken from fTools plugin

    :param geom: an arbitrary geometry feature
    :type geom: QgsGeometry
    :return: list of points
    :rtype: list
    """
    multi_geom = QgsGeometry()
    temp_geom = []
    # point geometry
    if geom.type() == 0:
        if geom.isMultipart():
            temp_geom = geom.asMultiPoint()
        else:
            temp_geom.append(geom.asPoint())
    # line geometry
    if geom.type() == 1:
        # if multipart feature explode to single part
        if geom.isMultipart():
            multi_geom = geom.asMultiPolyline()
            for i in multi_geom:
                temp_geom.extend(i)
        else:
            temp_geom = geom.asPolyline()
    # polygon geometry
    elif geom.type() == 2:
        # if multipart feature explode to single part
        if geom.isMultipart():
            multi_geom = geom.asMultiPolygon()
            # now single part polygons
            for i in multi_geom:
                # explode to line segments
                for j in i:
                    temp_geom.extend(j)
        else:
            multi_geom = geom.asPolygon()
            # explode to line segments
            for i in multi_geom:
                temp_geom.extend(i)
    return temp_geom

# TODO: translation required
def sort_by_angle(list_of_points, last_point, last_angle):
    """
    gibt die Punkte in list_of_points in absteigender Reihenfolge des Winkels zum letzten Segment der Hülle zurück,
    gemessen im Uhrzeigersinn. Es wird also immer der rechteste der benachbarten  Punkte ausgewählt. Der erste
    Punkt dieser Liste wird der nächste Punkt der Hülle.
    """
    def getkey(item):
        return angle_difference(last_angle, angle(last_point, item))

    vertex_list = sorted(list_of_points, key=getkey, reverse=True)
    return vertex_list


def concave_hull(points_list, k):
    """
    Calculates a valid concave hull polygon containing all given points. The algorithm searches for that
    point in the neighbourhood of k nearest neighbours which maximizes the rotation angle in clockwise direction
    without intersecting any previous line segments.

    This is an implementation of the algorithm described by Adriano Moreira and Maribel Yasmina Santos:
    CONCAVE HULL: A K-NEAREST NEIGHBOURS APPROACH FOR THE COMPUTATION OF THE REGION OCCUPIED BY A SET OF POINTS.
    GRAPP 2007 - International Conference on Computer Graphics Theory and Applications; pp 61-68.

    :param points_list: list of points as tuple (x, y)
    :type points_list: list
    :param k: number of nearest neighbours
    :type k: int
    :return: concave hull as list of points as tuple (x, y)
    :rtype: list
    """
    # return an empty list if not enough points are given
    if k > len(points_list):
        return None

    # the number of nearest neighbours k must be greater than or equal to 3
    # kk = max(k, 3)
    kk = max(k, 2)

    # delete duplicate points
    point_set = clean_list(points_list)

    # if point_set has less then 3 points no polygon can be created and an empty list will be returned
    if len(point_set) < 3:
        return None

    # if point_set has 3 points then these are already vertices of the hull. Append the first point to
    # close the hull polygon
    if len(point_set) == 3:
        return add_point(point_set, point_set[0])

    # make sure that k neighbours can be found
    kk = min(kk, len(point_set))

    # start with the point having the smallest y-coordinate (most southern point)
    first_point = find_min_y_point(point_set)

    # add this points as the first vertex of the hull
    hull = [first_point]

    # make the first vertex of the hull to the current point
    current_point = first_point

    # remove the point from the point_set, to prevent him being among the nearest points
    point_set = remove_point(point_set, first_point)
    previous_angle = math.pi

    # step counts the number of segments
    step = 2

    # as long as point_set is not empty or search is returning to the starting point
    while (current_point != first_point) or (step == 2) and (len(point_set) > 0):

        # after 3 iterations add the first point to point_set again, otherwise a hull cannot be closed
        if step == 5:
            point_set = add_point(point_set, first_point)

        # search the k nearest neighbours of the current point
        k_nearest_points = nearest_points(point_set, current_point, kk)

        # sort the candidates (neighbours) in descending order of right-hand turn. This way the algorithm progresses
        # in clockwise direction through as many points as possible
        c_points = sort_by_angle(k_nearest_points, current_point, previous_angle)

        its = True
        i = -1

        # search for the nearest point to which the connecting line does not intersect any existing segment
        while its is True and (i < len(c_points)-1):
            i += 1
            if c_points[i] == first_point:
                last_point = 1
            else:
                last_point = 0
            j = 2
            its = False

            while its is False and (j < len(hull) - last_point):
                its = intersect((hull[step-2], c_points[i]), (hull[step-2-j], hull[step-1-j]))
                j += 1

        # there is no candidate to which the connecting line does not intersect any existing segment, so the
        # for the next candidate fails. The algorithm starts again with an increased number of neighbours
        if its is True:
            return concave_hull(points_list, kk + 1)

        # the first point which complies with the requirements is added to the hull and gets the current point
        current_point = c_points[i]
        hull = add_point(hull, current_point)

        # calculate the angle between the last vertex and his precursor, that is the last segment of the hull
        # in reversed direction
        previous_angle = angle(hull[step - 1], hull[step - 2])

        # remove current_point from point_set
        point_set = remove_point(point_set, current_point)

        # increment counter
        step += 1

    all_inside = True
    i = len(point_set)-1

    # check if all points are within the created polygon
    while (all_inside is True) and (i >= 0):
        all_inside = point_in_polygon_q(point_set[i], hull)
        i -= 1

    # since at least one point is out of the computed polygon, try again with a higher number of neighbours
    if all_inside is False:
        return concave_hull(points_list, kk + 1)

    # a valid hull has been constructed
    return hull


class ConcaveHull:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'ConcaveHull_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Create the dialog (after translation) and keep reference
        self.dlg = ConcaveHullDialog()

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Concave Hull')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'ConcaveHull')
        self.toolbar.setObjectName(u'ConcaveHull')

        # get reference to the QGIS message bar
        self.msg_bar = self.iface.messageBar()

        # create reference to a vector layer and data provider; will be overridden when target layer is created
        self._concav_hull_layer = QgsVectorLayer()
        self._concav_hull_data_provider = None

        # added 2019/01/01
        self.table_data = []

        # setup fields of the output layer
        self.field_clause = '&field=id:integer&field=count:integer'

        # connect dialog widget signals
        self.dlg.buttonBox.accepted.connect(self.accepted)
        self.dlg.buttonBox.rejected.connect(self.rejected)

        btn = self.dlg.buttonBox.button(QDialogButtonBox.Apply)
        btn.clicked.connect(self.apply)

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :return: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('ConcaveHull', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :return: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/concavehull/icon.svg'
        self.add_action(
            icon_path,
            text=self.tr(u'Concave Hull'),
            callback=self.run,
            parent=self.iface.mainWindow())

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Concave Hull'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    def create_output_layer(self):
        """Creates the output layer _before_ generating hull geometries

        :return: True, if layer was successfully created, otherwise False
        :rtype: bool
        """
        if self.dlg.rb_shapefile.isChecked():
            shape_filename = self.dlg.ed_output_layer.text()
            layer_name = os.path.splitext(os.path.basename(str(shape_filename)))[0]
            if shape_filename == '':
                msg = self.msg_bar.createMessage(self.tr('No shapefile name specified'))
                self.msg_bar.pushWidget(msg, Qgis.Critical, 5)
                return False
        elif self.dlg.rb_new_memory_layer.isChecked():
            layer_name = self.dlg.ed_memory_layer.text()
        else:
            layer_name = self.dlg.cb_output.currentText()
            # TODO: if group_by_attribute is marked, check compatibility between origin attribute and layer attribute

        # if the layer does not exist it has to be created
        if self.dlg.rb_new_memory_layer.isChecked() or self.dlg.rb_shapefile.isChecked():
            srs = self.iface.mapCanvas().mapSettings().destinationCrs().authid()
            self._concav_hull_layer = QgsVectorLayer('Polygon?crs={}{}'.format(srs, self.field_clause),
                                                     layer_name,
                                                     'memory')
            self._concav_hull_data_provider = self._concav_hull_layer.dataProvider()

        # if the layer already exists
        else:
            self._concav_hull_layer = QgsProject.instance().mapLayersByName(layer_name)[0]
            self._concav_hull_data_provider = self._concav_hull_layer.dataProvider()

        return True

    def create_output_feature(self, geom, start_id=0, value=None):
        """Appends hull geometries one by one _before_ proceeding with next cluster

        :param geom: geometry of concave hull and count of enclosed objects
        :type geom: list(QgsGeometry, int)
        :param start_id: id to assign to first object in geom, increment for any further object
        :type start_id: int
        :param value: arbitrary value to take as
        :type value: str or int or float
        :return: count of hulls added
        :rtype: int
        """
        # add hull geometry to data provider
        fid = start_id
        for hull in geom:
            feature = QgsFeature()
            feature.setGeometry(hull[0])
            if self._concav_hull_layer.fields().indexFromName('id') > -1:
                if value:
                    feature.setAttributes([fid, hull[1], value])
                else:
                    feature.setAttributes([fid, hull[1]])
                fid += 1
            self._concav_hull_data_provider.addFeatures([feature])

        return fid - start_id
        
    def close_output_layer(self):
        """Write and close layer just _after_ generation of last hull geometry

        :return: True, if layer was successfully created, otherwise False
        :rtype: bool
        """
        # if new memory layer simply add memory layer to the map
        layer = self._concav_hull_layer
        if self.dlg.rb_new_memory_layer.isChecked():
            QgsProject.instance().addMapLayer(layer)

        # if features go to shapefile dump memory layer to shapefile
        elif self.dlg.rb_shapefile.isChecked():
            shape_filename = self.dlg.ed_output_layer.text()
            error = QgsVectorFileWriter.writeAsVectorFormat(layer, shape_filename, 'CP1250', layer.crs(), 'ESRI Shapefile')
            if error[0] != QgsVectorFileWriter.NoError:
                msg = self.msg_bar.createMessage(self.tr('Error writing shapefile: ' + str(error)))
                self.msg_bar.pushWidget(msg, Qgis.Critical, 5)
                return False

            # add the new layer to the map
            if self.dlg.cb_add_to_map.isChecked():
                base_name = os.path.splitext(os.path.basename(str(shape_filename)))[0]
                layer_name = QgsVectorLayer(shape_filename, base_name, 'ogr')
                QgsProject.instance().addMapLayer(layer_name)

        # because change of extent in provider is not propagated to the layer
        layer.updateExtents()
        layer.triggerRepaint()
        return True

    def create_noise_layer(self, geom, layer_name='ConcaveHullNoisePoints'):
        """Writes all noise points to a layer for quality checking

        Creates a memory layer named layer_name, default name ConcaveHullNoisePoints, using project CRS and
        suppressing the CRS settings dialog. The layer may be helpful when it comes to fine tune 
        algorithm parameters.

        :param geom: list of points
        :type geom: list
        :param layer_name: name of noise point layer
        :type layer_name: str
        :return: True, if layer was successfully created, otherwise False
        :rtype: bool
        """
        if self.dlg.rb_shapefile.isChecked():
            shape_filename = self.dlg.ed_output_layer.text() + 'NoisePoints'
            layer_name = 'ConcaveHullNoisePoints'
            if shape_filename == '':
                return False
        else:
            if self.dlg.rb_new_memory_layer.isChecked():
                layer_name = self.dlg.ed_memory_layer.text() + 'NoisePoints'
            else:
                layer_name = self.dlg.cb_output.currentText() + 'NoisePoints'

        # if the layer does not exist it has to be created
        if not QgsProject.instance().mapLayersByName(layer_name):
            srs = self.iface.mapCanvas().mapSettings().destinationCrs().authid()
            layer = QgsVectorLayer('Point?crs=' + str(srs) + '&field=id:integer', layer_name, 'memory')
            provider = layer.dataProvider()

        # if the layer already exists
        else:
            layer = QgsProject.instance().mapLayersByName(layer_name)[0]
            provider = layer.dataProvider()

        # add noise points to data provider
        fid = 0
        for np in geom:
            feature = QgsFeature()
            feature.setGeometry(QgsPoint(np[0], np[1]))
            if layer.fields().indexFromName('id') > -1:
                feature.setAttributes([fid])
                fid += 1
            provider.addFeatures([feature])

        # if new memory layer simply add memory layer to the map
        if self.dlg.rb_new_memory_layer.isChecked():
            QgsProject.instance().addMapLayer(layer)

        # if features go to shapefile dump memory layer to shapefile
        elif self.dlg.rb_shapefile.isChecked():
            error = QgsVectorFileWriter.writeAsVectorFormat(layer, shape_filename, 'CP1250', layer.crs(), 'ESRI Shapefile')
            if error != QgsVectorFileWriter.NoError:
                msg = self.msg_bar.createMessage(self.tr('Error writing shapefile: ' + str(error)))
                self.msg_bar.pushWidget(msg, Qgis.Critical, 5)
                return False

            # add the new layer to the map
            if self.dlg.cb_add_to_map.isChecked():
                base_name = os.path.splitext(os.path.basename(str(shape_filename)))[0]
                layer_name = QgsVectorLayer(shape_filename, base_name, 'ogr')
                QgsProject.instance().addMapLayer(layer_name)

        # because change of extent in provider is not propagated to the layer
        layer.updateExtents()
        layer.triggerRepaint()
        return True

    def create_cluster_layer(self, clusters, layer_name='SNNClusters'):
        """Writes all noise points to a layer for quality checking

        Creates a memory layer named layer_name, default name ConcaveHullNoisePoints, using project CRS and
        suppressing the CRS settings dialog. The layer may be helpful when it comes to fine tune
        algorithm parameters.

        :param clusters: clustered points
        :type clusters: dict of dicts
        :param layer_name: name of noise point layer
        :type layer_name: str
        :return: True, if layer was successfully created, otherwise False
        :rtype: bool
        """
        if self.dlg.rb_shapefile.isChecked():
            shape_filename = self.dlg.ed_output_layer.text()
            layer_name = 'SNNClusters'
            if shape_filename == '':
                return False
        else:
            if self.dlg.rb_new_memory_layer.isChecked():
                layer_name = self.dlg.ed_memory_layer.text()
            else:
                layer_name = self.dlg.cb_output.currentText()

        # if the layer does not exist it has to be created
        if not QgsProject.instance().mapLayersByName(layer_name):
            srs = self.iface.mapCanvas().mapSettings().destinationCrs().authid()
            layer = QgsVectorLayer('Point?crs={}{}'.format(srs, self.field_clause),
                                   layer_name,
                                   'memory')
            provider = layer.dataProvider()

        # if the layer already exists
        else:
            layer = QgsProject.instance().mapLayersByName(layer_name)[0]
            provider = layer.dataProvider()

        # add clustered points to data provider
        fid = 0
        features = []
        for group in clusters.keys():
            for cluster in clusters[group].keys():
                _len = len(clusters[group][cluster])
                for np in clusters[group][cluster]:
                    feature = QgsFeature()
                    feature.setGeometry(QgsPoint(np[0], np[1]))
                    if group:
                        feature.setAttributes([fid, _len, group])
                    else:
                        feature.setAttributes([fid, _len])
                    features.append(feature)
                fid += 1
        provider.addFeatures(features)

        # if new memory layer simply add memory layer to the map
        if self.dlg.rb_new_memory_layer.isChecked():
            QgsProject.instance().addMapLayer(layer)

        # if features go to shapefile dump memory layer to shapefile
        elif self.dlg.rb_shapefile.isChecked():
            error = QgsVectorFileWriter.writeAsVectorFormat(layer, shape_filename, 'CP1250', layer.crs(), 'ESRI Shapefile')
            if error != QgsVectorFileWriter.NoError:
                msg = self.msg_bar.createMessage(self.tr('Error writing shapefile: ' + str(error)))
                self.msg_bar.pushWidget(msg, Qgis.Critical, 5)
                return False

            # add the new layer to the map
            if self.dlg.cb_add_to_map.isChecked():
                base_name = os.path.splitext(os.path.basename(str(shape_filename)))[0]
                layer_name = QgsVectorLayer(shape_filename, base_name, 'ogr')
                QgsProject.instance().addMapLayer(layer_name)

        # because change of extent in provider is not propagated to the layer
        layer.updateExtents()
        layer.triggerRepaint()
        return True

    def get_vector_layers_by_type(self, geom_type=None, skip_active=False):
        """Returns map layers with specified geometry type

        Returns a dict of layers [name: id] in the project for the given geom_type.
        If skip_active is True the active layer is not included.
        Code taken from DigitizingTools plugin, (C) 2013 by Bernhard Stroebl

        :param geom_type: geomTypes, valid argument is 0: point, 1: line, 2: polygon
        :type geom_type: int
        :param skip_active: step over active layer
        :type skip_active: bool
        :return: map of layers of given geometry type and their layer_id
        :rtype: dict
        """
        layer_list = {}
        layers = [tree_layer.layer() for tree_layer in QgsProject.instance().layerTreeRoot().findLayers()]
        for layer in layers:
            if 0 == layer.type():   # vectorLayer
                if skip_active and (self.iface.mapCanvas().currentLayer().id() == layer.id()):
                    continue
                else:
                    if geom_type is not None:
                        if isinstance(geom_type,  int):
                            if layer.geometryType() == geom_type:
                                layer_list[layer.name()] = layer.id()
                        else:
                            layer_list[layer.name()] = layer.id()
        return layer_list

    def set_output_layer_combobox(self, geom_type=None, item=''):
        """
        Populates the ComboBox with all layers of the given geometry type geom_type, and sets
        currentIndex to the entry named index.

        :param geom_type: geomTypes, valid argument is 0: point, 1: line, 2: polygon
        :type geom_type: int
        :param item: name of the ComboBox entry to set currentIndex to
        :type item: str
        :return: None
        """
        self.dlg.cb_output.clear()
        layer_list = self.get_vector_layers_by_type(geom_type, False)
        if len(layer_list) > 0:
            for index, aName in enumerate(layer_list):
                self.dlg.cb_output.addItem('')
                self.dlg.cb_output.setItemText(index, aName)
                if aName == item:
                    self.dlg.cb_output.setCurrentIndex(index)
        return None

    def default_values(self):
        """Prepares dialog properties

        :return: None
        """
        # empty table data
        self.table_data = []
        # set dialog widgets
        self.dlg.ls_layers.clearSpans()

        self.dlg.buttonBox.button(QDialogButtonBox.Ok).setDisabled(True)
        has_selected_features = False

        # check if an active layer exists
        active_layer_name = ''
        active_layer_index = None
        if self.iface.activeLayer() is not None:
            active_layer_name = self.iface.activeLayer().name()

        # all vector layers get added to the list
        current_layer_index = 0
        layers = [tree_layer.layer() for tree_layer in QgsProject.instance().layerTreeRoot().findLayers()]
        for layer in layers:
            if layer.type() == QgsMapLayer.VectorLayer:
                # if there are selected features toggle has_selected_features
                if layer.selectedFeatureCount():
                    has_selected_features = True

                # get field list for group by attribute selection combo boxes
                field_list = ['...']
                field_name_map = layer.dataProvider().fieldNameMap()
                field_index_map = {index: name for name, index in field_name_map.items()}
                field_list.extend([field_index_map[index] for index in range(0, len(field_name_map))])
                self.table_data.append([layer.name(), field_list])

                # select the active layer by default
                if layer.name() == active_layer_name:
                    active_layer_index = current_layer_index

                current_layer_index += 1

        # fill table with data
        self.dlg.ls_layers.model().setTableData(self.table_data)

        # select the active layer by default
        if active_layer_index is not None:
            self.dlg.ls_layers.setCurrentIndex(self.dlg.ls_layers.model().index(active_layer_index, 0))
            self.dlg.ls_layers.selectionModel().select(self.dlg.ls_layers.model().index(active_layer_index, 0),
                                                       QItemSelectionModel.ClearAndSelect | QItemSelectionModel.Rows)
            self.dlg.ls_layers.selectionModel().selectionChanged.emit(QItemSelection(), QItemSelection())

        # if at least one vector layer has selected features enable checkbutton cb_selected_only
        if has_selected_features:
            self.dlg.cb_selected_only.setEnabled(True)
            self.dlg.cb_selected_only.setChecked(True)
        else:
            self.dlg.cb_selected_only.setChecked(False)
            self.dlg.cb_selected_only.setEnabled(False)

        # initialize cb_output
        # remember the layer being selected the last time
        last_index = self.dlg.cb_output.currentText()

        # populate the combo box with polygon layers listed in the current legend
        self.set_output_layer_combobox(2, last_index)

        return None

    def update_cb_selected_only(self, layer):
        """Updates _Use selected features only_ checkbox according changed feature selection set in any map layer
        
        :param layer: map layer whose selection has changed
        :type layer: QgsVectorLayer
        :return: None
        """
        selected = layer.selectedFeatureCount()
        self.dlg.cb_selected_only.setEnabled(selected)
        self.dlg.cb_selected_only.setChecked(selected)

    def run(self):
        self.default_values()
        self.iface.mapCanvas().selectionChanged.connect(self.update_cb_selected_only)
        self.iface.mapCanvas().layersChanged.connect(self.default_values)
        # show the dialog
        self.dlg.show()

        # Run the dialog event loop
        result = self.dlg.exec_()

        return result

    def execute(self):
        """Execute method performs all the real work

        :return: None
        """
        # in the beginning [...] the earth was waste and void
        points = {}

        # get all selected rows in the order of selection (!)
        rows = self.dlg.ls_layers.selectionModel().selectedRows()

        # required operating mode is group by attribute
        if self.dlg.cb_group_by_attribute.isChecked():

            # iterate over selected layers
            for row in rows:
                # get layer name from 1st column
                col = row.siblingAtColumn(0)
                layer_name = self.dlg.ls_layers.model().data(col)

                # get map layer
                active_layer = QgsProject.instance().mapLayersByName(layer_name)[0]

                # index of selected field of source layer, minus 1 because of the first dummy entry in combobox list
                field_index = int(self.dlg.ls_layers.model().data(row, Qt.UserRole)) - 1

                # update points dict with unique values as keys and empty list as values
                for unique in active_layer.uniqueValues(field_index):
                    if unique not in points:
                        points[unique] = []

                # get all or the currently selected features of map layer according to state of cb_selected_only
                # convert each feature to points
                if active_layer.selectedFeatureCount() and self.dlg.cb_selected_only.checkState():
                    for feat in active_layer.selectedFeatures():
                        # check feature for valid geometry
                        if feat.hasGeometry():
                            points[feat[field_index]].extend(extract_points(feat.geometry()))
                else:
                    for feat in active_layer.getFeatures():
                        # check feature for valid geometry
                        if feat.hasGeometry():
                            points[feat[field_index]].extend(extract_points(feat.geometry()))

            # some plausibility checks before proceeding
            num_points = 0
            values_to_remove = []
            for value, point_list in points.items():
                num_points_group = len(point_list)

                # processing is aborted due to given point set is empty
                if num_points_group == 0:
                    msg = self.msg_bar.createMessage(self.tr(
                        'Empty point set for group {}. Excluded from processing'.
                        format(value)))
                    self.msg_bar.pushWidget(msg, Qgis.Warning, 2)
                    values_to_remove.append(value)

                # processing is aborted due to given point set has less than 3 points (the minimum required to
                # build some polygon geometry (if group by attribute is selected, it will not be checked,
                # if these 3 points fall into the same group
                elif num_points_group < 3:
                    msg = self.msg_bar.createMessage(self.tr(
                        'A minimum of 3 points is required ({} are given for group {}). Excluded from processing'.
                        format(int(num_points_group), value)))
                    self.msg_bar.pushWidget(msg, Qgis.Warning, 2)
                    values_to_remove.append(value)

                # processing is aborted due to given point set has less points than
                # the minimum required neighbors specified for the concave hull construction algorithm
                elif num_points_group < self.dlg.sb_neighbors.value():
                    msg = self.msg_bar.createMessage(self.tr(
                        '{} neighbors are required, {} input points are given for group {}. \
                        Excluded from processing'.
                        format(int(self.dlg.sb_neighbors.value()), int(num_points_group), value)))
                    self.msg_bar.pushWidget(msg, Qgis.Warning, 2)
                    values_to_remove.append(value)

                else:
                    num_points += num_points_group

            # remove unique value groups without any element (eg no such selected feature was selected)
            if values_to_remove:
                for value in values_to_remove:
                    points.pop(value)

        # normal operating mode w/o group by attribute
        else:
            # initialize default group
            points[None] = []

            # iterate over selected layers
            for row in rows:
                # get layer name from 1st column
                col = row.siblingAtColumn(0)
                layer_name = self.dlg.ls_layers.model().data(col)

                # get map layer
                active_layer = QgsProject.instance().mapLayersByName(layer_name)[0]

                # get all or the currently selected features of map layer according to state of cb_selected_only
                # convert each feature to points
                if active_layer.selectedFeatureCount() and self.dlg.cb_selected_only.checkState():
                    for feat in active_layer.selectedFeatures():
                        # check feat on valid geometry
                        if feat.hasGeometry():
                            points[None].extend(extract_points(feat.geometry()))
                else:
                    for feat in active_layer.getFeatures():
                        # check feat on valid geometry
                        if feat.hasGeometry():
                            points[None].extend(extract_points(feat.geometry()))

            # some plausibility checks before proceeding
            num_points = len(points[None])

            # processing is aborted due to given point set is empty
            if num_points == 0:
                msg = self.msg_bar.createMessage(self.tr('Empty point set. Processing aborted'))
                self.msg_bar.pushWidget(msg, Qgis.Critical, 2)
                return None

            # processing is aborted due to given point set has less than 3 points (the minimum required to
            # build some polygon geometry (if group by attribute is selected, it will not be checked,
            # if these 3 points fall into the same group
            if num_points < 3:
                msg = self.msg_bar.createMessage(self.tr(
                    'A minimum of 3 points is required ({} are given). Processing aborted'.
                    format(int(num_points))))
                self.msg_bar.pushWidget(msg, Qgis.Critical, 2)
                return None

            # processing is aborted due to given point set has less points than
            # the minimum required neighbors specified for the concave hull construction algorithm
            if num_points < self.dlg.sb_neighbors.value():
                msg = self.msg_bar.createMessage(self.tr(
                    '{} neighbors are required, {} input points are given. Processing aborted'.
                    format(int(self.dlg.sb_neighbors.value()), int(num_points))))
                self.msg_bar.pushWidget(msg, Qgis.Critical, 2)
                return None

        # send WARNING to the message bar to inform about a probably long running time
        if num_points > 1000:
            msg = self.msg_bar.createMessage(self.tr(
                'Please be patient, processing of more then {} points may take a while'.
                format(int(num_points))))
            self.msg_bar.pushWidget(msg, Qgis.Warning, 2)

        # if more then 5000 points ask user to confirm
        if num_points > 5000:
            proceed = QMessageBox.question(None, self.tr(
                'Please confirm'), self.tr('Do you really want to proceed?'),
                QMessageBox.Yes | QMessageBox.No)
            if proceed == QMessageBox.No:
                QApplication.instance().setOverrideCursor(Qt.ArrowCursor)
                return None

        # get properties of the field grouping is based on. Properties are copied from the indicated field
        # of the first selected layers (in the order of selecting!). If more layers are selected no check will
        # be performed if properties of their selected fields differ from the first selected one
        if self.dlg.cb_group_by_attribute.isChecked():

            # get first selected row (layer)
            row = self.dlg.ls_layers.selectionModel().selectedRows()[0]

            # get layer name from 1st column
            col = row.siblingAtColumn(0)
            layer_name = self.dlg.ls_layers.model().data(col)

            # get map layer
            field_template_layer = QgsProject.instance().mapLayersByName(layer_name)[0]

            # index of selected field of source layer, minus 1 because of the first dummy entry in combobox list
            field_index = int(self.dlg.ls_layers.model().data(row, Qt.UserRole)) - 1

            # get field properties
            field_name = field_template_layer.fields().field(field_index).name()
            field_type = field_template_layer.fields().field(field_index).type()
            field_length = field_template_layer.fields().field(field_index).length()
            field_precision = field_template_layer.fields().field(field_index).precision()

            if field_type == QVariant.Int:
                field_type_str = 'integer'
            elif field_type == QVariant.Double:
                field_type_str = 'float'
            else:
                field_type_str = 'string({})'.format(field_length)

            # append the field definition to clause
            self.field_clause += '&field={}:{}'.format(field_name, field_type_str)

        # change cursor to inform user about ongoing processing
        QApplication.instance().setOverrideCursor(Qt.BusyCursor)

        # create_output_layer creates the output layer _before_ generating hull geometries
        self.create_output_layer()

        # generate create hull geometry, optionally based on clustering
        # and/or field classification according unique values
        created_hulls = 0

        # points not being member of some cluster
        noise_points = []

        # for output of clustered points only
        if self.dlg.cb_output_clusters_only.isChecked():
            _clusters = {}

        # process points with prior clustering
        if self.dlg.gb_clustering.isChecked():

            # iterate over grouped input points
            for value, geom in points.items():

                # generate point clusters
                ssnc = SSNClusters(geom, self.dlg.sb_neighborhood_list_size.value())
                clusters = ssnc.get_clusters()

                # check how many points are not member of some cluster (lower k-neighbours required)
                _noise_points = ssnc.get_outliers()
                if _noise_points:
                    noise_points.extend(_noise_points)

                # normal operation: cluster points and generate concave hull
                if not self.dlg.cb_output_clusters_only.isChecked():

                    # iterate over each individual hull
                    for cluster in clusters.keys():
                        try:
                            the_hull = concave_hull(clusters[cluster], self.dlg.sb_neighbors.value())
                            if the_hull:
                                # append hull geometry one by one before proceeding with next cluster
                                hulls = self.create_output_feature([[as_polygon(the_hull), len(clusters[cluster])]], created_hulls, value)
                                created_hulls += hulls

                        except:
                            if self.dlg.cb_group_by_attribute.isChecked():
                                msg = self.msg_bar.createMessage(
                                    self.tr('Error creating concave hull for group {}'.format(value)))
                            else:
                                msg = self.msg_bar.createMessage(self.tr('Error creating concave hull'))
                            self.msg_bar.pushWidget(msg, Qgis.Critical, 5)

                # output of clustered points only
                else:
                    _clusters[value] = clusters

            # check how many points are not member of some cluster (lower k-neighbours required)
            if _noise_points:
                msg = self.msg_bar.createMessage(self.tr(
                    '{} noise points not added to any cluster'.format(len(noise_points))))
                self.msg_bar.pushWidget(msg, Qgis.Info, 2)

                if self.dlg.cb_output_noise_points.isChecked():
                    self.create_noise_layer(noise_points)

            # report result in QGIS message bar and return for mode clustered points only
            if self.dlg.cb_output_clusters_only.isChecked():

                success = self.create_cluster_layer(_clusters)

                # reset cursor
                QApplication.instance().setOverrideCursor(Qt.ArrowCursor)

                if success:
                    msg = self.msg_bar.createMessage(self.tr('Clustered points layer created'))
                    self.msg_bar.pushWidget(msg, Qgis.Info, 5)

                else:
                    msg = self.msg_bar.createMessage(self.tr('Creating clustered points layer failed'))
                    self.msg_bar.pushWidget(msg, Qgis.Warning, 5)

                return success

        # process points without clustering
        else:

            for value, geom in points.items():

                try:
                    the_hull = concave_hull(geom, self.dlg.sb_neighbors.value())

                    # append hull geometry one by one before proceeding with next cluster
                    hulls = self.create_output_feature([[as_polygon(the_hull), len(geom)]], created_hulls, value)
                    created_hulls += hulls

                except:
                    if self.dlg.cb_group_by_attribute.isChecked():
                        msg = self.msg_bar.createMessage(
                            self.tr('Error creating concave hull for group {}'.format(value)))
                    else:
                        msg = self.msg_bar.createMessage(self.tr('Error creating concave hull'))
                    self.msg_bar.pushWidget(msg, Qgis.Critical, 5)

        # write and close layer after generation of last hull geometry
        success = self.close_output_layer()

        # reset cursor
        QApplication.instance().setOverrideCursor(Qt.ArrowCursor)

        # report result in QGIS message bar
        if success and created_hulls:
            msg = self.msg_bar.createMessage(self.tr('{} Concave hulls created successfully'.format(int(created_hulls))))
            self.msg_bar.pushWidget(msg, Qgis.Success, 5)

        elif created_hulls == 0:
            msg = self.msg_bar.createMessage(self.tr('No concave hull was created'))
            self.msg_bar.pushWidget(msg, Qgis.Info, 5)

        elif not success:
            msg = self.msg_bar.createMessage(self.tr('Creating/editing concave hulls layer failed'))
            self.msg_bar.pushWidget(msg, Qgis.Warning, 5)

        return success

    def apply(self):
        """Apply algorithms without closing the dialog. This allows multiple executions with same settings.
        """
        self.execute()
        return None

    def accepted(self):
        """Apply algorithms and close the dialog
        """
        self.iface.mapCanvas().selectionChanged.disconnect(self.update_cb_selected_only)
        self.iface.mapCanvas().layersChanged.disconnect(self.default_values)
        self.execute()
        self.dlg.close()
        return None

    def rejected(self):
        """Close dialog without any processing
        """
        self.iface.mapCanvas().selectionChanged.disconnect(self.update_cb_selected_only)
        self.iface.mapCanvas().layersChanged.disconnect(self.default_values)
        self.dlg.close()
        return None
