# -*- coding: utf-8 -*-
"""
/***************************************************************************
 CommutingAnalysis
                                 A QGIS plugin
 This plugin analysis and visualises commuting data.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2018-11-23
        git sha              : $Format:%H$
        copyright            : (C) 2018-2024 by Kristian Bergstrand
        email                : kristian.bergstrand@sweco.se
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt5.QtCore import QSettings, QTranslator, QVariant, qVersion, QCoreApplication, QObject
from PyQt5 import QtGui
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QAction, QFileDialog, QWidget
from qgis.core import Qgis, QgsField, QgsFeature, QgsFeatureRequest, \
    QgsProject, QgsMapLayer, \
    QgsVectorLayer, QgsSymbol, QgsRendererRange, QgsGraduatedSymbolRenderer, QgsPointXY, QgsPoint, QgsGeometry, \
    QgsProcessingUtils, QgsProcessingFeatureSourceDefinition
from qgis.analysis import QgsGraphBuilder, QgsGraphAnalyzer, QgsVectorLayerDirector, \
    QgsNetworkDistanceStrategy, QgsNetworkSpeedStrategy
import processing
import sqlite3
import os, time

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .commuting_analysis_dialog import CommutingAnalysisDialog
import os.path

class CommutingAnalysis:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'CommutingAnalysis_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Create the dialog (after translation) and keep reference
        self.dlg = CommutingAnalysisDialog()

        # Declare instance attributes
        self.actions = []
        self.my_range_list = []
        self.result_lyr = []

        self.points_a = None
        self.tbl_data_a = None
        self.points_b = None

        self.points_b = None
        self.network_b = None
        self.tbl_data_b = None

        self.geometry_c = None
        self.tbl_data_c = None

        self.menu = self.tr(u'&Commuting Analysis')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'CommutingAnalysis')
        self.toolbar.setObjectName(u'CommutingAnalysis')

        self.warning_a = self.tr(u'Field types of A1, B1 and B2 may not match!')
        self.warning_b = self.tr(u'Field types of A1, C1 and C2 may not match!')
        self.warning_c = self.tr(u'Field types of A1, B1 and B2 may not match!')
        self.warning_c_2 = self.tr(u'Selection on polygon grid layer is mandatory!')
        self.info_running = self.tr(u'Process running...')

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('CommutingAnalysis', message)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """
        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)
        return action

    """ Populate one or more comboboxes.
        @params layer_name <string>
                cbs <list> Comboboxes to update.
        @returns <void> 
    """
    def populate_cb(self, layer_name, cbs):
        layer = QgsProject.instance().mapLayersByName(layer_name)[0]
        prov = layer.dataProvider()
        field_names = [field.name() for field in prov.fields()]
        for cb in cbs:
            cb.clear()
            cb.addItems(field_names)

    """ Populate combo box cb_id_points_a with the fields from selected layer in cb_points_a.
        @params
        @returns <void> 
    """
    def populate_points_a(self):
        c = self.dlg.cb_points_a.currentText()
        if not c or (c and self.points_a and c == self.points_a):
            return
        self.populate_cb(c, [self.dlg.cb_id_points_a])
        self.points_a = c

    """ Populate combo boxes cb_from_a, cb_to_a and cb_nbr_a with the fields from selected layer in cb_tbl_data_a.
        @params
        @returns <void> 
    """
    def populate_tbl_data_a(self):
        c = self.dlg.cb_tbl_data_a.currentText()
        if not c or (c and self.tbl_data_a and c == self.tbl_data_a):
            return
        self.populate_cb(c, [self.dlg.cb_from_a, self.dlg.cb_to_a, self.dlg.cb_nbr_a])
        self.tbl_data_a = c

    """ Populate combo box cb_id_points_b from selected layer in cb_points_b.
        @params
        @returns <void> 
    """
    def populate_points_b(self):
        c = self.dlg.cb_points_b.currentText()
        if not c or (c and self.points_b and c == self.points_b):
            return
        self.populate_cb(c, [self.dlg.cb_id_points_b])
        self.points_b = c

    """ Populate combo box cb_speed_b from selected layer in cb_network_b.
        @params
        @returns <void> 
    """
    def populate_network_b(self):
        c = self.dlg.cb_network_b.currentText()
        if not c or (c and self.network_b and c == self.network_b):
            return
        self.populate_cb(c, [self.dlg.cb_speed_b])
        self.network_b = c

    """ Populate combo boxes cb_from_b, cb_to_b and cb_nbr_b from selected layer in cb_tbl_data_b.
        @params
        @returns <void> 
    """
    def populate_tbl_data_b(self):
        c = self.dlg.cb_tbl_data_b.currentText()
        if not c or (c and self.tbl_data_b and c == self.tbl_data_b):
            return
        self.populate_cb(c, [self.dlg.cb_from_b, self.dlg.cb_to_b, self.dlg.cb_nbr_b])
        self.tbl_data_b = c

    """ Populate combo box cb_geometry_id_c from selected layer in cb_geometry_c.
        @params
        @returns <void> 
    """
    def populate_squares_c(self):
        c = self.dlg.cb_geometry_c.currentText()
        if not c or (c and self.geometry_c and c == self.geometry_c):
            return
        self.populate_cb(c, [self.dlg.cb_geometry_id_c])
        self.geometry_c = c

    """ Populate combo boxes cb_live_c, cb_work_c and cb_value_c from selected layer in cb_tbl_data_c.
        @params
        @returns <void> 
    """
    def populate_tbl_data_c(self):
        c = self.dlg.cb_tbl_data_c.currentText()
        if not c or (c and self.tbl_data_c and c == self.tbl_data_c):
            return
        self.populate_cb(c, [self.dlg.cb_live_c, self.dlg.cb_work_c, self.dlg.cb_value_c])
        self.tbl_data_c = c

    """Create the menu entries and toolbar icons inside the QGIS GUI."""
    def initGui(self):
        icon_path = ':/plugins/commuting_analysis/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Commuting analysis'),
            callback=self.run,
            parent=self.iface.mainWindow())
        # A
        self.dlg.tb_points_a.clicked.connect(self.open_points_a)
        self.dlg.tb_tbl_data_a.clicked.connect(self.open_tbl_data_a)
        self.dlg.tb_network_a.clicked.connect(self.open_network_a)
        # B
        self.dlg.tb_points_b.clicked.connect(self.open_points_b)
        self.dlg.tb_network_b.clicked.connect(self.open_network_b)
        self.dlg.tb_tbl_data_b.clicked.connect(self.open_tbl_data_b)
        # C
        self.dlg.tb_geometry_c.clicked.connect(self.open_squares_c)
        self.dlg.tb_tbl_data_c.clicked.connect(self.open_tbl_data_c)
        # A
        self.dlg.cb_points_a.currentIndexChanged.connect(self.populate_points_a)
        self.dlg.cb_id_points_a.currentIndexChanged.connect(self.check_field_types_a)
        self.dlg.cb_from_a.currentIndexChanged.connect(self.check_field_types_a)
        self.dlg.cb_to_a.currentIndexChanged.connect(self.check_field_types_a)
        self.dlg.cb_network_a.currentIndexChanged.connect(self.check_field_types_a)
        self.dlg.cb_tbl_data_a.currentIndexChanged.connect(self.populate_tbl_data_a)
        # B
        self.dlg.cb_points_b.currentIndexChanged.connect(self.populate_points_b)
        self.dlg.cb_id_points_b.currentIndexChanged.connect(self.check_field_types_b)
        self.dlg.cb_from_b.currentIndexChanged.connect(self.check_field_types_b)
        self.dlg.cb_to_b.currentIndexChanged.connect(self.check_field_types_b)
        self.dlg.cb_network_b.currentIndexChanged.connect(self.populate_network_b)
        self.dlg.cb_tbl_data_b.currentIndexChanged.connect(self.populate_tbl_data_b)
        # C
        self.dlg.cb_geometry_c.currentIndexChanged.connect(self.populate_squares_c)
        self.dlg.cb_geometry_id_c.currentIndexChanged.connect(self.check_field_types_c)
        self.dlg.cb_live_c.currentIndexChanged.connect(self.check_field_types_c)
        self.dlg.cb_work_c.currentIndexChanged.connect(self.check_field_types_c)
        self.dlg.cb_tbl_data_c.currentIndexChanged.connect(self.populate_tbl_data_c)

        # Tab widget.
        self.dlg.tabWidget.currentChanged.connect(self.check_tab)
        self.dlg.pb_ok.clicked.connect(self.check_selection_c)

    def check_selection_c(self):
        if self.dlg.tabWidget.currentIndex() == 2:
            try:
                fts = QgsProject.instance().mapLayersByName(self.dlg.cb_geometry_c.currentText())[0].selectedFeatures()
            except IndexError:
                return
            if len(fts) == 0:
                self.dlg.txt_c_2.setPlainText(self.warning_c_2)
                return
            self.dlg.txt_c_2.setPlainText('')
        self.process_commuting()        
        #self.dlg.accept()

    """ Creates a symbology range.
        @params min_val, max_val, label, colour, opacity
        @returns <void> 
    """
    def make_symbology_range(self, min_val, max_val, label, colour, opacity):
        range_colour = QtGui.QColor(colour)
        range_symbol = QgsSymbol.defaultSymbol(2)
        range_symbol.setColor(range_colour)
        range_symbol.setOpacity(opacity)
        range_to_append = QgsRendererRange(min_val, max_val, range_symbol, label)
        self.my_range_list.append(range_to_append)

    """ Creates a symbology.
        @params include_selection <boolean>
        @returns <void> 
    """
    def make_symbology(self, include_selection):
        self.my_range_list = []
        # label:min,max,colour,opacity
        if include_selection:
            dict_ranges = {
                '0 SELECTION': [0, 0.00001, '#000000', 1],
                '1': [0.00002, 1, '#f7fcf0', 1],
                '2': [1.1, 2, '#e0f3db', 1],
                '3': [2.1, 3, '#ccebc5', 1],
                '4': [3.1, 4, '#a8ddb5', 1],
                '5-6': [4.1, 6, '#7bccc4', 1],
                '7-9': [6.1, 9, '#4eb3d3', 1],
                '10-19': [9.1, 19, '#2b8cbe', 1],
                '20-29': [19.1, 29, '#0868ac', 1],
                '30-49': [29.1, 49, '#084081', 1],
                '50-69': [49.1, 69, '#084087', 1],
                '70-99': [69.1, 99, '#084088', 1],
                '100-2000': [99.1, 2000, '#084089', 1]
            }
        else:
            dict_ranges = {
                '1': [0.00001, 1, '#f7fcf0', 1],
                '2': [1.1, 2, '#e0f3db', 1],
                '3': [2.1, 3, '#ccebc5', 1],
                '4': [3.1, 4, '#a8ddb5', 1],
                '5-6': [4.1, 6, '#7bccc4', 1],
                '7-9': [6.1, 9, '#4eb3d3', 1],
                '10-19': [9.1, 19, '#2b8cbe', 1],
                '20-29': [19.1, 29, '#0868ac', 1],
                '30-49': [29.1, 49, '#084081', 1],
                '50-69': [49.1, 69, '#084087', 1],
                '70-99': [69.1, 99, '#084088', 1],
                '100-2000': [99.1, 2000, '#084089', 1]
            }

        for k, v in dict_ranges.items():
            self.make_symbology_range(v[0], v[1], k, v[2], v[3])

        my_target_field = 'preTotal'
        my_renderer = QgsGraduatedSymbolRenderer('', self.my_range_list)
        my_renderer.setMode(QgsGraduatedSymbolRenderer.EqualInterval)
        my_renderer.setClassAttribute(my_target_field)
        self.result_lyr.setRenderer(my_renderer)
        QgsProject.instance().addMapLayer(self.result_lyr)

    """ Part C. Grid cells. """
    def run_part_c(self):
        # Fetch geometry layer.
        try:
            polygons_data_name = self.dlg.cb_geometry_c.currentText()
            polygons_data_lyr = QgsProject.instance().mapLayersByName(polygons_data_name)[0]
        except IndexError:
            return
        commute_data_name = self.dlg.cb_tbl_data_c.currentText()

        # Get fields from layers.
        polygon_id_field = self.dlg.cb_geometry_id_c.currentText()
        live_code_field = self.dlg.cb_live_c.currentText()
        work_code_field = self.dlg.cb_work_c.currentText()
        value_field = self.dlg.cb_value_c.currentText()

        # Get selected features
        fts = polygons_data_lyr.selectedFeatures()
        if len(fts) == 0:
            return
        a = [k[polygon_id_field] for k in fts]
        sel_squares = ','.join(a)

        if self.dlg.cb_analysis_c.currentIndex() == 1:
            # OUT-commuting
            query = """ 
                SELECT "{0}" as "{5}", sum("{2}") as Total /*:real*/ 
                FROM "{4}" 
                WHERE "{1}" IN({3}) AND "{0}" NOT IN ({3}) 
                GROUP BY "{0}" 
                """.format(work_code_field, live_code_field, value_field, sel_squares,
                           commute_data_name, polygon_id_field)
        else:
            # IN-commuting
            query = """ 
                SELECT "{1}" as "{5}", sum("{2}") as Total /*:real*/ 
                FROM "{4}" 
                WHERE "{1}" NOT IN({3}) AND "{0}" IN ({3}) 
                GROUP BY "{1}" 
                """.format(work_code_field, live_code_field, value_field, sel_squares,
                           commute_data_name, polygon_id_field)

        vlayer = QgsVectorLayer("?query={}".format(query), "vlayer", "virtual")
        vlayer_to_use = vlayer
        include_selection = False
        QgsProject.instance().addMapLayer(vlayer)
        if self.dlg.cbox_sel_c.isChecked():
            vlayer_name = vlayer.name()
            query2 = """ SELECT "{1}", Total /*:real*/ 
                         FROM "{2}" 
                         UNION 
                         SELECT "{1}", 0 as Total /*:real*/ 
                         FROM "{3}" 
                         WHERE "{1}" IN({0}) """.format(sel_squares, polygon_id_field, vlayer_name, polygons_data_name)

            vlayer2 = QgsVectorLayer("?query={}".format(query2), "vlayer", "virtual")
            vlayer_to_use = vlayer2
            include_selection = True
            QgsProject.instance().addMapLayer(vlayer2)

        parameters = {'INPUT': polygons_data_lyr, 'FIELD': polygon_id_field, 'INPUT_2': vlayer_to_use,
                      'FIELD_2': polygon_id_field,
                      'FIELDS_TO_COPY': 'Total', 'METHOD': 0, 'DISCARD_NONMATCHING': True, 'PREFIX': 'pre',
                      'OUTPUT': 'memory:resultLayer'}

        result = processing.run('qgis:joinattributestable', parameters)
        QgsProject.instance().removeMapLayer(vlayer)
        if include_selection:
            QgsProject.instance().removeMapLayer(vlayer2)
        self.result_lyr = result['OUTPUT']
        self.make_symbology(include_selection)

    """ Check if layers are added and field types for part A are OK.
        @params
        @returns <void> 
    """
    def check_field_types_a(self):
        # Try fetching geometry layer and commuting table data.
        try:
            point_lyr = QgsProject.instance().mapLayersByName(self.dlg.cb_points_a.currentText())[0]
            commute_data = QgsProject.instance().mapLayersByName(self.dlg.cb_tbl_data_a.currentText())[0]
            network_a = QgsProject.instance().mapLayersByName(self.dlg.cb_network_a.currentText())[0]
        except IndexError:
            return
        try:
            if point_lyr.fields().field(self.dlg.cb_id_points_a.currentText()).type() == \
                    commute_data.fields().field(self.dlg.cb_from_a.currentText()).type() == \
                    commute_data.fields().field(self.dlg.cb_to_a.currentText()).type():
                self.dlg.txt_a.setPlainText('')
                pass
            else:
                self.dlg.txt_a.setPlainText(self.warning_a)
                return
        except KeyError:
            return

    """ Check if layers are added and field types for part B are OK.
        @params
        @returns <void> 
    """
    def check_field_types_b(self):
        # Try fetching geometry layer, road network layer and commuting table data.
        try:
            point_lyr_b = QgsProject.instance().mapLayersByName(self.dlg.cb_points_b.currentText())[0]
            network_lyr = QgsProject.instance().mapLayersByName(self.dlg.cb_network_b.currentText())[0]
            commute_data_b = QgsProject.instance().mapLayersByName(self.dlg.cb_tbl_data_b.currentText())[0]
        except IndexError:
            return

        try:
            if point_lyr_b.fields().field(self.dlg.cb_id_points_b.currentText()).type() == \
                    commute_data_b.fields().field(self.dlg.cb_from_b.currentText()).type() == \
                    commute_data_b.fields().field(self.dlg.cb_to_b.currentText()).type():
                self.dlg.txt_b.setPlainText('')
            else:
                self.dlg.txt_b.setPlainText(self.warning_b)
                return
        except KeyError:
            return

    """ Check if layers are added and field types for part C are OK.
        @params
        @returns <void> 
    """
    def check_field_types_c(self):
        # Try fetching geometry layer, road network layer and commuting table data.
        try:
            polygons_data_lyr = QgsProject.instance().mapLayersByName(self.dlg.cb_geometry_c.currentText())[0]
            commute_data_c = QgsProject.instance().mapLayersByName(self.dlg.cb_tbl_data_c.currentText())[0]
        except IndexError:
            return
        try:
            if polygons_data_lyr.fields().field(self.dlg.cb_geometry_id_c.currentText()).type() == \
                    commute_data_c.fields().field(self.dlg.cb_live_c.currentText()).type() == \
                    commute_data_c.fields().field(self.dlg.cb_work_c.currentText()).type():
                self.dlg.txt_c.setPlainText('')
            else:
                self.dlg.txt_c.setPlainText(self.warning_c)
                return
        except KeyError:
            return

    """ Part A. Clusters. """
    def run_part_a(self):
        # Fetch geometry layer, commuting table data and network.
        try:
            point_lyr = QgsProject.instance().mapLayersByName(self.dlg.cb_points_a.currentText())[0]
            commute_data = QgsProject.instance().mapLayersByName(self.dlg.cb_tbl_data_a.currentText())[0]
            network_a = QgsProject.instance().mapLayersByName(self.dlg.cb_network_a.currentText())[0]
        except IndexError:
            return

        # Get fields from layers.
        points_code_field_a = self.dlg.cb_id_points_a.currentText()
        from_code_field = self.dlg.cb_from_a.currentText()
        to_code_field = self.dlg.cb_to_a.currentText()
        value_field = self.dlg.cb_nbr_a.currentText()

        # feature_list will be populated from the forms data and feature_list_2 from geometry data.
        feature_list = []
        feature_list2 = []

        for feature in commute_data.getFeatures():
            new_feature = (feature[from_code_field], feature[to_code_field], feature[value_field])
            feature_list.append(new_feature)

        for feature in point_lyr.getFeatures():
            new_feature = (feature[points_code_field_a])
            feature_list2.append([new_feature])        

        dynamic_param = self.dlg.sb_advparam_a.value()
        dynamic_param_b = self.dlg.sb_advparam_b.value()
        dynamic_param_c = self.dlg.sb_advparam_c.value()
        
        # Create an in-memory sqlite database to handle the form data
        db_conn = sqlite3.connect(":memory:")
        cur = db_conn.cursor()
        cur.execute(""" CREATE TABLE commutePoints(pointcode text NOT NULL) """)
        cur.executemany(""" INSERT INTO commutePoints VALUES (?) """, feature_list2)
        cur.execute(""" CREATE TABLE commute(from_code text NOT NULL, to_code text, commutevalue INT NOT NULL) """)
        cur.executemany(""" INSERT INTO commute VALUES (?,?,?) """, feature_list)
        
        # From the tables in the in-memory sqlite database, get the right results
        # q0: keep from_codes from table data where we have a match in the geometry layer
        # q1: group result from q0 by from_code and sum the value for number of commutants.
        # q2: calculate the percentage for each commute relation from-to in the table data. Only where
        # we have a match in the geometry-layer for to-code.
        # q3: look if we have any from-to relation over specified threshold. If not, mark the from_code as
        # being a LA. If we have ONE relation, then we mark the data of the to_code.
        # q4 and q5: if we have TWO relations, keep information on those two and which is the main to_code.
        # q6: structure all the data in a reasonable way
        # Finally select all from q6.

        cur.execute(""" 
        WITH q0 AS( 
            SELECT * 
            FROM commute 
            WHERE from_code IN(SELECT pointcode FROM commutePoints) 
        ), q1 AS( 
            SELECT SUM(commutevalue) AS nbrSum, from_code 
            FROM q0 
            GROUP BY from_code 
        ), q2 AS( 
            SELECT SUM(commutevalue) AS nbrSum2, from_code 
            FROM q0 
            WHERE from_code <> to_code 
            GROUP BY from_code 
        ), q3 AS( 
            SELECT t.from_code, t.to_code, 100.0 * t.commutevalue / aa.nbrSum AS Andel, 

            CASE WHEN (100.0 * t.commutevalue / aa.nbrSum) > {0}   
                 AND t.from_code <> t.to_code THEN t.to_code ELSE NULL END AS tempLAS

            ,CASE WHEN (100.0 * aaa.nbrSum2 / aa.nbrSum) > {1}
                 AND t.from_code <> t.to_code THEN 1 ELSE NULL END AS threshold_all
            FROM q0 t 
            INNER JOIN q1 aa ON t.from_code = aa.from_code
            INNER JOIN q2 aaa ON aa.from_code = aaa.from_code 
            WHERE t.to_code IN(SELECT pointcode FROM commutePoints) 
        ), q4 AS( 
            SELECT q3.from_code, count(q3.tempLAS) as count, 
            CASE WHEN ( count(q3.tempLAS) = 0 AND count(q3.threshold_all) = 0 ) THEN 'LA'
            WHEN ( count(q3.tempLAS) = 0 AND count(q3.threshold_all) > 0 ) THEN 'Solitare' 
            END AS LA 
            FROM q3 
            GROUP BY q3.from_code 
        ), q5 AS(            
            SELECT *, CASE WHEN commute_rank = 1 THEN tempLAS ELSE NULL END AS firstLAS
                    , CASE WHEN commute_rank = 2 THEN tempLAS ELSE NULL END AS secLAS 
            FROM (
                SELECT from_code, to_code, Andel, tempLAS
                    ,row_number() OVER (PARTITION BY from_code ORDER BY Andel DESC) AS commute_rank 
                FROM q3
                WHERE tempLAS is not null
            ) ranks
            WHERE commute_rank <= 2
        ), q6 AS( 
            SELECT from_code, group_concat(firstLAS) AS firstLAS2, group_concat(secLAS) AS secLAS2
            FROM q5
            GROUP BY from_code
        ), q7 AS(
            SELECT ii.from_code, ii.firstLAS2 AS commonLA
            FROM q6 ii
            INNER JOIN q6 iii ON ii.firstLAS2 = iii.from_code AND ii.from_code = iii.firstLAS2
        ), q8 AS( 
            SELECT q4.from_code, COALESCE(q4.LA, q6.firstLAS2) AS fLAS, q6.secLAS2, COALESCE(q7.commonLA,NULL) AS commonLA
            FROM q4 
            LEFT JOIN q6 ON q4.from_code = q6.from_code 
            LEFT JOIN q7 ON q4.from_code = q7.from_code
        ) 
        SELECT * from q8
        """.format(dynamic_param,dynamic_param_b))

        # Fetch result from the query
        #,CASE WHEN (((100.0 * t.commutevalue / aa.nbrSum) < {0}) AND ((100.0 * aaa.nbrSum2 / aa.nbrSum) > {1})) 
        #AND t.from_code <> t.to_code THEN 1 ELSE NULL END AS solitary
        sqlite_result = cur.fetchall()
        list_tmp = []
        temporary_list = []
        for rec in sqlite_result:
            y = list(rec)
            list_tmp.append(y)
            temporary_list.append(y[0])
        
        # Put all the from_codes in the list into a string with each member quoted.
        list_twns = ','.join('"{0}"'.format(w) for w in temporary_list)
        point_lyr_id = point_lyr.id()

        # Prepare a query for virtual layer creation
        query_vl = """ SELECT * from '{0}' WHERE {1} IN ({2})  """.format(
            point_lyr_id, points_code_field_a, list_twns)

        # Create virtual layer
        vlayer = QgsVectorLayer("?query={}".format(query_vl), "vlayer", "virtual")

        # Parameters for joinattributestable algorithm
        parameters = {'INPUT': point_lyr, 'FIELD': points_code_field_a,
                      'INPUT_2': vlayer, 'FIELD_2': points_code_field_a,
                      'FIELDS_TO_COPY': points_code_field_a,
                      'METHOD': 0, 'DISCARD_NONMATCHING': True, 'PREFIX': 'pre', 'OUTPUT': 'memory:conurbations'}

        # Join geometry layer and the virtual layer
        result = processing.run('qgis:joinattributestable', parameters)
        # rl is the resultlayer, eventually to be a output for the user.
        rl = result['OUTPUT']
        pr = rl.dataProvider()
        rl.startEditing()
        pr.addAttributes([QgsField("LA", QVariant.String),
                          QgsField("Sec_LA", QVariant.String),
                          QgsField("LAS1", QVariant.String),
                          QgsField("LAS2", QVariant.String)])
        field_index_la = pr.fieldNameIndex("LA")
        field_index_sec_la = pr.fieldNameIndex("Sec_LA")
        field_index_sec_las1 = pr.fieldNameIndex("LAS1")
        field_index_sec_las2 = pr.fieldNameIndex("LAS2")
        rl.commitChanges()
        rl.startEditing()
        la_dict = {}
        las_dict = {}
        common_la = []

        for f in list_tmp:
            from_code = f[0]
            point_type = f[1]
            sec_las = f[2]
            common_las = f[3]

            if common_las and from_code not in common_la:
                sec_las = None
                point_type = 'LA'
                common_la.append(common_las)
                
            elif common_las and from_code in common_la:
                sec_las = None
                point_type = common_las
                           
            if sec_las:
                query = " \"{0}\" = '{1}'".format(points_code_field_a, sec_las)
                selection = rl.getFeatures(QgsFeatureRequest().setFilterExpression(query))
                for feature in selection:
                    the_id = feature.id()
                    rl.changeAttributeValue(the_id, field_index_sec_la, sec_las)
                    if sec_las in common_la:
                        rl.changeAttributeValue(the_id, field_index_la, sec_las)
                    if sec_las not in las_dict:
                        geom = feature.geometry()
                        point_geom = geom.asPoint()
                        las_dict[sec_las] = point_geom
                query2 = " \"{0}\" = '{1}'".format(points_code_field_a, from_code)
                selection = rl.getFeatures(QgsFeatureRequest().setFilterExpression(query2))
                for feature in selection:
                    the_id = feature.id()
                    rl.changeAttributeValue(the_id, field_index_sec_las1, point_type)
                    rl.changeAttributeValue(the_id, field_index_sec_las2, sec_las)
            elif point_type == 'LA':
                query = " \"{0}\" = '{1}'".format(points_code_field_a, from_code)
                selection = rl.getFeatures(QgsFeatureRequest().setFilterExpression(query))
                for feature in selection:
                    the_id = feature.id()
                    rl.changeAttributeValue(the_id, field_index_la, from_code)
                    if from_code not in la_dict:
                        geom = feature.geometry()
                        point_geom = geom.asPoint()
                        la_dict[from_code] = point_geom
            elif point_type == 'Solitare':
                query = " \"{0}\" = '{1}'".format(points_code_field_a, from_code)
                selection = rl.getFeatures(QgsFeatureRequest().setFilterExpression(query))
                for feature in selection:
                    the_id = feature.id()
                    rl.changeAttributeValue(the_id, field_index_la, 'Solitare')
            elif point_type:
                query = " \"{0}\" = '{1}'".format(points_code_field_a, from_code)
                selection = rl.getFeatures(QgsFeatureRequest().setFilterExpression(query))
                for feature in selection:
                    the_id = feature.id()
                    rl.changeAttributeValue(the_id, field_index_sec_las1, point_type)
                    if from_code in common_la:
                        rl.changeAttributeValue(the_id, field_index_la, from_code)
        
        rl.removeSelection()
        rl.commitChanges()
        qml_points = os.path.dirname(__file__) + "/qml/style3.qml"
        rl.loadNamedStyle(qml_points)
        QgsProject.instance().addMapLayer(rl)

        params = {'DEFAULT_DIRECTION': 2, 'DEFAULT_SPEED': 5, 'DIRECTION_FIELD': None, 'END_POINT': '',
                  'INPUT': network_a, 'OUTPUT': 'memory:', 'SPEED_FIELD': None,
                  'START_POINTS': QgsProcessingFeatureSourceDefinition(rl.id(), True),
                  'STRATEGY': 0, 'TOLERANCE': 0, 'VALUE_BACKWARD': '', 'VALUE_BOTH': '', 'VALUE_FORWARD': ''}

        merged_layer = None
        field_index = None
        lyr_nbr = 0
        for k, v in la_dict.items():
            lyr_nbr += 1
            params['END_POINT'] = v
            sp_name = 'memory:lyr{0}'.format(lyr_nbr)
            params['OUTPUT'] = sp_name
            query = " \"LAS1\" = '{0}'".format(k)
            iterator = rl.getFeatures(QgsFeatureRequest().setFilterExpression(query))
            ids = [i.id() for i in iterator]
            if len(ids) == 0:
                continue

            rl.selectByIds(ids)
            params['START_POINTS'] = QgsProcessingFeatureSourceDefinition(rl.id(), True)
            params['END_POINT'] = str(v.x()) + ',' + str(v.y())
            # Algorithm shortestpathlayertopoint
            make_paths = processing.run('qgis:shortestpathlayertopoint', params)
            lyr_name = make_paths['OUTPUT']
            if isinstance(lyr_name, str):
                continue

            p_buffer = {'DISSOLVE': True, 'DISTANCE': dynamic_param_c, 'END_CAP_STYLE': 0, 'INPUT': make_paths['OUTPUT'],
                        'JOIN_STYLE': 0, 'MITER_LIMIT': 2, 'OUTPUT': 'memory:', 'SEGMENTS': 5}
            make_buffer = processing.run('qgis:buffer', p_buffer)

            p_holes = {'INPUT': make_buffer['OUTPUT'], 'MIN_AREA': 0, 'OUTPUT': 'memory:'}
            del_holes = processing.run('qgis:deleteholes', p_holes)

            buffer_lyr = del_holes['OUTPUT']

            if not merged_layer:
                prov = buffer_lyr.dataProvider()
                prov.addAttributes([QgsField("ClusterTyp", QVariant.String)])
                field_index = prov.fieldNameIndex("ClusterTyp")
                for f in buffer_lyr.getFeatures():
                    the_id = f.id()
                    buffer_lyr.changeAttributeValue(the_id, field_index, "Primary")
                buffer_lyr.commitChanges()
                buffer_lyr.updateExtents()
                merged_layer = buffer_lyr
            else:
                features = []
                for feature in buffer_lyr.getFeatures():
                    features.append(feature)
                merged_layer.startEditing()
                prov = merged_layer.dataProvider()
                prov.addFeatures(features)
                for f in merged_layer.getFeatures():
                    attrs = f.attributes()
                    if attrs[field_index] == 'Primary':
                        pass
                    else:
                        the_id = f.id()
                        merged_layer.changeAttributeValue(the_id, field_index, "Primary")
                merged_layer.commitChanges()
                merged_layer.updateExtents()

        for k, v in las_dict.items():
            lyr_nbr += 1
            params['END_POINT'] = v
            sp_name = 'memory:lyr{0}'.format(lyr_nbr)
            params['OUTPUT'] = sp_name
            query = " \"LAS2\" = '{0}'".format(k)
            iterator = rl.getFeatures(QgsFeatureRequest().setFilterExpression(query))
            ids = [i.id() for i in iterator]
            if len(ids) == 0:
                continue

            rl.selectByIds(ids)
            params['START_POINTS'] = QgsProcessingFeatureSourceDefinition(rl.id(), True)
            params['END_POINT'] = str(v.x()) + ',' + str(v.y())

            # Algorithm shortestpathlayertopoint
            make_paths = processing.run('qgis:shortestpathlayertopoint', params)
            # shortestpathlayertopoint, output layername
            lyr_name = make_paths['OUTPUT']
            if isinstance(lyr_name, str):
                continue

            # parameters buffer
            p_buffer = {'DISSOLVE': True, 'DISTANCE': dynamic_param_c, 'END_CAP_STYLE': 0, 'INPUT': make_paths['OUTPUT'],
                        'JOIN_STYLE': 0, 'MITER_LIMIT': 2, 'OUTPUT': 'memory:', 'SEGMENTS': 5}
            # buffer
            make_buffer = processing.run('qgis:buffer', p_buffer)

            p_holes = {'INPUT': make_buffer['OUTPUT'], 'MIN_AREA': 0, 'OUTPUT': 'memory:'}
            del_holes = processing.run('qgis:deleteholes', p_holes)
            buffer_lyr = del_holes['OUTPUT']

            if not merged_layer:
                prov = buffer_lyr.dataProvider()
                prov.addAttributes([QgsField("ClusterTyp", QVariant.String)])
                field_index = prov.fieldNameIndex("ClusterTyp")
                for f in buffer_lyr.getFeatures():
                    the_id = f.id()
                    buffer_lyr.changeAttributeValue(the_id, field_index, "Secondary")
                buffer_lyr.commitChanges()
                buffer_lyr.updateExtents()
                merged_layer = buffer_lyr
            else:
                features = []
                for feature in buffer_lyr.getFeatures():
                    features.append(feature)
                merged_layer.startEditing()
                prov = merged_layer.dataProvider()
                prov.addFeatures(features)
                for f in merged_layer.getFeatures():
                    attrs = f.attributes()
                    if attrs[field_index] == 'Primary' or attrs[field_index] == 'Secondary':
                        pass
                    else:
                        the_id = f.id()
                        merged_layer.changeAttributeValue(the_id, field_index, "Secondary")
                merged_layer.commitChanges()
                merged_layer.updateExtents()

        # Output the resulting pointlayer
        qml2use = os.path.dirname(__file__) + "/qml/style.qml"
        merged_layer.loadNamedStyle(qml2use)
        db_conn.close()
        rl.removeSelection()
        merged_layer.removeSelection()
        QgsProject.instance().addMapLayer(merged_layer)

    """ Analyze using dijkstra algorithm.
        @params self
                graph, start_point, stop_point, layer, nbr
        @returns <void> 
    """
    def dijkstras(self, graph, start_point, stop_point, layer, nbr):
        id_start = graph.findVertex(start_point)
        id_stop = graph.findVertex(stop_point)
        (tree, cost) = QgsGraphAnalyzer.dijkstra(graph, id_start, 0)

        if tree[id_stop] == -1:
            # Path not found
            pass
        else:
            p = []
            cur_pos = id_stop
            i = 0
            while cur_pos != id_start:
                i += 1
                p.append(graph.vertex(graph.edge(tree[cur_pos]).toVertex()).point())
                cur_pos = graph.edge(tree[cur_pos]).fromVertex()
            p.append(start_point)
            # Set the provider to accept the data source
            layer.startEditing()
            prov = layer.dataProvider()
            # Add a new feature and assign the geometry
            feat = QgsFeature()
            tmp_list = []
            for pnt in p:
                the_point = QgsPoint(pnt.x(), pnt.y())
                tmp_list.append(the_point)
            feat.setGeometry(QgsGeometry.fromPolyline(tmp_list))
            feat.setAttributes([nbr])
            prov.addFeatures([feat])
            layer.commitChanges()
            layer.updateExtents()

    """ Part B. Network. """
    def run_part_b(self):
        # Get geometry layer, road network layer and commuting table data.
        try:
            point_lyr_b = QgsProject.instance().mapLayersByName(self.dlg.cb_points_b.currentText())[0]
            network_lyr = QgsProject.instance().mapLayersByName(self.dlg.cb_network_b.currentText())[0]
            commute_data_b = QgsProject.instance().mapLayersByName(self.dlg.cb_tbl_data_b.currentText())[0]
        except IndexError:
            return

        # Fetch field names from layers.
        points_code_field_b = self.dlg.cb_id_points_b.currentText()
        network_speed_field = self.dlg.cb_speed_b.currentText()
        prov_network = network_lyr.dataProvider()
        field_names = [field.name() for field in prov_network.fields()]
        network_speed_field_index = field_names.index(network_speed_field)
        network_tolerance_val = self.dlg.sb_tolerance_b.value()
        from_code_field_b = self.dlg.cb_from_b.currentText()
        to_code_field_b = self.dlg.cb_to_b.currentText()
        value_field_b = self.dlg.cb_nbr_b.currentText()

        geo = []
        to = []
        tied_to = {}
        # Specify the geometry type
        layer = QgsVectorLayer('LineString?crs=epsg:3006', 'paths', 'memory')
        tmppr = layer.dataProvider()
        layer.startEditing()
        tmppr.addAttributes([QgsField("nbr", QVariant.Int)])
        layer.commitChanges()

        for f in point_lyr_b.getFeatures():
            geom = f.geometry()
            geo.append(QgsPointXY(geom.asPoint().x(), geom.asPoint().y()))
            to.append(f[points_code_field_b])

        # all roads are treated as two-way
        # vl, directionFieldId, directDirectionValue, reverseDirectionValue, bothDirectionValue, defaultDirection
        director = QgsVectorLayerDirector(network_lyr, -1, '', '', '', 2)
        # attributeId, defaultValue, toMetricFactor
        properter = QgsNetworkSpeedStrategy(network_speed_field_index, 50.00, 0.278)
        director.addStrategy(properter)
        builder = QgsGraphBuilder(network_lyr.crs(), True, network_tolerance_val)
        tied_points = director.makeGraph(builder, geo)
        graph = builder.graph()
        for i, p in enumerate(tied_points, start=0):
            tied_to[to[i]] = p

        for f in commute_data_b.getFeatures():
            tocode_from = f[from_code_field_b]
            tocode_to = f[to_code_field_b]
            nbr = f[value_field_b]
            if tocode_from == tocode_to:
                continue

            start_point = tied_to.get(tocode_from, None)
            stop_point = tied_to.get(tocode_to, None)

            if start_point is None or stop_point is None:
                # print("Startpoint or stopPoint = None! _Error! start: {}, stop: {}".format(tocode_from, tocode_to))
                continue
            self.dijkstras(graph, start_point, stop_point, layer, nbr)

        # Add the layer to the Layers panel
        QgsProject.instance().addMapLayer(layer)

        alg = 'qgis:explodelines'
        p_explode = {'INPUT': layer, 'OUTPUT': 'memory:'}
        ret = processing.run(alg, p_explode)
        splitted_lines = ret['OUTPUT']

        crs_to_use = layer.crs()
        epsg_to_use = crs_to_use.authid()
        epsg = epsg_to_use[-4:]
        splitted_lines.setCrs(crs_to_use)
        lyr_name = splitted_lines.name()
        QgsProject.instance().addMapLayer(splitted_lines)

        query_vl = """with q1 as( select st_union(geometry) as geometry, sum(nbr) as commuters0 from '{0}' 
            group by st_startpoint(geometry), st_endpoint(geometry)) 
            select st_union(geometry) as geometry /*:linestring:{1}*/, sum(commuters0) as nbrCommuters, 
            COALESCE(st_minx(geometry)+0.1, '') || COALESCE(st_miny(geometry)+0.1, '') || 
            COALESCE(st_maxx(geometry)+0.1, '') || COALESCE(st_maxy(geometry)+0.1, '') as gid 
            from q1 group by gid""".format(lyr_name, epsg)

        vlayer = QgsVectorLayer("?query={}".format(query_vl), "vlayer", "virtual")

        p_dissolve = {'FIELD': ['nbrCommuters'], 'INPUT': vlayer, 'OUTPUT': 'memory:'}
        dissolved_lyr = processing.run('qgis:dissolve', p_dissolve)
        dissolved_lyr_obj = dissolved_lyr['OUTPUT']
        QgsProject.instance().removeMapLayer(layer)
        QgsProject.instance().removeMapLayer(splitted_lines)

        qml_dissolve = os.path.dirname(__file__) + "/qml/style2.qml"
        dissolved_lyr_obj.loadNamedStyle(qml_dissolve)
        QgsProject.instance().addMapLayer(dissolved_lyr_obj)

    """ Get vector layers from the currently loaded layers.
        @params geom_filter <list>
        @returns vector_layers <list> 
    """
    def get_current_layers(self, geom_filter=[0, 1, 2, 3, 4]):
        layers = [layer for layer in QgsProject.instance().mapLayers().values()]
        v_layers = []
        # Find vector layers from the currently loaded layers.
        for layer in layers:
            if layer.type() == QgsMapLayer.VectorLayer and layer.geometryType() in geom_filter:
                v_layers.append(layer.name())
        return v_layers

    """ Load point layers for cb_points_a combo box in part A.
        @params
        @returns <void> 
    """
    def load_points_a(self):
        # Clear cb_points_a combo box.
        self.dlg.cb_points_a.clear()
        # Get currently loaded layers and populate cb_points_a combo box.
        vector_layers = self.get_current_layers(geom_filter=[0])
        self.dlg.cb_points_a.addItems(vector_layers)

    """ Load table data for cb_tbl_data_a combo box in part A.
        @params self
        @returns <void> 
    """
    def load_tbl_data_a(self):
        # Clear cb_tbl_data_a combo box.
        self.dlg.cb_tbl_data_a.clear()
        # Get currently loaded layers and populate cb_tbl_data_a combo box.
        vector_layers = self.get_current_layers(geom_filter=[4])
        self.dlg.cb_tbl_data_a.addItems(vector_layers)

    """ Load line layers for cb_network_a combo box in part A.
        @params self
        @returns <void> 
    """
    def load_network_a(self):
        # Clear cb_network_a combo box.
        self.dlg.cb_network_a.clear()
        # Get currently loaded layers and populate cb_network_a combo box.
        vector_layers = self.get_current_layers(geom_filter=[1])
        self.dlg.cb_network_a.addItems(vector_layers)

    """ Load point layers for cb_points_b combo box in part B.
        @params self
        @returns <void> 
    """
    def load_points_b(self):
        # Clear cb_points_b combo box.
        self.dlg.cb_points_b.clear()
        # Get currently loaded layers and populate cb_points_b combo box.
        vector_layers = self.get_current_layers(geom_filter=[0])
        self.dlg.cb_points_b.addItems(vector_layers)

    """ Load line layers for cb_network_b combo box in part B.
        @params
        @returns <void> 
    """
    def load_network_b(self):
        # Clear cb_network_b combo box.
        self.dlg.cb_network_b.clear()
        # Get currently loaded layers and populate cb_network_b combo box.
        vector_layers = self.get_current_layers(geom_filter=[1])
        self.dlg.cb_network_b.addItems(vector_layers)

    """ Load table data for cb_tbl_data_b combo box in part B.
        @params 
        @returns <void> 
    """
    def load_tbl_data_b(self):
        # Clear cb_tbl_data_b combo box.
        self.dlg.cb_tbl_data_b.clear()
        # Get currently loaded layers and populate cb_tbl_data_b combo box.
        vector_layers = self.get_current_layers(geom_filter=[4])
        self.dlg.cb_tbl_data_b.addItems(vector_layers)

    """ Load polygon (or point) layers for cb_geometry_c combo box in part C.
        @params 
        @returns <void> 
    """
    def load_squares_c(self):
        # Clear cb_geometry_c combo box.
        self.dlg.cb_geometry_c.clear()
        # Get currently loaded layers and populate cb_geometry_c combo box.
        vector_layers = self.get_current_layers(geom_filter=[0, 2])
        self.dlg.cb_geometry_c.addItems(vector_layers)

    """ Load table data for cb_tbl_data_c combo box in part C.
        @params
        @returns <void> 
    """
    def load_tbl_data_c(self):
        # Clear cb_tbl_data_c combo box.
        self.dlg.cb_tbl_data_c.clear()
        # Get currently loaded layers and populate cb_tbl_data_c combo box.
        vector_layers = self.get_current_layers(geom_filter=[4])
        self.dlg.cb_tbl_data_c.addItems(vector_layers)

    """ Load analyze types for cb_analysis_c combo box in part C.
        @params
        @returns <void> 
    """
    def load_analyze_types_c(self):
        # Clear cb_analysis_c combo box.
        self.dlg.cb_analysis_c.clear()
        analyze_types = [self.tr(u'Relationships TO selected cells'), self.tr(u'Relationships FROM selected cells')]
        self.dlg.cb_analysis_c.addItems(analyze_types)

    """ Open vector from file dialog and add to QGIS TOC. Re-populate combo box, point layers for part A.
        @params
        @returns <void>
    """
    def open_points_a(self):
        input_file = str(QFileDialog.getOpenFileName(caption="Open file", filter="All files (*.*)")[0])
        if input_file:
            self.iface.addVectorLayer(input_file, str.split(os.path.basename(input_file), ".")[0], "ogr")
            self.load_points_a()

    """ Open vector from file dialog and add to QGIS TOC. Re-populate combo box, table data layers for part A.
        @params
        @returns <void>
    """
    def open_tbl_data_a(self):
        input_file = str(QFileDialog.getOpenFileName(caption="Open table data", filter="All files (*.*)")[0])
        if input_file:
            self.iface.addVectorLayer(input_file, str.split(os.path.basename(input_file), ".")[0], "ogr")
            self.load_tbl_data_a()

    """ Open vector from file dialog and add to QGIS TOC. Re-populate combo box, line layers for part A.
        @params
        @returns <void>
    """
    def open_network_a(self):
        input_file = str(QFileDialog.getOpenFileName(caption="Open file", filter="All files (*.*)")[0])

        if input_file:
            self.iface.addVectorLayer(input_file, str.split(os.path.basename(input_file), ".")[0], "ogr")
            self.load_network_a()

    """ Open vector from file dialog and add to QGIS TOC. Re-populate combo box, point layers for part B.
        @params
        @returns <void>
    """
    def open_points_b(self):
        input_file = str(QFileDialog.getOpenFileName(caption="Open file", filter="All files (*.*)")[0])
        if input_file:
            self.iface.addVectorLayer(input_file, str.split(os.path.basename(input_file), ".")[0], "ogr")
            self.load_points_b()

    """ Open vector from file dialog and add to QGIS TOC. Re-populate combo box, line layers for part B.
        @params self
        @returns <void>
    """
    def open_network_b(self):
        input_file = str(QFileDialog.getOpenFileName(caption="Open file", filter="All files (*.*)")[0])
        if input_file:
            self.iface.addVectorLayer(input_file, str.split(os.path.basename(input_file), ".")[0], "ogr")
            self.load_network_b()

    """ Open vector from file dialog and add to QGIS TOC. Re-populate combo box, table data layers for part B.
        @params self
        @returns <void>
    """
    def open_tbl_data_b(self):
        input_file = str(QFileDialog.getOpenFileName(caption="Open table data", filter="All files (*.*)")[0])
        if input_file:
            self.iface.addVectorLayer(input_file, str.split(os.path.basename(input_file), ".")[0], "ogr")
            self.load_tbl_data_b()

    """ Open vector from file dialog and add to QGIS TOC. Re-populate combo box, polygon/point layers for part C.
        @params self
        @returns <void>
    """
    def open_squares_c(self):
        input_file = str(QFileDialog.getOpenFileName(caption="Open file", filter="All files (*.*)")[0])
        if input_file:
            self.iface.addVectorLayer(input_file, str.split(os.path.basename(input_file), ".")[0], "ogr")
            self.load_squares_c()

    """ Open vector from file dialog and add to QGIS TOC. Re-populate combo box, table data layers for part C.
        @params self
        @returns <void>
    """
    def open_tbl_data_c(self):
        input_file = str(QFileDialog.getOpenFileName(caption="Open table data", filter="All files (*.*)")[0])
        if input_file:
            self.iface.addVectorLayer(input_file, str.split(os.path.basename(input_file), ".")[0], "ogr")
            self.load_tbl_data_c()

    """Removes the plugin menu item and icon from QGIS GUI."""
    def unload(self):
        for action in self.actions:
            self.iface.removePluginMenu(self.tr(u'&Commuting Analysis'), action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    def check_tab(self):
        index_tab = self.dlg.tabWidget.currentIndex()
        if index_tab == 0:
            self.check_field_types_a()
        elif index_tab == 1:
            self.check_field_types_b()
        elif index_tab == 2:
            self.check_field_types_c()

    def process_commuting(self):
        #self.iface.messageBar().pushMessage(self.tr(u'Info'), self.tr('Process started...'), level=Qgis.Info)
        index_tab = self.dlg.tabWidget.currentIndex()
        if index_tab == 0:
            self.dlg.txt_a.setPlainText(self.info_running)
            self.run_part_a()
            #self.dlg.txt_a.setPlainText('')
        elif index_tab == 1:
            #self.dlg.txt_b.setPlainText(self.info_running)
            self.run_part_b()
            #self.dlg.txt_b.setPlainText('')
        elif index_tab == 2:
            #self.dlg.txt_c_2.setPlainText(self.info_running)
            self.run_part_c()
            #self.dlg.txt_c_2.setPlainText('')
    
    def run(self):
        """Run method that performs all the real work"""
        # show the dialog
        self.dlg.show()
        # Load layers into plugin-GUI
        self.load_points_a()
        self.load_tbl_data_a()
        self.load_network_a()
        self.load_points_b()
        self.load_network_b()
        self.load_tbl_data_b()
        self.load_squares_c()
        self.load_tbl_data_c()
        self.load_analyze_types_c()

        self.check_tab()

        # Run the dialog event loop
        result = self.dlg.exec_()

        # See if OK was pressed
        if result:
            pass