# -*- coding: utf-8 -*-
"""
/***************************************************************************
 CommuneCadastre
                                 A QGIS plugin
 Permet d'identifier une commune ou une parcelle cadastrale sur la carte
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-07-04
        git sha              : $Format:%H$
        copyright            : (C) 2023 by Augustin Roche
        email                : augustin.roche@enertrag.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QUrl, QUrlQuery
from qgis.PyQt.QtGui import QIcon, QIntValidator, QColor
from qgis.PyQt.QtWidgets import QAction, QFormLayout
from qgis.core import QgsNetworkContentFetcher, Qgis, QgsWkbTypes
from qgis.gui import QgsRubberBand

from .commune_selector import CommuneSelector
from .utils import geoJson2geom

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the DockWidget
from .commune_cadastre_dockwidget import CommuneCadastreDockWidget
import os.path
import json


class CommuneCadastre:
    """QGIS Plugin Implementation."""

    CADASTRE_API_URL = "https://apicarto.ign.fr/api/cadastre/"

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'CommuneCadastre_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Commune Cadastre')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.pluginToolBar()

        #print "** INITIALIZING CommuneCadastre"

        self.pluginIsActive = False
        self.dockwidget = None
        self.commune_widget = None
        self.parcelles = []
        self.parcelleRubberBand = None


    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('CommuneCadastre', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action


    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/commune_cadastre/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Recherche commune et parcelle'),
            callback=self.run,
            parent=self.iface.mainWindow())

    #--------------------------------------------------------------------------

    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        #print "** CLOSING CommuneCadastre"

        if self.rubberBand:
            self.rubberBand.reset()

        # disconnects
        self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)

        # remove this statement if dockwidget is to remain
        # for reuse if plugin is reopened
        # Commented next statement since it causes QGIS crashe
        # when closing the docked window:
        # self.dockwidget = None

        self.pluginIsActive = False


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        #print "** UNLOAD CommuneCadastre"

        if self.rubberBand:
            self.rubberBand.reset()

        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Commune Cadastre'),
                action)
            self.iface.removeToolBarIcon(action)

    #--------------------------------------------------------------------------

    def run(self):
        """Run method that loads and starts the plugin"""

        if not self.pluginIsActive:
            self.pluginIsActive = True

            #print "** STARTING CommuneCadastre"

            # dockwidget may not exist if:
            #    first run of plugin
            #    removed on close (see self.onClosePlugin method)
            if self.dockwidget is None:
                # Create the dockwidget (after translation) and keep reference
                self.dockwidget = CommuneCadastreDockWidget()
                self.commune_widget = CommuneSelector(self.dockwidget)
                self.dockwidget.formLayout.setWidget(0, QFormLayout.FieldRole, self.commune_widget)

            # connect to provide cleanup on closing of dockwidget
            self.dockwidget.closingPlugin.connect(self.onClosePlugin)                
            
            self.dockwidget.numeroLineEdit.setValidator(QIntValidator(0, 9999))

            self.sections_manager = QgsNetworkContentFetcher()
            self.sections_manager.finished.connect(self.on_sections_dl)
            self.parcelles_manager = QgsNetworkContentFetcher()
            self.parcelles_manager.finished.connect(self.on_parcelle_dl)

            self.commune_widget.communeChanged.connect(self.get_sections)
            self.commune_widget.textEdited.connect(self.on_commune_text_changed)
            self.dockwidget.sectionComboBox.currentIndexChanged.connect(self.on_section_changed)

            self.dockwidget.pushButton.clicked.connect(self.search)
            self.dockwidget.numeroLineEdit.returnPressed.connect(self.search)

            self.rubberBand = QgsRubberBand(self.iface.mapCanvas(), QgsWkbTypes.PolygonGeometry)
            self.rubberBand.setStrokeColor(QColor(65, 135, 240))
            self.rubberBand.setWidth(3)
            self.rubberBand.setBrushStyle(Qt.NoBrush)


            # show the dockwidget
            # TODO: fix to allow choice of dock location
            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dockwidget)
            self.dockwidget.show()

    def setStatusText(self, txt: str):
        self.dockwidget.statusLabel.setText(txt)

    def on_commune_text_changed(self):
        self.dockwidget.sectionComboBox.clear()
        self.dockwidget.sectionComboBox.setEnabled(False)

    def get_sections(self):
        self.setStatusText(self.tr("Loading sections..."))
        insee_com = self.commune_widget.get_insee()
        params = QUrlQuery()
        params.addQueryItem('code_insee', insee_com)
        url = QUrl(self.CADASTRE_API_URL + 'feuille')
        url.setQuery(params)
        self.sections_manager.fetchContent(url)

    def on_sections_dl(self):
        self.setStatusText("")
        content = self.sections_manager.contentAsString()
        try:
            results = json.loads(content)['features']
        except (json.decoder.JSONDecodeError, KeyError):
            self.iface.messageBar().pushMessage(self.tr("Could not parse cadastre sections content"), 
                                                level=Qgis.Warning, duration=5)
            return
        self.dockwidget.sectionComboBox.clear()
        self.dockwidget.sectionComboBox.setEnabled(True)
        self.dockwidget.numeroLineEdit.clear()

        sections = set(' ')
        for li in results:
            sections.add(li['properties']['section'])
        for sec in sorted(sections):
            self.dockwidget.sectionComboBox.addItem(sec)
        self.dockwidget.numeroLineEdit.setEnabled(True)

    def on_section_changed(self):
        val =self.dockwidget.sectionComboBox.currentText()
        if val != ' ':
            self.dockwidget.numeroLineEdit.setEnabled(True)

    def search(self):
        """ Looks for the parcelle or commune entered """
        insee_com = self.commune_widget.get_insee()
        if not insee_com:
            return
        section = self.dockwidget.sectionComboBox.currentText()
        
        if section == ' ' or not section or not self.dockwidget.numeroLineEdit.hasAcceptableInput():
            # show commune
            geom = self.commune_widget.get_geom()
            self.rubberBand.setToGeometry(geom)
            self.iface.mapCanvas().zoomToFeatureExtent(geom.boundingBox())
            return

        self.setStatusText(self.tr("Searching parcelle...")) 
        numero = self.dockwidget.numeroLineEdit.text()
        numero = numero.rjust(4, '0')
        insee_com = self.commune_widget.get_insee()
        params = QUrlQuery()
        params.addQueryItem('code_insee', insee_com)
        params.addQueryItem('section', section)
        params.addQueryItem('numero', numero)
        url = QUrl(self.CADASTRE_API_URL + 'parcelle')
        url.setQuery(params)
        self.parcelles_manager.fetchContent(url)

    def on_parcelle_dl(self):
        content = self.parcelles_manager.contentAsString()
        try:
            parcelles = json.loads(content)['features']
            parcelles.sort(key=lambda l: int(l['properties']['numero']))
        except (json.decoder.JSONDecodeError, KeyError):
            self.iface.messageBar().pushMessage(self.tr("Could not parse cadastre parcelles content"), 
                                                level=Qgis.Warning, duration=5)
            return
        if len(parcelles) == 0:
            self.setStatusText(self.tr("Parcelle not found."))
            return
        parcelle = parcelles[0]
        geom = geoJson2geom(parcelle['geometry'])
        
        self.rubberBand.setToGeometry(geom)
        self.iface.mapCanvas().zoomToFeatureExtent(geom.boundingBox())
        self.setStatusText("")  

