# -*- coding: utf-8 -*-
"""
/***************************************************************************
 ColorMappingTool
                                 A QGIS plugin
 Easily apply color codes to layers based on selected field values
 Generated by Plugin Builder
                              -------------------
        begin                : 2024-11-13
        copyright            : (C) 2024 by Mehmet Ünüvar
        email                : unuvarmehmet88@gmail.com
 ***************************************************************************/
 """

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, QObject
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMessageBox, QMenu, QToolButton, QWidgetAction, QScrollArea, QVBoxLayout, QPushButton, QWidget
from qgis.core import QgsProject
from qgis.gui import QgsMapToolIdentifyFeature
import os


class ColorMappingTool(QObject):
    """QGIS Plugin for assigning colors based on field values."""

    def __init__(self, iface):
        super().__init__()
        self.iface = iface
        self.plugin_dir = os.path.dirname(__file__)

        # Plugin settings
        self.selected_field = None
        self.adresspoint_layer = None
        self.color_index = 0
        self.colors = []
        self.main_action = None

    def initGui(self):
        """Initializes the plugin GUI."""
        if self.main_action:  # Avoid adding multiple icons
            return

        self.tool_button = QToolButton()
        self.tool_button.setIcon(QIcon(os.path.join(self.plugin_dir, 'icon.png')))
        self.tool_button.setPopupMode(QToolButton.MenuButtonPopup)
        self.tool_button.setToolTip("Field Color Coding Tool")
        self.menu = QMenu()
        self.tool_button.setMenu(self.menu)

        self.main_action = QWidgetAction(self.iface.mainWindow())
        self.main_action.setDefaultWidget(self.tool_button)
        self.iface.addToolBarIcon(self.main_action)

        self.menu.aboutToShow.connect(self.populate_field_menu)
        self.tool_button.clicked.connect(self.toggle_tool)

    def unload(self):
        """Removes plugin icons and menu items."""
        if self.main_action:
            self.iface.removeToolBarIcon(self.main_action)
        self.menu.clear()
        self.tool_button = None

    def populate_field_menu(self):
        """Populates the dropdown menu with available color code fields."""
        self.adresspoint_layer = self.iface.activeLayer()
        self.menu.clear()

        if not self.adresspoint_layer or not self.adresspoint_layer.isValid():
            QMessageBox.warning(None, "Layer Not Found", "Please select an address layer first.")
            return

        scroll_area = QScrollArea()
        scroll_area.setWidgetResizable(True)
        scroll_area.setFixedWidth(200)

        widget = QWidget()
        layout = QVBoxLayout(widget)
        widget.setFixedWidth(180)

        for field in self.adresspoint_layer.fields():
            display_name = field.name()[:20]
            button = QPushButton(display_name)
            button.clicked.connect(lambda checked, f=field.name(): self.set_field(f))
            layout.addWidget(button)

        scroll_area.setWidget(widget)
        scroll_area.setFixedHeight(200)

        action_scroll_area = QWidgetAction(self.menu)
        action_scroll_area.setDefaultWidget(scroll_area)
        self.menu.addAction(action_scroll_area)

    def set_field(self, field_name):
        """Sets the selected field and loads color codes."""
        self.selected_field = field_name
        self.save_last_field(field_name)
        self.load_colors_from_field()
        self.menu.close()

    def load_colors_from_field(self):
        """Loads color codes from the selected field's Value Map."""
        if not self.selected_field or not self.adresspoint_layer:
            QMessageBox.warning(None, "Field Selection", "Please select a valid field first.")
            return

        field_index = self.adresspoint_layer.fields().indexFromName(self.selected_field)
        if field_index == -1:
            QMessageBox.warning(None, "Field Not Found", "The selected field is invalid.")
            return

        field = self.adresspoint_layer.fields().field(field_index)
        editor_widget = field.editorWidgetSetup()
        config = editor_widget.config()

        if editor_widget.type() == "ValueMap" and isinstance(config, dict) and "map" in config:
            self.colors = [list(item.values())[0] for item in config["map"] if isinstance(item, dict)]
            self.color_index = 0
        else:
            self.colors = []
            QMessageBox.warning(None, "Invalid Color Codes", "No valid Value Map found in the selected field.")

    def save_last_field(self, field_name):
        """Saves the last selected field."""
        QSettings().setValue("color_tool/last_selected_field", field_name)

    def toggle_tool(self):
        """Activates or deactivates the tool."""
        self.adresspoint_layer = self.iface.activeLayer()

        if self.adresspoint_layer:
            if not self.selected_field:
                self.selected_field = QSettings().value("color_tool/last_selected_field", "")
                self.load_colors_from_field()

            if not self.selected_field:
                QMessageBox.warning(None, "No Color Field Selected", "Please select a color field first.")
                return

            self.color_index = 0
            self.map_tool = ColorMappingSelectTool(self.iface, self)
            self.iface.mapCanvas().setMapTool(self.map_tool)
        else:
            QMessageBox.warning(None, "Layer Not Found", "Please select an address layer.")


class ColorMappingSelectTool(QgsMapToolIdentifyFeature):
    def __init__(self, iface, plugin):
        super().__init__(iface.mapCanvas())
        self.iface = iface
        self.plugin = plugin
        self.adresspoint_layer = plugin.adresspoint_layer

    def canvasReleaseEvent(self, event):
        if not self.plugin.colors:
            QMessageBox.warning(None, "No Color Codes Available", "Selected field has no color codes defined.")
            return

        results = self.identify(event.x(), event.y(), [self.adresspoint_layer], self.TopDownStopAtFirst)
        if results:
            feature = results[0].mFeature
            if event.button() == 1:  # Left-click for next color
                color = self.plugin.colors[self.plugin.color_index]
                self.plugin.color_index = (self.plugin.color_index + 1) % len(self.plugin.colors)
            elif event.button() == 2:  # Right-click for previous color
                self.plugin.color_index = (self.plugin.color_index - 1) % len(self.plugin.colors)
                color = self.plugin.colors[self.plugin.color_index]
            self.change_color(feature, color)

    def change_color(self, feature, color):
        try:
            self.adresspoint_layer.startEditing()
            feature[self.plugin.selected_field] = color
            self.adresspoint_layer.updateFeature(feature)
            self.adresspoint_layer.triggerRepaint()
        except KeyError:
            QMessageBox.warning(None, "Error", f"Field '{self.plugin.selected_field}' could not be updated.")
