# -*- coding: utf-8 -*-
"""
/***************************************************************************
 NetworkContinuity
                                 A QGIS plugin
 This plugin deduces natural continuity and hierarchy of street network.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2020-03-23
        git sha              : $Format:%H$
        copyright            : (C) 2020 by Pratyush Tripathy
        email                : pratkrt@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from PyQt5.QtCore import QSettings, QTranslator, qVersion, QCoreApplication
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QAction, QFileDialog, QListWidgetItem
from .streetHierarchy import network

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .coins_dialog import NetworkContinuityDialog
import os.path, time, shutil, math, glob

def timeTaken(t1, t2, stepName):
    print('It took %.2f seconds to %s' % ((t2 - t1), stepName))

class NetworkContinuity:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'NetworkContinuity_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)

            if qVersion() > '4.3.3':
                QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&COINS - Continuity in Street Networks')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('NetworkContinuity', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToVectorMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/coins/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Continuity in Street Networks'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(
                self.tr(u'&COINS - Continuity in Street Networks'),
                action)
            self.iface.removeToolBarIcon(action)

    def select_input_file(self):
        self.infile, _filter = QFileDialog.getOpenFileName(self.dlg, 'Open file', 'F:\\StreetNetworkLongest\\QGISplugin',"ESRI Shapefile (*.shp)")
        self.dlg.comboBox.addItem(self.infile)
        time.sleep(0.25)
        index = self.dlg.comboBox.findText(self.infile, QtCore.Qt.MatchFixedString)
        self.dlg.comboBox.setCurrentIndex(index)
        time.sleep(0.25)
        self.dlg.outputHeirarchyFileName.setText(self.infile.replace('.shp', '_hierarchy.shp'))

    def select_output_file(self):
        filename, _filter = QFileDialog.getSaveFileName(self.dlg, "Select output file ","", '*.shp')
        self.dlg.outputHeirarchyFileName.setText(filename)
        
    def select_premerge_file(self):
        filename, _filter = QFileDialog.getSaveFileName(self.dlg, "Select output file ","", '*.shp')
        self.dlg.SelectPreMergeOutputButton.setText(filename)

    def update_output_file(self):
        self.infile = self.dlg.comboBox.currentText()
        time.sleep(0.25)
        self.dlg.outputHeirarchyFileName.setText(self.infile.replace('.shp', '_hierarchy.shp'))
        
    def activate_premerge_field(self):
        if not self.infile == None:
            if self.dlg.preMergeCheckBox.isChecked():
                self.dlg.outputPreMergeFileName.setDisabled(False)
                self.dlg.SelectPreMergeOutputButton.setDisabled(False)
                self.dlg.outputPreMergeFileName.setText(self.infile.replace('.shp', '_preMerge.shp'))
            else:
                self.dlg.outputPreMergeFileName.setDisabled(True)
                self.dlg.SelectPreMergeOutputButton.setDisabled(True)
        else:
            pass
            
    def update_threshold(self):
        try:
            self.angleThreshold = float(self.dlg.angleThresholdValue.text())
        except:
            pass

    def startProcessing(self):
        #self.iface.messageBar().pushMessage("Running COINS tool..", level=0, duration=3)
        #self.iface.mainWindow().repaint()
        #self.iface.statusBarIface().showMessage("Processed {} %".format(int(60)))
        self.processNetworkVector()
        
        #time.sleep(4)
        #self.iface.messageBar().pushMessage("Processing complete.", level=3, duration=3)
        self.dlg.close()
        
    def processNetworkVector(self):
        t1 = time.time()
        inFile = self.infile
        outMergedFile = self.dlg.outputHeirarchyFileName.text()
        outPreMergedFile = self.dlg.outputPreMergeFileName.text()
        
        myNetwork = network(inFile)
        #Split lines
        myNetwork.splitLines1()
        
        #Create unique ID
        myNetwork.uniqueID()
        
        #Compute connectivity table
        myNetwork.getLinks()
        
        #Find best link at every point for both lines
        myNetwork.bestLink()
        
        #Cross check best links
        myNetwork.crossCheckLinks(self.angleThreshold)
        
        #Merge finalised links
        myNetwork.mergeLines()
        
        #Export lines
        if self.dlg.preMergeCheckBox.isChecked():
            myNetwork.exportPreMerged(outPreMergedFile)
        myNetwork.exportMerged(outMergedFile)
        
        t2 = time.time()
        print('Processing completed in %.2f' % (t2-t1))
        minutes = math.floor((t2-t1) / 60)
        seconds = (t2 - t1) % 60
        message = "Process complete in %d minutes %.2f seconds." % (minutes, seconds)
        self.iface.messageBar().pushMessage(message, level=3, duration=3)
        """
        temp = myNetwork.tempDirectory
        time.sleep(1)
        myNetwork = None
                
        os.chdir(self.plugin_dir)
        tempFiles = glob.glob('*.npy')
        tempFiles.append(glob.glob('*.json')[0])
        for file in tempFiles:
            shutil.rmtree(file)
        
        if os.path.exists(temp):
            try:
                shutil.rmtree(temp)
            except:
                pass
        """

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = NetworkContinuityDialog()
            self.angleThreshold = 0
        
            #method_list = [func for func in dir(self.iface.statusBarIface) if callable(getattr(self.iface.statusBarIface, func))]
            #print(method_list)
            
            
            self.dlg.SelectInputLayerButton.clicked.connect(self.select_input_file)
            self.dlg.SelectOutputLayerButton.clicked.connect(self.select_output_file)
            self.dlg.preMergeCheckBox.stateChanged.connect(self.activate_premerge_field)
            self.dlg.processButton.clicked.connect(self.startProcessing)
            self.dlg.cancelButton.clicked.connect(self.dlg.close)
            self.dlg.comboBox.setEditable(True)
            self.dlg.comboBox.currentIndexChanged.connect(self.update_output_file)
            
            #Clear all the fields in the GUI at the start-up
            self.dlg.outputHeirarchyFileName.setText("")
            self.dlg.SelectPreMergeOutputButton.setText("")
            self.dlg.progressBar.setValue(0)
            # Get all the layers already added to the sesison
            self.layers = self.iface.mapCanvas().layers()
            self.layersList = list()
            for layer in self.layers:
                self.layersList.append(layer.source())
            self.dlg.comboBox.clear()
            self.dlg.comboBox.addItems(self.layersList)
            self.dlg.angleThresholdValue.textChanged.connect(self.update_threshold)

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        
        if result:
            pass
