"""
/***************************************************************************
                                 A QGIS plugin
 CLUZ for QGIS
                             -------------------
        begin                : 2022-26-08
        copyright            : (C) 2022 by Bob Smith, DICE
        email                : r.j.smith@kent.ac.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from os import remove, rename
from csv import reader, writer

from .cluz_make_file_dicts import formatRawValueList_IdentifyNumericalCols, returnTempPathName
from .cluz_messages import warningMessage


def makeZonesTargetDict(setupObject):
    targetDict = dict()
    targetCSVFilePath = setupObject.targetPath
    try:
        with open(targetCSVFilePath, 'rt') as f:
            targetReader = reader(f)

            origHeaderList = next(targetReader)
            headerList = list()  # convert to lowercase so it doesn't matter whether the headers or lowercase, uppercase or a mix
            for aHeader in origHeaderList:
                headerList.append(aHeader.lower())

            for aRow in targetReader:
                featID = int(aRow[headerList.index('id')])
                featList = makeZonesTargetDictRowFeatList(setupObject, aRow, headerList)
                targetDict[featID] = featList

    except ValueError:
        warningMessage('Target table error', 'The Target table is incorrectly formatted. Please use the Troubleshoot all CLUZ files function to identify the problem.')
        targetDict = 'blank'

    return targetDict


def makeZonesTargetDictRowFeatList(setupObject, aRow, headerList):
    featName = str(aRow[headerList.index('name')])
    featType = int(aRow[headerList.index('type')])
    featSpf = float(aRow[headerList.index('spf')])
    featTarget = float(aRow[headerList.index('target')])
    featEarLock = float(aRow[headerList.index('ear+lock')])
    featTotal = float(aRow[headerList.index('total')])
    featPc_Target = float(aRow[headerList.index('pc_target')])
    standardFeatList = [featName, featType, featSpf, featTarget, featEarLock, featTotal, featPc_Target]
    zonesFeatList = makeZonesFeatList(setupObject, aRow, headerList)
    featList = standardFeatList + zonesFeatList

    return featList


def makeZonesFeatList(setupObject, aRow, headerList):
    zonesFeatList = list()
    for zoneID in list(setupObject.zonesDict):
        aZoneTarget = float(aRow[headerList.index('z' + str(zoneID) + '_prop')])
        zonesFeatList.append(aZoneTarget)

    for zoneID in list(setupObject.zonesDict):
        aZoneTarget = float(aRow[headerList.index('z' + str(zoneID) + '_target')])
        zonesFeatList.append(aZoneTarget)

    for zoneID in list(setupObject.zonesDict):
        aZoneTarget = float(aRow[headerList.index('z' + str(zoneID) + '_ear+lock')])
        zonesFeatList.append(aZoneTarget)

    return zonesFeatList


def makeZonesTargetDialogRowList(setupObject):
    targetFileHeaderList = makeZonesTargetFileHeaderList(setupObject)
    targetFileDataRowCount = returnZonesTargetFileDataRowCount(setupObject)

    rawZonesTargetDialogDict = makeZonesRawTargetDialogDict(setupObject)
    zonesTargetDialogRowList = list()
    zonesTargetDialogRowList.append(targetFileHeaderList)

    numericColsList = list()

    zonesTargetDialogDict = dict()
    for colName in targetFileHeaderList:
        rawValueList = rawZonesTargetDialogDict[colName]
        valueList, ColTypeIntOrFloat = formatRawValueList_IdentifyNumericalCols(setupObject, rawValueList)
        zonesTargetDialogDict[colName] = valueList
        if ColTypeIntOrFloat:
            numericColsList.append(colName.lower())  # Turned to lower case to allow later comparisons

    for aRow in range(0, targetFileDataRowCount):
        rowList = list()
        for colName in targetFileHeaderList:
            rowList.append(zonesTargetDialogDict[colName][aRow])
        zonesTargetDialogRowList.append(rowList)

    return zonesTargetDialogRowList, numericColsList


def makeZonesTargetFileHeaderList(setupObject):
    zonesTargetCSVFilePath = setupObject.targetPath
    try:
        with open(zonesTargetCSVFilePath, 'rt') as f:
            targetReader = reader(f)
            zonesTargetFileHeaderList = next(targetReader)

    except ValueError:
        zonesTargetFileHeaderList = 'blank'

    return zonesTargetFileHeaderList


def returnZonesTargetFileDataRowCount(setupObject):
    zonesTargetCSVFilePath = setupObject.targetPath
    with open(zonesTargetCSVFilePath, 'rt') as f:
        countReader = reader(f)
        next(countReader)
        targetFileDataRowCount = len(list(countReader))

    return targetFileDataRowCount


def makeZonesDict(setupObject):
    zonesDict = dict()
    zonesCSVFilePath = setupObject.zonesPath
    try:
        with open(zonesCSVFilePath, 'rt') as f:
            zonesReader = reader(f)
            next(zonesReader)
            for aRow in zonesReader:
                zoneID = int(aRow[0])
                zoneName = aRow[1]
                zonesDict[zoneID] = zoneName

    except ValueError:
        # warningMessage('Zones table error', 'The Zones table is incorrectly formatted. Please use the Troubleshoot all CLUZ files function to identify the problem.')
        zonesDict = 'blank'

    return zonesDict


def makeZonesPropDict(setupObject):
    zonesPropDict = dict()
    zonesHeaderNameList = makeZonesHeaderNameList(setupObject, '_Prop')
    targetCSVFilePath = setupObject.targetPath
    try:
        with open(targetCSVFilePath, 'rt') as f:
            targetReader = reader(f)

            origHeaderList = next(targetReader)
            lowercaseHeaderList = list()  # convert to lowercase so it doesn't matter whether the headers or lowercase, uppercase or a mix
            for aHeader in origHeaderList:
                lowercaseHeaderList.append(aHeader.lower())

            featIDColValue = lowercaseHeaderList.index('id')
            zonesColValueDict = makeZonesColValueDict(lowercaseHeaderList, zonesHeaderNameList)

            for aRow in targetReader:
                featID = int(aRow[featIDColValue])
                for zonesHeaderName in zonesHeaderNameList:
                    try:
                        zonesPropValueDict = zonesPropDict[zonesHeaderName]
                    except KeyError:
                        zonesPropValueDict = dict()
                    zonesColValue = zonesColValueDict[zonesHeaderName]
                    zonesPropValueDict[featID] = float(aRow[zonesColValue])
                    zonesPropDict[zonesHeaderName] = zonesPropValueDict

    except ValueError:
        # warningMessage('Target table error', 'The Target table is incorrectly formatted. Please use the Troubleshoot all CLUZ files function to identify the problem.')
        zonesPropDict = 'blank'

    return zonesPropDict


def makeZonesTargetZonesDict(setupObject):
    zonesTargetDict = dict()
    zonesHeaderNameList = makeZonesHeaderNameList(setupObject, '_Target')
    targetCSVFilePath = setupObject.targetPath
    try:
        with open(targetCSVFilePath, 'rt') as f:
            targetReader = reader(f)

            origHeaderList = next(targetReader)
            lowercaseHeaderList = list()  # convert to lowercase so it doesn't matter whether the headers or lowercase, uppercase or a mix
            for aHeader in origHeaderList:
                lowercaseHeaderList.append(aHeader.lower())

            featIDColValue = lowercaseHeaderList.index('id')
            zonesColValueDict = makeZonesColValueDict(lowercaseHeaderList, zonesHeaderNameList)

            for aRow in targetReader:
                featID = int(aRow[featIDColValue])
                for zonesHeaderName in zonesHeaderNameList:
                    try:
                        zonesTargetValueDict = zonesTargetDict[zonesHeaderName]
                    except KeyError:
                        zonesTargetValueDict = dict()
                    zonesColValue = zonesColValueDict[zonesHeaderName]
                    zonesTargetValueDict[featID] = float(aRow[zonesColValue])
                    zonesTargetDict[zonesHeaderName] = zonesTargetValueDict

    except ValueError:
        # warningMessage('Target table error', 'The Target table is incorrectly formatted. Please use the Troubleshoot all CLUZ files function to identify the problem.')
        zonesTargetDict = 'blank'

    return zonesTargetDict


def makeZonesHeaderNameList(setupObject, headerSuffixText):
    headerColNumList = list()
    for zoneID in setupObject.zonesDict:
        headerColNumList.append('Z' + str(zoneID) + headerSuffixText)

    return headerColNumList


def makeZonesColValueDict(lowercaseHeaderList, zonesHeaderNameList):
    zonesColDict = dict()
    for zonesHeaderName in zonesHeaderNameList:
        lowercaseZonesHeaderName = zonesHeaderName.lower()
        zonesColDict[zonesHeaderName] = lowercaseHeaderList.index(lowercaseZonesHeaderName)

    return zonesColDict


def makeZonesBLMDictFromSetupFile(setupDict):
    zonesBLMDict = dict()
    for setupFileRowName in setupDict:
        if 'BLM_Zone' in setupFileRowName:
            zonesBLMDictKey = setupFileRowName[4:].replace('_', ' ')
            zonesBLMDict[zonesBLMDictKey] = float(setupDict[setupFileRowName])

    return zonesBLMDict


def makeZonesRawTargetDialogDict(setupObject):
    rawZonesTargetDialogDict = dict()
    zonesTargetCSVFilePath = setupObject.targetPath
    with open(zonesTargetCSVFilePath, 'rt') as f:
        targetReader = reader(f)
        targetFileHeaderList = next(targetReader)
        for aRow in targetReader:
            for aCol in range(0, len(targetFileHeaderList)):
                colName = targetFileHeaderList[aCol]
                try:
                    rawColumnValuesList = rawZonesTargetDialogDict[colName]
                except KeyError:
                    rawColumnValuesList = list()
                rawColumnValuesList.append(aRow[aCol])
                rawZonesTargetDialogDict[colName] = rawColumnValuesList

    if len(rawZonesTargetDialogDict) == 0:
        with open(zonesTargetCSVFilePath, 'rt') as f:
            targetReader = reader(f)
            targetFileHeaderList = next(targetReader)
            for aCol in range(0, len(targetFileHeaderList)):
                colName = targetFileHeaderList[aCol]
                rawZonesTargetDialogDict[colName] = list()

    return rawZonesTargetDialogDict


def updateZonesTargetCSVFromTargetDict(setupObject, targetDict):
    decPrec = setupObject.decimalPlaces
    targetCSVFilePath = setupObject.targetPath
    textRows = list()
    with open(targetCSVFilePath, 'rt') as in_file:
        targetReader = reader(in_file)
        origHeaderList = next(targetReader)
        textRows.append(origHeaderList)
        lowerHeaderList = list()  # convert to lowercase so it doesn't matter whether the headers or lowercase, uppercase or a mix
        for aHeader in origHeaderList:
            lowerHeaderList.append(aHeader.lower())

        for aRow in targetReader:
            featID = int(aRow[lowerHeaderList.index('id')])
            featTarget = float(aRow[lowerHeaderList.index('target')])
            pcTarget = returnPCTargetValueForTargetTable(targetDict, featID, featTarget, decPrec)

            aRow[lowerHeaderList.index('ear+lock')] = targetDict[featID][4]
            aRow[lowerHeaderList.index('total')] = targetDict[featID][5]
            aRow[lowerHeaderList.index('pc_target')] = pcTarget

            zoneCount = len(setupObject.zonesDict)
            zoneIDList = list(setupObject.zonesDict)
            for zoneID in setupObject.zonesDict:
                targetListPos = 6 + (2 * zoneCount) + zoneIDList.index(zoneID) + 1
                aRow[lowerHeaderList.index('z' + str(zoneID) + '_ear+lock')] = targetDict[featID][targetListPos]

            textRows.append(aRow)

    with open(targetCSVFilePath, 'w', newline='') as out_file:
        targetWriter = writer(out_file)
        for bRow in textRows:
            targetWriter.writerow(bRow)


def returnPCTargetValueForTargetTable(targetDict, featID, featTarget, decPrec):
    if featTarget > 0:
        pcTarget = targetDict[featID][4] / featTarget
        pcTarget *= 100
        pcTarget = round(float(pcTarget), decPrec)
        pcTarget = format(pcTarget, "." + str(decPrec) + "f")
    else:
        pcTarget = -1

    return pcTarget


# Add data to target table from Add data from Vec, raster, csv files #######################################

def zonesAddFeaturesToTargetCsvFile(setupObject, featIDList):
    tempTargetPath = returnTempPathName(setupObject.targetPath, 'csv')
    with open(tempTargetPath, 'w', newline='') as tempTargetFile:
        tempTargetWriter = writer(tempTargetFile)
        with open(setupObject.targetPath, 'rt') as f:
            targetReader = reader(f)
            targetFileHeaderList = next(targetReader)
            tempTargetWriter.writerow(targetFileHeaderList)
            for row in targetReader:
                tempTargetWriter.writerow(row)

            addTargetList = featIDList
            addTargetList.sort()
            for featID in addTargetList:
                row = zonesMakeNewRowTargetCSVFromAddTargetList(targetFileHeaderList, featID)
                tempTargetWriter.writerow(row)

    tempTargetFile.close()
    remove(setupObject.targetPath)
    rename(tempTargetPath, setupObject.targetPath)


def zonesMakeNewRowTargetCSVFromAddTargetList(targetFileHeaderList, featID):
    newRowList = [''] * len(targetFileHeaderList)
    for aTargetHeaderCol in range(0, len(targetFileHeaderList)):
        aTargetHeaderName = targetFileHeaderList[aTargetHeaderCol]
        if aTargetHeaderName.lower() == 'id':
            newRowList[aTargetHeaderCol] = str(featID)
        elif aTargetHeaderName.lower() == 'name':
            newRowList[aTargetHeaderCol] = 'blank'
        elif aTargetHeaderName.lower() == 'ear+lock':
            newRowList[aTargetHeaderCol] = '0'
        elif aTargetHeaderName.lower() == 'total':
            newRowList[aTargetHeaderCol] = '0'
        elif aTargetHeaderName.lower() == 'pc_target':
            newRowList[aTargetHeaderCol] = '-1'
        elif aTargetHeaderName.lower() in ['type', 'target', 'spf']:
            newRowList[aTargetHeaderCol] = '0'
        elif '_prop' in aTargetHeaderName.lower():
            newRowList[aTargetHeaderCol] = '1'
        elif '_target' in aTargetHeaderName.lower():
            newRowList[aTargetHeaderCol] = '0'
        elif '_ear+lock' in aTargetHeaderName.lower():
            newRowList[aTargetHeaderCol] = '0'

    return newRowList
