"""
/***************************************************************************
                                 A QGIS plugin
 CLUZ for QGIS
                             -------------------
        begin                : 2022-26-08
        copyright            : (C) 2022 by Bob Smith, DICE
        email                : r.j.smith@kent.ac.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.core import QgsVectorLayer, QgsSpatialIndex, QgsField
from qgis.PyQt.QtCore import QVariant

from csv import reader, writer
from os import path, sep
from pathlib import Path
from statistics import median

from .cluz_messages import clearProgressBar, makeProgressBar, warningMessage

# Produce Marxan input files #######################################################


def createZonesFeatDatFile(setupObject):
    zonesFeatDatFile = setupObject.inputPath + sep + 'feat.dat'
    with open(zonesFeatDatFile, 'w', newline='') as out_file:
        specDatWriter = writer(out_file)
        specDatWriter.writerow(['id', 'name', 'target', 'spf'])

        targetDict = setupObject.targetDict
        featList = list(targetDict)
        featList.sort()

        progressBar = makeProgressBar('Making a new feat.dat file')
        rowTotalCount = len(featList)
        rowCount = 1

        for aFeat in featList:
            progressBar.setValue((rowCount/rowTotalCount) * 100)
            rowCount += 1

            featList = targetDict[aFeat]
            rawFeatName = featList[0]
            changeBool, featName = convertFeatNameByChangingIncompatibleTextCharacters(rawFeatName)
            featTarget = featList[3]
            featSpf = featList[2]
            specDatWriter.writerow([aFeat, featName, featTarget, featSpf])
    clearProgressBar()


def convertFeatNameByChangingIncompatibleTextCharacters(rawFeatName):
    changeBool = False
    featName = rawFeatName.replace(' ', '_')
    featName = featName.replace('.', '')

    if rawFeatName != featName:
        changeBool = True

    return changeBool, featName


def createZonesTargetDatFile(setupObject):
    zonesFeatDatFile = setupObject.inputPath + sep + 'zonetarget.dat'
    with open(zonesFeatDatFile, 'w', newline='') as out_file:
        zonesTargetDatWriter = writer(out_file)
        zonesTargetDatWriter.writerow(['zoneid', 'featureid', 'target'])

        featList = list(setupObject.targetDict)
        featList.sort()

        progressBar = makeProgressBar('Making a new feat.dat file')
        rowTotalCount = len(featList)
        rowCount = 1

        for zonesTargetTypeName in setupObject.zonesTargetDict:
            progressBar.setValue((rowCount/rowTotalCount) * 100)
            rowCount += 1

            zonesIDPrefix = zonesTargetTypeName.split('_')[0]
            zonesID = int(zonesIDPrefix[1:])
            zonesFeatTargetDict = setupObject.zonesTargetDict[zonesTargetTypeName]
            for featID in zonesFeatTargetDict:
                zonesTargetDatWriter.writerow([zonesID, featID, zonesFeatTargetDict[featID]])
    clearProgressBar()


def createZonesPropDatFile(setupObject):
    zonesPropDatFile = setupObject.inputPath + sep + 'zonecontrib.dat'
    with open(zonesPropDatFile, 'w', newline='') as out_file:
        zonesPropDatWriter = writer(out_file)
        zonesPropDatWriter.writerow(['zoneid', 'featureid', 'fraction'])

        featList = list(setupObject.targetDict.keys())
        featList.sort()

        progressBar = makeProgressBar('Making a new feat.dat file')
        rowTotalCount = len(featList)
        rowCount = 1

        for zonesPropTypeName in setupObject.zonesPropDict:
            progressBar.setValue((rowCount/rowTotalCount) * 100)
            rowCount += 1

            zonesIDPrefix = zonesPropTypeName.split('_')[0]
            zonesID = int(zonesIDPrefix[1:])
            zonesPropTargetDict = setupObject.zonesPropDict[zonesPropTypeName]
            for featID in zonesPropTargetDict:
                zonesPropDatWriter.writerow([zonesID, featID, zonesPropTargetDict[featID]])
    clearProgressBar()


def createZonesPuDatFile(setupObject):
    puZonesDatPathName = setupObject.inputPath + sep + 'pu.dat'

    puLayer = QgsVectorLayer(setupObject.puPath, 'Planning units', 'ogr')
    puFeatures = puLayer.getFeatures()
    puIDField = puLayer.fields().indexFromName('Unit_ID')

    progressBar = makeProgressBar('Making a new pu.dat file')
    polyCount = 1
    polyTotalCount = puLayer.featureCount()

    with open(puZonesDatPathName, 'w', newline='') as out_file:
        puDatWriter = writer(out_file)
        zonesPuDatFileHeaderList = ['id'] + makeZonesHeaderList(setupObject, '_Cost')
        puDatWriter.writerow(zonesPuDatFileHeaderList)

        zonesPuCostFieldList = makeZonesFieldList(setupObject, puLayer, '_Cost')
        for puFeature in puFeatures:
            progressBar.setValue((polyCount/polyTotalCount) * 100)
            polyCount += 1
            puDatRowList = makePUDatRowList(setupObject, puFeature, puIDField, zonesPuCostFieldList)
            puDatWriter.writerow(puDatRowList)
    clearProgressBar()


def makePUDatRowList(setupObject, puFeature, puIDField, zonesPuCostFieldList):
    decPrec = setupObject.decimalPlaces
    puAttributes = puFeature.attributes()
    puID = puAttributes[puIDField]
    puDatRowList = [puID]
    for costField in zonesPuCostFieldList:
        rawCostValue = puAttributes[costField]
        costValue = round(float(rawCostValue), decPrec)
        costValue = format(costValue, "." + str(decPrec) + "f")
        puDatRowList.append(costValue)

    return puDatRowList


def createZonesPUStatusDict(setupObject):
    zonesPUStatusDict = dict()
    puLayer = QgsVectorLayer(setupObject.puPath, 'Planning units', 'ogr')
    puFeatures = puLayer.getFeatures()
    puIDField = puLayer.fields().indexFromName('Unit_ID')
    zoneIDList = list(setupObject.zonesDict)

    for puFeature in puFeatures:
        puAttributes = puFeature.attributes()
        puID = puAttributes[puIDField]
        for zoneID in zoneIDList:
            statusField = puLayer.fields().indexFromName('Z' + str(zoneID) + '_Status')
            statusValue = puAttributes[statusField]
            try:
                puZonesPUStatusDict = zonesPUStatusDict[puID]
            except KeyError:
                puZonesPUStatusDict = dict()
            puZonesPUStatusDict[zoneID] = statusValue
            zonesPUStatusDict[puID] = puZonesPUStatusDict

    return zonesPUStatusDict


def createPuLockDatFile(setupObject, zonesPUStatusDict):
    puLockDatPathName = setupObject.inputPath + sep + 'pulock.dat'

    progressBar = makeProgressBar('Making a new pulock.dat file')
    lineCount = 1
    lineTotalCount = len(zonesPUStatusDict)

    with open(puLockDatPathName, 'w', newline='') as out_file:
        puLockDatWriter = writer(out_file)
        puLockDatWriter.writerow(['puid', 'zoneid'])
        puIDList = list(zonesPUStatusDict.keys())
        puIDList.sort()
        for puID in puIDList:
            progressBar.setValue((lineCount/lineTotalCount) * 100)
            lineCount += 1

            puZonesPUStatusDict = zonesPUStatusDict[puID]
            for zoneID in puZonesPUStatusDict:
                statusValue = puZonesPUStatusDict[zoneID]
                if statusValue == 'Locked' or statusValue == 'Earmarked':
                    puLockDatWriter.writerow([puID, zoneID])

    clearProgressBar()


def createPuZoneDatFile(setupObject, zonesPUStatusDict):
    puZonesDatPathName = setupObject.inputPath + sep + 'puzone.dat'

    progressBar = makeProgressBar('Making a new puzone.dat file')
    lineCount = 1
    lineTotalCount = len(zonesPUStatusDict)

    with open(puZonesDatPathName, 'w', newline='') as out_file:
        puZonesDatWriter = writer(out_file)
        puZonesDatWriter.writerow(['puid', 'zoneid'])
        puIDList = list(zonesPUStatusDict.keys())
        puIDList.sort()
        for puID in puIDList:
            progressBar.setValue((lineCount/lineTotalCount) * 100)
            lineCount += 1

            puZonesPUStatusDict = zonesPUStatusDict[puID]
            puZonesPUStatusList = puZonesPUStatusDict.values()
            if 'Excluded' in puZonesPUStatusList:
                restrictedToZoneList = makeRestrictedToZoneList(puZonesPUStatusDict)
                for zoneID in restrictedToZoneList:
                    puZonesDatWriter.writerow([puID, zoneID])

    clearProgressBar()


def makeRestrictedToZoneList(puZonesPUStatusDict):
    restrictedToZoneList = list()
    for zoneID in puZonesPUStatusDict:
        statusValue = puZonesPUStatusDict[zoneID]
        if statusValue != 'Excluded':
            restrictedToZoneList.append(zoneID)

    return restrictedToZoneList


def makeZonesHeaderList(setupObject, nameSuffix):
    zonesPuDatFileHeaderList = list()
    for zoneID in setupObject.zonesDict:
        newZoneHeader = 'Z' + str(zoneID) + nameSuffix
        zonesPuDatFileHeaderList.append(newZoneHeader)

    return zonesPuDatFileHeaderList


def makeZonesFieldList(setupObject, puLayer, nameSuffix):
    zonesPuCostFieldList = list()
    for zoneID in setupObject.zonesDict:
        zonesCostFieldName = 'Z' + str(zoneID) + nameSuffix
        zonesCostField = puLayer.fields().indexFromName(zonesCostFieldName)
        zonesPuCostFieldList.append(zonesCostField)

    return zonesPuCostFieldList


def createCostsDatFile(setupObject):
    costsDatPathName = setupObject.inputPath + sep + 'costs.dat'

    progressBar = makeProgressBar('Making a new costs.dat file')
    rowCount = 1
    totalRowCount = len(setupObject.zonesDict)

    with open(costsDatPathName, 'w', newline='') as out_file:
        costsDatWriter = writer(out_file)
        costsDatWriter.writerow(['costid', 'costname'])

        for zoneID in setupObject.zonesDict:
            progressBar.setValue((rowCount/totalRowCount) * 100)
            rowCount += 1
            costsDatWriter.writerow([zoneID, 'Z' + str(zoneID) + '_Cost'])
    clearProgressBar()


def createZonesDatFile(setupObject):
    zonesDatPathName = setupObject.inputPath + sep + 'zones.dat'

    progressBar = makeProgressBar('Making a new zones.dat file')
    rowCount = 1
    totalRowCount = len(setupObject.zonesDict)

    with open(zonesDatPathName, 'w', newline='') as out_file:
        costsDatWriter = writer(out_file)
        costsDatWriter.writerow(['zoneid', 'zonename'])

        for zoneID in setupObject.zonesDict:
            progressBar.setValue((rowCount/totalRowCount) * 100)
            costsDatWriter.writerow([zoneID, setupObject.zonesDict[zoneID]])
    clearProgressBar()


def createZonecostDatFile(setupObject):
    zonecostDatPathName = setupObject.inputPath + sep + 'zonecost.dat'

    progressBar = makeProgressBar('Making a new zonecost.dat file')
    rowCount = 1
    totalRowCount = len(setupObject.zonesDict)

    with open(zonecostDatPathName, 'w', newline='') as out_file:
        zonecostDatWriter = writer(out_file)
        zonecostDatWriter.writerow(['zoneid', 'costid', 'multiplier'])

        for zoneID in setupObject.zonesDict:
            progressBar.setValue((rowCount/totalRowCount) * 100)
            for costID in setupObject.zonesDict:
                if zoneID == costID:
                    zonecostDatWriter.writerow([zoneID, costID, 1])
                else:
                    zonecostDatWriter.writerow([zoneID, costID, 0])
    clearProgressBar()


# Marxan dialog ###########################


def returnZonesOutputName(setupObject):
    oldOutputName = setupObject.outputName
    outputPath = setupObject.outputPath
    oldOutputBestName = outputPath + sep + oldOutputName + '_best.csv'

    oldOutputNameStem = ''
    numValueBool = True
    for aNum in range(len(oldOutputName), 0, -1):
        aChar = oldOutputName[aNum - 1]
        try:
            int(aChar)
        except ValueError:
            numValueBool = False
        if numValueBool is False:
            oldOutputNameStem = aChar + oldOutputNameStem

    if path.isfile(oldOutputBestName):
        nameSuffix = 1
        newName = outputPath + sep + oldOutputNameStem + str(nameSuffix) + '_best.csv'
        while path.isfile(newName):
            nameSuffix += 1
            newName = outputPath + sep + oldOutputNameStem + str(nameSuffix) + '_best.csv'

        outputName = oldOutputNameStem + str(nameSuffix)
    else:
        outputName = oldOutputName

    return outputName


def createZonesBLMFile(ZonesMarxanDialog, setupObject):
    zonesBLMValueDict = makeZonesBLMValueDictFromDialog(ZonesMarxanDialog)
    setupObject.zonesBLMDict = zonesBLMValueDict

    puZonesBLMDatPathName = setupObject.inputPath + sep + 'zoneboundcost.dat'

    progressBar = makeProgressBar('Making a new zoneboundcost.dat file')
    lineCount = 1
    lineTotalCount = len(zonesBLMValueDict) + len(setupObject.zonesDict)

    with open(puZonesBLMDatPathName, 'w', newline='') as out_file:
        puZonesBoundCostDatWriter = writer(out_file)
        puZonesBoundCostDatWriter.writerow(['zoneid1', 'zoneid2', 'cost'])
        puZonesLabelList = list(zonesBLMValueDict.keys())
        puZonesLabelList.sort()
        for aZone in setupObject.zonesDict:
            progressBar.setValue((lineCount/lineTotalCount) * 100)
            lineCount += 1
            puZonesBoundCostDatWriter.writerow([aZone, aZone, 0])
        for aZoneLabel in puZonesLabelList:
            progressBar.setValue((lineCount/lineTotalCount) * 100)
            lineCount += 1
            zonesSplitLabel = aZoneLabel.split(' vs ')
            zoneID1 = zonesSplitLabel[0][5:]
            zoneID2 = zonesSplitLabel[1][5:]
            puZonesBoundCostDatWriter.writerow([zoneID1, zoneID2, zonesBLMValueDict[aZoneLabel]])
    clearProgressBar()

    return setupObject


def makeZonesBLMValueDictFromDialog(ZonesMarxanDialog):
    zonesBLMValueDict = dict()
    for aRow in range(0, ZonesMarxanDialog.blmTableWidget.rowCount()):
        labelValue = ZonesMarxanDialog.blmTableWidget.item(aRow, 0).text()
        blmValue = float(ZonesMarxanDialog.blmTableWidget.item(aRow, 1).text())
        zonesBLMValueDict[labelValue] = blmValue

    return zonesBLMValueDict


def makeZonesMarxanInputFile(setupObject, marxanParameterDict, addZoneTargetDatBool):
    if marxanParameterDict['extraOutputsBool']:
        extraOutputValue = '3'
    else:
        extraOutputValue = '0'
    if path.isfile(marxanParameterDict['marxanPath']):
        writeZonesMarxanInputFile(setupObject, marxanParameterDict, extraOutputValue, addZoneTargetDatBool)


def checkIfAddZoneTargetDatNeededBool(setupObject):
    zoneCount = len(setupObject.zonesDict)
    addZoneTargetDatBool = False
    for featID in setupObject.targetDict:
        targetListBegPos = 6 + (1 * zoneCount) + 1
        targetListEndPos = 6 + (2 * zoneCount) + 1
        zoneTargetList = setupObject.targetDict[featID][targetListBegPos:targetListEndPos]

        for aValue in zoneTargetList:
            if float(aValue) > 0:
                addZoneTargetDatBool = True

    return addZoneTargetDatBool


def writeZonesMarxanInputFile(setupObject, zonesMarxanParameterDict, extraOutputValue, addZoneTargetDatBool):
    with open(zonesMarxanParameterDict['marxanSetupPath'], 'w', newline='') as marxanFile:
        marxanWriter = writer(marxanFile)

        header1 = 'Input file for Marxan program, written by Ian Ball, Hugh Possingham and Matt Watts.'
        header2 = 'This file was generated using CLUZ, written by Bob Smith'
        marxanWriter.writerow([header1])
        marxanWriter.writerow([header2])
        marxanWriter.writerow([])

        marxanWriter.writerow(['General Parameters'])
        marxanWriter.writerow(['BLM ' + str(zonesMarxanParameterDict['blmValue'])])
        marxanWriter.writerow(['PROP  ' + str(zonesMarxanParameterDict['initialProp'])])
        marxanWriter.writerow(['RANDSEED -1'])
        marxanWriter.writerow(['NUMREPS ' + str(zonesMarxanParameterDict['numRun'])])
        marxanWriter.writerow(['AVAILABLEZONE  1'])  # "The available zone is treated as an unprotected zone in Marxan Z."

        marxanWriter.writerow([])

        marxanWriter.writerow(['Annealing Parameters'])
        marxanWriter.writerow(['NUMITNS ' + str(zonesMarxanParameterDict['numIter'])])
        marxanWriter.writerow(['STARTTEMP -1'])
        marxanWriter.writerow(['COOLFAC  -1'])
        marxanWriter.writerow(['NUMTEMP 10000'])
        marxanWriter.writerow([])

        marxanWriter.writerow(['Cost Threshold'])
        marxanWriter.writerow(['COSTTHRESH  0'])
        marxanWriter.writerow(['THRESHPEN1  0'])
        marxanWriter.writerow(['THRESHPEN2  0'])
        marxanWriter.writerow([])

        marxanWriter.writerow(['Input Files'])
        marxanWriter.writerow(['INPUTDIR ' + setupObject.inputPath])
        marxanWriter.writerow(['PUNAME pu.dat'])
        marxanWriter.writerow(['FEATNAME feat.dat'])
        marxanWriter.writerow(['PUVSPRNAME puvspr2.dat'])
        marxanWriter.writerow(['ZONESNAME zones.dat'])
        marxanWriter.writerow(['COSTSNAME costs.dat'])
        marxanWriter.writerow(['ZONECOSTNAME zonecost.dat'])
        marxanWriter.writerow(['BOUNDNAME bound.dat'])
        if setupObject.zonesBoundFlag:
            marxanWriter.writerow(['ZONEBOUNDCOSTNAME zoneboundcost.dat'])
        marxanWriter.writerow(['PUZONENAME puzone.dat'])
        marxanWriter.writerow(['PULOCKNAME pulock.dat'])
        if addZoneTargetDatBool:
            marxanWriter.writerow(['ZONETARGETNAME zonetarget.dat'])
        marxanWriter.writerow(['ZONECONTRIBNAME zonecontrib.dat'])

        marxanWriter.writerow([])

        marxanWriter.writerow(['Save Files'])
        marxanWriter.writerow(['SCENNAME ' + zonesMarxanParameterDict['outputName']])
        marxanWriter.writerow(['SAVERUN ' + extraOutputValue])
        marxanWriter.writerow(['SAVEBEST 3'])
        marxanWriter.writerow(['SAVESUMMARY 3'])
        marxanWriter.writerow(['SAVESCEN ' + extraOutputValue])
        marxanWriter.writerow(['SAVETARGMET 3'])
        marxanWriter.writerow(['SAVESUMSOLN 3'])

        marxanWriter.writerow(['SAVESOLUTIONSMATRIX 0'])
        marxanWriter.writerow(['SOLUTIONSMATRIXHEADERS 0'])
        marxanWriter.writerow(['SAVEPENALTY 0'])
        marxanWriter.writerow(['SAVELOG 3'])
        marxanWriter.writerow(['SAVEANNEALINGTRACE 0'])
        marxanWriter.writerow(['ANNEALINGTRACEROWS 0'])
        marxanWriter.writerow(['SAVEITIMPTRACE 0'])
        marxanWriter.writerow(['ITIMPTRACEROWS 0'])
        marxanWriter.writerow(['SAVEZONECONNECTIVITYSUM 0'])
        marxanWriter.writerow(['OUTPUTDIR ' + setupObject.outputPath])
        marxanWriter.writerow([])

        marxanWriter.writerow(['Program control.'])
        marxanWriter.writerow(['RUNMODE 1'])
        marxanWriter.writerow(['MISSLEVEL  ' + str(zonesMarxanParameterDict['missingProp'])])
        marxanWriter.writerow(['ITIMPTYPE 0'])
        marxanWriter.writerow(['VERBOSITY 3'])
        marxanWriter.writerow([])


def zonesMarxanUpdateSetupObject(ZonesMarxanDialog, setupObject, marxanParameterDict):
    setupObject.outputName = marxanParameterDict['outputName']
    setupObject.numIter = marxanParameterDict['numIter']
    setupObject.numRuns = marxanParameterDict['numRun']
    setupObject.blmValue = marxanParameterDict['blmValue']
    setupObject.boundFlag = ZonesMarxanDialog.boundCheckBox.isChecked()
    setupObject.extraOutputsFlag = ZonesMarxanDialog.extraCheckBox.isChecked()
    setupObject.zonesBoundFlag = ZonesMarxanDialog.boundZoneCheckBox.isChecked()
    setupObject.startProp = marxanParameterDict['initialProp']
    setupObject.targetProp = marxanParameterDict['missingProp']

    return setupObject


def addBestZonesMarxanOutputToPUShapefile(setupObject, bestZonesOutputFilePath, bestZonesFieldName):
    bestZonesDict = makeBestZonesDict(bestZonesOutputFilePath)
    puLayer = QgsVectorLayer(setupObject.puPath, "Planning units", "ogr")
    idFieldIndex = puLayer.fields().indexFromName("Unit_ID")

    bestZonesFieldIndex = puLayer.fields().indexFromName(bestZonesFieldName)
    provider = puLayer.dataProvider()
    if bestZonesFieldIndex == -1:
        provider.addAttributes([QgsField(bestZonesFieldName, QVariant.Int)])
        puLayer.updateFields()
    bestZonesFieldIndex = provider.fieldNameIndex(bestZonesFieldName)

    progressBar = makeProgressBar('Loading best output results')
    polyTotalCount = puLayer.featureCount()
    polyCount = 1

    puFeatures = puLayer.getFeatures()
    puLayer.startEditing()
    for puFeature in puFeatures:
        progressBar.setValue((polyCount/polyTotalCount) * 100)
        polyCount += 1

        puRow = puFeature.id()
        puAttributes = puFeature.attributes()
        puID = puAttributes[idFieldIndex]
        bestZone = bestZonesDict[puID]
        puLayer.changeAttributeValue(puRow, bestZonesFieldIndex, bestZone)
    puLayer.commitChanges()
    clearProgressBar()


def makeBestZonesDict(bestOutputFilePath):
    bestZonesDict = dict()
    with open(bestOutputFilePath, 'rt') as f:
        bestOutputReader = reader(f)
        next(bestOutputReader, None)  # skip the headers
        for row in bestOutputReader:
            puID = int(float(row[0]))
            bestZone = int(float(row[1]))
            bestZonesDict[puID] = bestZone

    return bestZonesDict


def addSummedZonesMarxanOutputToPUShapefile(setupObject, summedOutputFilePath):
    summedScoreDict = makeZonesSummedScoresDict(summedOutputFilePath)

    puLayer = QgsVectorLayer(setupObject.puPath, 'Planning units', 'ogr')
    provider = puLayer.dataProvider()
    idFieldIndex = provider.fieldNameIndex('Unit_ID')

    for zoneID in list(setupObject.zonesDict):
        zonesSummedFieldName = 'Z' + str(zoneID) + '_' + 'SFreq'
        summedFieldIndex = provider.fieldNameIndex(zonesSummedFieldName)
        if summedFieldIndex == -1:
            provider.addAttributes([QgsField(zonesSummedFieldName, QVariant.Int)])
            puLayer.updateFields()

    progressBar = makeProgressBar('Loading summed solution output results')
    polyTotalCount = puLayer.featureCount()
    polyCount = 1

    puFeatures = puLayer.getFeatures()
    puLayer.startEditing()
    for puFeature in puFeatures:
        progressBar.setValue((polyCount/polyTotalCount) * 100)
        polyCount += 1

        puRow = puFeature.id()
        puAttributes = puFeature.attributes()
        puID = puAttributes[idFieldIndex]
        for zoneID in list(setupObject.zonesDict):
            zonesSummedFieldName = 'Z' + str(zoneID) + '_' + 'SFreq'
            zoneSFScoreFieldIndex = provider.fieldNameIndex(zonesSummedFieldName)
            zoneNameText = setupObject.zonesDict[zoneID]
            sfScore = summedScoreDict[zoneNameText][puID]

            puLayer.changeAttributeValue(puRow, zoneSFScoreFieldIndex, sfScore)

    puLayer.commitChanges()
    clearProgressBar()


def makeZonesSummedScoresDict(summedOutputFile):
    zonesSummedScoreDict = dict()
    with open(summedOutputFile, 'rt') as f:
        zonesSFFileReader = reader(f)
        zonesSFFileHeader = next(zonesSFFileReader)
        zonesSFFileZoneNameList = zonesSFFileHeader[2:]
        for row in zonesSFFileReader:
            puID = int(row[0])
            zonesSFFileZoneSFScoreList = row[2:]
            for aCol in range(0, len(zonesSFFileZoneNameList)):
                headerName = zonesSFFileZoneNameList[aCol]
                sfScore = int(zonesSFFileZoneSFScoreList[aCol])

                try:
                    aZoneSummedScoreDict = zonesSummedScoreDict[headerName]
                except KeyError:
                    aZoneSummedScoreDict = dict()
                aZoneSummedScoreDict[puID] = sfScore
                zonesSummedScoreDict[headerName] = aZoneSummedScoreDict

    return zonesSummedScoreDict


def zonesLoadBestMarxanOutputToPUShapefile(ZonesLoadDialog, setupObject):
    zonesLoadBestOutputFilePath = ZonesLoadDialog.zonesBestLineEdit.text()
    zonesLoadBestCsvFileName = Path(zonesLoadBestOutputFilePath).stem
    zonesLoadBestFieldName = ZonesLoadDialog.zonesBestNameLineEdit.text()
    zonesLoadBestDict = zonesMakeBestValuesDict(zonesLoadBestOutputFilePath)
    puLayer = QgsVectorLayer(setupObject.puPath, 'Planning units', "ogr")
    idFieldIndex = puLayer.fields().indexFromName('Unit_ID')

    provider = puLayer.dataProvider()
    provider.addAttributes([QgsField(zonesLoadBestFieldName, QVariant.String)])
    puLayer.updateFields()
    bestLoadFieldIndex = provider.fieldNameIndex(zonesLoadBestFieldName)

    progressBar = makeProgressBar('Loading best output results')
    polyTotalCount = puLayer.featureCount()
    polyCount = 1

    puFeatures = puLayer.getFeatures()
    puLayer.startEditing()
    for puFeature in puFeatures:
        progressBar.setValue((polyCount/polyTotalCount) * 100)
        polyCount += 1

        puRow = puFeature.id()
        puAttributes = puFeature.attributes()
        puID = puAttributes[idFieldIndex]
        bestZoneValue = zonesLoadBestDict[puID]
        puLayer.changeAttributeValue(puRow, bestLoadFieldIndex, bestZoneValue)
    puLayer.commitChanges()
    clearProgressBar()

    return zonesLoadBestCsvFileName, zonesLoadBestFieldName


def zonesMakeBestValuesDict(bestOutputFilePath):
    zonesBestScoresDict = dict()
    with open(bestOutputFilePath, 'rt') as f:
        bestOutputReader = reader(f)
        next(bestOutputReader, None)  # skip the headers
        for row in bestOutputReader:
            puID = int(float(row[0]))
            bestZoneValue = int(float(row[1]))
            zonesBestScoresDict[puID] = bestZoneValue

    return zonesBestScoresDict


def zonesLoadSummedMarxanOutputToPUShapefile(ZonesLoadDialog, setupObject):
    summedOutputFilePath = ZonesLoadDialog.zonesSummedLineEdit.text()
    summedScoreDict = makeZonesSummedScoresDict(summedOutputFilePath)
    zonesLoadSFNamePrefix = ZonesLoadDialog.zonesSummedNameLineEdit.text()
    zonesLoadSFCsvFileName = Path(summedOutputFilePath).stem

    puLayer = QgsVectorLayer(setupObject.puPath, 'Planning units', 'ogr')
    provider = puLayer.dataProvider()
    idFieldIndex = provider.fieldNameIndex('Unit_ID')

    for zoneID in list(setupObject.zonesDict):
        zonesSummedFieldName = zonesLoadSFNamePrefix + '_Z' + str(zoneID) + '_' + 'SF'
        provider.addAttributes([QgsField(zonesSummedFieldName, QVariant.Int)])
        puLayer.updateFields()

    progressBar = makeProgressBar('Loading summed solution output results')
    polyTotalCount = puLayer.featureCount()
    polyCount = 1

    puFeatures = puLayer.getFeatures()
    puLayer.startEditing()
    maxSFScore = 0
    for puFeature in puFeatures:
        progressBar.setValue((polyCount/polyTotalCount) * 100)
        polyCount += 1

        puRow = puFeature.id()
        puAttributes = puFeature.attributes()
        puID = puAttributes[idFieldIndex]
        for zoneID in list(setupObject.zonesDict):
            zonesSummedFieldName = zonesLoadSFNamePrefix + '_Z' + str(zoneID) + '_SF'
            zoneSFScoreFieldIndex = provider.fieldNameIndex(zonesSummedFieldName)
            zoneNameText = setupObject.zonesDict[zoneID]
            sfScore = summedScoreDict[zoneNameText][puID]
            if sfScore > maxSFScore:
                maxSFScore = sfScore

            puLayer.changeAttributeValue(puRow, zoneSFScoreFieldIndex, sfScore)
    puLayer.commitChanges()
    clearProgressBar()

    return zonesLoadSFCsvFileName, zonesLoadSFNamePrefix, maxSFScore


def makeZonesCalibrateResultsDict(setupObject, marxanParameterDict):
    zonesCalibrateResultsDict = dict()
    scoreList = list()
    costList = list()
    connectivityCostList = list()
    penaltyList = list()
    mpmList = list()

    summaryTextPath = setupObject.outputPath + sep + marxanParameterDict['outputName'] + '_sum.csv'
    if path.isfile(summaryTextPath):
        with open(summaryTextPath, 'rt') as f:
            summaryReader = reader(f)
            headerList = next(summaryReader)
            for aRow in summaryReader:
                scoreValue = float(aRow[headerList.index('Score')])
                costValue = float(aRow[headerList.index('Cost')])
                connectivityCostValue = float(aRow[headerList.index('Connection Strength')])
                penaltyValue = float(aRow[headerList.index('Penalty')])
                mpmValue = float(aRow[headerList.index('MPM')])

                scoreList.append(scoreValue)
                costList.append(costValue)
                connectivityCostList.append(connectivityCostValue)
                penaltyList.append(penaltyValue)
                mpmList.append(mpmValue)

        medianScore = median(scoreList)
        medianCost = median(costList)
        medianConnectivity = median(connectivityCostList)
        medianPenalty = median(penaltyList)
        medianMPM = median(mpmList)

        zonesCalibrateResultsDict['numIter'] = marxanParameterDict['numIter']
        zonesCalibrateResultsDict['numRun'] = marxanParameterDict['numRun']
        zonesCalibrateResultsDict['blmValue'] = marxanParameterDict['blmValue']
        zonesCalibrateResultsDict['spfValue'] = marxanParameterDict['spfValue']
        zonesCalibrateResultsDict['outputName'] = str(marxanParameterDict['outputName'])

        zonesCalibrateResultsDict['medianScore'] = medianScore
        zonesCalibrateResultsDict['medianCost'] = medianCost
        zonesCalibrateResultsDict['medianConnectivity'] = medianConnectivity
        zonesCalibrateResultsDict['medianPenalty'] = medianPenalty
        zonesCalibrateResultsDict['medianMPM'] = medianMPM

    else:
        warningMessage('No files found', 'The Marxan summary file was not found and so this process will terminate.')

    return zonesCalibrateResultsDict
