"""
/***************************************************************************
                                 A QGIS plugin
 CLUZ for QGIS
                             -------------------
        begin                : 2022-26-08
        copyright            : (C) 2022 by Bob Smith, DICE
        email                : r.j.smith@kent.ac.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.core import QgsProject, QgsVectorLayer, QgsCategorizedSymbolRenderer, QgsFillSymbol, QgsRendererCategory
from qgis.core import QgsRendererRange, QgsGraduatedSymbolRenderer, QgsCoordinateReferenceSystem, QgsRectangle
from qgis.utils import iface


def addZonesPULayers(setupObject, legendPosition):
    allLayers = iface.mapCanvas().layers()
    layerNameList = list()
    for aLayer in allLayers:
        layerNameList.append(aLayer.name)

    root = QgsProject.instance().layerTreeRoot()
    for zoneID in list(setupObject.zonesDict)[::-1]:
        zonePULayerName = 'Z' + str(zoneID) + ' ' + setupObject.zonesDict[zoneID] + ' Planning units'
        statusField = 'Z' + str(zoneID) + '_Status'

        if not QgsProject.instance().mapLayersByName(zonePULayerName):
            puLayer = QgsVectorLayer(setupObject.puPath, zonePULayerName, 'ogr')
            categoryList = makeZonesPULayerLegendCategory()
            myRenderer = QgsCategorizedSymbolRenderer('', categoryList)
            myRenderer.setClassAttribute(statusField)
            puLayer.setRenderer(myRenderer)

            QgsProject.instance().addMapLayer(puLayer, False)
            root.insertLayer(legendPosition, puLayer)
            iface.setActiveLayer(puLayer)

            layerCRSText = puLayer.crs().authid()
            layerCRS = QgsCoordinateReferenceSystem(layerCRSText)
            iface.mapCanvas().setDestinationCrs(layerCRS)
            iface.mapCanvas().zoomToFeatureExtent(puLayer.extent())
            iface.mapCanvas().refresh()


def checkZonesPULayerPresent(setupObject):
    allLayers = iface.mapCanvas().layers()
    puLayerPresentBool = True
    zonePUNameList = list()
    for zoneID in setupObject.zonesDict:
        zonePULayerName = 'Z' + str(zoneID) + ' ' + setupObject.zonesDict[zoneID] + ' Planning units'
        zonePUNameList.append(zonePULayerName)

    zonesLayerCount = 0
    for aLayer in allLayers:
        if aLayer.name() in zonePUNameList:
            zonesLayerCount += 1
            zonePUNameList.remove(aLayer.name())

    if zonesLayerCount != 3:
        puLayerPresentBool = False

    return puLayerPresentBool


def makeZonesPULayerLegendCategory():
    categoryList = []
    # Set category 1
    cat1Value = 'Locked'
    cat1Label = 'Locked'
    cat1Symbol = QgsFillSymbol.createSimple({'style': 'solid', 'color': '#006633', 'color_border': '#006633'})
    myCat1 = QgsRendererCategory(cat1Value, cat1Symbol, cat1Label)
    categoryList.append(myCat1)

    # Set category 2
    cat2Value = 'Excluded'
    cat2Label = 'Excluded'
    cat2Symbol = QgsFillSymbol.createSimple({'style': 'solid', 'color': '#730083', 'color_border': '#730083'})
    myCat2 = QgsRendererCategory(cat2Value, cat2Symbol, cat2Label)
    categoryList.append(myCat2)

    # Set category 3
    cat3Value = 'Unassigned'
    cat3Label = 'Unassigned'
    cat3Symbol = QgsFillSymbol.createSimple({'style': 'solid', 'color': '#d9ffd9', 'color_border': '#d9ffd9'})
    myCat3 = QgsRendererCategory(cat3Value, cat3Symbol, cat3Label)
    categoryList.append(myCat3)

    # Set category 4
    cat4Value = 'Earmarked'
    cat4Label = 'Earmarked'
    cat4Symbol = QgsFillSymbol.createSimple({'style': 'solid', 'color': '#33cc33', 'color_border': '#33cc33'})
    myCat4 = QgsRendererCategory(cat4Value, cat4Symbol, cat4Label)
    categoryList.append(myCat4)

    return categoryList


def displayZonesBestOutput(setupObject, bestLayerName, zonesBestFieldName):
    bestZonesLayer = QgsVectorLayer(setupObject.puPath, bestLayerName, 'ogr')

    categoryList = list()
    zoneIDList = list(setupObject.zonesDict)
    zoneIDList.sort()

    for zoneID in setupObject.zonesDict:
        catLabel = 'Zone ' + str(zoneID) + ' ' + setupObject.zonesDict[zoneID]
        catSymbol = returnZoneCatSymbol(list(setupObject.zonesDict).index(zoneID))
        myCat = QgsRendererCategory(zoneID, catSymbol, catLabel)
        categoryList.append(myCat)

    myRenderer = QgsCategorizedSymbolRenderer('', categoryList)

    myRenderer.setClassAttribute(zonesBestFieldName)
    bestZonesLayer.setRenderer(myRenderer)
    QgsProject.instance().addMapLayer(bestZonesLayer)

    iface.mapCanvas().refresh()


def returnZoneCatSymbol(zoneID):
    colourDict = {0: '#c05000', 1: '#ffb070', 2: '#d0e0ff', 3: '#33a02c', 4: '#fb9a99', 5: '#e31a1c'}
    if zoneID < 6:
        catColour = colourDict[zoneID]
    else:
        catColour = '#e6e6e6'
    catSymbol = QgsFillSymbol.createSimple({'style': 'solid', 'color': catColour, 'color_border': catColour})

    return catSymbol


def reloadZonesPULayer(setupObject):
    root = QgsProject.instance().layerTreeRoot()

    layers = QgsProject.instance().mapLayers()
    nameList = list()
    for QGISFullname, layer in layers.items():
        layerName = str(layer.name())
        nameList.append(layerName)
        if str(layerName.endswith('Planning units')):
            QgsProject.instance().removeMapLayer(layer.id())

    for zoneID in list(setupObject.zonesDict):
        puLayerPosition = nameList.index('Z' + str(zoneID) + ' ' + setupObject.zonesDict[zoneID] + ' Planning units')
        puLayer = QgsVectorLayer(setupObject.puPath, 'Z' + str(zoneID) + ' ' + setupObject.zonesDict[zoneID] + ' Planning units', 'ogr')
        categoryList = makeZonesPULayerLegendCategory()
        myRenderer = QgsCategorizedSymbolRenderer('', categoryList)
        myRenderer.setClassAttribute('Z' + str(zoneID) + '_Status')
        puLayer.setRenderer(myRenderer)

        QgsProject.instance().addMapLayer(puLayer, False)
        root.insertLayer(puLayerPosition, puLayer)


def removePreviousZonesMarxanLayers():
    layers = QgsProject.instance().mapLayers()
    for QGISFullname, layer in layers.items():
        layerName = layer.name()
        if str(layerName)[0:6] == 'Best (' or str(layerName)[0:10] == 'SF_Score (':
            QgsProject.instance().removeMapLayer(layer.id())


def displayZonesSFLayer(setupObject, runNumber, outputName, fieldNamePrefix, fieldNameSuffix):
    colourList = ['#C5C2C5', '#CDCEB4', '#DEDEA3', '#EEE894', '#FFFA8B', '#FFE273', '#FFAA52', '#FF8541', '#FF6D31', '#FF0000']

    for zoneID in list(setupObject.zonesDict)[::-1]:
        zoneLayerName = str(zoneID) + ' ' + setupObject.zonesDict[zoneID] + ' SF' + ' (' + outputName + ')'
        graduatedLayer = QgsVectorLayer(setupObject.puPath, zoneLayerName, 'ogr')
        fieldName = fieldNamePrefix + str(zoneID) + fieldNameSuffix

        rangeList = list()
        minValue = 0
        incValue = float(runNumber) / 10

        for aValue in range(0, 10):
            maxValue = minValue + incValue
            if aValue == 9:
                maxValue = runNumber
            myColour = colourList[aValue]
            mySymbol = QgsFillSymbol.createSimple({'style': 'solid', 'color': myColour, 'color_border': myColour})
            minValueDisplay = round(minValue, setupObject.decimalPlaces)
            maxValueDisplay = round(minValue + incValue, setupObject.decimalPlaces)
            theRange = QgsRendererRange(minValueDisplay, maxValueDisplay, mySymbol, str(minValueDisplay) + ' - ' + str(maxValueDisplay))
            minValue = maxValue
            rangeList.insert(0, theRange)

        myRenderer = QgsGraduatedSymbolRenderer('', rangeList)
        myRenderer.setMode(QgsGraduatedSymbolRenderer.EqualInterval)
        myRenderer.setClassAttribute(fieldName)
        graduatedLayer.setRenderer(myRenderer)
        QgsProject.instance().addMapLayer(graduatedLayer)

    iface.mapCanvas().refresh()
