"""
/***************************************************************************
                                 A QGIS plugin
 CLUZ for QGIS
                             -------------------
        begin                : 2022-26-08
        copyright            : (C) 2022 by Bob Smith, DICE
        email                : r.j.smith@kent.ac.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.core import QgsProject, QgsVectorLayer

import csv
from os import sep, mkdir, path
from shutil import copy
from itertools import combinations

from .cluz_messages import warningMessage
from .zcluz_functions2 import convertOld2NewTargetDict, createZoneTargetCSVFile, createZonesPULayer, createZoneZonesCSVFile


# Transform to Marxan with Zones ########################################

def updateMarxanWithZonesTransformObject(zonesTransformDialog, zonesTransformObject):
    zonesTransformObject.updated = False
    zonesTransformObjectErrorCount = 0

    mwzPath = zonesTransformDialog.mwzInputLineEdit.text()
    if transformCheckMwzPathOK(mwzPath) is False:
        zonesTransformObjectErrorCount += 1
    blankFolderPath = zonesTransformDialog.folderLineEdit.text()
    if transformCheckBlankFolderPathOK(blankFolderPath) is False:
        zonesTransformObjectErrorCount += 1

    transformSetupFileName = zonesTransformDialog.setupLineEdit.text()
    transformPULayerFileName = zonesTransformDialog.puLineEdit.text()
    transformTargetFileName = zonesTransformDialog.targetLineEdit.text()
    transformZonesFileName = zonesTransformDialog.zonesLineEdit.text()

    fileNameList = [transformSetupFileName, transformPULayerFileName, transformTargetFileName, transformZonesFileName]
    fileTypeList = ['CLUZ setup file', 'Planning unit shapefile', 'Target csv file', 'Zones csv file']
    if transformFileNames(fileNameList, fileTypeList) is False:
        zonesTransformObjectErrorCount += 1

    if zonesTransformObjectErrorCount == 0:
        zonesTransformObject.updated = True
        zonesTransformObject.marxanWithZonesPath = mwzPath
        zonesTransformObject.transformFolderPath = blankFolderPath
        zonesTransformObject.setupFileName = transformSetupFileName.strip()  # Removes any whitespace from around name
        zonesTransformObject.puLayerName = transformPULayerFileName.strip()  # Removes any whitespace from around name
        zonesTransformObject.zonesTargetCSVFileName = transformTargetFileName.strip()  # Removes any whitespace from around name
        zonesTransformObject.zonesZonesCSVFileName = transformZonesFileName.strip()  # Removes any whitespace from around name
        zonesTransformObject.zoneNum = int(zonesTransformDialog.zoneComboBox.currentText())

    return zonesTransformObject


def transformCheckMwzPathOK(mwzPath):
    checkMwzPathOK = True
    if mwzPath == '':
        warningMessage('Marxan with Zones pathway', 'The Marxan with Zones pathway is blank - please specify a correct pathway.')
        checkMwzPathOK = False
    elif path.isfile(mwzPath):
        pass
    else:
        warningMessage('Marxan with Zones pathway', 'The Marxan with Zones pathway is not valid - please specify a correct pathway.')
        checkMwzPathOK = False

    return checkMwzPathOK


def transformCheckBlankFolderPathOK(blankFolderPath):
    checkFolderPathOK = True
    if blankFolderPath == '':
        warningMessage('Folder pathway', 'The empty folder pathway is blank - please specify a correct pathway.')
        checkFolderPathOK = False
    elif path.isdir(blankFolderPath):
        pass
    else:
        warningMessage('Folder pathway', 'The empty folder pathway is not valid - please specify a correct pathway.')
        checkFolderPathOK = False

    return checkFolderPathOK


def transformFileNames(fileNameList, fileTypeList):
    checkFileNamesOK = True
    for aNum in range(0, 4):
        aFileName = fileNameList[aNum]
        aFileNameType = fileTypeList[aNum]
        aStrippedFileName = aFileName.strip()
        if aStrippedFileName == '':
            warningMessage(aFileNameType + ' format problem', 'The ' + aFileNameType + ' name is blank. Please specify a name.')
            checkFileNamesOK = False

    return checkFileNamesOK


def zonesTransformMakeInputFolder(zonesTransformObject):
    transformInputFolderPath = zonesTransformObject.transformFolderPath + sep + 'input'
    mkdir(transformInputFolderPath)


def zonesTransformMakeOutputFolder(zonesTransformObject):
    transformOutputFolderPath = zonesTransformObject.transformFolderPath + sep + 'output'
    mkdir(transformOutputFolderPath)


def zonesTransformCopyPuvSpr2File(setupObject, zonesTransformObject):
    origPuvspr2DatFile = setupObject.inputPath + sep + 'puvspr2.dat'
    transformInputFolderPath = zonesTransformObject.transformFolderPath + sep + 'input'
    copy(origPuvspr2DatFile, transformInputFolderPath)


def zonesTransformPULayer(setupObject, zonesTransformObject):
    createZonesPULayer(setupObject, zonesTransformObject)


def zonesTransformTargetCSVFile(setupObject, zonesTransformObject):
    zoneTargetDict = convertOld2NewTargetDict(setupObject, zonesTransformObject.zoneNum)
    createZoneTargetCSVFile(zoneTargetDict, zonesTransformObject)


def zonesTransformZonesCSVFile(zonesTransformObject):
    createZoneZonesCSVFile(zonesTransformObject)


def zonesTransformCreateClzSetupFile(setupObject, zonesTransformObject):
    transformClzSetupFilePath = zonesTransformObject.transformFolderPath + sep + zonesTransformObject.setupFileName + '.clz'
    with open(transformClzSetupFilePath, 'w', newline='') as out_file:
        targetWriter = csv.writer(out_file)

        targetWriter.writerow(['analysis_type = MarxanWithZones'])
        targetWriter.writerow(['decimal_places = ' + str(setupObject.decimalPlaces)])
        targetWriter.writerow(['marxan_path = ' + zonesTransformObject.marxanWithZonesPath])
        targetWriter.writerow(['input_dir = ' + zonesTransformObject.transformFolderPath + sep + 'input'])
        targetWriter.writerow(['output_dir = ' + zonesTransformObject.transformFolderPath + sep + 'output'])
        targetWriter.writerow(['unit_theme = ' + zonesTransformObject.transformFolderPath + sep + zonesTransformObject.puLayerName + '.shp'])
        targetWriter.writerow(['target_table = ' + zonesTransformObject.transformFolderPath + sep + zonesTransformObject.zonesTargetCSVFileName + '.csv'])
        targetWriter.writerow(['zones_table = ' + zonesTransformObject.transformFolderPath + sep + zonesTransformObject.zonesZonesCSVFileName + '.csv'])
        targetWriter.writerow(['output_name = output1'])
        targetWriter.writerow(['num_iterations = 1000000'])
        targetWriter.writerow(['num_runs = 10'])
        targetWriter.writerow(['bound_flag = True'])
        targetWriter.writerow(['blm = 0.0'])
        targetWriter.writerow(['extra_flag = True'])
        targetWriter.writerow(['start_prop = 0.2'])
        targetWriter.writerow(['target_prop = 1.0'])
        targetWriter.writerow(['zones_bound_flag = False'])

        for (zoneA, zoneB) in list(combinations(range(1, zonesTransformObject.zoneNum + 1), 2)):
            targetWriter.writerow(['BLM_Zone_' + str(zoneA) + '_vs_Zone_' + str(zoneB) + ' = 0.0'])


# Import Vec data ###############################################################################################################

def zonesLoadVecThemesList(setupObject):
    zonesZonesPuLayerNameList = list()
    for zoneID in list(setupObject.zonesDict):
        zonesPuLayerName = 'Z' + str(zoneID) + ' ' + setupObject.zonesDict[zoneID] + ' Planning units'
        zonesZonesPuLayerNameList.append(zonesPuLayerName)

    listMapItems = QgsProject.instance().mapLayers()
    zonesLayerNameList = list()
    for nameCode, layer in listMapItems.items():
        layerName = layer.name()
        try:
            layerGeomType = layer.geometryType()
            if layerName in zonesZonesPuLayerNameList:
                pass
            else:
                if layerGeomType != 0:
                    zonesLayerNameList.append(str(layerName))
        except AttributeError:
            pass

    return zonesLayerNameList
