# """
# /***************************************************************************
#                                  A QGIS plugin
#  CLUZ for QGIS
#                              -------------------
#         begin                : 2022-26-08
#         copyright            : (C) 2022 by Bob Smith, DICE
#         email                : r.j.smith@kent.ac.uk
#  ***************************************************************************/
#
# /***************************************************************************
#  *                                                                         *
#  *   This program is free software; you can redistribute it and/or modify  *
#  *   it under the terms of the GNU General Public License as published by  *
#  *   the Free Software Foundation; either version 2 of the License, or     *
#  *   (at your option) any later version.                                   *
#  *                                                                         *
#  ***************************************************************************/
# """
#
from qgis.PyQt.QtWidgets import QFileDialog
from qgis.core import QgsVectorLayer
from qgis.utils import iface

from os import path
from csv import reader
from time import ctime

from .cluz_messages import warningMessage
from .zcluz_display import addZonesPULayers


def createAndCheckZonesFile(setupObject, checkBool):
    zonesCSVFilePath = setupObject.zonesPath
    zonesFileFieldNameList = ['id', 'name']
    try:
        with open(zonesCSVFilePath, 'rt') as f:
            zonesReader = reader(f)
            origHeaderList = next(zonesReader)

        lowercaseHeaderList = list()
        for aHeader in origHeaderList:
            lowercaseHeader = aHeader.lower()
            lowercaseHeaderList.append(lowercaseHeader)

        for aHeader in zonesFileFieldNameList:
            if lowercaseHeaderList.count(aHeader) == 0:
                warningMessage('Formatting error:', 'the Zones table is missing a ' + aHeader + ' field. Please select a table with the correct format.')
                checkBool = False
    except FileNotFoundError:
        checkBool = False

    return checkBool


def createAndCheckZonesTargetFile(setupObject, checkBool):
    targetCSVFilePath = setupObject.targetPath
    setupObject.targetFileDate = ctime(path.getmtime(targetCSVFilePath))
    targetFileFieldNameList = ['id', 'name', 'type', 'spf', 'target', 'ear+lock', 'total', 'pc_target']
    zoneIDList = setupObject.zonesDict.keys()
    try:
        with open(targetCSVFilePath, 'rt') as f:
            targetReader = reader(f)
            origHeaderList = next(targetReader)

        lowercaseHeaderList = list()
        for aHeader in origHeaderList:
            lowercaseHeader = aHeader.lower()
            lowercaseHeaderList.append(lowercaseHeader)

        for aHeader in targetFileFieldNameList:
            if lowercaseHeaderList.count(aHeader) == 0:
                warningMessage('Formatting error:', 'the Target table is missing a ' + aHeader + ' field. Please select a table with the correct format.')
                checkBool = False

        zCodeMismatchErrorCount = 0
        for aHeader in lowercaseHeaderList:
            if aHeader[0:1] == 'z':
                zoneIDText = aHeader.split('_')[0].replace('z', '')
                if int(zoneIDText) not in zoneIDList:
                    zCodeMismatchErrorCount += 1
        if zCodeMismatchErrorCount > 0:
            warningMessage('Formatting error:', 'there is a mismatch between the Zone ID numeric codes used in the Target table and those specified in the Zones table.')
            checkBool = False

    except FileNotFoundError:
        checkBool = False

    return checkBool


def createAndCheckZonesPuLayerFile(setupObject, checkBool):
    puLayer = QgsVectorLayer(setupObject.puPath, 'Planning units', 'ogr')
    fields = puLayer.fields()
    fieldDetailsList = list()
    titleText = 'Formatting error: '
    mainText = 'the planning unit shapefile must contain a field named '
    for aField in fields:
        fieldDetailsList.append((str(aField.name()), str(aField.typeName())))
    if fieldDetailsList.count(('Unit_ID', 'Integer')) == 0 and fieldDetailsList.count(('Unit_ID', 'Integer64')) == 0:
        warningMessage(titleText, mainText + 'Unit_ID containing integer values.')
        checkBool = False
    if fieldDetailsList.count(('Area', 'Real')) == 0:
        warningMessage(titleText, mainText + 'Area containing real number values.')
        checkBool = False

    for aZoneNum in list(setupObject.zonesDict):
        costFieldName = 'Z' + str(aZoneNum) + '_Cost'
        statusFieldName = 'Z' + str(aZoneNum) + '_Status'
        if fieldDetailsList.count((costFieldName, 'Real')) == 0:
            warningMessage(titleText, mainText + costFieldName + ' containing real number values.')
            checkBool = False
        if fieldDetailsList.count((statusFieldName, 'String')) == 0:
            warningMessage(titleText, mainText + statusFieldName + ' containing text values.')
            checkBool = False

    return checkBool


def checkAddZonesPULayer(setupObject):
    if setupObject.setupStatus == 'files_checked':
        if not checkZonesPULayerPresent(setupObject):
            addZonesPULayers(setupObject, 0)  # 0 = Position


def checkZonesPULayerPresent(setupObject):
    allLayers = iface.mapCanvas().layers()
    puLayerPresentBool = True
    zonePUNameList = list()
    for aZoneNum in list(setupObject.zonesDict):
        zonePULayerName = 'Z' + str(aZoneNum) + ' Planning units'
        zonePUNameList.append(zonePULayerName)

    zonesLayerCount = 0
    for aLayer in allLayers:
        if aLayer.name() in zonePUNameList:
            zonesLayerCount += 1
            zonePUNameList.remove(aLayer.name())

    if zonesLayerCount != len(setupObject.zonesDict):
        puLayerPresentBool = False

    return puLayerPresentBool
