"""
/***************************************************************************
                                 A QGIS plugin
 CLUZ for QGIS
                             -------------------
        begin                : 2022-26-08
        copyright            : (C) 2022 by Bob Smith, DICE
        email                : r.j.smith@kent.ac.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.core import QgsVectorLayer

from csv import reader
from os import listdir, path, remove, sep
from subprocess import Popen
from time import sleep

from .cluz_functions5 import checkNumRunsParaDict, makeCalibrateParameterValueList, addBestMarxanOutputToPUShapefile, makeMarxanBatFile
from .cluz_functions5 import checkInitialPropValueParaDict, checkMissingPropValueParaDict, makeCalibrateResultsDict, addSummedMarxanOutputToPUShapefile, waitingForMarxan
from .cluz_functions5 import returnOutputName, makeMarxanInputFile, checkNumIterParaDict, checkPermissionToUseMarxanFolderParaDict, checkBlmValueParaDict
from .cluz_functions5 import makeCalibrateSpecDatFile

from .zcluz_functions5 import makeZonesMarxanInputFile, checkIfAddZoneTargetDatNeededBool, makeZonesCalibrateResultsDict

from .cluz_display import displayGraduatedLayer, reloadPULayer, displayBestOutput
from .cluz_messages import warningMessage


# Marxan Dialog###########################################
def setDialogParameters(MarxanDialog, setupObject):
    MarxanDialog.boundLineEdit.setVisible(False)

    MarxanDialog.iterLineEdit.setText(str(setupObject.numIter))
    MarxanDialog.runLineEdit.setText(str(setupObject.numRuns))
    outputName = returnOutputName(setupObject)
    MarxanDialog.outputLineEdit.setText(outputName)
    MarxanDialog.boundLineEdit.setText(str(setupObject.blmValue))

    if setupObject.boundFlag:
        MarxanDialog.boundCheckBox.setChecked(True)
        MarxanDialog.boundLineEdit.setVisible(True)

    if setupObject.extraOutputsFlag:
        MarxanDialog.extraCheckBox.setChecked(True)

    MarxanDialog.missingLineEdit.setText(str(setupObject.targetProp))
    MarxanDialog.propLineEdit.setText(str(setupObject.startProp))


def makeMarxanRawParameterDict(MarxanDialog, setupObject):
    numIterString = MarxanDialog.iterLineEdit.text()
    numRunString = MarxanDialog.runLineEdit.text()
    outputName = str(MarxanDialog.outputLineEdit.text())
    setupObject.outputName = outputName
    if MarxanDialog.boundCheckBox.isChecked():
        blmValueString = MarxanDialog.boundLineEdit.text()
    else:
        blmValueString = "0"
    missingPropString = MarxanDialog.missingLineEdit.text()
    initialPropString = MarxanDialog.propLineEdit.text()
    extraOutputsBool = MarxanDialog.extraCheckBox.isChecked()

    marxanRawParameterDict = dict()
    marxanRawParameterDict['numIterString'] = numIterString
    marxanRawParameterDict['numRunString'] = numRunString
    marxanRawParameterDict['blmValueString'] = blmValueString
    marxanRawParameterDict['missingPropString'] = missingPropString
    marxanRawParameterDict['initialPropString'] = initialPropString
    marxanRawParameterDict['extraOutputsBool'] = extraOutputsBool
    marxanRawParameterDict['specName'] = 'spec.dat'
    marxanRawParameterDict['outputName'] = outputName
    marxanRawParameterDict['marxanPath'] = setupObject.marxanPath

    return marxanRawParameterDict


def makeMarxanParameterDict(setupObject, marxanRawParameterDict):
    marxanParameterDict = dict()
    marxanParameterDict['numIter'] = int(marxanRawParameterDict['numIterString'])
    marxanParameterDict['numRun'] = int(marxanRawParameterDict['numRunString'])
    marxanParameterDict['blmValue'] = float(marxanRawParameterDict['blmValueString'])
    marxanParameterDict['missingProp'] = float(marxanRawParameterDict['missingPropString'])
    marxanParameterDict['initialProp'] = float(marxanRawParameterDict['initialPropString'])

    marxanParameterDict['extraOutputsBool'] = marxanRawParameterDict['extraOutputsBool']
    marxanParameterDict['specName'] = marxanRawParameterDict['specName']
    marxanParameterDict['outputName'] = marxanRawParameterDict['outputName']
    marxanParameterDict['extraOutputsBool'] = marxanRawParameterDict['extraOutputsBool']

    marxanPath = setupObject.marxanPath
    marxanFolderName = path.dirname(marxanPath)
    marxanSetupPath = str(marxanFolderName) + sep + 'input.dat'
    marxanParameterDict["marxanPath"] = marxanPath
    marxanParameterDict["marxanSetupPath"] = marxanSetupPath

    return marxanParameterDict


def returnMarxanInputValuesOKBool(marxanParameterDict):
    marxanInputValuesOKBool = checkNumIterParaDict(marxanParameterDict['numIterString'])
    marxanInputValuesOKBool = checkNumRunsParaDict(marxanParameterDict['numRunString'], marxanInputValuesOKBool)
    marxanInputValuesOKBool = checkBlmValueParaDict(marxanParameterDict['blmValueString'], marxanInputValuesOKBool)
    marxanInputValuesOKBool = checkMissingPropValueParaDict(marxanParameterDict['missingPropString'], marxanInputValuesOKBool)
    marxanInputValuesOKBool = checkInitialPropValueParaDict(marxanParameterDict['initialPropString'], marxanInputValuesOKBool)
    marxanInputValuesOKBool = checkPermissionToUseMarxanFolderParaDict(marxanParameterDict, marxanInputValuesOKBool)

    return marxanInputValuesOKBool


def checkMarxanFilesExistBool(setupObject):
    marxanFilesExistBool = True
    puDatPath = setupObject.inputPath + sep + 'pu.dat'
    specDatPath = setupObject.inputPath + sep + 'spec.dat'
    puvspr2DatPath = setupObject.inputPath + sep + 'puvspr2.dat'

    if path.exists(puDatPath) is False:
        warningMessage('Missing Marxan file', 'There is no pu.dat file in the specified Marxan input folder. Please create the file using the Create Marxan input files function')
        marxanFilesExistBool = False
    if path.exists(specDatPath) is False:
        warningMessage('Missing Marxan file', 'There is no spec.dat file in the specified Marxan input folder. Please create the file using the Create Marxan input files function')
        marxanFilesExistBool = False
    if path.exists(puvspr2DatPath) is False:
        warningMessage('Missing Marxan file', 'There is no puvspr2.dat file in the specified Marxan input folder. Please create one')
        marxanFilesExistBool = False

    return marxanFilesExistBool


def launchMarxanAnalysis(setupObject, marxanParameterDict):
    makeMarxanInputFile(setupObject, marxanParameterDict)
    marxanBatFileName = makeMarxanBatFile(setupObject)
    Popen([marxanBatFileName])
    waitingForMarxan(setupObject, marxanParameterDict['outputName'])
    bestOutputFile = setupObject.outputPath + sep + marxanParameterDict['outputName'] + '_best.txt'
    summedOutputFile = setupObject.outputPath + sep + marxanParameterDict['outputName'] + '_ssoln.txt'

    return bestOutputFile, summedOutputFile


# Load previous results ########################################################

def returnInitialLoadFieldNames(setupObject):
    puLayer = QgsVectorLayer(setupObject.puPath, 'Planning units', 'ogr')
    fieldNameList = [field.name() for field in puLayer.fields()]
    bestName = 'IMP_BEST'
    bestSuffix = ''
    if bestName in fieldNameList:
        bestSuffix = 1
        while (bestName + str(bestSuffix)) in fieldNameList:
            bestSuffix += 1
    finalBestName = bestName + str(bestSuffix)

    summedName = 'IMP_SUM'
    summedSuffix = ''
    if summedName in fieldNameList:
        summedSuffix = 1
        while (summedName + str(summedSuffix)) in fieldNameList:
            summedSuffix += 1
    finalSummedName = summedName + str(summedSuffix)

    return finalBestName, finalSummedName


def check_LoadBestMarxanResult(LoadDialog, setupObject):
    bestFieldName = LoadDialog.bestNameLineEdit.text()
    if LoadDialog.bestCheckBox.isChecked():
        bestPath = LoadDialog.bestLineEdit.text()
    else:
        bestPath = 'blank'
    progressBool = checkImportBestFieldName(setupObject, bestFieldName)
    if progressBool:
        LoadDialog.close()
        if bestPath != 'blank':
            if path.isfile(bestPath):
                with open(bestPath, 'rt') as f:
                    bestReader = reader(f)
                    bestHeader = next(bestReader, None)  # skip the headers
                if bestHeader == setupObject.bestHeadingFieldNames:
                    addBestMarxanOutputToPUShapefile(setupObject, bestPath, bestFieldName)
                    bestShapefileName = bestFieldName
                    displayBestOutput(setupObject, bestFieldName, bestShapefileName)
                else:
                    warningMessage('Invalid file', 'The specified Marxan best output file is incorrectly formatted. It must contain only two fields named planning_unit and zone.')
            else:
                warningMessage('Incorrect pathname', 'The specified pathname for the Marxan best output is invalid. Please choose another one.')
        if bestPath != 'blank':
            reloadPULayer(setupObject)


def checkImportBestFieldName(setupObject, bestFieldName):
    puLayer = QgsVectorLayer(setupObject.puPath, 'Planning units', 'ogr')
    fieldNameList = [field.name() for field in puLayer.fields()]
    progressBool = True
    if bestFieldName in fieldNameList:
        warningMessage('Best field name duplication', 'The planning unit theme already contains a field named ' + bestFieldName + '. Please choose another name.')
        progressBool = False
    if len(bestFieldName) > 10:
        warningMessage('Invalid field name', 'The Best field name cannot be more than 10 characters long.')
        progressBool = False
        
    return progressBool


def check_LoadSummedMarxanResult(LoadDialog, setupObject):
    summedFieldName = LoadDialog.summedNameLineEdit.text()
    if LoadDialog.summedCheckBox.isChecked():
        summedPath = LoadDialog.summedLineEdit.text()
    else:
        summedPath = 'blank'
        
    progressBool = checkImportSummedFieldName(setupObject, summedFieldName)
    if progressBool:
        LoadDialog.close()
        if summedPath != 'blank':
            if path.isfile(summedPath):
                with open(summedPath, 'rt') as f:
                    summedReader = reader(f)
                    summedHeader = next(summedReader, None)  # skip the headers
                if summedHeader == setupObject.summedHeadingFieldNames:
                    addSummedMarxanOutputToPUShapefile(setupObject, summedPath, summedFieldName)
                    summedShapefileName = summedFieldName
                    displayGraduatedLayer(setupObject, summedFieldName, summedShapefileName, 1)  # 1 is SF legend code
                else:
                    warningMessage('Invalid file', 'The specified Marxan summed output file is incorrectly formatted. It must contain only two fields named planning_unit and number')
            else:
                warningMessage('Incorrect pathname', 'The specified pathname for the Marxan summed output is invalid. Please choose another one')
        if summedPath != 'blank':
            reloadPULayer(setupObject)


def checkImportSummedFieldName(setupObject, summedFieldName):
    puLayer = QgsVectorLayer(setupObject.puPath, 'Planning units', 'ogr')
    fieldNameList = [field.name() for field in puLayer.fields()]
    progressBool = True
    if summedFieldName in fieldNameList:
        warningMessage('Summed field name duplication', 'The planning unit theme already contains a field named ' + summedFieldName + '. Please choose another name.')
        progressBool = False
    if len(summedFieldName) > 10:
        warningMessage('Invalid field name', 'The Summed field name cannot be more than 10 characters long.')
        progressBool = False

    return progressBool


# Calibrate Dialog #######################################################

def setInitialValuesCalibrateDialog(CalibrateDialog, setupObject):
    CalibrateDialog.paraComboBox.addItems(['BLM', 'Number of iterations', 'Number of runs', 'SPF'])
    CalibrateDialog.iterLineEdit.setText(str(setupObject.numIter))
    CalibrateDialog.runLineEdit.setText(str(setupObject.numRuns))
    CalibrateDialog.boundLineEdit.setText(str(setupObject.blmValue))
    CalibrateDialog.boundLabel.setEnabled(False)
    CalibrateDialog.boundLineEdit.setEnabled(False)
    CalibrateDialog.boundLineEdit.setText('Specified above')
    CalibrateDialog.spfLabel.setEnabled(False)
    CalibrateDialog.spfLineEdit.setEnabled(False)
    CalibrateDialog.spfLineEdit.setText('As specified in spec.dat file')


def makeMarxanCalibrateRawParameterDict(CalibrateDialog):
    calibrateRawParameterDict = dict()
    calibrateRawParameterDict['numAnalysesText'] = CalibrateDialog.numberLineEdit.text()
    calibrateRawParameterDict['minAnalysesText'] = CalibrateDialog.minLineEdit.text()
    calibrateRawParameterDict['maxAnalysesText'] = CalibrateDialog.maxLineEdit.text()
    calibrateRawParameterDict['iterAnalysesText'] = CalibrateDialog.iterLineEdit.text()
    calibrateRawParameterDict['runAnalysesText'] = CalibrateDialog.runLineEdit.text()
    calibrateRawParameterDict['blmAnalysesText'] = CalibrateDialog.boundLineEdit.text()
    calibrateRawParameterDict['spfAnalysesText'] = CalibrateDialog.spfLineEdit.text()

    calibrateRawParameterDict['outputNameBase'] = CalibrateDialog.outputLineEdit.text()
    calibrateRawParameterDict['resultPathText'] = CalibrateDialog.resultsLineEdit.text()
    
    return calibrateRawParameterDict


def checkCalibrateAnalysisParameters(CalibrateDialog, calibrateRawParameterDict):
    numRunList = list()
    numIterList = list()
    blmValueList = list()
    spfList = list()
    checkBool = True

    checkBool = checkCalibrateOutputNameBase(calibrateRawParameterDict, checkBool)
    checkBool = checkCalibrateOutputNumAnalyses(calibrateRawParameterDict, checkBool)
    checkBool = checkCalibrateOutputMinMaxValues(CalibrateDialog, calibrateRawParameterDict, checkBool)

    if checkBool:
        exponentialBool = CalibrateDialog.expCheckBox.isChecked()
        parameterValueList = makeCalibrateParameterValueList(calibrateRawParameterDict, exponentialBool)
        numIterList, checkBool = checkCalibrateNumIterValue(CalibrateDialog, calibrateRawParameterDict, parameterValueList, checkBool)
        numRunList, checkBool = checkCalibrateNumRunValue(CalibrateDialog, calibrateRawParameterDict, parameterValueList, checkBool)
        blmValueList, checkBool = checkCalibrateBlmValue(CalibrateDialog, calibrateRawParameterDict, parameterValueList, checkBool)
        spfList, checkBool = checkCalibrateSpfValue(CalibrateDialog, calibrateRawParameterDict, parameterValueList, checkBool)

    return checkBool, numRunList, numIterList, blmValueList, spfList

        
def checkCalibrateOutputNameBase(calibrateRawParameterDict, checkBool):
    if calibrateRawParameterDict['outputNameBase'] == '':
        warningMessage('Incorrect output basename', 'The specified basename for the Marxan output files is blank. Please choose another one')
        checkBool = False

    return checkBool


def checkCalibrateOutputNumAnalyses(calibrateRawParameterDict, checkBool):
    try:
        numAnalyses = int(calibrateRawParameterDict['numAnalysesText'])
        if numAnalyses < 1:
            warningMessage('Incorrect format', 'The specified number of analysis is incorrectly formatted. It must be an integer and greater than 1.')
            checkBool = False
    except ValueError:
        warningMessage('Incorrect format', 'The specified number of analysis is incorrectly formatted. It must be an integer and greater than 1.')
        checkBool = False

    return checkBool


def checkCalibrateOutputMinMaxValues(CalibrateDialog, calibrateRawParameterDict, checkBool):
    if CalibrateDialog.paraComboBox.currentText() == 'BLM' or CalibrateDialog.paraComboBox.currentText() == 'SPF':
        valueFloor = 0
        minWarningMessageString = 'The specified minimum value is incorrectly formatted. It must be a number greater than 0.'
        maxWarningMessageString = 'The specified maximum value is incorrectly formatted. It must be a number greater than 0.'
    else:
        valueFloor = 1
        minWarningMessageString = 'The specified minimum value is incorrectly formatted. It must be a number greater than 1.'
        maxWarningMessageString = 'The specified maximum value is incorrectly formatted. It must be a number greater than 1.'
    try:
        testMinValue = float(calibrateRawParameterDict['minAnalysesText'])
        if testMinValue < valueFloor:
            warningMessage('Incorrect format', minWarningMessageString)
            checkBool = False
        if CalibrateDialog.paraComboBox.currentText() == 'Number of iterations' or CalibrateDialog.paraComboBox.currentText() == 'Number of runs':
            if testMinValue.is_integer() is False:
                warningMessage('Incorrect format', 'The specified minimum value has to be an integer.')
                checkBool = False
    except ValueError:
        warningMessage('Incorrect format', minWarningMessageString)
        checkBool = False
    try:
        testMaxValue = float(calibrateRawParameterDict['maxAnalysesText'])
        if testMaxValue < valueFloor:
            warningMessage('Incorrect format', maxWarningMessageString)
            checkBool = False
        if CalibrateDialog.paraComboBox.currentText() == 'Number of iterations' or CalibrateDialog.paraComboBox.currentText() == 'Number of runs':
            if testMaxValue.is_integer() is False:
                warningMessage('Incorrect format', 'The specified maximum value has to be an integer.')
                checkBool = False
    except ValueError:
        warningMessage('Incorrect format', maxWarningMessageString)
        checkBool = False
    try:
        if float(calibrateRawParameterDict['minAnalysesText']) > float(calibrateRawParameterDict['maxAnalysesText']):
            warningMessage('Incorrect format', 'The specified minimum value has to be lower than the specified maximum value.')
            checkBool = False
    except ValueError:
        checkBool = False

    return checkBool


def checkCalibrateNumIterValue(CalibrateDialog, calibrateRawParameterDict, parameterValueList, checkBool):
    numIterList = parameterValueList
    if CalibrateDialog.iterLineEdit.isEnabled():
        numIterText = CalibrateDialog.iterLineEdit.text()
        try:
            numIter = int(numIterText)
            numIterList = [numIter] * int(calibrateRawParameterDict['runAnalysesText'])
            if numIter < 10000:
                warningMessage('Incorrect format', 'The specified number of iterations is incorrectly formatted. It must be an integer greater than 10000 (Marxan uses 10000 temperature drops in the simulated annealing process in these analyses and the number of iterations must be greater than the number of temperature drops).')
                checkBool = False
        except ValueError:
            warningMessage('Incorrect format', 'The specified number of iterations is incorrectly formatted. It must be a positive integer.')
            checkBool = False

    return numIterList, checkBool


def checkCalibrateNumRunValue(CalibrateDialog, calibrateRawParameterDict, parameterValueList, checkBool):
    numRunList = parameterValueList
    if CalibrateDialog.runLineEdit.isEnabled():
        numRunText = CalibrateDialog.runLineEdit.text()
        try:
            numRun = int(numRunText)
            numRunList = [numRun] * int(calibrateRawParameterDict['runAnalysesText'])
            if numRun < 1:
                warningMessage('Incorrect format', 'The specified number of runs is incorrectly formatted. It must be a positive integer.')
                checkBool = False
        except ValueError:
            warningMessage('Incorrect format', 'The specified number of runs is incorrectly formatted. It must be a positive integer.')
            checkBool = False
        
    return numRunList, checkBool


def checkCalibrateBlmValue(CalibrateDialog, calibrateRawParameterDict, parameterValueList, checkBool):
    blmValueList = parameterValueList
    if CalibrateDialog.boundLineEdit.isEnabled():
        blmValueText = CalibrateDialog.boundLineEdit.text()
        try:
            blmValue = float(blmValueText)
            blmValueList = [blmValue] * int(calibrateRawParameterDict['runAnalysesText'])
            if blmValue < 0:
                warningMessage('Incorrect format', 'The specified BLM value is incorrectly formatted. It must be a positive number.')
                checkBool = False
        except ValueError:
            warningMessage('Incorrect format', 'The specified BLM value is incorrectly formatted. It must be a positive number.')
            checkBool = False

    return blmValueList, checkBool


def checkCalibrateSpfValue(CalibrateDialog, calibrateRawParameterDict, parameterValueList, checkBool):
    if CalibrateDialog.spfLineEdit.text() == 'As specified in spec.dat file':
        spfValueList = ['As specified in spec.dat file'] * int(calibrateRawParameterDict['runAnalysesText'])
    else:
        spfValueList = parameterValueList

    return spfValueList, checkBool


def runCalibrateMarxan(setupObject, calibrateRawParameterDict, numRunList, numIterList, blmValueList, spfValueList):
    calibrateResultsDict = dict()
    for analysisNumber in range(0, int(calibrateRawParameterDict['numAnalysesText'])):
        marxanParameterDict = makeCalibrateMarxanParameterDict(setupObject, calibrateRawParameterDict, numIterList, numRunList, blmValueList, spfValueList, analysisNumber)
        if len(set(spfValueList)) != 1:
            spfValue = spfValueList[analysisNumber]
            calibrateSpecDatFileName = 'calib_del_later_spec' + str(analysisNumber) + '.dat'
            marxanParameterDict['specName'] = calibrateSpecDatFileName
            makeCalibrateSpecDatFile(setupObject, calibrateSpecDatFileName, spfValue)
        if setupObject.analysisType != 'MarxanWithZones':
            makeMarxanInputFile(setupObject, marxanParameterDict)
        else:
            addZoneTargetDatBool = checkIfAddZoneTargetDatNeededBool(setupObject)
            makeZonesMarxanInputFile(setupObject, marxanParameterDict, addZoneTargetDatBool)
        marxanBatFileName = makeMarxanBatFile(setupObject)
        Popen([marxanBatFileName])
        sleep(2)
        waitingForMarxan(setupObject, calibrateRawParameterDict['outputNameBase'] + str(analysisNumber + 1))

        if setupObject.analysisType != 'MarxanWithZones':
            calibrateResultsDict[analysisNumber] = makeCalibrateResultsDict(setupObject, marxanParameterDict)
        else:
            calibrateResultsDict[analysisNumber] = makeZonesCalibrateResultsDict(setupObject, marxanParameterDict)
        if len(set(spfValueList)) != 1:
            removeCalibrateSpecDatFile(setupObject)

    return calibrateResultsDict


def makeCalibrateMarxanParameterDict(setupObject, calibrateRawParameterDict, numIterList, numRunList, blmValueList, spfValueList, analysisNumber):
    missingPropValue = 1.0
    initialPropValue = 0.2
    calibrateMarxanParameterDict = dict()
    calibrateMarxanParameterDict['numIter'] = numIterList[analysisNumber]
    calibrateMarxanParameterDict['numRun'] = numRunList[analysisNumber]
    calibrateMarxanParameterDict['blmValue'] = blmValueList[analysisNumber]
    calibrateMarxanParameterDict['spfValue'] = spfValueList[analysisNumber]
    calibrateMarxanParameterDict['missingProp'] = missingPropValue
    calibrateMarxanParameterDict['initialProp'] = initialPropValue

    calibrateMarxanParameterDict['outputName'] = calibrateRawParameterDict['outputNameBase'] + str(analysisNumber + 1)
    calibrateMarxanParameterDict['extraOutputsBool'] = True
    calibrateMarxanParameterDict['specName'] = 'spec.dat'

    marxanPath = setupObject.marxanPath
    marxanFolderName = path.dirname(marxanPath)
    marxanSetupPath = str(marxanFolderName) + sep + 'input.dat'
    calibrateMarxanParameterDict['marxanPath'] = marxanPath
    calibrateMarxanParameterDict['marxanSetupPath'] = marxanSetupPath

    return calibrateMarxanParameterDict


def removeCalibrateSpecDatFile(setupObject):
    inputFileList = listdir(setupObject.inputPath)
    for aFile in inputFileList:
        if 'calib_del_later_spec' in aFile:
            removeFilePath = setupObject.inputPath + sep + aFile
            remove(removeFilePath)
