"""
/***************************************************************************
                                 A QGIS plugin
 CLUZ for QGIS
                             -------------------
        begin                : 2022-26-08
        copyright            : (C) 2022 by Bob Smith, DICE
        email                : r.j.smith@kent.ac.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from qgis.PyQt.QtWidgets import QDialog, QFileDialog

from os import path
from sys import platform

from .cluz_setup import updateClzSetupFile
from .cluz_dialog5_code import check_LoadSummedMarxanResult, makeMarxanParameterDict, check_LoadBestMarxanResult, makeMarxanRawParameterDict
from .cluz_dialog5_code import returnMarxanInputValuesOKBool, setDialogParameters, launchMarxanAnalysis, returnInitialLoadFieldNames, checkMarxanFilesExistBool
from .cluz_dialog5_code import setInitialValuesCalibrateDialog, makeMarxanCalibrateRawParameterDict, checkCalibrateAnalysisParameters, runCalibrateMarxan
from .cluz_display import displayGraduatedLayer, removePreviousMarxanLayers, reloadPULayer, displayBestOutput
from .cluz_functions5 import createPuDatFile, marxanUpdateSetupObject, addBestMarxanOutputToPUShapefile, createBoundDatFile, createSpecDatFile, addSummedMarxanOutputToPUShapefile
from .cluz_functions5 import makeCalibrateOutputFile
from .cluz_messages import criticalMessage, successMessage

from cluz_form_inputs import Ui_inputsDialog
from cluz_form_marxan import Ui_marxanDialog
from cluz_form_load import Ui_loadDialog
from cluz_form_calibrate import Ui_calibrateDialog


class inputsDialog(QDialog, Ui_inputsDialog):
    def __init__(self, iface, setupObject):
        QDialog.__init__(self)
        self.iface = iface
        self.setupUi(self)
        self.boundextBox.setEnabled(False)
        self.okButton.clicked.connect(lambda: self.setCreateMarxanInputFiles(setupObject))

    def setCreateMarxanInputFiles(self, setupObject):
        messageStringList = list()
        if self.targetBox.isChecked():
            createSpecDatFile(setupObject)
            messageStringList.append('spec.dat')

        if self.puBox.isChecked():
            createPuDatFile(setupObject)
            messageStringList.append('pu.dat')

        if self.boundBox.isChecked():
            if self.boundextBox.isChecked() and self.boundextBox.isEnabled():
                extEdgeBool = True
            else:
                extEdgeBool = False
            createBoundDatFile(setupObject, extEdgeBool)
            messageStringList.append('bound.dat')

        if len(messageStringList) > 0:
            messageString = ''
            for aString in messageStringList:
                messageString += aString + ', '
            finalMessageString = messageString[:-2]
            successMessage('Marxan files:', 'the following files have been produced: ' + finalMessageString)

        self.close()


def checkCluzIsNotRunningOnMac():
    marxanBool = True
    if platform.startswith('darwin'):
        criticalMessage('CLUZ and MacOS', 'The current version of CLUZ cannot run Marxan on Mac computers. Sorry about that. Instead, you can run Marxan indepedently and load the results into CLUZ.')
        marxanBool = False
        
    return marxanBool


def checkMarxanPath(setupObject, marxanBool):
    if setupObject.marxanPath == 'blank':
        criticalMessage('Marxan path missing', 'The location of Marxan has not been specified. CLUZ will now open the CLUZ setup dialog box, so please specify a correct version.')
        marxanBool = False
    if path.exists(setupObject.marxanPath) is False:
        criticalMessage('Incorrect Marxan path', 'Marxan cannot be found at the specified pathway. CLUZ will now open the CLUZ setup dialog box, so please specify a correct version.')
        marxanBool = False
        
    return marxanBool


class marxanDialog(QDialog, Ui_marxanDialog):
    def __init__(self, iface, setupObject):
        QDialog.__init__(self)
        self.iface = iface
        self.setupUi(self)
        setDialogParameters(self, setupObject)
        self.startButton.clicked.connect(lambda: self.runMarxan(setupObject))

    def runMarxan(self, setupObject):
        marxanRawParameterDict = makeMarxanRawParameterDict(self, setupObject)
        marxanInputValuesOKBool = returnMarxanInputValuesOKBool(marxanRawParameterDict)
        marxanFilesExistBool = checkMarxanFilesExistBool(setupObject)
        if marxanInputValuesOKBool and marxanFilesExistBool:
            marxanParameterDict = makeMarxanParameterDict(setupObject, marxanRawParameterDict)
            createSpecDatFile(setupObject)

            setupObject = marxanUpdateSetupObject(self, setupObject, marxanParameterDict)
            updateClzSetupFile(setupObject, True)  # saveSuccessfulBool = True
            self.close()

            bestLayerName = 'Best (' + marxanParameterDict['outputName'] + ')'
            summedLayerName = 'SF_Score (' + marxanParameterDict['outputName'] + ')'
            bestOutputFile, summedOutputFile = launchMarxanAnalysis(setupObject, marxanParameterDict)

            addBestMarxanOutputToPUShapefile(setupObject, bestOutputFile, 'Best')
            addSummedMarxanOutputToPUShapefile(setupObject, summedOutputFile, 'SF_Score')

            reloadPULayer(setupObject)
            removePreviousMarxanLayers()
            displayGraduatedLayer(setupObject, 'SF_Score', summedLayerName, 1)  # 1 is SF legend code
            displayBestOutput(setupObject, 'Best', bestLayerName)  # Added second to be on top

            setupObject.TargetsMetAction.setEnabled(True)


class loadDialog(QDialog, Ui_loadDialog):
    def __init__(self, iface, setupObject):
        QDialog.__init__(self)
        self.iface = iface
        self.setupUi(self)

        self.bestLabel.setVisible(False)
        self.bestLineEdit.setVisible(False)
        self.bestNameLineEdit.setVisible(False)
        self.bestButton.setVisible(False)
        self.summedLabel.setVisible(False)
        self.summedLineEdit.setVisible(False)
        self.summedNameLineEdit.setVisible(False)
        self.summedButton.setVisible(False)

        bestName, summedName = returnInitialLoadFieldNames(setupObject)
        self.bestNameLineEdit.setText(bestName)
        self.summedNameLineEdit.setText(summedName)

        self.bestButton.clicked.connect(self.setBestPath)
        self.summedButton.clicked.connect(self.setSummedPath)
        self.okButton.clicked.connect(lambda: self.loadPreviousMarxanResults(setupObject))

    def setBestPath(self):
        (bestPathNameText, fileTypeDetailsText) = QFileDialog.getOpenFileName(self, 'Select Marxan best portfolio output', '*.txt')
        if bestPathNameText is not None:
            self.bestLineEdit.setText(bestPathNameText)

    def setSummedPath(self):
        (summedPathNameText, fileTypeDetailsText) = QFileDialog.getOpenFileName(self, 'Select Marxan summed solution output', '*.txt')
        if summedPathNameText is not None:
            self.summedLineEdit.setText(summedPathNameText)

    def loadPreviousMarxanResults(self, setupObject):
        check_LoadBestMarxanResult(self, setupObject)
        check_LoadSummedMarxanResult(self, setupObject)


class calibrateDialog(QDialog, Ui_calibrateDialog):
    def __init__(self, iface, setupObject):
        QDialog.__init__(self)
        self.iface = iface
        self.setupUi(self)
        setInitialValuesCalibrateDialog(self, setupObject)
        self.paraComboBox.activated.connect(lambda: self.combo_chosen(setupObject))
        if setupObject.analysisType == 'MarxanWithZones':
            self.setWindowTitle('Calibrate Marxan with Zones parameters')
            self.outputLabel.setText('Marxan with Zones output files base name     ')

        self.saveResultsButton.clicked.connect(self.saveResultsFile)
        self.runButton.clicked.connect(lambda: self.runCalibrateAnalysis(setupObject))

    def combo_chosen(self, setupObject):
        self.iterLineEdit.setText(str(setupObject.numIter))
        self.runLineEdit.setText(str(setupObject.numRuns))
        self.boundLineEdit.setText(str(setupObject.blmValue))
        self.iterLabel.setEnabled(True)
        self.iterLineEdit.setEnabled(True)
        self.runLabel.setEnabled(True)
        self.runLineEdit.setEnabled(True)
        self.boundLabel.setEnabled(True)
        self.boundLineEdit.setEnabled(True)
        parameterText = self.paraComboBox.currentText()
        if parameterText == 'Number of iterations':
            self.iterLabel.setEnabled(False)
            self.iterLineEdit.setEnabled(False)
            self.iterLineEdit.setText('Specified above')
            self.spfLineEdit.setText('As specified in spec.dat file')
        elif parameterText == 'Number of runs':
            self.runLabel.setEnabled(False)
            self.runLineEdit.setEnabled(False)
            self.runLineEdit.setText('Specified above')
            self.spfLineEdit.setText('As specified in spec.dat file')
        elif parameterText == 'BLM':
            self.boundLabel.setEnabled(False)
            self.boundLineEdit.setEnabled(False)
            self.boundLineEdit.setText('Specified above')
            self.spfLineEdit.setText('As specified in spec.dat file')
        elif parameterText == 'SPF':
            self.spfLabel.setEnabled(False)
            self.spfLineEdit.setEnabled(False)
            self.spfLineEdit.setText('Specified above')

    def saveResultsFile(self):
        (resultsFilePath, fileTypeDetailsText) = QFileDialog.getSaveFileName(self, 'Save Calibration results file', '*.csv')
        self.resultsLineEdit.setText(resultsFilePath)

    def runCalibrateAnalysis(self, setupObject):
        calibrateRawParameterDict = makeMarxanCalibrateRawParameterDict(self)
        checkBool, numRunList, numIterList, blmValueList, spfList = checkCalibrateAnalysisParameters(self, calibrateRawParameterDict)
        if checkBool:
            calibrateResultsDict = runCalibrateMarxan(setupObject, calibrateRawParameterDict, numRunList, numIterList, blmValueList, spfList)
            makeCalibrateOutputFile(calibrateRawParameterDict['resultPathText'], calibrateResultsDict)
            self.close()
