# -*- coding: utf-8 -*-
"""
/***************************************************************************
 CensusDownloaderDialog
                                 A QGIS plugin
 Downloads Census Data
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2024-08-06
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Amer Islam
        email                : aislam@ccrpc.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import pandas as pd
import geopandas as gpd
import requests
from qgis.PyQt import uic
from qgis.PyQt import QtWidgets
from qgis.PyQt.QtWidgets import QMessageBox, QTableWidgetItem
from PyQt5.QtWidgets import QInputDialog, QTableWidget
from qgis.core import QgsVectorLayer, QgsProject

# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'census_downloader_dialog_base.ui'))


class CensusDownloaderDialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, parent=None):
        """Constructor."""
        super(CensusDownloaderDialog, self).__init__(parent)
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)
        self.check_and_set_api_key() # Run the API check at the start
        self.filter_comboboxes() # Start by filtering the combo boxes to the appropriate selections
        self.progressBar.reset() # Reset the progress bar, this also helps avoid any cached progress bar values being rendered
        """
        Every time a combo box selection is changed it will run the filter method to select a valid selection and disable or invalid
        selections. Variables and variable groups will be cleared as appropriate as well.
        """
        # Combo Boxes
        self.cb_dataset.currentTextChanged.connect(self.filter_comboboxes)
        self.cb_dataset.currentTextChanged.connect(self.clear_variable)
        self.cb_dataset.currentTextChanged.connect(self.clear_variable_group)
        self.cb_table.currentTextChanged.connect(self.filter_comboboxes)
        self.cb_table.currentTextChanged.connect(self.clear_variable_group)
        self.cb_geography.currentTextChanged.connect(self.filter_comboboxes)
        self.cb_geography.currentTextChanged.connect(self.clear_variable)
        self.cb_predefined.currentTextChanged.connect(self.filter_comboboxes)
        self.cb_predefined.currentTextChanged.connect(self.clear_variable_group)
        self.cb_county.currentTextChanged.connect(self.filter_comboboxes)
        self.cb_county.currentTextChanged.connect(self.clear_variable)
        self.cb_year.currentTextChanged.connect(self.filter_comboboxes)
        self.cb_year.currentTextChanged.connect(self.clear_variable)
        self.cb_year.currentTextChanged.connect(self.clear_variable_group)
        # Table Widgets
        self.tw_census_view.setSelectionBehavior(QTableWidget.SelectRows) # Select selection behaviour to select entire row instead of each cell
        self.tw_selected_variables.setSelectionBehavior(QTableWidget.SelectRows) # Select selection behaviour to select entire row instead of each cell
        # Push Buttons
        self.pb_variables.clicked.connect(self.get_variables) # Get the list of variables for the specified group
        self.pb_selection.clicked.connect(self.add_to_variable_table) # Add variable from the varaible list
        self.pb_remove.clicked.connect(self.remove_from_variable_table) # Delete variable from the variable list
        self.pb_query.clicked.connect(self.execute_query) # Run the API calls

    # Check if census api key is an environment variable, if it is prompt user to input it
    # otherwise set the existing API key as census_key
    def check_and_set_api_key(self):
        self.census_key_dir = os.path.dirname(__file__) # Set path to working directory (this is the plugin directory)
        locale_path = os.path.join(self.census_key_dir, '.env') # define locale_path as '.env'

        if os.path.exists(locale_path) is False: # Look for the .env file defined above
            title = "Enter Census Key"
            message = "Please enter your Census API Key:"
            key = QInputDialog.getText(None, title, message) # Produce a dialogue box that takes the users API token and save it as key
            with open(locale_path, 'w') as file: # Write the key to a new file
                file.write(key[0])
        else:
            with open(locale_path, 'r') as file: # Read the key from the existing file
                return file.read() # Return the API key

    """
    This function is resonsible for making sure that all selections are valid and will result in a valid Census and TIGERweb API call.
    Nested dictionaries are combed with forloops based on the current selections.
    """
    def filter_comboboxes(self):
        # Dictionary of allowed tables for a given dataset
        allowed_tables = {
            "ACS5": ["Detailed Tables", "Subject Tables", "Data Profile"],
            "ACS1": ["Detailed Tables", "Subject Tables", "Data Profile"],
            "Decennial": ["Demographic and Housing Characteristics File", 'Demographic Profile']
        }
        # Dictionary of allowed geographies for a given dataset and table
        allowed_geography = {
            "ACS5": {
                "Detailed Tables": ['ZCTA','Census Tract','Subdivision (Township)','Urban Areas','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)', 'School District (unified)','Block Group'],
                "Subject Tables": ['ZCTA','Census Tract','Subdivision (Township)','Urban Areas','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)', 'School District (unified)'],
                "Data Profile": ['ZCTA','Census Tract','Subdivision (Township)','Urban Areas','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)', 'School District (unified)']
            },
            "ACS1": {
                "Detailed Tables": ['Subdivision (Township)','Urban Areas','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)', 'School District (unified)'],
                "Subject Tables": ['Subdivision (Township)','Urban Areas','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)', 'School District (unified)'],
                "Data Profile": ['Subdivision (Township)','Urban Areas','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)', 'School District (unified)']
            },
            "Decennial": {
                "Demographic and Housing Characteristics File": ['ZCTA','Census Tract','Subdivision (Township)','Urban Areas','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)', 'School District (unified)','Block Group'],
                'Demographic Profile': ['ZCTA','Census Tract','Subdivision (Township)','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)', 'School District (unified)']
            }
        }
        # Dictionary of allowed counties with a given geography
        allowed_counties = {
            'Census Tract': [
                'Champaign, IL',
                'Clark, IL',
                'Coles, IL',
                'Cumberland, IL',
                'De Witt, IL',
                'Douglas, IL',
                'Edgar, IL',
                'Macon, IL',
                'Moultrie, IL',
                'Piatt, IL',
                'Shelby, IL',
                'Vermilion, IL',
                'Ford, IL',
                'Iroquois, IL'
            ],
            'Subdivision (Township)': [
                'Champaign, IL',
                'Clark, IL',
                'Coles, IL',
                'Cumberland, IL',
                'De Witt, IL',
                'Douglas, IL',
                'Edgar, IL',
                'Macon, IL',
                'Moultrie, IL',
                'Piatt, IL',
                'Shelby, IL',
                'Vermilion, IL',
                'Ford, IL',
                'Iroquois, IL'
            ],
            'Urban Areas': [
                'Champaign, IL',
                'Coles, IL',
                'De Witt, IL',
                'Douglas, IL',
                'Edgar, IL',
                'Macon, IL',
                'Moultrie, IL',
                'Piatt, IL',
                'Shelby, IL',
                'Vermilion, IL',
                'Ford, IL',
                'Iroquois, IL'
            ],
            'Metropolitan Statistical Area': ['Champaign, IL','Macon, IL'],
            'Designated & Incorporated Place': [
                'Champaign, IL',
                'Clark, IL',
                'Coles, IL',
                'Cumberland, IL',
                'De Witt, IL',
                'Douglas, IL',
                'Edgar, IL',
                'Macon, IL',
                'Moultrie, IL',
                'Piatt, IL',
                'Shelby, IL',
                'Vermilion, IL',
                'Ford, IL',
                'Iroquois, IL'
            ],
            'School District (elementary)':[
                'Champaign, IL',
                'De Witt, IL',
                'Vermilion, IL',
                'Iroquois, IL'
            ],
            'School District (secondary)':[
                'Champaign, IL',
                'De Witt, IL',
                'Vermilion, IL',
                'Iroquois, IL'
            ],
            'School District (unified)':[
                'Champaign, IL',
                'Clark, IL',
                'Coles, IL',
                'Cumberland, IL',
                'De Witt, IL',
                'Douglas, IL',
                'Edgar, IL',
                'Macon, IL',
                'Moultrie, IL',
                'Piatt, IL',
                'Shelby, IL',
                'Vermilion, IL',
                'Ford, IL',
                'Iroquois, IL'
            ],
            'ZCTA':[
                'Champaign, IL',
                'Clark, IL',
                'Coles, IL',
                'Cumberland, IL',
                'De Witt, IL',
                'Douglas, IL',
                'Edgar, IL',
                'Macon, IL',
                'Moultrie, IL',
                'Piatt, IL',
                'Shelby, IL',
                'Vermilion, IL',
                'Ford, IL',
                'Iroquois, IL'
            ],
            'Block Group':[
                'Champaign, IL',
                'Clark, IL',
                'Coles, IL',
                'Cumberland, IL',
                'De Witt, IL',
                'Douglas, IL',
                'Edgar, IL',
                'Macon, IL',
                'Moultrie, IL',
                'Piatt, IL',
                'Shelby, IL',
                'Vermilion, IL',
                'Ford, IL',
                'Iroquois, IL'
            ]
        }
        # Dictionary of allowed groups based on given datasets and tables
        allowed_predefined = {
            "ACS5": {
                "Detailed Tables": ["Sex By Age","Race","Hispanic or Latino Origin","Receipt of Food Stamps/SNAP in the Past 12 Months by Disability Status for Households","Poverty Status of Individuals in the Past 12 Months by Living Arrangement","Means of Transportation to Work","Travel Time to Work","Age by Language Spoken at Home by Ability to Speak English for the Population 5 Years and Over","Tenure by Vehicles Available", "Language Spoken at Home by Ability to Speak English for the Population 5 Years and Over"],
                "Subject Tables": ["Age and Sex","Household and Families"],
                "Data Profile": ["Selected Social Characteristics in the United States","Selected Economic Characteristics","Selected Housing Characteristics"]
            },
            'ACS1': {
                "Detailed Tables": ["Sex By Age","Race","Hispanic or Latino Origin","Receipt of Food Stamps/SNAP in the Past 12 Months by Disability Status for Households","Poverty Status of Individuals in the Past 12 Months by Living Arrangement","Means of Transportation to Work","Travel Time to Work","Age by Language Spoken at Home by Ability to Speak English for the Population 5 Years and Over","Tenure by Vehicles Available", "Language Spoken at Home by Ability to Speak English for the Population 5 Years and Over"],
                "Subject Tables": ["Age and Sex","Household and Families"],
                "Data Profile": ["Selected Social Characteristics in the United States","Selected Economic Characteristics","Selected Housing Characteristics"]
            },
            "Decennial":{
                "Demographic and Housing Characteristics File": ["Sex by Age for Selected Age Categories"],
                'Demographic Profile': ["Profile of General Population and Housing Characteristics"]
            }
        }
        # Dictionary of allowed years based on given datasets, tables, and geographies
        allowed_years = {
            "ACS5": {
                "Detailed Tables": {
                    "ZCTA": list(map(str, range(2011, 2025))), # Create a range from the bottom year to your max year + 1 (The max is excluded), then convert these string with map, and then convert to a list.
                    "Block Group": list(map(str, range(2013, 2025))),
                    "Census Tract": list(map(str, range(2009, 2025))),
                    "Subdivision (Township)": list(map(str, range(2009, 2025))),
                    'Urban Areas': list(map(str, range(2009, 2011))) + list(map(str, range(2012, 2025))),
                    'Metropolitan Statistical Area': list(map(str, range(2009, 2022))) + list(map(str, range(2023, 2025))),
                    'Designated & Incorporated Place': list(map(str, range(2009, 2025))),
                    'School District (elementary)': list(map(str, range(2009, 2025))),
                    'School District (secondary)': list(map(str, range(2009, 2025))),
                    'School District (unified)': list(map(str, range(2009, 2025))),
                },
                "Subject Tables": {
                    "ZCTA": list(map(str, range(2011, 2025))),
                    "Census Tract": list(map(str, range(2010, 2025))),
                    "Subdivision (Township)": list(map(str, range(2010, 2025))),
                    'Urban Areas': list(map(str, range(2009, 2011))) + list(map(str, range(2012, 2025))),
                    'Metropolitan Statistical Area': list(map(str, range(2010, 2022))) + list(map(str, range(2023, 2025))),
                    'Designated & Incorporated Place': list(map(str, range(2010, 2025))),
                    'School District (elementary)': list(map(str, range(2010, 2025))),
                    'School District (secondary)': list(map(str, range(2010, 2025))),
                    'School District (unified)': list(map(str, range(2010, 2025))),
                },
                "Data Profile": {
                    "ZCTA": list(map(str, range(2011, 2025))),
                    "Census Tract": list(map(str, range(2010, 2025))),
                    "Subdivision (Township)": list(map(str, range(2010, 2025))),
                    'Urban Areas': list(map(str, range(2010, 2011))) + list(map(str, range(2012, 2025))),
                    'Metropolitan Statistical Area': list(map(str, range(2010, 2022))) + list(map(str, range(2023, 2025))),
                    'Designated & Incorporated Place': list(map(str, range(2010, 2025))),
                    'School District (elementary)': list(map(str, range(2010, 2025))),
                    'School District (secondary)': list(map(str, range(2010, 2025))),
                    'School District (unified)': list(map(str, range(2010, 2025))),
                },
            },
            "ACS1":{
                "Detailed Tables": {
                    "Subdivision (Township)": list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                    'Urban Areas': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                    'Metropolitan Statistical Area': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2022))) + list(map(str, range(2023, 2025))),
                    'Designated & Incorporated Place': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                    'School District (elementary)': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                    'School District (secondary)': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                    'School District (unified)': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                },
                "Subject Tables": {
                    "Subdivision (Township)": list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                    'Urban Areas': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                    'Metropolitan Statistical Area': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2022))) + list(map(str, range(2023, 2025))),
                    'Designated & Incorporated Place': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                    'School District (elementary)': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                    'School District (secondary)': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                    'School District (unified)': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                },
                "Data Profile": {
                    "Subdivision (Township)": list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                    'Urban Areas': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                    'Metropolitan Statistical Area': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2022))) + list(map(str, range(2023, 2025))),
                    'Designated & Incorporated Place': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                    'School District (elementary)': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                    'School District (secondary)': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                    'School District (unified)': list(map(str, range(2014, 2020))) + list(map(str, range(2021, 2025))),
                },
            },
            "Decennial": {
                "Demographic and Housing Characteristics File": {
                    "ZCTA": "2020",
                    "Block Group": "2020",
                    "Census Tract": "2020",
                    "Subdivision (Township)": "2020",
                    'Urban Areas': "2020",
                    'Metropolitan Statistical Area': "2020",
                    'Designated & Incorporated Place': "2020",
                    'School District (elementary)': "2020",
                    'School District (secondary)': "2020",
                    'School District (unified)': "2020",
                },
                'Demographic Profile': {
                    "ZCTA": "2020",
                    "Census Tract": "2020",
                    "Subdivision (Township)": "2020",
                    'Metropolitan Statistical Area': "2020",
                    'Designated & Incorporated Place': "2020",
                    'School District (elementary)': "2020",
                    'School District (secondary)': "2020",
                    'School District (unified)': "2020",
                },
            }
        }
        """
        For a given combo_box the method will first select a valid selection from the allowed dictionary.
        Then it loops through all the options to disable all the invalid ones, and enable all valid choices.
        """
        def enable_and_select(combo_box,allowed):
            # If the current selection does not match a value in the allowed list
            # Loop through the entire combo box until you fid a value that
            # is in the allowed list. Once you find one set that as the current selection.
            if combo_box.currentText() not in allowed:
                for i in range(combo_box.count()):
                    if combo_box.itemText(i) in allowed:
                        combo_box.setCurrentIndex(i)
            # Loop through the entire combo box again. For all values that match
            # a value in the allowed list, set them as enabled. For all that do
            # not match a value, set them as disabled.
            for i in range(combo_box.count()):
                name = combo_box.itemText(i)
                if name in allowed:
                    combo_box.model().item(i).setEnabled(True)
                else:
                    combo_box.model().item(i).setEnabled(False)
        # Take the current combo box and match it with the appropriate allowed dictionary, and current selection and place them as inputs in the method defined above.
        # While typically you can store the current text as variables themselvse I opted not ot do that as the selections update is such quick succesion when being
        # rearranged that simply having it pull a fresh one whenever is it inputted back into the method worked best to prevent any errors.
        enable_and_select(self.cb_table,allowed_tables[self.cb_dataset.currentText()]) # Filters the table combobox
        enable_and_select(self.cb_geography,allowed_geography[self.cb_dataset.currentText()][self.cb_table.currentText()])# Filters the geography combobox
        enable_and_select(self.cb_county,allowed_counties[self.cb_geography.currentText()])# Filters the county combobox
        enable_and_select(self.cb_predefined,allowed_predefined[self.cb_dataset.currentText()][self.cb_table.currentText()])# Filters the predefined groups combobox
        enable_and_select(self.cb_year,allowed_years[self.cb_dataset.currentText()][self.cb_table.currentText()][self.cb_geography.currentText()])# Filters the year combobox
    """
    This method is responsible for gathering all of the variables when the user clicks "submit"
    and populating the left QWidgetTable. It works by taking the combo box values and plugging
    them into nested dictionaries to get a portion of the census url string. This is them all
    put together in an fstring that passes the completed url to the requests fucntion which
    then scrapes the page and presents as a table.
    """
    def get_variables(self):
        # Store all of the required combobox selections as variables
        selected_year = self.cb_year.currentText()
        selected_dataset = self.cb_dataset.currentText()
        selected_table = self.cb_table.currentText()
        selected_group = self.cb_predefined.currentText()
        # Dictionary for dataset portion of URL
        dataset = {
            "ACS5": "acs/acs5",
            "ACS1": "acs/acs1",
            "Decennial": "dec",
        }
        # Dictionary for table portion of URL
        table = {
            "Detailed Tables": "",
            "Subject Tables": "/subject",
            "Data Profile": "/profile",
            "Demographic and Housing Characteristics File": "/dhc",
            "Demographic Profile": "/dp",
        }
        # Dictionary for group portion of URL
        predefined_groups = {
            "Sex By Age": "B01001",
            "Race": "B02001",
            "Hispanic or Latino Origin": "B03003",
            "Receipt of Food Stamps/SNAP in the Past 12 Months by Disability Status for Households": "B22010",
            "Poverty Status of Individuals in the Past 12 Months by Living Arrangement": "B17021",
            "Means of Transportation to Work": "B08301",
            "Travel Time to Work": "B08303",
            "Age by Language Spoken at Home by Ability to Speak English for the Population 5 Years and Over": "B16004",
            "Tenure by Vehicles Available": "B25044",
            "Age and Sex": "S0101",
            "Household and Families": "S1101",
            "Selected Social Characteristics in the United States": "DP02",
            "Selected Economic Characteristics": "DP03",
            "Selected Housing Characteristics": "DP04",
            "Sex by Age for Selected Age Categories": "P12",
            "Profile of General Population and Housing Characteristics": "DP1",
        }

        self.census_url = f'https://api.census.gov/data/{selected_year}/{dataset[selected_dataset]}{table[selected_table]}/variables.json' # The fstring responsible for pulling the variables
        print(f"Census Variable Url: {self.census_url}") # Print the Census URL to their QGIS python console. This is for trouble shooting in case seomething goes wrong with pulling it, or just in case they want the api link for any reason

        self.data = requests.get(self.census_url) # Query the link
        # Status code 200 means that the request of succesful.
        # This confirms that the census is up and that subsequent
        # pulls will be successful.
        if self.data.status_code == 200:
            self.data = self.data.json() # convert the data into parsable JSON format
            # Initialize the lists and loop through to fill them with with the appropriate
            # date if they match the selected group
            variables = []
            labels = []
            for key, value in self.data['variables'].items(): # Interate through the JSON for each key, and all of its values as well
                if value.get('group') == predefined_groups[selected_group]: # Cehck each value's group metadata, and if it is equal to the code we are looking for
                    variables.append(key) # Add the first iterator to the variable list
                    labels.append(value['label']) # Add the second iterator's label metadata to the label list
            # Create a Dataframe and give it the columns Variables, Labels, and table
            # Lists will fill the content of these columns.
            self.df = pd.DataFrame(
                {
                    'Variables': variables,
                    'Labels': labels,
                    'Table': selected_table
                }
            )
            self.df = self.df.sort_values('Variables', ascending=True) # Sort the dataframe to ascending order based on the Variables column
            self.write_df_to_qtable(self.df, self.tw_census_view) # Convert the dataframe to a qtable object and population the left Qtable oject in the UI
        else:
            QMessageBox.warning(self,"No Variables","The Census API is Currently Offline")
            return False # Adding this return as a way to quit running the plugin in case of the Census being down. This could be done with a try/catch/exception instead by I'm going with this. Feel free to change.
    # this method clears the left Qtable Widget
    def clear_variable_group(self):
        self.tw_census_view.clear()
        self.tw_census_view.setRowCount(0)
        self.tw_census_view.setColumnCount(0)
    # this method clears the right Qtable Widget
    def clear_variable(self):
        self.tw_selected_variables.clear()
        self.tw_selected_variables.setRowCount(0)
        self.tw_selected_variables.setColumnCount(0)
    """
    This handles the addition of items from the left Qtable to the right Qtable.
    I came back to this method after a while and realized hash maps are much quicker
    than using lists, O(1) vs O(n). Similar efficiency gains could probably be made
    elsewhere by utilizing sets in other methods.
    """
    def add_to_variable_table(self):
        # Initialize table and set header names
        headers = ["Variables","Labels","Table"]
        self.tw_selected_variables.setColumnCount(3)
        self.tw_selected_variables.setHorizontalHeaderLabels(headers)
        # Iterate through each cell in the QTable object tw_census_view and extract its row.
        # Then store it in set selected_rows
        selected_rows = {item.row() for item in self.tw_census_view.selectedItems()}
        # Store the value of the first cell of each row in set selected_data
        selected_data = {
            self.tw_selected_variables.item(row, 0).text()
            for row in range(self.tw_selected_variables.rowCount())
            if self.tw_selected_variables.item(row, 0) # Null Check
        }
        # Store each cell(if it exists) from the left QTableWidget and
        # store it into row_data_list
        row_data_list = [
            [
                self.tw_census_view.item(row, col).text() if self.tw_census_view.item(row, col) else ""
                for col in range(min(self.tw_census_view.columnCount(), 3))
            ]
            for row in selected_rows
        ]
        # Loop through the row_data_list, if it is not already in the selected dataset
        # then insert it as a row into the right QTableWidget
        current_row = self.tw_selected_variables.rowCount()
        for row_data in (rd for rd in row_data_list if rd[0] not in selected_data):
            self.tw_selected_variables.insertRow(current_row)
            for col, data in enumerate(row_data[:3]):
                self.tw_selected_variables.setItem(current_row, col, QTableWidgetItem(data))
            current_row += 1
    # Method to remove variables from right QTableWidget
    def remove_from_variable_table(self):
        selected_items = self.tw_selected_variables.selectedItems() # Store selected
        # Iterate through each cell in the QTable object tw_census_view and extract its row.
        # Then store it in set selected_rows
        selected_rows = set(item.row() for item in selected_items)
        for row in sorted(selected_rows, reverse=True):
            self.tw_selected_variables.removeRow(row)
    # Converts dataframe to Qtable for easier handling
    def write_df_to_qtable(self,df,table):
        # Inherit df headers and dimensions
        headers = list(df)
        table.setRowCount(df.shape[0])
        table.setColumnCount(df.shape[1])
        table.setHorizontalHeaderLabels(headers)
        df_array = df.values # Create an array for the dataframe values
        # Loop through the dataframe and set each value as a QTableWidgetItem
        for row in range(df.shape[0]):
            for col in range(df.shape[1]):
                table.setItem(row, col, QTableWidgetItem(str(df_array[row,col])))
    # Converts Qtable to dataframe
    def qtablewidget_to_dataframe(self, qtw):
        # Collect QTableWidget dimensions
        num_cols = qtw.columnCount()
        num_rows = qtw.rowCount()
        data = []
        # Loop through QTableWidget and get all the items
        # and place them into the data list
        for row in range(num_rows):
            row_data = []
            for col in range(num_cols):
                item = qtw.item(row, col)
                if item:
                    row_data.append(str(item.text()))
                else:
                    row_data.append('')
            data.append(row_data)
        df = pd.DataFrame(data, columns=['Variables','Labels','Table']) # Set column names
        return df
    """
    The method that runs when clicking the 'Run' button. Checks that everything is valid and will produce a valid
    Census result before running.
    """
    def execute_query(self):
        variable_df = self.qtablewidget_to_dataframe(self.tw_selected_variables)
        # Check that either groupcall is checked or there is at least one variable selected
        if self.cb_groupcall.isChecked() is False and self.tw_selected_variables.rowCount() == 0:
            QMessageBox.warning(self,"Group Call or Manual","Make sure either group call is selected or variables have been inputted into the Text Box")
        # Ensure MPA can only be selected for Champaign County
        elif self.cb_mpa.isChecked() is True and self.cb_county.currentText() != 'Champaign, IL':
            QMessageBox.warning(self,"MPA","MPA can only be selected for Champaign County")
        # Check for duplicate label names
        elif variable_df['Labels'].duplicated().any():
            QMessageBox.warning(self,"DUPLICATE LABELS","Cannot Have Duplicate Label names")
        else:
            self.update_url() # If all checks pass then run the census query
    """
    The main core of the plugin. This method creates the TIGER and Census API from
    dictionaries with an fstring. First the TIGERweb geometries are pulled. If the
    Census API feaures require filtering, then they use these geometries as a filter.
    They are both stored as dataframes and merged and then served as a QGIS layer in
    memory, or as a local geopackage if specified.
    """
    def update_url(self):
        # Store values necessary for creating the API links
        census_key = self.check_and_set_api_key()
        selected_year = self.cb_year.currentText()
        selected_dataset = self.cb_dataset.currentText()
        selected_table = self.cb_table.currentText()
        selected_geography = self.cb_geography.currentText()
        selected_group = self.cb_predefined.currentText()
        selected_county = self.cb_county.currentText()
        # Dictionary for County Codes
        county_code = {
            'Champaign, IL': '019',
            'Clark, IL': '023',
            'Coles, IL': '029',
            'Cumberland, IL': '035',
            'De Witt, IL': '039',
            'Douglas, IL': '041',
            'Edgar, IL': '045',
            'Macon, IL': '115',
            'Moultrie, IL': '139',
            'Piatt, IL': '147',
            'Shelby, IL': '173',
            'Vermilion, IL': '183',
            'Ford, IL': '053',
            'Iroquois, IL': '075'
        }
        # An ESRI envelope in CRS 4326 that encompasses all of the counties in the plugin
        bounding_area='-89.2347,+39.1056,-87.4688,+41.0453'
        # Centroid point of each county in CRS 4326
        centroid = {
            'Champaign, IL':'-88.1908,+40.1891',
            'Clark, IL': '-87.6901,+39.3980',
            'Coles, IL': '-88.1845,+39.4966, ',
            'Cumberland, IL': '-88.2425,+39.2739',
            'De Witt, IL': '-88.9700,+40.1589',
            'Douglas, IL': '-88.2903,+39.8033',
            'Edgar, IL': '-87.6984,+39.6104',
            'Macon, IL':'-88.9530,+39.8355',
            'Moultrie, IL': '-88.61109,+39.60027',
            'Piatt, IL': '-88.5718,+40.0302',
            'Shelby, IL': '-88.7984,+39.4123',
            'Vermilion, IL': '-87.6338,+40.1316',
            'Ford, IL': '-88.1117,+40.4631',
            'Iroquois, IL': '-87.7492,+40.7774'
        }
        # Dictionary of the last portion of the TIGERweb REST API link. It depends on year and geography.
        query = {
            '2014': {
                'Census Tract': f"ACS2014/MapServer/8/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Subdivision (Township)': f"ACS2014/MapServer/20/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Urban Areas' :{
                    'Areas': f'ACS2014/MapServer/62/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                    'Clusters': f'ACS2014/MapServer/64/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson'
                },
                'Metropolitan Statistical Area': f'ACS2014/MapServer/78/query?timeRelation=esriTimeRelationOverlaps&geometry={centroid[selected_county]}&geometryType=esriGeometryPoint&inSR=4326&spatialRel=esriSpatialRelIntersects&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Designated & Incorporated Place' :{
                    'Designated': f'ACS2014/MapServer/28/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                    'Incorporated': f'ACS2014/MapServer/26/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson'
                },
                'School District (elementary)': f'ACS2014/MapServer/16/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (secondary)': f'ACS2014/MapServer/14/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (unified)': f'ACS2014/MapServer/12/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'ZCTA': f'ACS2014/MapServer/2/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Block Group': f"ACS2014/MapServer/10/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
            },
            '2015': {
                'Census Tract': f"ACS2015/MapServer/8/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Subdivision (Township)': f"ACS2015/MapServer/20/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Urban Areas' :{
                    'Areas': f'ACS2015/MapServer/62/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                    'Clusters': f'ACS2015/MapServer/64/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson'
                },
                'Metropolitan Statistical Area': f'ACS2015/MapServer/78/query?timeRelation=esriTimeRelationOverlaps&geometry={centroid[selected_county]}&geometryType=esriGeometryPoint&inSR=4326&spatialRel=esriSpatialRelIntersects&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Designated & Incorporated Place' :{
                    'Designated': f'ACS2015/MapServer/28/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                    'Incorporated': f'ACS2015/MapServer/26/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson'
                },
                'School District (elementary)': f'ACS2015/MapServer/16/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (secondary)': f'ACS2015/MapServer/14/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (unified)': f'ACS2015/MapServer/12/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'ZCTA': f'ACS2015/MapServer/2/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Block Group': f"ACS2015/MapServer/10/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
            },
            '2016': {
                'Census Tract': f"ACS2016/MapServer/8/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Subdivision (Township)': f"ACS2016/MapServer/20/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Urban Areas' :{
                    'Areas': f'ACS2016/MapServer/62/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                    'Clusters': f'ACS2016/MapServer/64/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson'
                },
                'Metropolitan Statistical Area': f'ACS2016/MapServer/78/query?timeRelation=esriTimeRelationOverlaps&geometry={centroid[selected_county]}&geometryType=esriGeometryPoint&inSR=4326&spatialRel=esriSpatialRelIntersects&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Designated & Incorporated Place' :{
                    'Designated': f'ACS2016/MapServer/28/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                    'Incorporated': f'ACS2016/MapServer/26/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson'
                },
                'School District (elementary)': f'ACS2016/MapServer/16/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (secondary)': f'ACS2016/MapServer/14/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (unified)': f'ACS2016/MapServer/12/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'ZCTA': f'ACS2016/MapServer/2/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Block Group': f"ACS2016/MapServer/10/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
            },
            '2017': {
                'Census Tract': f"ACS2017/MapServer/8/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Subdivision (Township)': f"ACS2017/MapServer/20/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Urban Areas' :{
                    'Areas': f'ACS2017/MapServer/62/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                    'Clusters': f'ACS2017/MapServer/64/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson'
                },
                'Metropolitan Statistical Area': f'ACS2017/MapServer/78/query?timeRelation=esriTimeRelationOverlaps&geometry={centroid[selected_county]}&geometryType=esriGeometryPoint&inSR=4326&spatialRel=esriSpatialRelIntersects&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Designated & Incorporated Place' :{
                    'Designated': f'ACS2017/MapServer/28/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                    'Incorporated': f'ACS2017/MapServer/26/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson'
                },
                'School District (elementary)': f'ACS2017/MapServer/16/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (secondary)': f'ACS2017/MapServer/14/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (unified)': f'ACS2017/MapServer/12/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'ZCTA': f'ACS2017/MapServer/2/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Block Group': f"ACS2017/MapServer/10/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
            },
            '2018': {
                'Census Tract': f"ACS2018/MapServer/8/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Subdivision (Township)': f"ACS2018/MapServer/20/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Urban Areas' :{
                    'Areas': f'ACS2018/MapServer/62/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                    'Clusters': f'ACS2018/MapServer/64/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson'
                },
                'Metropolitan Statistical Area': f'ACS2018/MapServer/78/query?timeRelation=esriTimeRelationOverlaps&geometry={centroid[selected_county]}&geometryType=esriGeometryPoint&inSR=4326&spatialRel=esriSpatialRelIntersects&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Designated & Incorporated Place' :{
                    'Designated': f'ACS2018/MapServer/28/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                    'Incorporated': f'ACS2018/MapServer/26/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson'
                },
                'School District (elementary)': f'ACS2018/MapServer/16/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (secondary)': f'ACS2018/MapServer/14/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (unified)': f'ACS2018/MapServer/12/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'ZCTA': f'ACS2018/MapServer/2/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Block Group': f"ACS2018/MapServer/10/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
            },
            '2019': {
                'Census Tract': f"ACS2019/MapServer/8/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Subdivision (Township)': f"ACS2019/MapServer/20/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Urban Areas' :{
                    'Areas': f'ACS2019/MapServer/62/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                    'Clusters': f'ACS2019/MapServer/64/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson'
                },
                'Metropolitan Statistical Area': f'ACS2019/MapServer/78/query?timeRelation=esriTimeRelationOverlaps&geometry={centroid[selected_county]}&geometryType=esriGeometryPoint&inSR=4326&spatialRel=esriSpatialRelIntersects&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Designated & Incorporated Place' :{
                    'Designated': f'ACS2019/MapServer/28/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                    'Incorporated': f'ACS2019/MapServer/26/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson'
                },
                'School District (elementary)': f'ACS2019/MapServer/16/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (secondary)': f'ACS2019/MapServer/14/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (unified)': f'ACS2019/MapServer/12/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'ZCTA': f'ACS2019/MapServer/2/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Block Group': f"ACS2019/MapServer/10/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
            },
            '2020': {
                'Census Tract': f"Census2020/MapServer/6/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Subdivision (Township)': f"Census2020/MapServer/20/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Urban Areas': f'Census2020/MapServer/88/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Metropolitan Statistical Area': f'Census2020/MapServer/76/query?timeRelation=esriTimeRelationOverlaps&geometry={centroid[selected_county]}&geometryType=esriGeometryPoint&inSR=4326&spatialRel=esriSpatialRelIntersects&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Designated & Incorporated Place' :{
                    'Designated': f'Census2020/MapServer/28/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                    'Incorporated': f'Census2020/MapServer/26/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson'
                },
                'School District (elementary)': f'Census2020/MapServer/16/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (secondary)': f'Census2020/MapServer/14/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (unified)': f'Census2020/MapServer/12/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'ZCTA': f'Census2020/MapServer/84/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Block Group': f"Census2020/MapServer/8/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
            },
            '2021': {
                'Census Tract': f"ACS2021/MapServer/6/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Subdivision (Township)': f"ACS2021/MapServer/18/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Urban Areas': f'Census2020/MapServer/88/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Metropolitan Statistical Area': f'ACS2021/MapServer/72/query?timeRelation=esriTimeRelationOverlaps&geometry={centroid[selected_county]}&geometryType=esriGeometryPoint&inSR=4326&spatialRel=esriSpatialRelIntersects&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Designated & Incorporated Place' :{
                    'Designated': f'ACS2021/MapServer/26/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                    'Incorporated': f'ACS2021/MapServer/24/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson'
                },
                'School District (elementary)': f'ACS2021/MapServer/14/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (secondary)': f'ACS2021/MapServer/12/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (unified)': f'ACS2021/MapServer/10/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'ZCTA': f'ACS2021/MapServer/0/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Block Group': f"ACS2021/MapServer/8/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
            },
            '2022': {
                'Census Tract': f"ACS2022/MapServer/6/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Subdivision (Township)': f"ACS2022/MapServer/18/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Urban Areas': f'Census2020/MapServer/88/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Designated & Incorporated Place' :{
                    'Designated': f'ACS2022/MapServer/26/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                    'Incorporated': f'ACS2022/MapServer/24/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson'
                },
                'School District (elementary)': f'ACS2022/MapServer/14/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (secondary)': f'ACS2022/MapServer/12/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (unified)': f'ACS2022/MapServer/10/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'ZCTA': f'ACS2022/MapServer/0/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Block Group': f"ACS2022/MapServer/8/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
            },
            '2023': {
                'Census Tract': f"ACS2023/MapServer/8/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Subdivision (Township)': f"ACS2023/MapServer/22/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
                'Urban Areas': f'ACS2023/MapServer/88/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Metropolitan Statistical Area': f'ACS2023/MapServer/93/query?timeRelation=esriTimeRelationOverlaps&geometry={centroid[selected_county]}&geometryType=esriGeometryPoint&inSR=4326&spatialRel=esriSpatialRelIntersects&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Designated & Incorporated Place' :{
                    'Designated': f'ACS2023/MapServer/30/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                    'Incorporated': f'ACS2023/MapServer/28/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson'
                },
                'School District (elementary)': f'ACS2023/MapServer/18/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (secondary)': f'ACS2023/MapServer/16/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'School District (unified)': f'ACS2023/MapServer/14/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'ZCTA': f'ACS2023/MapServer/2/query?timeRelation=esriTimeRelationOverlaps&geometry={bounding_area}&geometryType=esriGeometryEnvelope&inSR=4326&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson',
                'Block Group': f"ACS2023/MapServer/10/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson",
            }
        }
        # Lists of geographies that require county specific clipping, and codes for filter respectively
        requires_clipping = ['Urban Areas','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)','School District (unified)','ZCTA']
        requires_codes = ['Urban Areas','Metropolitan Statistical Area','ZCTA']
        # Dictionary of geographies that require combining multiple layers
        combination = {
            'Urban Areas': list(map(str, range(2014, 2020))),
            'Designated & Incorporated Place': list(map(str, range(2014, 2024)))
        }
        geometry = gpd.GeoDataFrame()# Initialize geometry Geodataframe
        self.progressBar.setValue(10)# This is the progress bar progress. 0 Is min, 100 is max. Increase it as the method runs to track progress
        if selected_geography in combination and selected_year in combination[selected_geography]:
            # Determine if the geography needs to be combined from two different sources
            # or can be pulled directly.
            self.progressBar.setValue(20)
            for value in (list(query[selected_year][selected_geography].values())):
                # Loop through the all the urls within the selected geography dictionary
                self.tiger_url = f"https://tigerweb.geo.census.gov/arcgis/rest/services/TIGERweb/tigerWMS_{value}"
                tiger_data = requests.get(self.tiger_url)
                print(f"Tiger URLs: {self.tiger_url}")
                if tiger_data.status_code == 200: # Check if url is valid
                    temp_gdf = gpd.read_file(self.tiger_url) # Store first url as geodataframe
                    geometry = pd.concat([geometry, temp_gdf]) # Combine subsequent gdfs into geometry geodataframe
                else:
                    QMessageBox.warning(self,"TIGERWEB ERROR","Invalid Tiger URL or TIGER servers are currently busy") # If the url isn't valid put a Warning message
                    self.progressBar.reset() # Reset the progress bar
                    return False # Return false to break out of program
            self.progressBar.setValue(30)
        else:
            self.progressBar.setValue(20)
            self.tiger_url = f"https://tigerweb.geo.census.gov/arcgis/rest/services/TIGERweb/tigerWMS_{query[selected_year][selected_geography]}" # Finish url from dictionary
            print(f"Tiger URL: {self.tiger_url}")
            tiger_data = requests.get(self.tiger_url)
            self.progressBar.setValue(30)
            if tiger_data.status_code == 200:# Check if url is valid
                geometry = gpd.read_file(self.tiger_url)# Store as gdf
            else:
                QMessageBox.warning(self,"TIGERWEB ERROR","Invalid Tiger URL or TIGER servers are currently busy")# Throw error message
                self.progressBar.reset()# Reset progress bar
                return False# Return false to break out of program
        # Extract only the GEOID and geometry column
        geometry = geometry[['GEOID','geometry']]
        # Determine if geography requires to clip geom data to county outlines
        if selected_geography in requires_clipping:
            # Dictionary for layer number for every year
            layer = {
                '2014': '84',
                '2015': '84',
                '2016': '84',
                '2017': '84',
                '2018': '84',
                '2019': '84',
                '2020': '82',
                '2021': '78',
                '2022': '78',
                '2023': '82',
            }
            # 2020 uses a different url than the rest.
            if selected_year != '2020':
                county_url = f"https://tigerweb.geo.census.gov/arcgis/rest/services/TIGERweb/tigerWMS_ACS{selected_year}/MapServer/{layer[selected_year]}/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson"
            else:
                county_url = f"https://tigerweb.geo.census.gov/arcgis/rest/services/TIGERweb/tigerWMS_Census{selected_year}/MapServer/{layer[selected_year]}/query?where=STATE='17'+AND+COUNTY='{county_code[selected_county]}'&outFields=GEOID&returnGeometry=true&outSR=3435&f=geojson"
            print("County URL:" + county_url) # Print Census URL to puthon console for trouble shooting
            county_data = requests.get(county_url)
            if county_data.status_code == 200: # Check if url is valid
                county_geometry = gpd.read_file(county_url)
                geometry = geometry[geometry.intersects(county_geometry.unary_union)] # Filter merge to only include features that intersect with county_geometry
            else:
                # Thow message error, reset progress bar, and return false to break out of program
                QMessageBox.warning(self,"TIGERWEB ERROR","Invalid Tiger URL or TIGER servers are currently busy")
                self.progressBar.reset()
                return False

        self.progressBar.setValue(40)

        if self.cb_mpa.isChecked() is True:
            # If MPA checkbox is selected
            # Find it in the local plugin filesystem
            # and clip it with the geometry gdf defined earlier
            directory = os.path.dirname(os.path.abspath(__file__))
            mpa_path = os.path.join(directory, 'MPA.gpkg')
            mpa_boundary = gpd.read_file(mpa_path)
            mpa_boundary = mpa_boundary.to_crs(epsg = 3435)
            geometry = geometry.overlay(mpa_boundary)

        self.progressBar.setValue(50)

        # Some census urls cannot be constructed to be for the counties specifically so instead
        # we will extract all the GEOIDs from the tiger geometry and use that as an input
        # in the census url in order to receive the correct census data.
        codes=[]
        if selected_geography in requires_codes:
            codes = ','.join(str(x) for x in geometry['GEOID'])# Save geoids as a string
        self.progressBar.setValue(60)
        # Dictionary of census datasets
        dataset = {
            "ACS5": "acs/acs5",
            "ACS1": "acs/acs1",
            "Decennial": "dec",
        }
        # Dictionary of census tables
        table = {
            "Detailed Tables": "",
            "Subject Tables": "/subject",
            "Data Profile": "/profile",
            "Demographic and Housing Characteristics File": "/dhc",
            "Demographic Profile": "/dp",
        }
        # Dictionary of predefined groups
        predefined_groups = {
            "Sex By Age": "B01001",
            "Race": "B02001",
            "Hispanic or Latino Origin": "B03003",
            "Receipt of Food Stamps/SNAP in the Past 12 Months by Disability Status for Households": "B22010",
            "Poverty Status of Individuals in the Past 12 Months by Living Arrangement": "B17021",
            "Means of Transportation to Work": "B08301",
            "Travel Time to Work": "B08303",
            "Age by Language Spoken at Home by Ability to Speak English for the Population 5 Years and Over": "B16004",
            "Tenure by Vehicles Available": "B25044",
            "Age and Sex": "S0101",
            "Household and Families": "S1101",
            "Selected Social Characteristics in the United States": "DP02",
            "Selected Economic Characteristics": "DP03",
            "Selected Housing Characteristics": "DP04",
            "Sex by Age for Selected Age Categories": "P12",
            "Profile of General Population and Housing Characteristics": "DP1",
        }
        # Dictionary of geographies 2020 or after
        geography_post_2020 = {
            'ACS5': {
                'Census Tract': f'tract:*&in=state:17&in=county:{county_code[selected_county]}',
                'Subdivision (Township)': f'county%20subdivision:*&in=state:17&in=county:{county_code[selected_county]}',
                'Urban Areas': f'urban%20area:{codes}', #Will have to get a list of urban area codes for given counties
                'Metropolitan Statistical Area': f'metropolitan%20statistical%20area/micropolitan%20statistical%20area:{codes}',
                'Designated & Incorporated Place': 'place:*&in=state:17', #Will need place codes for given counties
                'School District (elementary)': 'school%20district%20(elementary):*&in=state:17', #Needs codes for schools
                'School District (secondary)': 'school%20district%20(secondary):*&in=state:17', #Need codes for schools
                'School District (unified)': 'school%20district%20(unified):*&in=state:17', #Need codes for schools
                'ZCTA': f'zip%20code%20tabulation%20area:{codes}', #Need zipcodes for given counties
                'Block Group': f'block%20group:*&in=state:17%20county:{county_code[selected_county]}',
            },
            'ACS1': {
                'Subdivision (Township)': f'county%20subdivision:*&in=state:17&in=county:{county_code[selected_county]}',
                'Urban Areas': f'urban%20area:{codes}', #Need Urban Area codes for given counties
                'Metropolitan Statistical Area': f'metropolitan%20statistical%20area/micropolitan%20statistical%20area:{codes}',
                'Designated & Incorporated Place': 'place:*&in=state:17', #Need codes for places for given counties
                'School District (elementary)': 'school%20district%20(elementary):*&in=state:17', #Needs codes for schools
                'School District (secondary)': 'school%20district%20(secondary):*&in=state:17', #Needs codes for schools
                'School District (unified)': 'school%20district%20(unified):*&in=state:17', #Needs codes for schools
            },
            'Decennial': {
                'Census Tract': f'tract:*&in=state:17&in=county:{county_code[selected_county]}',
                'Subdivision (Township)': f'county%20subdivision:*&in=state:17&in=county:{county_code[selected_county]}',
                'Urban Areas': f'urban%20area:{codes}', #Later add a set of selected urban area code for the given counties
                'Metropolitan Statistical Area': f'metropolitan%20statistical%20area/micropolitan%20statistical%20area:{codes}',
                'Designated & Incorporated Place': f'place:*&in=state:17', #There is a more specific filter for dhp table but didn't want to complicate it right now
                'School District (elementary)': 'school%20district%20(elementary):*&in=state:17', #Later add a set of selected school codes to filter by county
                'School District (secondary)': 'school%20district%20(secondary):*&in=state:17', #Later add a set of selected school codes to filter by county
                'School District (unified)': 'school%20district%20(unified):*&in=state:17', #Later add a set of selected school codes to filter by county
                'ZCTA': f'zip%20code%20tabulation%20area:{codes}', #There is a more specific filter for dhp table but didn't want to complicate it right now, can be resolved by having predetermined codes
                'Block Group': f'block%20group:*&in=state:17%20county:{county_code[selected_county]}',
            }
        }
        # Dictionary of geographies pre 2020
        geography_pre_2020 = {
            'ACS5': {
                'Census Tract': f'tract:*&in=state:17&in=county:{county_code[selected_county]}',
                'Subdivision (Township)': f'county%20subdivision:*&in=state:17&in=county:{county_code[selected_county]}',
                'Urban Areas': f'urban%20area:{codes}', #Will have to get a list of urban area codes for given counties
                'Metropolitan Statistical Area': f'metropolitan%20statistical%20area/micropolitan%20statistical%20area:{codes}',
                'Designated & Incorporated Place': 'place:*&in=state:17', #Will need place codes for given counties
                'School District (elementary)': 'school%20district%20(elementary):*&in=state:17', #Needs codes for schools
                'School District (secondary)': 'school%20district%20(secondary):*&in=state:17', #Need codes for schools
                'School District (unified)': 'school%20district%20(unified):*&in=state:17', #Need codes for schools
                'ZCTA': f'zip%20code%20tabulation%20area:{codes}&in=state:17', #Need zipcodes for given counties
                'Block Group': f'block%20group:*&in=state:17%20county:{county_code[selected_county]}',
            },
            'ACS1': {
                'Subdivision (Township)': f'county%20subdivision:*&in=state:17&in=county:{county_code[selected_county]}',
                'Urban Areas': f'urban%20area:{codes}', #Need Urban Area codes for given counties
                'Metropolitan Statistical Area': f'metropolitan%20statistical%20area/micropolitan%20statistical%20area:{codes}',
                'Designated & Incorporated Place': 'place:*&in=state:17', #Need codes for places for given counties
                'School District (elementary)': 'school%20district%20(elementary):*&in=state:17', #Needs codes for schools
                'School District (secondary)': 'school%20district%20(secondary):*&in=state:17', #Needs codes for schools
                'School District (unified)': 'school%20district%20(unified):*&in=state:17', #Needs codes for schools
            }
        }

        def create_variable_lists(df):
            # Create a dictionary of variales from an inputted dataframe
            variable_lists = {}
            for table_type, group in df.groupby('Table'):
                variable_lists[table_type] = group['Variables'].tolist()

            return variable_lists
        #Lists of which variables GEOID column is constructed from the last 4, 3 or last 2 columns in their dataframe
        GEOID_4 = ['Block Group']
        GEOID_3 = ["Census Tract",'Subdivision (Township)']
        GEOID_2 = ['Designated & Incorporated Place','School District (elementary)','School District (secondary)','School District (unified)']
        self.progressBar.setValue(70)

        if self.cb_groupcall.isChecked() is True:
            # Run if Groupcall is checked
            # Build the Census URL based on the year
            if int(selected_year) >=2020:
                self.census_url = f"https://api.census.gov/data/{selected_year}/{dataset[selected_dataset]}{table[selected_table]}?get=group({predefined_groups[selected_group]})&for={geography_post_2020[selected_dataset][selected_geography]}&key={census_key}"
            else:
                self.census_url = f"https://api.census.gov/data/{selected_year}/{dataset[selected_dataset]}{table[selected_table]}?get=group({predefined_groups[selected_group]})&for={geography_pre_2020[selected_dataset][selected_geography]}&key={census_key}"
            print(f"Census API URL: {self.census_url}") # Print Census API link

            data = requests.get(self.census_url)
            if data.status_code == 200: # Check if census url is valid
                data = data.json() # Convert the url data to JSON format
                group_df = pd.DataFrame(data[1:], columns=data[0]) # Exclude the first row and set the first row as headers
                group_df['GEOID'] = group_df['GEO_ID'].str.split('US').str[1] # Add a new column calld GEOID that is equal t othe GEO_ID column for everything in the content that occurs after the string 'US'
                df = group_df # Set the group call df to the variable df
            else:
                # Throw error warning, reset progress bar, and return false to break out of program
                QMessageBox.warning(self,"CENSUS ERROR","Invalid Census URL or Census servers are currently busy")
                self.progressBar.reset()
                return False
            self.progressBar.setValue(80)
        else:
            # If groupcall is not checked, get the variables
            # manually from the variable table.
            variable_df = self.qtablewidget_to_dataframe(self.tw_selected_variables) # Convert the right QTableWidget to a dataframe
            variable_lists = create_variable_lists(variable_df) # Convert the dataframe into a list of variables
            variable_labels = {
                 # Take each row in the variable dataframe and extract the variable
                 # and create a dictionary where Variables is Key, Labels is value
                row['Variables']: row['Labels']
                for _, row in variable_df.iterrows()
            }
            # Initialize all_dfs ad a dataframe
            all_dfs = pd.DataFrame()

            for key in variable_lists:
                # Go through all the keys in the variable_lists
                if int(selected_year) >= 2020: # Grab the correct census url for hte right year
                    temp_url = f"https://api.census.gov/data/{selected_year}/{dataset[selected_dataset]}{table[key]}?get={','.join(map(str, variable_lists[key]))}&for={geography_post_2020[selected_dataset][selected_geography]}&key={census_key}"
                else:
                    temp_url = f"https://api.census.gov/data/{selected_year}/{dataset[selected_dataset]}{table[key]}?get={','.join(map(str, variable_lists[key]))}&for={geography_pre_2020[selected_dataset][selected_geography]}&key={census_key}"
                print(f"Census API URLs: {temp_url}")
                data = requests.get(temp_url)
                if data.status_code == 200: # Check of url is valid
                    data = data.json() # Convert data to JSON
                    temp_df = pd.DataFrame(data[1:], columns=data[0]) # Remove the first row and set it as the header instead
                    if selected_geography in GEOID_4: # Construct GEOID column based on the last 4 columns
                        temp_df['GEOID'] = temp_df[temp_df.columns[-4]] + temp_df[temp_df.columns[-3]] + temp_df[temp_df.columns[-2]] + temp_df[temp_df.columns[-1]]
                    elif selected_geography in GEOID_3:# Construct GEOID column based on the last 3 columns
                        temp_df['GEOID'] = temp_df[temp_df.columns[-3]] + temp_df[temp_df.columns[-2]] + temp_df[temp_df.columns[-1]]
                    elif selected_geography in GEOID_2:# Construct GEOID column based on the last 2 columns
                        temp_df['GEOID'] = temp_df[temp_df.columns[-2]] + temp_df[temp_df.columns[-1]]
                    else:# Construct GEOID column based on the last column
                        temp_df['GEOID'] = temp_df[temp_df.columns[-1]]

                    if all_dfs.empty is True:
                        # If the dataframe is empty set it equal to the first all_dfs dataframe
                        all_dfs = temp_df
                    else:
                        # Left-merge `all_dfs` with selected columns from `temp_df` (GEOID + variables), preserving all original rows
                        all_dfs = pd.merge(all_dfs, temp_df[['GEOID'] + variable_lists[key]], on='GEOID', how='left')
                else:
                    # If invalid link throw an error message, reset progress bar, and return false to break out of the program
                    QMessageBox.warning(self,"CENSUS ERROR","Invalid Census URL or Census servers are currently busy")
                    self.progressBar.reset()
                    return False

            all_dfs.rename(columns=variable_labels, inplace=True) # Update the Column names with Variable labels dictionary
            df = all_dfs
            self.progressBar.setValue(80)

        merge = geometry.merge(df, on = "GEOID") # Merge geometry and census data
        merge = merge.apply(pd.to_numeric, errors='ignore') # convert all valid cells to integer type from string, ignore errors (letters)

        if merge.empty:
            QMessageBox.warning(self,"EMPTY DATAFRAME","No data within the requested geography") # If there is no overlap in the merge throw an error

        self.progressBar.setValue(90)

        if self.cb_local.isChecked() is False:
            # If the local checkbox is not checked save it as a memory layer in QGIS
            vl = QgsVectorLayer(merge.to_json(),f"{selected_dataset}_{selected_table}_{selected_geography}_{selected_county}_{selected_group}_{selected_year}","ogr")
            QgsProject.instance().addMapLayer(vl)
        else:
            # If it is selected than bring up a file dialog and choose save location in filesystem
            save_directory = QtWidgets.QFileDialog.getSaveFileName(
                parent = None,
                caption = "Select Directory",
                directory = "file.gpkg",
                filter="GeoPackage Files (*.gpkg)"
            )
            # Save it to the file system with the chosen name
            merge.to_file(
                save_directory[0],
                driver="GPKG",
                layer="clipped_data"
            )
            uri = f"{save_directory[0]}|layername=clipped_data" # Set filelocation to uri
            layer_name = os.path.splitext(os.path.basename(save_directory[0]))[0]
            vl = QgsVectorLayer(uri,f"{layer_name}","ogr") # Load as layer into QGIS will file location
            QgsProject.instance().addMapLayer(vl)

        # Once the method is done reach 100 and then reset
        self.progressBar.setValue(100)
        self.progressBar.reset()
