# -*- coding: utf-8 -*-
"""
/***************************************************************************
 CensusDownloaderDialog
                                 A QGIS plugin
 Downloads Census Data
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2024-08-06
        git sha              : $Format:%H$
        copyright            : (C) 2024 by Amer Islam
        email                : aislam@ccrpc.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os
import pandas as pd
import geopandas as gpd
import requests
from qgis.PyQt import uic
from qgis.PyQt import QtWidgets
from qgis.PyQt.QtWidgets import QMessageBox, QTableWidgetItem
from PyQt5.QtWidgets import QInputDialog, QTableWidget
from qgis.core import QgsVectorLayer, QgsProject

# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'census_downloader_dialog_base.ui'))


class CensusDownloaderDialog(QtWidgets.QDialog, FORM_CLASS):
    def __init__(self, parent=None):
        """Constructor."""
        super(CensusDownloaderDialog, self).__init__(parent)
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)
        self.check_and_set_api_key()
        self.filter_comboboxes()
        self.progressBar.reset()
        self.cb_dataset.currentTextChanged.connect(self.filter_comboboxes)
        self.cb_dataset.currentTextChanged.connect(self.clear_variable)
        self.cb_table.currentTextChanged.connect(self.filter_comboboxes)
        self.cb_geography.currentTextChanged.connect(self.filter_comboboxes)
        self.cb_geography.currentTextChanged.connect(self.clear_variable)
        self.cb_predefined.currentTextChanged.connect(self.filter_comboboxes)
        self.cb_county.currentTextChanged.connect(self.filter_comboboxes)
        self.cb_county.currentTextChanged.connect(self.clear_variable)
        self.cb_year.currentTextChanged.connect(self.filter_comboboxes)
        self.cb_year.currentTextChanged.connect(self.clear_variable)
        self.pb_variables.clicked.connect(self.census_group_call)
        self.pb_selection.clicked.connect(self.add_to_variable_table)
        self.pb_remove.clicked.connect(self.remove_from_variable_table)
        self.pb_query.clicked.connect(self.execute_query)
        
        
    
    # Check if census api key is an environment variable, if it is prompt user to input it
    # otherwise set the existing api key to census_key
    def check_and_set_api_key(self):
        
        self.census_key_dir = os.path.dirname(__file__)
        locale_path = os.path.join(self.census_key_dir, 'census.env')
        if os.path.exists(locale_path):
            with open('census.env', 'r') as file:
                census_key = file.read()
        else:
            title = "Enter Census Key"
            message = "Please enter your Census API Key:"
            key = QInputDialog.getText(None, title, message)
            with open(locale_path, 'w') as file:
                file.write(key[0])
            census_key = key[0]
        return census_key

    #This first filter method makes sure that the combo boxes have valid combinations
    def filter_comboboxes(self): 
        
        allowed_tables = {
            "ACS5": ["Detailed Tables", "Subject Tables", "Data Profile"],
            "ACS1": ["Detailed Tables", "Subject Tables", "Data Profile"],
            "Decennial": ["Demographic and Housing Characteristics File", 'Demographic Profile']
        }
             
        allowed_geography = {
            ("ACS5", "Detailed Tables"): ['ZCTA','Census Tract','Subdivision (Township)','Urban Areas','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)', 'School District (unified)','Block Group'],
            ("ACS5", "Subject Tables"): ['ZCTA','Census Tract','Subdivision (Township)','Urban Areas','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)', 'School District (unified)'],
            ("ACS5", "Data Profile"): ['ZCTA','Census Tract','Subdivision (Township)','Urban Areas','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)', 'School District (unified)'],
            ("ACS1", "Detailed Tables"): ['Subdivision (Township)','Urban Areas','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)', 'School District (unified)'],
            ("ACS1", "Subject Tables"): ['Subdivision (Township)','Urban Areas','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)', 'School District (unified)'],
            ("ACS1", "Data Profile"): ['Subdivision (Township)','Urban Areas','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)', 'School District (unified)'],
            ("Decennial", "Demographic and Housing Characteristics File"): ['ZCTA','Census Tract','Subdivision (Township)','Urban Areas','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)', 'School District (unified)','Block Group'],
            ("Decennial", 'Demographic Profile'): ['ZCTA','Census Tract','Subdivision (Township)','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)', 'School District (unified)']
        }
        
        allowed_predefined = {
            ("ACS5", "Detailed Tables"): ["Sex By Age","Race","Hispanic or Latino Origin","Receipt of Food Stamps/SNAP in the Past 12 Months by Disability Status for Households","Poverty Status of Individuals in the Past 12 Months by Living Arrangement","Means of Transportation to Work","Travel Time to Work","Age by Language Spoken at Home by Ability to Speak English for the Population 5 Years and Over","Tenure by Vehicles Available", "Language Spoken at Home by Ability to Speak English for the Population 5 Years and Over"],
            ("ACS5", "Subject Tables"): ["Age and Sex","Household and Families"],
            ("ACS5", "Data Profile"): ["Selected Social Characteristics in the United States","Selected Economic Characteristics","Selected Housing Characteristics"],
            ('ACS1', "Detailed Tables"): ["Sex By Age","Race","Hispanic or Latino Origin","Receipt of Food Stamps/SNAP in the Past 12 Months by Disability Status for Households","Poverty Status of Individuals in the Past 12 Months by Living Arrangement","Means of Transportation to Work","Travel Time to Work","Age by Language Spoken at Home by Ability to Speak English for the Population 5 Years and Over","Tenure by Vehicles Available", "Language Spoken at Home by Ability to Speak English for the Population 5 Years and Over"],
            ('ACS1', "Subject Tables"): ["Age and Sex","Household and Families"],
            ('ACS1', "Data Profile"): ["Selected Social Characteristics in the United States","Selected Economic Characteristics","Selected Housing Characteristics"],
            ("Decennial", "Demographic and Housing Characteristics File"): ["Sex by Age for Selected Age Categories"],
            ("Decennial", 'Demographic Profile'): ["Profile of General Population and Housing Characteristics"]
        }
        #Then nested dictionary structure here is easier to read and the rest of the dictionaries should be changed to reflect this
        #Range is not inclusive of the last value so 2022 is the highest year being returned despite the upper end being 2023
        # Could clean up the dictionary by making the range int oa predefined list, but ti would make editing years harder from a glance
        allowed_years = {
            "ACS5": {
                "Detailed Tables": {
                    "ZCTA": list(map(str, range(2011, 2023))),
                    "Block Group": list(map(str, range(2013, 2023))),
                    "Census Tract": list(map(str, range(2009, 2023))),
                    "Subdivision (Township)": list(map(str, range(2009, 2023))),
                    'Urban Areas': list(map(str, range(2009, 2011))) + list(map(str, range(2012, 2023))),
                    'Metropolitan Statistical Area': list(map(str, range(2009, 2022))), 
                    'Designated & Incorporated Place': list(map(str, range(2009, 2023))),
                    'School District (elementary)': list(map(str, range(2009, 2023))),
                    'School District (secondary)': list(map(str, range(2009, 2023))),
                    'School District (unified)': list(map(str, range(2009, 2023))),
                },
                "Subject Tables": {
                    "ZCTA": list(map(str, range(2011, 2023))),
                    "Census Tract": list(map(str, range(2010, 2023))),
                    "Subdivision (Township)": list(map(str, range(2010, 2023))),
                    'Urban Areas': list(map(str, range(2009, 2011))) + list(map(str, range(2012, 2023))),
                    'Metropolitan Statistical Area': list(map(str, range(2010, 2022))),
                    'Designated & Incorporated Place': list(map(str, range(2010, 2023))),
                    'School District (elementary)': list(map(str, range(2010, 2023))),
                    'School District (secondary)': list(map(str, range(2010, 2023))),
                    'School District (unified)': list(map(str, range(2010, 2023))),
                },
                "Data Profile": {
                    "ZCTA": list(map(str, range(2011, 2023))),
                    "Census Tract": list(map(str, range(2010, 2023))),
                    "Subdivision (Township)": list(map(str, range(2010, 2023))),
                    'Urban Areas': list(map(str, range(2010, 2011))) + list(map(str, range(2012, 2023))),
                    'Metropolitan Statistical Area': list(map(str, range(2010, 2022))),
                    'Designated & Incorporated Place': list(map(str, range(2010, 2023))),
                    'School District (elementary)': list(map(str, range(2010, 2023))),
                    'School District (secondary)': list(map(str, range(2010, 2023))),
                    'School District (unified)': list(map(str, range(2010, 2023))),
                },
            },
            "ACS1":{
                "Detailed Tables": {
                    "Subdivision (Township)": list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2023))),
                    'Urban Areas': list(map(str, range(2010, 2011))) + list(map(str, range(2012, 2020))) + list(map(str, range(2021, 2023))),
                    'Metropolitan Statistical Area': list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2022))),
                    'Designated & Incorporated Place': list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2023))),
                    'School District (elementary)': list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2023))),
                    'School District (secondary)': list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2023))),
                    'School District (unified)': list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2023))),
                },
                "Subject Tables": {
                    "Subdivision (Township)": list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2023))),
                    'Urban Areas': list(map(str, range(2010, 2011))) + list(map(str, range(2012, 2020))) + list(map(str, range(2021, 2023))),
                    'Metropolitan Statistical Area': list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2022))),
                    'Designated & Incorporated Place': list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2023))),
                    'School District (elementary)': list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2023))),
                    'School District (secondary)': list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2023))),
                    'School District (unified)': list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2023))),
                },
                "Data Profile": {
                    "Subdivision (Township)": list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2023))),
                    'Urban Areas': list(map(str, range(2010, 2011))) + list(map(str, range(2012, 2020))) + list(map(str, range(2021, 2023))),
                    'Metropolitan Statistical Area': list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2022))),
                    'Designated & Incorporated Place': list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2023))),
                    'School District (elementary)': list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2023))),
                    'School District (secondary)': list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2023))),
                    'School District (unified)': list(map(str, range(2010, 2020))) + list(map(str, range(2021, 2023))),
                },
            },
            "Decennial": {
                "Demographic and Housing Characteristics File": {
                    "ZCTA": "2020",
                    "Block Group": "2020",
                    "Census Tract": "2020",
                    "Subdivision (Township)": "2020",
                    'Urban Areas': "2020",
                    'Metropolitan Statistical Area': "2020",
                    'Designated & Incorporated Place': "2020",
                    'School District (elementary)': "2020",
                    'School District (secondary)': "2020",
                    'School District (unified)': "2020",
                },
                'Demographic Profile': {
                    "ZCTA": "2020",
                    "Census Tract": "2020",
                    "Subdivision (Township)": "2020",
                    'Metropolitan Statistical Area': "2020",
                    'Designated & Incorporated Place': "2020",
                    'School District (elementary)': "2020",
                    'School District (secondary)': "2020",
                    'School District (unified)': "2020",
                },
            }
        }
                       
        #Below is a lot of forloops and if statements, These should be condensed into different methods but for the time being I'm leaving them as is
        selected_year = self.cb_year.currentText()
        selected_dataset = self.cb_dataset.currentText() 
        selected_table = self.cb_table.currentText() 
        selected_geography = self.cb_geography.currentText() 
        selected_group = self.cb_predefined.currentText()
        selected_county = self.cb_county.currentText()
        
        #Disable options in the drop down that are not in the allowed dictionaries listed above.
        for i in range(self.cb_table.count()):
            table_name = self.cb_table.itemText(i)
            if table_name in allowed_tables[selected_dataset]:
                self.cb_table.model().item(i).setEnabled(True)
            else:
                self.cb_table.model().item(i).setEnabled(False)
        #The above loops disabled the drop downs but did not assign a new selection. This loop detects if the current table is valid. If it isn't then it loops through the dropdown entires
        #until it finds a valid one.  
        if selected_table not in allowed_tables[selected_dataset]:
            for i in range(self.cb_table.count()):
                if self.cb_table.itemText(i) in allowed_tables[selected_dataset]:
                    self.cb_table.setCurrentIndex(i)
                    selected_table = self.cb_table.currentText() #make sure to update the current selection after changing it
        #Disable invalid options based on previous selections
        for i in range(self.cb_geography.count()):
            geography_name = self.cb_geography.itemText(i)
            if geography_name in allowed_geography[selected_dataset, selected_table]:
                self.cb_geography.model().item(i).setEnabled(True)
            else:
                self.cb_geography.model().item(i).setEnabled(False)
        #Update selection to a valid one        
        if selected_geography not in allowed_geography[selected_dataset, selected_table]:
            for i in range(self.cb_geography.count()):
                if self.cb_geography.itemText(i) in allowed_geography[selected_dataset, selected_table]:
                    self.cb_geography.setCurrentIndex(i)
                    selected_geography = self.cb_geography.currentText() 
        #Disable invalid options based on previous selections           
        for i in range(self.cb_predefined.count()):
            group_name = self.cb_predefined.itemText(i)
            if group_name in allowed_predefined[selected_dataset, selected_table]:
                self.cb_predefined.model().item(i).setEnabled(True)
            else:
                self.cb_predefined.model().item(i).setEnabled(False)
        #Update selection to a valid one   
        if selected_group not in allowed_predefined[selected_dataset, selected_table]:
            for i in range(self.cb_predefined.count()):
                if self.cb_predefined.itemText(i) in allowed_predefined[selected_dataset, selected_table]:
                    self.cb_predefined.setCurrentIndex(i)
                    selected_group = self.cb_predefined.currentText()
        #Disable invalid options based on previous selections            
        for i in range(self.cb_year.count()):
            year = self.cb_year.itemText(i)
            if year in allowed_years[selected_dataset][selected_table][selected_geography]:
                self.cb_year.model().item(i).setEnabled(True)
            else:
                self.cb_year.model().item(i).setEnabled(False)
        #Update selection to a valid one           
        if selected_year not in allowed_years[selected_dataset][selected_table][selected_geography]:
            for i in range(self.cb_year.count()):
                if self.cb_year.itemText(i) in allowed_years[selected_dataset][selected_table][selected_geography]:
                    self.cb_year.setCurrentIndex(i)
                    selected_year = self.cb_year.currentText()
                    
    def write_df_to_qtable(self,df,table):
            headers = list(df)
            table.setRowCount(df.shape[0])
            table.setColumnCount(df.shape[1])
            table.setHorizontalHeaderLabels(headers)       
            
            df_array = df.values
            for row in range(df.shape[0]):
                for col in range(df.shape[1]):
                    table.setItem(row, col, QTableWidgetItem(str(df_array[row,col])))
    
    def qtablewidget_to_dataframe(self, qtw):
        num_cols = qtw.columnCount()
        num_rows = qtw.rowCount()
        data = []
        for row in range(num_rows):
            row_data = []
            for col in range(num_cols):
                item = qtw.item(row, col)
                if item:
                    row_data.append(str(item.text()))
                else:
                    row_data.append('')
            data.append(row_data)
        
        df = pd.DataFrame(data, columns=['Variables','Labels','Table'])
        return df
    
    def census_group_call(self):
        selected_year = self.cb_year.currentText()
        selected_dataset = self.cb_dataset.currentText() 
        selected_table = self.cb_table.currentText() 
        selected_geography = self.cb_geography.currentText() 
        selected_group = self.cb_predefined.currentText()
        selected_county = self.cb_county.currentText() 
        census_key = self.check_and_set_api_key()
        
        dataset = {
            "ACS5": "acs/acs5",
            "ACS1": "acs/acs1",
            "Decennial": "dec",
        }

        table = {
            "Detailed Tables": "",
            "Subject Tables": "/subject",
            "Data Profile": "/profile",
            "Demographic and Housing Characteristics File": "/dhc",
            "Demographic Profile": "/dp",
        }

        predefined_groups = {
            "Sex By Age": "B01001",
            "Race": "B02001",
            "Hispanic or Latino Origin": "B03003",
            "Receipt of Food Stamps/SNAP in the Past 12 Months by Disability Status for Households": "B22010",
            "Poverty Status of Individuals in the Past 12 Months by Living Arrangement": "B17021",
            "Means of Transportation to Work": "B08301",
            "Travel Time to Work": "B08303",
            "Age by Language Spoken at Home by Ability to Speak English for the Population 5 Years and Over": "B16004",
            "Tenure by Vehicles Available": "B25044",
            "Language Spoken at Home by Ability to Speak English for the Population 5 Years and Over": "B16001",
            "Age and Sex": "S0101",
            "Household and Families": "S1101",
            "Selected Social Characteristics in the United States": "DP02",
            "Selected Economic Characteristics": "DP03",
            "Selected Housing Characteristics": "DP04",
            "Sex by Age for Selected Age Categories": "P12",
            "Profile of General Population and Housing Characteristics": "DP1",
        }

        self.census_url = f'https://api.census.gov/data/{selected_year}/{dataset[selected_dataset]}{table[selected_table]}/variables.json'
        print(f"Census Variable Url: {self.census_url}")
        self.data = requests.get(self.census_url)
        self.data = self.data.json()
        
        keys = list(self.data['variables'].keys())
        labels = [item['label'] for item in self.data['variables'].values()]

        keys = [key for key, value in self.data['variables'].items() if value.get('group') == predefined_groups[selected_group]]
        labels = [value['label'] for key, value in self.data['variables'].items() if value.get('group') == predefined_groups[selected_group]]
    
        self.df = pd.DataFrame({'Variables': keys, 'Labels': labels, 'Table': selected_table})
        self.df = self.df.sort_values('Variables', ascending=True)

        
        self.write_df_to_qtable(self.df, self.tw_census_view)
        self.tw_census_view.setSelectionBehavior(QTableWidget.SelectRows)

    def execute_query(self):
        if self.cb_groupcall.isChecked() is False and self.tw_selected_variables.rowCount() == 0:
            QMessageBox.warning(self,"Group Call or Manual","Make sure either group call is selected or variables have been inputted into the Text Box")
        elif self.cb_mpa.isChecked() is True and self.cb_county.currentText() != 'Champaign, IL':
            QMessageBox.warning(self,"MPA","MPA can only be selected for Champaign County")
        else:
            self.update_url()
    
    def update_url(self):
        census_key = self.check_and_set_api_key()
        
        selected_year = self.cb_year.currentText()
        selected_dataset = self.cb_dataset.currentText() 
        selected_table = self.cb_table.currentText() 
        selected_geography = self.cb_geography.currentText() 
        selected_group = self.cb_predefined.currentText()
        selected_county = self.cb_county.currentText() 
        
        dataset = {
            "ACS5": "acs/acs5",
            "ACS1": "acs/acs1",
            "Decennial": "dec",
        }

        table = {
            "Detailed Tables": "",
            "Subject Tables": "/subject",
            "Data Profile": "/profile",
            "Demographic and Housing Characteristics File": "/dhc",
            "Demographic Profile": "/dp",
        }

        predefined_groups = {
            "Sex By Age": "B01001",
            "Race": "B02001",
            "Hispanic or Latino Origin": "B03003",
            "Receipt of Food Stamps/SNAP in the Past 12 Months by Disability Status for Households": "B22010",
            "Poverty Status of Individuals in the Past 12 Months by Living Arrangement": "B17021",
            "Means of Transportation to Work": "B08301",
            "Travel Time to Work": "B08303",
            "Age by Language Spoken at Home by Ability to Speak English for the Population 5 Years and Over": "B16004",
            "Tenure by Vehicles Available": "B25044",
            "Language Spoken at Home by Ability to Speak English for the Population 5 Years and Over": "B16001",
            "Age and Sex": "S0101",
            "Household and Families": "S1101",
            "Selected Social Characteristics in the United States": "DP02",
            "Selected Economic Characteristics": "DP03",
            "Selected Housing Characteristics": "DP04",
            "Sex by Age for Selected Age Categories": "P12",
            "Profile of General Population and Housing Characteristics": "DP1",
        }

        county_code = {
            'Champaign, IL': '019',
            'Clark, IL': '023',
            'Coles, IL': '029',
            'Cumberland, IL': '035',
            'De Witt, IL': '039',
            'Douglas, IL': '041',
            'Edgar, IL': '045',
            'Macon, IL': '115',
            'Moultrie, IL': '139',
            'Piatt, IL': '147',
            'Shelby, IL': '173',
            'Vermilion, IL': '183',
            'Ford, IL': '053',
            'Iroquois, IL': '075',
        }

        # I could make allowed lists for the all the geographies that don't have an option ot defaul through county code and feed in
        # predetermined codes. However, I am opting to download the whole dataset, to the geometrical join, and then clip by county geography
        # It is possible that this is inefficient and I may update this behaviour later if performance suffers.
        # I ended up making all the datasets have the same geography links so this dictionary can be collapsed later
        geography = {
            'ACS5': {
                'Census Tract': f'tract:*&in=state:17&in=county:{county_code[selected_county]}',
                'Subdivision (Township)': f'county%20subdivision:*&in=state:17&in=county:{county_code[selected_county]}',
                'Urban Areas': 'urban%20area:*', #Will have to get a list of urban area codes for given counties
                'Metropolitan Statistical Area': f'metropolitan%20statistical%20area/micropolitan%20statistical%20area:*',
                'Designated & Incorporated Place': 'place:*&in=state:17', #Will need place codes for given counties
                'School District (elementary)': 'school%20district%20(elementary):*&in=state:17', #Needs codes for schools
                'School District (secondary)': 'school%20district%20(secondary):*&in=state:17', #Need codes for schools
                'School District (unified)': 'school%20district%20(unified):*&in=state:17', #Need codes for schools
                'ZCTA': 'zip%20code%20tabulation%20area:*', #Need zipcodes for given counties
                'Block Group': f'block%20group:*&in=state:17%20county:{county_code[selected_county]}',
            },
            'ACS1': {
                'Subdivision (Township)': f'county%20subdivision:*&in=state:17&in=county:{county_code[selected_county]}',
                'Urban Areas': 'urban%20area:*', #Need Urban Area codes for given counties
                'Metropolitan Statistical Area': f'metropolitan%20statistical%20area/micropolitan%20statistical%20area:*',
                'Designated & Incorporated Place': 'place:*&in=state:17', #Need codes for places for given counties
                'School District (elementary)': 'school%20district%20(elementary):*&in=state:17', #Needs codes for schools
                'School District (secondary)': 'school%20district%20(secondary):*&in=state:17', #Needs codes for schools
                'School District (unified)': 'school%20district%20(unified):*&in=state:17', #Needs codes for schools
                'ZCTA': 'zip%20code%20tabulation%20area:*', #Need zipcodes for given counties
                'Block Group': f'block%20group:*&in=state:17%20county:{county_code[selected_county]}',
            },
            'Decennial': {
                'Census Tract': f'tract:*&in=state:17&in=county:{county_code[selected_county]}',
                'Subdivision (Township)': f'county%20subdivision:*&in=state:17&in=county:{county_code[selected_county]}',
                'Urban Areas': 'urban%20area%20(or%20part):*&in=state:17', #Later add a set of selected urban area code for the given counties
                'Metropolitan Statistical Area': f'metropolitan%20statistical%20area/micropolitan%20statistical%20area:*',
                'Designated & Incorporated Place': 'place:*&in=state:17', #There is a more specific filter for dhp table but didn't want to complicate it right now
                'School District (elementary)': 'school%20district%20(elementary):*&in=state:17', #Later add a set of selected school codes to filter by county
                'School District (secondary)': 'school%20district%20(secondary):*&in=state:17', #Later add a set of selected school codes to filter by county
                'School District (unified)': 'school%20district%20(unified):*&in=state:17', #Later add a set of selected school codes to filter by county
                'ZCTA': 'zip%20code%20tabulation%20area:*', #There is a more specific filter for dhp table but didn't want to complicate it right now, can be resolved by having predetermined codes
                'Block Group': f'block%20group:*&in=state:17%20county:{county_code[selected_county]}',
            }
        }
        
        def create_variable_lists(df):
            variable_lists = {}

            for table_type, group in df.groupby('Table'):
                variable_lists[table_type] = group['Variables'].tolist()
            
            return variable_lists
        
        #Lists of which variables GEOID column is constructed from the last 4, 3 or last 2 columns in their dataframe
        GEOID_4 = ['Block Group']
        GEOID_3 = ["Census Tract",'Subdivision (Township)']
        GEOID_2 = ['Designated & Incorporated Place','School District (elementary)','School District (secondary)','School District (unified)']
        self.progressBar.setValue(10)
        if self.cb_groupcall.isChecked() is True:
            self.census_url = f"https://api.census.gov/data/{selected_year}/{dataset[selected_dataset]}{table[selected_table]}?get=group({predefined_groups[selected_group]})&for={geography[selected_dataset][selected_geography]}&key={census_key}"
            print(f"Census API URL: {self.census_url}")
            data = requests.get(self.census_url)
            if data.status_code == 200:
                data = data.json()
                group_df = pd.DataFrame(data[1:], columns=data[0])
                group_df['GEOID'] = group_df['GEO_ID'].str.split('US').str[1]
                df = group_df
            else:
                QMessageBox.warning(self,"CENSUS ERROR","Invalid Census URL or Census servers are currently busy")
        else:
            variable_df = self.qtablewidget_to_dataframe(self.tw_selected_variables)
            variable_lists = create_variable_lists(variable_df)
            variable_labels = {row['Variables']: row['Labels'] for _, row in variable_df.iterrows()}
            all_dfs = pd.DataFrame()
            for key in variable_lists:
                temp_url = f"https://api.census.gov/data/{selected_year}/{dataset[selected_dataset]}{table[key]}?get={','.join(map(str, variable_lists[key]))}&for={geography[selected_dataset][selected_geography]}&key={census_key}"
                print(f"Census API URLs: {temp_url}")
                data = requests.get(temp_url)
                if data.status_code == 200:
                    data = data.json()
                    temp_df = pd.DataFrame(data[1:], columns=data[0])
                    if selected_geography in GEOID_4:
                        temp_df['GEOID'] = temp_df[temp_df.columns[-4]] + temp_df[temp_df.columns[-3]] + temp_df[temp_df.columns[-2]] + temp_df[temp_df.columns[-1]]
                    elif selected_geography in GEOID_3:
                        temp_df['GEOID'] = temp_df[temp_df.columns[-3]] + temp_df[temp_df.columns[-2]] + temp_df[temp_df.columns[-1]]
                    elif selected_geography in GEOID_2:
                        temp_df['GEOID'] = temp_df[temp_df.columns[-2]] + temp_df[temp_df.columns[-1]]
                    else:
                        temp_df['GEOID'] = temp_df[temp_df.columns[-1]]
                    if all_dfs.empty is True:
                        all_dfs = temp_df
                    else:
                        all_dfs = pd.merge(all_dfs, temp_df[['GEOID'] + variable_lists[key]], on='GEOID', how='left')
                else:
                    QMessageBox.warning(self,"CENSUS ERROR","Invalid Census URL or Census servers are currently busy")
            combined_df = all_dfs
            combined_df.rename(columns=variable_labels, inplace=True)
            df = combined_df
           
        
        self.progressBar.setValue(20)
                
        #There are quite a few mismatches between the years that data is available and the years that geomtries are availalbe.
        #I feel like there's a way to decrease the size of these dictionaries but I'm not sure how. Using ranges instead of straight year inputs
        #seems to heavily impact performance.

        #This can probably be condensed into just back to the geographies as keys. As for the differnces in UAC and ZCTAs changing after
        #2020 these can just handled with 2 if statements instead. Should look into optimizing for these large dictionaries later.
        tiger_geography = {
            '2010': {
                "Census Tract": "TRACT/2010",
                'Subdivision (Township)': "COUSUB/2010",
                'Urban Areas': 'UA/2010',
                'Metropolitan Statistical Area': 'CBSA/2010',
                'Designated & Incorporated Place': 'PLACE/2010',
                'School District (elementary)': 'ELSD/2010',
                'School District (secondary)': 'SCSD/2010',
                'School District (unified)': 'UNSD/2010',
                'ZCTA': 'ZCTA5/2010', 
                'Block Group': 'BG/2010'
            },
            '2011': {
                "Census Tract": "TRACT",
                'Subdivision (Township)': "COUSUB",
                'Urban Areas': 'UA',
                'Metropolitan Statistical Area': 'CBSA',
                'Designated & Incorporated Place': 'PLACE',
                'School District (elementary)': 'ELSD',
                'School District (secondary)': 'SCSD',
                'School District (unified)': 'UNSD',
                'ZCTA': 'ZCTA5', 
                'Block Group': 'BG'
            },
            '2012': {
                "Census Tract": "TRACT",
                'Subdivision (Township)': "COUSUB",
                'Urban Areas': 'UAC',
                'Metropolitan Statistical Area': 'CBSA',
                'Designated & Incorporated Place': 'PLACE',
                'School District (elementary)': 'ELSD',
                'School District (secondary)': 'SCSD',
                'School District (unified)': 'UNSD',
                'ZCTA': 'ZCTA5', 
                'Block Group': 'BG'
            },
            '2013': {
                "Census Tract": "TRACT",
                'Subdivision (Township)': "COUSUB",
                'Urban Areas': 'UAC',
                'Metropolitan Statistical Area': 'CBSA',
                'Designated & Incorporated Place': 'PLACE',
                'School District (elementary)': 'ELSD',
                'School District (secondary)': 'SCSD',
                'School District (unified)': 'UNSD',
                'ZCTA': 'ZCTA5', 
                'Block Group': 'BG'
            },
            '2014': {
                "Census Tract": "TRACT",
                'Subdivision (Township)': "COUSUB",
                'Urban Areas': 'UAC',
                'Metropolitan Statistical Area': 'CBSA',
                'Designated & Incorporated Place': 'PLACE',
                'School District (elementary)': 'ELSD',
                'School District (secondary)': 'SCSD',
                'School District (unified)': 'UNSD',
                'ZCTA': 'ZCTA5', 
                'Block Group': 'BG'
            },
            '2015': {
                "Census Tract": "TRACT",
                'Subdivision (Township)': "COUSUB",
                'Urban Areas': 'UAC',
                'Metropolitan Statistical Area': 'CBSA',
                'Designated & Incorporated Place': 'PLACE',
                'School District (elementary)': 'ELSD',
                'School District (secondary)': 'SCSD',
                'School District (unified)': 'UNSD',
                'ZCTA': 'ZCTA5', 
                'Block Group': 'BG'
            },
            '2016': {
                "Census Tract": "TRACT",
                'Subdivision (Township)': "COUSUB",
                'Urban Areas': 'UAC',
                'Metropolitan Statistical Area': 'CBSA',
                'Designated & Incorporated Place': 'PLACE',
                'School District (elementary)': 'ELSD',
                'School District (secondary)': 'SCSD',
                'School District (unified)': 'UNSD',
                'ZCTA': 'ZCTA5', 
                'Block Group': 'BG'
            },
            '2017': {
                "Census Tract": "TRACT",
                'Subdivision (Township)': "COUSUB",
                'Urban Areas': 'UAC',
                'Metropolitan Statistical Area': 'CBSA',
                'Designated & Incorporated Place': 'PLACE',
                'School District (elementary)': 'ELSD',
                'School District (secondary)': 'SCSD',
                'School District (unified)': 'UNSD',
                'ZCTA': 'ZCTA5', 
                'Block Group': 'BG'
            },
            '2018': {
                "Census Tract": "TRACT",
                'Subdivision (Township)': "COUSUB",
                'Urban Areas': 'UAC',
                'Metropolitan Statistical Area': 'CBSA',
                'Designated & Incorporated Place': 'PLACE',
                'School District (elementary)': 'ELSD',
                'School District (secondary)': 'SCSD',
                'School District (unified)': 'UNSD',
                'ZCTA': 'ZCTA5', 
                'Block Group': 'BG'
            },
            '2019': {
                "Census Tract": "TRACT",
                'Subdivision (Township)': "COUSUB",
                'Urban Areas': 'UAC',
                'Metropolitan Statistical Area': 'CBSA',
                'Designated & Incorporated Place': 'PLACE',
                'School District (elementary)': 'ELSD',
                'School District (secondary)': 'SCSD',
                'School District (unified)': 'UNSD',
                'ZCTA': 'ZCTA5', 
                'Block Group': 'BG'
            },
            '2020': {
                "Census Tract": "TRACT",
                'Subdivision (Township)': "COUSUB",
                'Urban Areas': 'UAC',
                'Metropolitan Statistical Area': 'CBSA',
                'Designated & Incorporated Place': 'PLACE',
                'School District (elementary)': 'ELSD',
                'School District (secondary)': 'SCSD',
                'School District (unified)': 'UNSD',
                'ZCTA': 'ZCTA520', 
                'Block Group': 'BG'
            },
            '2021': {
                "Census Tract": "TRACT",
                'Subdivision (Township)': "COUSUB",
                'Urban Areas': 'UAC',
                'Metropolitan Statistical Area': 'CBSA',
                'Designated & Incorporated Place': 'PLACE',
                'School District (elementary)': 'ELSD',
                'School District (secondary)': 'SCSD',
                'School District (unified)': 'UNSD',
                'ZCTA': 'ZCTA520', 
                'Block Group': 'BG'
            },
            '2022': {
                "Census Tract": "TRACT",
                'Subdivision (Township)': "COUSUB",
                'Urban Areas': 'UAC',
                'Metropolitan Statistical Area': 'CBSA',
                'Designated & Incorporated Place': 'PLACE',
                'School District (elementary)': 'ELSD',
                'School District (secondary)': 'SCSD',
                'School District (unified)': 'UNSD',
                'ZCTA': 'ZCTA520', 
                'Block Group': 'BG'
            }
        }
        #Going to have to make a decisions about what to do with areas that don't have geometries for a given year but have data
        #For most will probably just substitue with 2010 census geographies
        #Geometires pre 2011-2012 are also going to require some more tooling but just putting down general geometries for the time being

        #This can probably be condensed into just back to the geographies as keys. As for the differnces in UAC and ZCTAs changing after
        #2020 these can just handled with 2 if statements instead. Should look into optimizing for these large dictionaries later.
        tiger_file = {
            '2010': {
                "Census Tract": f"tl_{selected_year}_17{county_code[selected_county]}_tract10",
                'Subdivision (Township)': f"tl_{selected_year}_17{county_code[selected_county]}_cousub10",
                'Urban Areas': f'tl_{selected_year}_us_uac10', 
                'Metropolitan Statistical Area': f'tl_{selected_year}_us_cbsa10',
                'Designated & Incorporated Place': f'tl_{selected_year}_17{county_code[selected_county]}_place10',
                'School District (elementary)': f'tl_{selected_year}_17{county_code[selected_county]}_elsd10',
                'School District (secondary)': f'tl_{selected_year}_17{county_code[selected_county]}_scsd10',
                'School District (unified)': f'tl_{selected_year}_17{county_code[selected_county]}_unsd10',
                'Block Group': f'tl_{selected_year}_17{county_code[selected_county]}_bg10',
            },
            '2011': {
                "Census Tract": f"tl_{selected_year}_17_tract",
                'Subdivision (Township)': f"tl_{selected_year}_17_cousub",
                'Urban Areas': f'tl_{selected_year}_us_uac10', 
                'Metropolitan Statistical Area': f'tl_{selected_year}_us_cbsa',
                'Designated & Incorporated Place': f'tl_{selected_year}_17_place',
                'School District (elementary)': f'tl_{selected_year}_17_elsd',
                'School District (secondary)': f'tl_{selected_year}_17_scsd',
                'School District (unified)': f'tl_{selected_year}_17_unsd',
                'Block Group': f'tl_{selected_year}_17_bg',
            },
            '2012': {
                "Census Tract": f"tl_{selected_year}_17_tract",
                'Subdivision (Township)': f"tl_{selected_year}_17_cousub",
                'Urban Areas': f'tl_{selected_year}_us_uac10', 
                'Metropolitan Statistical Area': f'tl_{selected_year}_us_cbsa',
                'Designated & Incorporated Place': f'tl_{selected_year}_17_place',
                'School District (elementary)': f'tl_{selected_year}_17_elsd.',
                'School District (secondary)': f'tl_{selected_year}_17_scsd',
                'School District (unified)': f'tl_{selected_year}_17_unsd',
                'Block Group': f'tl_{selected_year}_17_bg'
            },
            '2013': {
                "Census Tract": f"tl_{selected_year}_17_tract",
                'Subdivision (Township)': f"tl_{selected_year}_17_cousub",
                'Urban Areas': f'tl_{selected_year}_us_uac10', 
                'Metropolitan Statistical Area': f'tl_{selected_year}_us_cbsa',
                'Designated & Incorporated Place': f'tl_{selected_year}_17_place',
                'School District (elementary)': f'tl_{selected_year}_17_elsd',
                'School District (secondary)': f'tl_{selected_year}_17_scsd',
                'School District (unified)': f'tl_{selected_year}_17_unsd',
                'ZCTA': f'tl_{selected_year}_us_zcta510', 
                'Block Group': f'tl_{selected_year}_17_bg'
            },
            '2014': {
                "Census Tract": f"tl_{selected_year}_17_tract",
                'Subdivision (Township)': f"tl_{selected_year}_17_cousub",
                'Urban Areas': f'tl_{selected_year}_us_uac10', 
                'Metropolitan Statistical Area': f'tl_{selected_year}_us_cbsa',
                'Designated & Incorporated Place': f'tl_{selected_year}_17_place',
                'School District (elementary)': f'tl_{selected_year}_17_elsd.',
                'School District (secondary)': f'tl_{selected_year}_17_scsd',
                'School District (unified)': f'tl_{selected_year}_17_unsd',
                'ZCTA': f'tl_{selected_year}_us_zcta510', 
                'Block Group': f'tl_{selected_year}_17_bg'
            },
            '2015': {
                "Census Tract": f"tl_{selected_year}_17_tract",
                'Subdivision (Township)': f"tl_{selected_year}_17_cousub",
                'Urban Areas': f'tl_{selected_year}_us_uac10', 
                'Metropolitan Statistical Area': f'tl_{selected_year}_us_cbsa',
                'Designated & Incorporated Place': f'tl_{selected_year}_17_place',
                'School District (elementary)': f'tl_{selected_year}_17_elsd',
                'School District (secondary)': f'tl_{selected_year}_17_scsd.',
                'School District (unified)': f'tl_{selected_year}_17_unsd.',
                'ZCTA': f'tl_{selected_year}_us_zcta510', 
                'Block Group': f'tl_{selected_year}_17_bg'
            },
            '2016': {
                "Census Tract": f"tl_{selected_year}_17_tract",
                'Subdivision (Township)': f"tl_{selected_year}_17_cousub",
                'Urban Areas': f'tl_{selected_year}_us_uac10', 
                'Metropolitan Statistical Area': f'tl_{selected_year}_us_cbsa',
                'Designated & Incorporated Place': f'tl_{selected_year}_17_place',
                'School District (elementary)': f'tl_{selected_year}_17_elsd',
                'School District (secondary)': f'tl_{selected_year}_17_scsd',
                'School District (unified)': f'tl_{selected_year}_17_unsd',
                'ZCTA': f'tl_{selected_year}_us_zcta510', 
                'Block Group': f'tl_{selected_year}_17_bg'
            },
            '2017': {
                "Census Tract": f"tl_{selected_year}_17_tract",
                'Subdivision (Township)': f"tl_{selected_year}_17_cousub",
                'Urban Areas': f'tl_{selected_year}_us_uac10', 
                'Metropolitan Statistical Area': f'tl_{selected_year}_us_cbsa',
                'Designated & Incorporated Place': f'tl_{selected_year}_17_place',
                'School District (elementary)': f'tl_{selected_year}_17_elsd',
                'School District (secondary)': f'tl_{selected_year}_17_scsd',
                'School District (unified)': f'tl_{selected_year}_17_unsd',
                'ZCTA': f'tl_{selected_year}_us_zcta510', 
                'Block Group': f'tl_{selected_year}_17_bg'
            },
            '2018': {
                "Census Tract": f"tl_{selected_year}_17_tract",
                'Subdivision (Township)': f"tl_{selected_year}_17_cousub",
                'Urban Areas': f'tl_{selected_year}_us_uac10', 
                'Metropolitan Statistical Area': f'tl_{selected_year}_us_cbsa',
                'Designated & Incorporated Place': f'tl_{selected_year}_17_place',
                'School District (elementary)': f'tl_{selected_year}_17_elsd',
                'School District (secondary)': f'tl_{selected_year}_17_scsd',
                'School District (unified)': f'tl_{selected_year}_17_unsd',
                'ZCTA': f'tl_{selected_year}_us_zcta510', 
                'Block Group': f'tl_{selected_year}_17_bg'
            },
            '2019': {
                "Census Tract": f"tl_{selected_year}_17_tract",
                'Subdivision (Township)': f"tl_{selected_year}_17_cousub",
                'Urban Areas': f'tl_{selected_year}_us_uac10', 
                'Metropolitan Statistical Area': f'tl_{selected_year}_us_cbsa',
                'Designated & Incorporated Place': f'tl_{selected_year}_17_place',
                'School District (elementary)': f'tl_{selected_year}_17_elsd',
                'School District (secondary)': f'tl_{selected_year}_17_scsd',
                'School District (unified)': f'tl_{selected_year}_17_unsd',
                'ZCTA': f'tl_{selected_year}_us_zcta510', 
                'Block Group': f'tl_{selected_year}_17_bg'
            },
            '2020': {
                "Census Tract": f"tl_{selected_year}_17_tract",
                'Subdivision (Township)': f"tl_{selected_year}_17_cousub",
                'Urban Areas': f'tl_{selected_year}_us_uac20', 
                'Metropolitan Statistical Area': f'tl_{selected_year}_us_cbsa',
                'Designated & Incorporated Place': f'tl_{selected_year}_17_place',
                'School District (elementary)': f'tl_{selected_year}_17_elsd',
                'School District (secondary)': f'tl_{selected_year}_17_scsd',
                'School District (unified)': f'tl_{selected_year}_17_unsd',
                'ZCTA': f'tl_{selected_year}_us_zcta520', 
                'Block Group': f'tl_{selected_year}_17_bg'
            },
            '2021': {
                "Census Tract": f"tl_{selected_year}_17_tract",
                'Subdivision (Township)': f"tl_{selected_year}_17_cousub",
                'Urban Areas': f'tl_{selected_year}_us_uac10', #this seems like a mistake on the part of the census I'd assume this should be uac20, but there is none available
                'Metropolitan Statistical Area': f'tl_{selected_year}_us_cbsa',
                'Designated & Incorporated Place': f'tl_{selected_year}_17_place',
                'School District (elementary)': f'tl_{selected_year}_17_elsd',
                'School District (secondary)': f'tl_{selected_year}_17_scsd',
                'School District (unified)': f'tl_{selected_year}_17_unsd',
                'ZCTA': f'tl_{selected_year}_us_zcta520', 
                'Block Group': f'tl_{selected_year}_17_bg'
            },
            '2022': {
                "Census Tract": f"tl_{selected_year}_17_tract",
                'Subdivision (Township)': f"tl_{selected_year}_17_cousub",
                'Urban Areas': f'tl_{selected_year}_us_uac20', 
                'Metropolitan Statistical Area': f'tl_{selected_year}_us_cbsa',
                'Designated & Incorporated Place': f'tl_{selected_year}_17_place',
                'School District (elementary)': f'tl_{selected_year}_17_elsd',
                'School District (secondary)': f'tl_{selected_year}_17_scsd',
                'School District (unified)': f'tl_{selected_year}_17_unsd',
                'ZCTA': f'tl_{selected_year}_us_zcta520', 
                'Block Group': f'tl_{selected_year}_17_bg'
            },
        }

        requires_clipping = ['Urban Areas','Metropolitan Statistical Area','Designated & Incorporated Place','School District (elementary)','School District (secondary)','School District (unified)','ZCTA']

        self.tiger_url = f"https://www2.census.gov/geo/tiger/TIGER{selected_year}/{tiger_geography[selected_year][selected_geography]}/{tiger_file[selected_year][selected_geography]}.zip"
        print(f"Tiger URL: {self.tiger_url}")
        self.progressBar.setValue(30)
        tiger_data = requests.get(self.tiger_url)
        self.progressBar.setValue(40)
        if tiger_data.status_code == 200:
            geography = gpd.read_file(self.tiger_url)
            self.progressBar.setValue(50)
            #Some geometries have GEOID20/GEOID10 instea of GEOID as a column name, this is correcting for that
            if 'GEOID10' in geography.columns:
                geography = geography.rename(columns={'GEOID10': 'GEOID'})
            if 'GEOID20' in geography.columns:
                geography = geography.rename(columns={'GEOID20': 'GEOID'})
            geography = geography.to_crs(epsg = 3435)
            geography = geography[['GEOID','geometry']]
            self.progressBar.setValue(60)            
            merge = geography.merge(df, on = "GEOID")
            self.progressBar.setValue(70)
            if self.cb_mpa.isChecked() is True:
                directory = os.path.dirname(os.path.abspath(__file__))
                mpa_path = os.path.join(directory, 'MPA.gpkg')
                mpa_boundary = gpd.read_file(mpa_path)
                mpa_boundary = mpa_boundary.to_crs(epsg = 3435)
                mpa = merge.overlay(mpa_boundary)
                vl = QgsVectorLayer(mpa.to_json(),f"{selected_dataset}_{selected_table}_{selected_geography}_{selected_county}_{selected_group}_{selected_year}_mpa","ogr")
                QgsProject.instance().addMapLayer(vl)
                self.progressBar.setValue(90)
            elif selected_geography in requires_clipping:
                county_url = f"https://www2.census.gov/geo/tiger/TIGER{selected_year}/COUNTY/tl_{selected_year}_us_county.zip"
                county_geography = gpd.read_file(county_url)
                #Out of the entire nations county map only take the ones from illinois with the following county ids
                filtered_county = county_geography[
                    (county_geography['STATEFP'] == '17') & 
                    (county_geography['COUNTYFP'] == county_code[selected_county])
                ]
                filtered_county = filtered_county.to_crs(epsg = 3435)
                #Only include whatever features are within the overlay of the county's shape
                self.progressBar.setValue(80)
                clipped = merge.overlay(filtered_county) # INSTEAD OF CLIPPED I CAN USE THE SAME METHOD THAT I USED TO ISOLATE THE COUNTY TO ISOLATE ALL THE OTHER ONES ASIDE FROM ZIPCODE
                vl = QgsVectorLayer(clipped.to_json(),f"{selected_dataset}_{selected_table}_{selected_geography}_{selected_county}_{selected_group}_{selected_year}","ogr")
                QgsProject.instance().addMapLayer(vl)
                self.progressBar.setValue(90)
            else:
                self.progressBar.setValue(80)
                vl = QgsVectorLayer(merge.to_json(),f"{selected_dataset}_{selected_table}_{selected_geography}_{selected_county}_{selected_group}_{selected_year}","ogr")
                QgsProject.instance().addMapLayer(vl)
                self.progressBar.setValue(90)
        else:
            QMessageBox.warning(self,"TIGERWEB ERROR","Invalid Tiger URL or TIGER servers are currently busy")
            
        self.progressBar.setValue(100)
        self.progressBar.reset()
    
    def clear_variable(self):
        self.tw_selected_variables.clear()
        self.tw_selected_variables.setRowCount(0)
        self.tw_selected_variables.setColumnCount(0)
        
    def add_to_variable_table(self):
        self.tw_selected_variables.setSelectionBehavior(QTableWidget.SelectRows)
        headers = ["Variables","Labels","Table"]
        self.tw_selected_variables.setColumnCount(3)
        self.tw_selected_variables.setHorizontalHeaderLabels(headers)
        row_count = self.tw_selected_variables.rowCount()
        if self.tw_census_view.rowCount() == 0:
            QMessageBox.information(self,"Populate Variables","You must run 'Get Variables' to pull an item from the table")
        else:
            selected_row = self.tw_census_view.currentRow()
            row_data = []
            for col in range(self.tw_census_view.columnCount()):
                item = self.tw_census_view.item(selected_row, col)
                if item:
                    row_data.append(item.text())
                    
            selected_data = []
            for row in range(self.tw_selected_variables.rowCount()):
                item = self.tw_selected_variables.item(row, 0)
                if item:
                    selected_data.append(item.text())
            
            if row_data[0] in selected_data:
                QMessageBox.warning(self,"Duplicate","Duplicate Entries are not permitted")
            else:
                for col in range(0,3):
                    self.tw_selected_variables.setRowCount(row_count+1)
                    self.tw_selected_variables.setItem(row_count,col,QTableWidgetItem(row_data[col]))
            
                
    def remove_from_variable_table(self):
        selected_row = self.tw_selected_variables.currentRow()
        self.tw_selected_variables.removeRow(selected_row)