# -*- coding: utf-8 -*-
"""
/***************************************************************************
 CartoDruidSync
                                 A QGIS plugin
 Plugin to synchronize SQLite databases to Cartodruid Synchronization services at ITACyL
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-01-20
        git sha              : $Format:%H$
        copyright            : (C) 2023 by ITACyL
        email                : gustavo.rio@itacyl.es
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import logging
import os.path
import shutil
import tempfile
from zipfile import ZipFile

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
# Import the code for the dialog
from qgis.core import QgsMessageLog, Qgis, QgsProject

from .cartodruid_sync_dialog import CartoDruidSyncDialog
from .crtsync_client import CrtDrdSyncClient, ApiClientListener

# Initialize Qt resources from file resources.py
# from qgis._core import QgsMessageLog, Qgis

logging.basicConfig(level=logging.INFO,
                    format='%(asctime)s - %(message)s',
                    datefmt='%Y-%m-%d %H:%M:%S')

from . import plugin_settings as stt


class CartoDruidSync:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'CartoDruidSync_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&CartoDruid Synchronizer')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        ## custom properties
        self.listener = SyncMessageListener()
        self.plugin_conf = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('CartoDruidSync', message)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/cartodruid_sync/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'CartoDruid Sync'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&CartoDruid Synchronizer'),
                action)
            self.iface.removeToolBarIcon(action)

    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.plugin_conf = stt.read_config(QgsProject.instance(), self.listener)
            print("cargada configuracion {}".format(self.plugin_conf))
            self.dlg = CartoDruidSyncDialog()
            self._initDialog()
            self.first_start = False

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()

        # See if OK was pressed
        if result:
            # get dialog info
            wks = self.dlg.wksId.text()
            username = self.dlg.userName.text()
            apikey = self.dlg.userApikey.text()
            endpoint = self.dlg.endpoint.text()
            print(f"{wks} {apikey} {endpoint}")
            file_path = self.dlg.fileWidget.filePath()
            print(f"{file_path}")

            # update plugin config
            self.plugin_conf.endpoints = [endpoint]
            self.plugin_conf.wks_configs = [stt.WksConfig(file_path, wks, username, apikey, endpoint)]
            stt.write_endpoints(QgsProject.instance(), self.plugin_conf.endpoints, self.listener)
            stt.write_sync_configs(QgsProject.instance(), self.plugin_conf.wks_configs, self.listener)

            api = CrtDrdSyncClient(endpoint, {"wks": wks, "user": username, "password": apikey})
            api.set_listener(self.listener)
            try:
                downloaded = api.exec(file_path)
                QgsMessageLog.logMessage("Downloaded database: {}".format(downloaded))
            except:
                logging.exception("Error while trying to synchronized")
                raise BaseException("Error during sync process.")
            # uncompress downloded file
            uncompressed_db = self._extract(downloaded, tempfile.gettempdir())
            # backup original file
            QgsMessageLog.logMessage("Backing up current data base to {}".format(file_path + ".backup"))
            shutil.copy(file_path, file_path + ".backup")

            # copy downloaded file to origin
            shutil.copy(uncompressed_db, file_path)

            self.iface.messageBar().pushMessage("CartoDruid Sync", "Synchronization finished successfully!",
                                                level=Qgis.Info)

    def _extract(self, file, dest_folder):
        zref = ZipFile(file, 'r')
        # sqlites in the file
        lst_db_files = list((filter(lambda x: x.filename.endswith(".sqlite"), zref.filelist)))
        if len(lst_db_files) == 0:
            raise BaseException("The downloaded file {} contains no database.".format(file))
        if len(lst_db_files) > 1:
            raise BaseException("The downloaded file {} contains more than one database.".format(file))

        # get files from zip file
        db_filename = lst_db_files[0].filename

        with ZipFile(file, 'r') as zip_ref:
            zip_ref.extractall(dest_folder)

        db_filename = os.path.join(dest_folder, db_filename)
        # check exists
        if not os.path.exists(db_filename):
            raise BaseException(
                "Something went wrong during the zip extraction, the expected file {} doesn't exists".format(
                    db_filename))
        return db_filename

    def _initDialog(self):
        # Load values from settings
        if self.plugin_conf.wks_configs:
            wks_conf = self.plugin_conf.wks_configs[0]
            self.dlg.fileWidget.setFilePath(wks_conf.db_file)
            self.dlg.wksId.setText(wks_conf.wks)
            self.dlg.userName.setText(wks_conf.username)
            self.dlg.userApikey.setText(wks_conf.apikey)
            self.dlg.endpoint.setText(wks_conf.endpoint)
        elif self.plugin_conf.endpoints:
            self.dlg.endpoint.setText(self.plugin_conf.endpoints[0])


class SyncMessageListener(ApiClientListener):

    def notify(self, message):
        logging.info(message)
        QgsMessageLog.logMessage(message, level=Qgis.Info)

    def info(self, message):
        logging.info(message)
        QgsMessageLog.logMessage(message, level=Qgis.Info)

    def error(self, message):
        print(message)
        logging.error(message)
        QgsMessageLog.logMessage(message, level=Qgis.Critical)
