# -*- coding: utf-8 -*-
"""
/***************************************************************************
 CartoDruidSyncDialog
                                 A QGIS plugin
 Plugin to synchronize SQLite databases to Cartodruid Synchronization services at ITACyL
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2023-01-20
        git sha              : $Format:%H$
        copyright            : (C) 2023 by ITACyL
        email                : gustavo.rio@itacyl.es
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os
import sqlite3

from PyQt5 import QtWidgets, QtCore
from PyQt5.QtCore import Qt
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QListWidgetItem, QDialog, QTextEdit
from qgis.PyQt import uic
from qgis.core import QgsApplication

if os.environ.get("TEST_RUNNING", 0):
    import db_manage as dbm
    from plugin_settings import resolve_path
else:
    from . import db_manage as dbm
    from .plugin_settings import resolve_path

ICONS = {
    "empty": QIcon(resolve_path("assets/transparent.png")),
    "bolt": QIcon(resolve_path("assets/lightning-bolt-24.png")),
    "time": QIcon(resolve_path("assets/time-8-24.png")),
    "layer": QIcon(resolve_path("assets/layers-24.png")),
}


class TableFilterScreen:
    def __init__(self, dialog, listener=None):
        """Constructor."""
        self.dlg = dialog
        self.listener = listener
        self.__create_help_dialog()
        self.__connect_actions()

        self.wks_config = None
        self.table_list = None
        self.selection_list = None
        self.plugin_confs = None

    def __connect_actions(self):
        # connect actions
        self.dlg.btn_add.clicked.connect(self.__add_selected)
        self.dlg.btn_add_all.clicked.connect(self.__add_all)
        self.dlg.btn_remove.clicked.connect(self.__remove_selected)
        self.dlg.btn_reload.clicked.connect(self.reload_tables)
        self.dlg.chk_apply_filter.clicked.connect(self.__enable_form_fields)
        self.dlg.btn_help.clicked.connect(self.__show_help_dialog)

    def __create_help_dialog(self):
        dialog = QDialog()
        dialog.setFixedSize(475, 300)
        text_edit = QTextEdit()
        text_edit.setReadOnly(True)
        # get current user language to find proper info.
        lang = QgsApplication.instance().locale()
        if lang not in ["en", "es"]:  # TODO: make this dynamic
            lang = "en"
        html_file = resolve_path(f"i18n/table_filter_help.{lang}.html")
        with open(html_file, "r") as file:
            help_content = file.read().replace("\n", "")

        text_edit.insertHtml(help_content.format(layer=resolve_path("assets/layers-24.png"),
                                                 bolt=resolve_path("assets/lightning-bolt-24.png"),
                                                 time=resolve_path("assets/time-8-24.png")))

        layout = QtWidgets.QVBoxLayout()
        layout.addWidget(text_edit)
        dialog.setLayout(layout)
        dialog.setModal(True)

        # Establece el tamaño mínimo y máximo del layout para que se ajuste al tamaño del dialog
        layout.setContentsMargins(0, 0, 0, 0)
        layout.setSizeConstraint(QtWidgets.QLayout.SetMinAndMaxSize)

        self.help_dialog = dialog

    def __show_help_dialog(self):
        self.help_dialog.show()

    def load_settings(self, wks_config):
        is_filter_checked = wks_config is not None and wks_config.table_filter is not None
        self.dlg.chk_apply_filter.setChecked(is_filter_checked)
        self.__enable_form_fields()

        table_filter = wks_config.table_filter
        is_checked = table_filter is not None
        self.dlg.chk_apply_filter.setChecked(is_checked)
        # load table_names into selection combo
        if table_filter and len(table_filter) > 0:
            self.__add_tables_to_selection(table_filter)
            # for table_name in table_filter:
            #     self.dlg.lstw_selectionList.addItem(QListWidgetItem(table_name))

    def __add_selected(self):
        # get selected tables and check if some of them don't have the trigger and ask to add them
        ok_tables = []
        no_trigger_tables = []
        for item_idx in self.dlg.lstw_tableList.selectedIndexes():
            table_name, has_index = self.table_list[item_idx.row()]
            ok_tables.append(table_name)
            # if has_index:
            # else:
            #     no_trigger_tables.append(table_name)

        # add ok tables to selected list
        self.__add_tables_to_selection(ok_tables)

        # msg = "Estas tablas no tienen asociado un trigger de actualización, este trigger es necesario para registrar " \
        #       "los cambios: \n {} \n¿Deseas crearlos?".format("\n".join(no_trigger_tables))
        #
        # result = QMessageBox.question(None, "Ha seleccionado tablas no monitorizadas", msg,
        #                               QMessageBox.Yes | QMessageBox.No, QMessageBox.No)
        # if result == QMessageBox.Yes:
        #     # create trigger and add tables to list
        #     conn = sqlite3.connect(self.get_db_file())
        #     try:
        #         dbm.create_update_trigger()
        #     finally:
        #         if conn:
        #             conn.close()

    def __add_tables_to_selection(self, table_list):
        current_selection = self.__get_current_selection()
        for table_name in table_list:
            if table_name not in current_selection:
                self.dlg.lstw_selectionList.addItem(QListWidgetItem(table_name))

    def __get_current_selection(self):
        current_selection = []
        for i in range(self.dlg.lstw_selectionList.count()):
            current_selection.append(self.dlg.lstw_selectionList.item(i).text())
        return current_selection

    def __enable_form_fields(self):
        enable = self.dlg.chk_apply_filter.isChecked()
        fields = ['btn_add', 'btn_add_all', 'btn_remove', 'btn_reload', 'lstw_selectionList', 'lstw_tableList']
        for f in fields:
            widget = getattr(self.dlg, f)
            widget.setEnabled(enable)

    def __add_all(self):
        if not self.table_list:
            return
        all_tables = [name for name, has_idx in self.table_list]
        self.__add_tables_to_selection(all_tables)

    def __remove_selected(self):
        current_selection = self.dlg.lstw_selectionList.selectedIndexes()
        for idx in reversed(current_selection):
            self.dlg.lstw_selectionList.takeItem(idx.row())

    # def __select_layer(self):
    #     # disable file widget
    #     self.fileWidget.setEnabled(False)
    #     self.combo_layer_name.setEnabled(True)
    #
    # def __select_file(self):
    #     self.fileWidget.setEnabled(True)
    #     self.combo_layer_name.setEnabled(False)

    def reload_tables(self):
        self.dlg.lstw_tableList.clear()
        file_path = self.dlg.fileWidget.filePath()
        if not file_path:
            self.dlg.lstw_tableList.clear()
            self.dlg.lstw_selectionList.clear()
            return
        file_path = file_path.replace('\\', '/')
        # leer las tablas de la BD
        geo_tables = dbm.get_geo_layers(file_path)
        conn = sqlite3.connect(file_path)
        try:
            table_names = dbm.get_table_list(conn)
            self.table_list = []
            item_list = []
            # check if tablas has an update trigger
            for name in table_names:
                is_geo = name in geo_tables
                has_trigger = dbm.has_update_trigger(conn, name)
                has_update_col = dbm.get_update_col(conn, name)
                self.table_list.append((name, has_trigger))
                item, widget = self.__create_list_item(self.dlg.lstw_tableList, name, is_geo, has_trigger,
                                                       has_update_col)
                self.dlg.lstw_tableList.addItem(item)
                self.dlg.lstw_tableList.setItemWidget(item, widget)

        finally:
            if conn:
                conn.close()

    def __create_list_item(self, list_widget, name, isgeo, has_trigger, has_update_coll):
        icon1 = ICONS["layer"] if isgeo else ICONS["empty"]
        icon2 = ICONS["bolt"] if has_trigger else ICONS["empty"]
        icon3 = ICONS["time"] if has_update_coll else ICONS["empty"]
        item = QtWidgets.QListWidgetItem(list_widget)
        item.setSizeHint(QtCore.QSize(0, 25))
        widget = IconTextWidget(name, icon1, icon2, icon3)
        return item, widget

    def get_table_filter(self):
        if not self.dlg.chk_apply_filter.isChecked():
            return None
        else:
            return self.__get_current_selection()


class IconTextWidget(QtWidgets.QWidget):
    def __init__(self, text, icon1, icon2, icon3, parent=None):
        super().__init__(parent)
        self.text_label = QtWidgets.QLabel(text)
        self.icon1_label = QtWidgets.QLabel()
        self.icon1_label.setPixmap(icon1.pixmap(16, 16))
        self.icon2_label = QtWidgets.QLabel()
        self.icon2_label.setPixmap(icon2.pixmap(16, 16))
        self.icon3_label = QtWidgets.QLabel()
        self.icon3_label.setPixmap(icon3.pixmap(16, 16))
        layout = QtWidgets.QHBoxLayout()
        # izquierdo, arriba, derecho y abajo
        layout.setContentsMargins(0, 0, 0, 0)
        layout.setAlignment(Qt.AlignLeft)
        layout.addWidget(self.text_label)
        layout.addWidget(self.icon1_label)
        layout.addWidget(self.icon2_label)
        layout.addWidget(self.icon3_label)
        self.setLayout(layout)
