# -*- coding: utf-8 -*-

"""
/***************************************************************************
 CartAGen4QGIS
                                 A QGIS plugin
 Cartographic generalization
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-05-11
        copyright            : (C) 2023 by Guillaume Touya, Justin Berli & Paul Bourcier
        email                : guillaume.touya@ign.fr
 ***************************************************************************/
"""

__author__ = 'Guillaume Touya, Justin Berli & Paul Bourcier'
__date__ = '2023-05-11'
__copyright__ = '(C) 2023 by Guillaume Touya, Justin Berli & Paul Bourcier'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessing, QgsFeatureSink, QgsProcessingAlgorithm,
    QgsFeature, QgsGeometry, QgsProcessingParameterDefinition, QgsWkbTypes
)
from qgis.core import (
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterBoolean,
    QgsProcessingParameterNumber,
    QgsProcessingParameterDistance,
    QgsProcessingParameterField,
    QgsProcessingParameterEnum,
    QgsProcessingParameterMultipleLayers
)

class HullDelaunay(QgsProcessingAlgorithm):
    """
    Create the hull of a set of points using the Delaunay triangulation.

    This algorithm was described by Duckham et al. [1] and is also described in Galton & Duckham [2] 
    It first computes the Delaunay triangulation of the points, and then removes iteratively the boundary edges that are longer than a given parameter.

    Parameters:

    points (list of Point) – The set of points.

    length (float) – The length in meters below which the Delaunay triangles edges are removed.

    Returns:
    
    Polygon
    
    Warning : This algorithm cannot create multiple polygons, unlike hull_swinging_arm(). Using a length too low can produce an invalid geometry.

    See also : hull_swinging_arm
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    INPUT = 'INPUT'

    LENGTH = 'LENGTH'
 
    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Hull Delaunay'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Points'

    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        from cartagen4qgis import get_plugin_icon
        return get_plugin_icon()

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr("Create the hull of a set of points using the Delaunay triangulation.\nThis algorithm first computes the Delaunay triangulation of the points, and then removes iteratively the boundary edges that are longer than a given parameter.\nLength: the length in meters below which the Delaunay triangles edges are removed.\nWarning : this algorithm cannot create multiple polygons, unlike hull_swinging_arm(). Using a length too low can produce an invalid geometry.")
        
    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return HullDelaunay()

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr('Input point layer'),
                [QgsProcessing.TypeVectorPoint]
            )
        )

        length = QgsProcessingParameterNumber(
            self.LENGTH,
            self.tr('Length threshold'),
            type=QgsProcessingParameterNumber.Double,
            defaultValue=2000,
            optional=False
        )
        self.addParameter(length)
        
        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Delaunay hull')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        import geopandas as gpd
        import pandas
        from cartagen4qgis import get_plugin_icon
        from cartagen import hull_delaunay, hull_swinging_arm
        from cartagen4qgis.src.tools import list_to_qgis_feature
        from shapely import Polygon
        from shapely.wkt import loads

        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.
        source = self.parameterAsSource(parameters, self.INPUT, context)
        #transform the source into a GeoDataFrame
        points = gpd.GeoDataFrame.from_features(source.getFeatures())
        
        # Compute the number of steps to display within the progress bar and
        # get features from source
        # total = 100.0 / source.featureCount() if source.featureCount() else 0
        
        feedback.setProgress(1) #set the loading bar to 1%

        #retrieve the other input parameter
        length = self.parameterAsDouble(parameters, self.LENGTH, context)

        #transform the gdf to a list of geometry, to fit the CartAGen algorithm
        points = list(points.geometry)

        #perform the CG algorithm
        res = hull_delaunay(points, length=length)

        feedback.setProgress(90) #set the loading bar to 90%
        
        # Convert the result to a gdf, the gdf to a dictionnary, and the dictionnary to a lsit of QgsFeature()
        res = gpd.GeoDataFrame(geometry=gpd.GeoSeries(res))
        res = res.to_dict('records')
        res = list_to_qgis_feature(res)
        
        # Define the ouput sink
        (sink, dest_id) = self.parameterAsSink(parameters, self.OUTPUT,
                context, res[0].fields(), QgsWkbTypes.Polygon, source.sourceCrs())
        
        # Add a feature in the sink
        sink.addFeatures(res, QgsFeatureSink.FastInsert)
        feedback.setProgress(100)
        return {
            self.OUTPUT: dest_id
        }

class HullSwingingArm(QgsProcessingAlgorithm):
    """
    Create the hull of a set of points using the swinging arm algorithm.

    This algorithm is described in Galton & Duckham. [1] 
    Starting at the most northen point, iteratively create the border of a polygon by adding the first point intersecting the line (the arm) rotating around the point.

    Parameters:

    points (list of Point) – The set of points.

    length (int) – The length of the line (or arm) that will rotate around the point.

    direction (str, optional) – The direction of the arm’s rotation. Can be ‘ccw’ or ‘cw’.

    Returns:

    list of Polygon

    Create the hull of a set of points using the Delaunay triangulation.

    Notes : This algorithm can create multiple polygons, unlike hull_delaunay()
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    INPUT = 'INPUT'

    DIRECTION = 'DIRECTION'
    LENGTH = 'LENGTH'
 
    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Hull swinging arm'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Points'

    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        from cartagen4qgis import get_plugin_icon
        return get_plugin_icon()

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr("Create the hull of a set of points using the swinging arm algorithm.\nStarting at the most northen point, iteratively create the border of a polygon by adding the first point intersecting the line (the arm) rotating around the point.\nLength : the length of the line (or arm) that will rotate around the point.\nDirection : the direction of the arm’s rotation. Can be ‘ccw’ or ‘cw’.\nNotes : This algorithm can create multiple polygons, unlike Hull Delaunay")
        
    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return HullSwingingArm()

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr('Input point layer'),
                [QgsProcessing.TypeVectorPoint]
            )
        )

        length = QgsProcessingParameterNumber(
            self.LENGTH,
            self.tr('Length threshold'),
            type=QgsProcessingParameterNumber.Double,
            defaultValue=2000,
            optional=False
        )
        self.addParameter(length)


        directions = ['cw', 'ccw']
        direction = QgsProcessingParameterEnum(
                self.DIRECTION,
                'Rotation direction of the arm',
                directions,
                defaultValue=0  
            )
        direction.setFlags(direction.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(direction)
        
        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Swinging arm hull(s)')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """ 
        import geopandas as gpd
        import pandas
        from cartagen4qgis import get_plugin_icon
        from cartagen import hull_delaunay, hull_swinging_arm
        from cartagen4qgis.src.tools import list_to_qgis_feature
        from shapely import Polygon
        from shapely.wkt import loads
        
        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.
        source = self.parameterAsSource(parameters, self.INPUT, context)
        #transform the source to a GeoDataFrame
        points = gpd.GeoDataFrame.from_features(source.getFeatures())
        
        # Compute the number of steps to display within the progress bar and
        # get features from source
        # total = 100.0 / source.featureCount() if source.featureCount() else 0
        
        # retrieve the values of the parameters
        length = self.parameterAsDouble(parameters, self.LENGTH, context)
        directions = ['cw', 'ccw']
        direction = self.parameterAsString(parameters, self.DIRECTION, context)

        feedback.setProgress(1) #set loading bar to 1 %

        # transform the GDF into a list of geometry to fit the algorithm
        points = list(points.geometry)

        # Perform CG algorithm
        res = hull_swinging_arm(points, length=length, direction=directions[int(direction)])

        feedback.setProgress(90) #set loading bar to 90 %
        
        #transform the result into a gdf, the gdf into a dictionnary, and the dictionnary into a list of QgsFeature()
        res = gpd.GeoDataFrame(geometry=gpd.GeoSeries(res))
        res = res.to_dict('records')
        res = list_to_qgis_feature(res)
     
        #Define the output sink
        (sink, dest_id) = self.parameterAsSink(parameters, self.OUTPUT,
                context, res[0].fields(), QgsWkbTypes.Polygon, source.sourceCrs())
        
        # Add a feature in the sink
        sink.addFeatures(res, QgsFeatureSink.FastInsert)
        
        feedback.setProgress(100) #set loading bar to 100 %
        
        return {
            self.OUTPUT: dest_id
        }
