# -*- coding: utf-8 -*-

"""
/***************************************************************************
 CartAGen4QGIS
                                 A QGIS plugin
 Cartographic generalization
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-05-11
        copyright            : (C) 2023 by Guillaume Touya, Justin Berli & Paul Bourcier
        email                : guillaume.touya@ign.fr
 ***************************************************************************/
"""

__author__ = 'Guillaume Touya, Justin Berli & Paul Bourcier'
__date__ = '2023-05-11'
__copyright__ = '(C) 2023 by Guillaume Touya, Justin Berli & Paul Bourcier'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication, QVariant
from qgis.core import (
    QgsProcessing, QgsFeatureSink, QgsProcessingAlgorithm,
    QgsFeature, QgsGeometry, QgsProcessingParameterDefinition,
    QgsWkbTypes, QgsVectorLayer, QgsField, QgsFields,
    QgsProcessingParameterField
)
from qgis.core import (
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterBoolean,
    QgsProcessingParameterNumber,
    QgsProcessingParameterDistance,
)
from qgis.PyQt.QtWidgets import QMessageBox

class RuralTraffic(QgsProcessingAlgorithm):
    """
        Detect central roads inside a network using traffic simulation.

    This algorithm proposed by Touya detects the most used road inside a network by simulating traffic between attraction points.
    Thos points can be provided or they can randomly be extracted from the network.

    Parameters
    ----------
    roads : GeoDataFrame of LineString
        The road network to analyze.
    min_traffic : int, optional 
        The minimum number of time a road must be used to be kept.
    attraction_points : GeoDataFrame of Point, optional
        The attraction points between which traffic will be calculated. 
        If provided, the points doesn’t need to be on the network, 
        they will be snapped to the closest node (i.e. intersection) on the network. 
        Default to None, meaning a random set a nodes will be taken.
    max_distance : int, optional  
        If attraction points are provided, it is the maximum distance from which the provided
        attraction points are snapped to the road network nodes. If further from this distance, 
        the point will not be used. Default to None, meaning the point is snapped no matter its
        distance from the network.
    sample_size : int, optional : 
        If no attraction points are provided, it is the number of nodes to take into account
        when calculating traffic. This means the calculation between all pairs of nodes,
        this can be really resource-intensive.
    export_samples : bool, optional
        If set to True, returns a tuple of two GeoDataFrame instead of a single GeoDataFrame.
        The second is a point GeoDataFrame of the samples used for the traffic calculation.
        If attraction points were provided, returns the snapped points on the nodes, 
        else returns the random node sample.
    cost : str, optional 
        The name of the attribute giving the cost of the road section.
        Make sure the attribute is a number. Default to None, which means the length
        of the road is used as the cost.

    Returns
    -------
    roads : GeoDataFrame of LineString
         The network with simulated traffic information.
    samples : GeoDataFrame of Point, optional 
        The random sample if wanted, and if no attraction points were provided.
    
    See Also
    --------
    rural_betweeness
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    ROAD_OUTPUT = 'ROAD_OUTPUT'
    ROAD_INPUT = 'ROAD_INPUT'
    POINT_INPUT = 'POINT_INPUT'
    POINT_OUTPUT = 'POINT_OUTPUT'
    MIN_TRAFFIC = 'MIN_TRAFFIC'
    MAX_DISTANCE = 'MAX_DISTANCE'
    SAMPLE_SIZE = 'SAMPLE_SIZE'
    EXPORT_SAMPLE = 'EXPORT_SAMPLE'
    COST = 'COST'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.ROAD_INPUT,
                self.tr('Input road network'),
                [QgsProcessing.TypeVectorLine]
            )
        )
        
        min_traffic = QgsProcessingParameterNumber(
                self.MIN_TRAFFIC,
                self.tr('road attendance threshold'),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=None,
                optional=True
            )
        min_traffic.setFlags(min_traffic.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(min_traffic)
        
        attraction_points = QgsProcessingParameterFeatureSource(
                self.POINT_INPUT,
                self.tr('Input attraction points'),
                [QgsProcessing.TypeVectorPoint],
                optional=True
            )
        #attraction_points.setFlags(attraction_points.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(attraction_points)
        
        max_distance = QgsProcessingParameterNumber(
                self.MAX_DISTANCE,
                self.tr('point snapping max distance'),
                type=QgsProcessingParameterNumber.Double,
                defaultValue=None,
                optional=True
            )
        max_distance.setFlags(max_distance.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(max_distance)
        
        sample_size = QgsProcessingParameterNumber(
                self.SAMPLE_SIZE,
                self.tr('Random attraction points sample'),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=10,
                optional=True
            )
        #sample_size.setFlags(sample_size.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(sample_size)
        
        export_samples = QgsProcessingParameterBoolean(
                self.EXPORT_SAMPLE,
                self.tr('Export point sample'),
                defaultValue=False,
                optional=True
            )
        export_samples.setFlags(export_samples.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(export_samples)
        
        cost = QgsProcessingParameterField(
            self.COST,
            self.tr('Attributes of cost of road section'),
            None, 
            'ROAD_INPUT', 
            QgsProcessingParameterField.Numeric, allowMultiple = False, optional = True)
        cost.setFlags(cost.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(cost)
        

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.ROAD_OUTPUT,
                self.tr('Network with traffic info')
            )
        )
        
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.POINT_OUTPUT,
                self.tr('Attraction points used')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        from cartagen import rural_traffic
        from cartagen4qgis.src.tools import list_to_qgis_feature, list_to_qgis_feature_2
        import geopandas as gpd
        from shapely import Polygon
        from shapely.wkt import loads

        # Get the QGIS source from the parameters
        source = self.parameterAsSource(parameters, self.ROAD_INPUT, context)
        gdf = gpd.GeoDataFrame.from_features(source.getFeatures())
        points = self.parameterAsSource(parameters, self.POINT_INPUT, context)
        cost = self.parameterAsFields(parameters, self.COST, context)
        min_traffic = self.parameterAsInt(parameters, self.MIN_TRAFFIC, context)
        sample_size = self.parameterAsInt(parameters, self.SAMPLE_SIZE, context)
        export_samples = self.parameterAsBoolean(parameters, self.EXPORT_SAMPLE, context)
        max_distance = self.parameterAsDouble(parameters, self.MAX_DISTANCE, context)
        if max_distance == 0:
            max_distance = None
        
        try:
            point_gdf = gpd.GeoDataFrame.from_features(points.getFeatures())
            gdf_final = rural_traffic(gdf, attraction_points = point_gdf, cost = cost[0], min_traffic=min_traffic, max_distance=max_distance, sample_size=sample_size, 
            export_samples= export_samples)
            if not export_samples:
                gdf_final = (gdf_final, 0)
            
        except IndexError: #si pas de champs cost indiqué
            gdf_final = rural_traffic(gdf, attraction_points = point_gdf, min_traffic=min_traffic, max_distance=max_distance, sample_size=sample_size, 
            export_samples=export_samples)
            if not export_samples:
                gdf_final = (gdf_final, 0)
        
        except:
            try:
                gdf_final = rural_traffic(gdf, cost = cost[0], min_traffic=min_traffic, max_distance=max_distance, sample_size=sample_size, 
                export_samples=export_samples)
                if not export_samples:
                    gdf_final = (gdf_final, 0)

            except:
                gdf_final = rural_traffic(gdf, min_traffic=min_traffic, max_distance=max_distance, sample_size=sample_size, 
                export_samples=export_samples)
                if not export_samples:
                    gdf_final = (gdf_final, 0)
            
        # Convert the source to GeoDataFrame, get the list of records and the number of entities
        
        
        
       # records = gdf.to_dict('records')
      #  count = len(records)
        feedback.setProgress(1) # set the loading bar to 1 %

        # Compute the number of steps to display within the progress bar and
        #total = 100.0 / count if count > 0 else 0
        
        # Retrieve parameters
        #area = self.parameterAsInt(parameters, self.AREA, context)
        #miller = self.parameterAsDouble(parameters, self.MILLER, context)

        # Actual algorithm
       # roads = []
       # for road in records:
       #     roads.append(road['geometry'])

       # faces = network_faces(roads, convex_hull=False)

      #  roundabouts = []
      #  index = 0
      #  for current, face in enumerate(faces):
            # Stop the algorithm if cancel button has been clicked
        #    if feedback.isCanceled():
         #       break

         #   add, infos = is_roundabout(face, area, miller)
         #   if add:
           #     infos['cid'] = index
           #     roundabouts.append(infos)
           #     index += 1

            # Update the progress bar
          #  feedback.setProgress(int(current * total))
           
      #  if len(roundabouts) == 0: #manually create fields if no roundabout is detected
       #     fields = QgsFields()
           # fields.append(QgsField("index", QVariant.Double))
        #    fields.append(QgsField("cid",  QVariant.Int))
          #  res = [QgsFeature(fields)] #create a list of empty QgsFeature with the created fields

            # QMessageBox.warning(None, "Warning", "No roundabouts detected, output layer is empty.")

       # else:    
            #convert the result to a GeoDataFrame, and this gdf into a list of dicts and then a list of QgsFeature()
          #  gdf_final = gpd.GeoDataFrame(roundabouts, crs = source.sourceCrs().authid())
          #  res = gdf_final.to_dict('records')
           # res = list_to_qgis_feature(res)
       
        res1 = gdf_final[0].to_dict('records')
        res1 = list_to_qgis_feature(res1)
     
        #Create the feature sink
        (sink, dest_id) = self.parameterAsSink(
            parameters, self.ROAD_OUTPUT, context,
            fields=res1[0].fields(),
            geometryType=QgsWkbTypes.LineString,
            crs=source.sourceCrs()
        )

        #Add features to the sink
        sink.addFeatures(res1, QgsFeatureSink.FastInsert)

        if export_samples:
            res2 = gdf_final[1].to_dict('records')
            res2 = list_to_qgis_feature(res2)
            (sink2, dest_id2) = self.parameterAsSink(
                parameters, self.POINT_OUTPUT, context,
                fields=res2[0].fields(),
                geometryType=QgsWkbTypes.Point,
                crs=source.sourceCrs())
            sink2.addFeatures(res2, QgsFeatureSink.FastInsert)
            return {
                self.ROAD_OUTPUT: dest_id,
                self.POINT_OUTPUT: dest_id2
            }
        else: 
            return {
                self.ROAD_OUTPUT: dest_id
            }

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Rural traffic'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Network'

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr("Detect central roads inside a network using traffic simulation.\nThis algorithm proposed by Touya detects the most used road inside a network by simulating traffic between attraction points. \n Those points can be provided or they can randomly be extracted from the network. \n Road network : the road network to analyze. \n Attraction points : The attraction points between which traffic will be calculated. If provided, the points doesn't need to be on the network, they will be snapped to the closest node (*i.e.* intersection) on the network. Default to None, meaning a random set a nodes will be taken. \n Sample size : If no attraction points are provided, it is the number of nodes to take into account when calculating traffic. This means the calculation between all pairs of nodes, this can be really resource-intensive. \n Road attendance threshold : the minimum number of time a road must be used to be kept. \n Max distance : if attraction points are provided, it is the maximum distance from which the provided attraction points are snapped to the road network nodes. If further from this distance, the point will not be used. Default to None, meaning the point is snapped no matter its distance from the network. \n Export samples : if set to True, returns the point used for the traffic calculation (provided or randomly generated), in addition to the enriched road network. \n  Cost : the name of the attribute giving the cost of the road section. Default to None, which means the length of the road is used as the cost.")
        
    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        from cartagen4qgis import get_plugin_icon
        return get_plugin_icon()

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return RuralTraffic()
