# -*- coding: utf-8 -*-
"""
/***************************************************************************
CartAGen4QGIS
A QGIS plugin
Cartographic generalization
Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
-------------------
        begin                : 2023-05-11
        copyright            : (C) 2023 by Guillaume Touya, Justin Berli & Paul Bourcier
        email                : guillaume.touya@ign.fr
 ***************************************************************************/
"""
__author__ = 'Guillaume Touya, Justin Berli & Paul Bourcier'
__date__ = '2023-05-11'
__copyright__ = '(C) 2023 by Guillaume Touya, Justin Berli & Paul Bourcier'
# This will get replaced with a git SHA1 when you do a git archive
__revision__ = '$Format:%H$'

import os
import sys
import inspect

cmd_folder = os.path.split(inspect.getfile(inspect.currentframe()))[0]
if cmd_folder not in sys.path:
    sys.path.insert(0, cmd_folder)


class CartAGen4QGISPlugin(object):
    
    def __init__(self):
        self.provider = None
        self.plugin_dir = os.path.dirname(__file__)
        self.initialized = False
    
    def initProcessing(self):
        """Init Processing provider for QGIS >= 3.8."""
        if self.initialized:
            return  # Éviter la double initialisation
            
        try:
            # Import retardé pour éviter les problèmes de chargement
            from qgis.core import QgsApplication
            from cartagen4qgis.src.provider import CartAGen4QGISProvider
            self.provider = CartAGen4QGISProvider()
            QgsApplication.processingRegistry().addProvider(self.provider)
            self.initialized = True
        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(
                f"There was an error during the initialization of the Processing provider: {str(e)}",
                'CartAGen',
                Qgis.Critical
            )
    
    def initGui(self):
        # Delay qt modules import
        try:
            from qgis.core import QgsMessageLog, Qgis

            # Check dependencies
            if not self.check_dependencies():
                if self.is_flatpak():
                    self.show_flatpak_instructions()
                else:
                    self.prompt_install_dependencies()

            if not self.initialized:
                self.initProcessing()
                    
        except Exception as e:
            # Use a try catch to avoid QGIS crashing
            try:
                from qgis.core import QgsMessageLog, Qgis
                QgsMessageLog.logMessage(
                    f"There was an error while loading the CartAGen plugin: {str(e)}",
                    'CartAGen',
                    Qgis.Critical
                )
            except:
                # If logging fail, send to stderr
                print(f"CRITICAL ERROR WITH CARTAGEN4QGIS PLUGIN: {str(e)}", file=sys.stderr)
    
    def unload(self):
        try:
            from qgis.core import QgsApplication
            if self.provider and self.initialized:
                QgsApplication.processingRegistry().removeProvider(self.provider)
                self.provider = None
                self.initialized = False
        except Exception as e:
            # Ignorer les erreurs au déchargement
            pass
    
    # ========== GESTION DES DÉPENDANCES ==========
    
    def is_flatpak(self):
        """Check if QGIS is runnning within a flatpak environment"""
        return os.path.exists('/.flatpak-info') or \
            'flatpak' in sys.executable.lower() or \
            '/app/' in sys.prefix
    
    def check_dependencies(self):
        """Check if cartagen have been installed already"""
        try:
            __import__('cartagen')
        except ImportError:
            return False
        return True
    
    def prompt_install_dependencies(self):
        """Display a prompt to ask for the installation of cartagen"""
        try:
            from qgis.PyQt.QtWidgets import QMessageBox
            
            msg = QMessageBox()
            msg.setIcon(QMessageBox.Information)
            msg.setWindowTitle("Missing dependencies for CartAGen")
            msg.setText("This plugin requires the CartAGen Python library to properly work.")
            
            msg.setInformativeText(
                f"CartAGen will be installed within the Python environment used by QGIS.\n\n"
                f"Proceed?"
            )
            
            msg.setStandardButtons(QMessageBox.Yes | QMessageBox.No)
            msg.setDefaultButton(QMessageBox.Yes)
            
            reply = msg.exec_()
            
            if reply == QMessageBox.Yes:
                self.install_dependencies()
            else:
                QMessageBox.warning(
                    None,
                    "Installation cancelled",
                    "CartAGen won't work without the proper dependencies.\n"
                    "You can try again later by reloading the plugin."
                )
        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(
                f"There was an error during the dialog display: {str(e)}",
                'CartAGen',
                Qgis.Critical
            )
    
    def install_dependencies(self):
        """Install deendencies with a progress bar"""
        import subprocess
        
        try:
            from qgis.PyQt.QtWidgets import QProgressDialog, QMessageBox, QApplication
            from qgis.PyQt.QtCore import Qt
            from qgis.core import QgsMessageLog, Qgis
            
            progress = QProgressDialog(
                "Installing dependencies...\n"
                "This can take several minutes.",
                None, 0,
                0  # Undefined mode
            )
            progress.setWindowTitle("Dependencies installation")
            progress.setWindowModality(Qt.WindowModal)
            progress.setCancelButton(None)  # deactivate cancel
            progress.show()
            QApplication.processEvents()
            
            # Prepare the installation command
            cmd = [ sys.executable, '-m', 'pip', 'install', '--user', 'cartagen' ]
            
            # Installation using pip
            result = subprocess.run(cmd, capture_output=True, text=True, timeout=300)
            # End progress bar
            progress.close()
            
            if result.returncode == 0:
                QMessageBox.information(
                    None,
                    "Installation successful",
                    "Dependencies have been installed successfully.\n"
                    "Please restart QGIS before using the plugin."
                )
                QgsMessageLog.logMessage(
                    "Dependencies installed successfully",
                    'CartAGen',
                    Qgis.Success
                )
            else:
                raise Exception(result.stderr)
                
        except subprocess.TimeoutExpired:
            if 'progress' in locals():
                progress.close()
            QMessageBox.critical(
                None,
                "Installation error",
                "The installation process took too long and was interupted.\n"
                "Check your internet connection."
            )
            QgsMessageLog.logMessage(
                "Timeout during dependencies installation",
                'CartAGen',
                Qgis.Critical
            )
            
        except Exception as e:
            if 'progress' in locals():
                progress.close()
            error_msg = str(e)
            QMessageBox.critical(
                None,
                "Intallation error",
                f"An error occured during the installation:\n\n{error_msg}\n\n"
                f"You can try to manually install dependencies using the following command within QGIS's Python environment:\n"
                f"pip install --user cartagen"
            )
            QgsMessageLog.logMessage(
                f"Installation error: {error_msg}",
                'CartAGen',
                Qgis.Critical
            )
    
    def show_flatpak_instructions(self):
        """Display Flatpak information"""
        try:
            from qgis.PyQt.QtWidgets import QDialog, QVBoxLayout, QTextBrowser, QPushButton, QMessageBox, QApplication
            
            dialog = QDialog()
            dialog.setWindowTitle("Missing dependencies for the CartAGen plugin")
            dialog.setMinimumWidth(650)
            dialog.setMinimumHeight(500)
            
            layout = QVBoxLayout()
            
            text_browser = QTextBrowser()
            text_browser.setOpenExternalLinks(True)
            text_browser.setHtml("""
                <h2>Detected QGIS installed with Flatpak - Dependencies required for CartAGen plugin</h2>
                
                <p>CartAGen needs some Python dependencies to properly work. You need
                to install them within the Python environment used by QGIS.</p>
                
                <h3>Instructions :</h3>
                
                <ol>
                    <li><b>Close QGIS</b></li>
                    <li>Open a terminal</li>
                    <li>Execute the following commands:</li>
                </ol>
                
                <pre style="background-color: #f5f5f5; padding: 10px; border-radius: 5px; font-family: monospace;">
flatpak run --devel --command=pip3 org.qgis.qgis install cartagen
                </pre>
                
                <p><b>Note :</b> If you get an error with <code>--devel</code>, 
                you should first install the SDK (replace VERSION with the version returned in the error):</p>
                
                <pre style="background-color: #f5f5f5; padding: 10px; border-radius: 5px; font-family: monospace;">
flatpak install org.kde.Sdk/x86_64/VERSION
                </pre>
                
                <p>Once installed, restart QGIS.</p>
                
                <p>More information on the CartAGen documentation page: 
                <a href="https://cartagen.readthedocs.io/en/latest/qgis.html">Documentation CartAGen</a></p>
            """)
            
            layout.addWidget(text_browser)
            
            # Button to copy command
            copy_button = QPushButton("Copy the installation command")
            copy_button.clicked.connect(lambda: self.copy_flatpak_commands())
            layout.addWidget(copy_button)
            
            close_button = QPushButton("Close")
            close_button.clicked.connect(dialog.close)
            layout.addWidget(close_button)
            
            dialog.setLayout(layout)
            dialog.exec_()
        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(
                f"Error during the display of Flatpak instructions: {str(e)}",
                'CartAGen',
                Qgis.Critical
            )
    
    def copy_flatpak_commands(self):
        """Copy flatpak commands in the clipboard"""
        try:
            from qgis.PyQt.QtWidgets import QMessageBox, QApplication
            commands = """flatpak run --devel --command=pip3 org.qgis.qgis install cartagen"""
            QApplication.clipboard().setText(commands)
            QMessageBox.information(
                None,
                "Copied",
                "The Flatpak installation command was copied in your clipboard.\n"
                "Paste them inside a terminal using Ctrl+Shift+V"
            )
        except Exception as e:
            from qgis.core import QgsMessageLog, Qgis
            QgsMessageLog.logMessage(
                f"Error during flatpak command copying: {str(e)}",
                'CartAGen',
                Qgis.Critical
            )
