# -*- coding: utf-8 -*-

"""
/***************************************************************************
 CartAGen4QGIS
                                 A QGIS plugin
 Cartographic generalization
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-06-18
        copyright            : (C) 2024 by Guillaume Touya, Justin Berli & Paul Bourcier
        email                : guillaume.touya@ign.fr
 ***************************************************************************/
"""

__author__ = 'Guillaume Touya, Justin Berli & Paul Bourcier'
__date__ = '2024-06-18'
__copyright__ = '(C) 2024 by Guillaume Touya, Justin Berli & Paul Bourcier'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.core import (
    QgsProject,
    QgsVectorLayer,
    QgsField,
    QgsFeature,
    QgsGeometry,
    QgsFields,
)
from qgis.utils import iface
from qgis.PyQt.QtCore import QVariant

from qgis.PyQt.QtWidgets import QMessageBox

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessing, QgsFeatureSink, QgsProcessingAlgorithm,
    QgsFeature, QgsGeometry, QgsProcessingParameterDefinition,
    QgsProcessingException, QgsWkbTypes,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterBoolean,
    QgsProcessingParameterNumber,
    QgsProcessingParameterDistance,
    QgsProcessingParameterString,
    QgsProcessingParameterField
)

from cartagen.enrichment import detect_dual_carriageways
from cartagen import collapse_dual_carriageways

from cartagen4qgis import PLUGIN_ICON
from cartagen4qgis.src.tools import list_to_qgis_feature, list_to_qgis_feature_2

import geopandas as gpd

class DetectDualCarriageways(QgsProcessingAlgorithm):
    """
    Detect dual carriageways based on geometric properties.

    This algorithm proposed by Touya :footcite:p:`touya:2010`
    detects the network faces as road separator (*i.e.* separation between
    dual carriageways) when the polygon meets the geometric requirements.
    Those values can be tweaked to fine-tune the detection, but complex interchange will
    nonetheless cause wrong characterization.

    Parameters
    ----------
    roads : GeoDataFrame of LineString
        Road network to analyze.
    importance : str, optional
        The attribute name of the data on which road importance is based.
        Default value is set to None which means every road is taken for the network face calculation.
    value : int, optional
        Maximum value of the importance attribute.
        Roads with an importance higher than this value will not be taken.
    concavity : float, optional
        Maximum concavity.
    elongation : float, optional
        Minimum elongation.
    compactness : float, optional
        Maximum compactness.
    area : float, optional
        Area factor to detect very long motorways.
    width : float, optional
        Maximum width of the the :func:`minimum_rotated_rectangle <shapely.minimum_rotated_rectangle>`.
    huber : int, optional
        Huber width for long motorways.

    Notes
    -----
    - **concavity** is the area of the polygon divided by the area of its convex hull.
    - **elongation** is the length of the :func:`minimum_rotated_rectangle <shapely.minimum_rotated_rectangle>`
      divided by its width.
    - **compactness** is calculated using :math:`(4·pi·area)/(perimeter^2)`
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    INPUT = 'INPUT'
    OUTPUT = 'OUTPUT'
    IMPORTANCE = 'IMPORTANCE'
    VALUE = 'VALUE'
    CONCAVITY = 'CONCAVITY'
    ELONGATION = 'ELONGATION' 
    COMPACTNESS = 'COMPACTNESS'
    AREA = 'AREA'
    WIDTH = 'WIDTH'
    HUBER = 'HUBER' 

    def tr(self, string):
        """
        Returns a translatable string with the self.tr() function.
        """
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return DetectDualCarriageways()

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Detect dual carriageways'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Network'

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr("Detect dual carriageways based on geometric properties.\nThis algorithm proposed by Touya detects the network faces as road separator (*i.e.* separation between dual carriageways) when the polygon meets the geometric requirements. Those values can be tweaked to fine-tune the detection, but complex interchange will nonetheless cause wrong characterization.\nImportance : the attribute name of the data on which road importance is based. Default value is set to None which means every road is taken for the network face calculation.\nValue : maximum value of the importance attribute. Roads with an importance higher than this value will not be taken.\nConcavity : maximum concavity. (concavity is the area of the polygon divided by the area of its convex hull)\nElongation : minimum elongation. (elongation is the length of the minimum rotated rectangle divided by its width)\nCompactness : maximum compactness. (compactness is calculated using (4*pi*area)/(perimeter^2))\nArea : area factor to detect very long motorways.\nWidth : maximum width of the the minimum rotated rectangle.\nHuber : Huber width for long motorways")
        
    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return PLUGIN_ICON

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. It can have any kind of
        # geometry.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr('Input road network'),
                [QgsProcessing.TypeVectorLine]
            )
        )
  		        
        importance = QgsProcessingParameterString(
            self.IMPORTANCE,
            self.tr('Importance attribute name'),
            optional=True,
            defaultValue = 'None'
        )    
        importance.setFlags(importance.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(importance)	     
	

        value = QgsProcessingParameterNumber(
            self.VALUE,
            self.tr('Maximum value of the importance attribute'),
            type=QgsProcessingParameterNumber.Integer,
            optional=True,
            defaultValue = 99
        )
        value.setFlags(value.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(value)

        concavity = QgsProcessingParameterNumber(
            self.CONCAVITY,
            self.tr('Maximum concavity'),
            type=QgsProcessingParameterNumber.Double,
            optional=True,
            defaultValue = 0.85
        )
        concavity.setFlags(concavity.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(concavity)

        elongation = QgsProcessingParameterNumber(
            self.ELONGATION,
            self.tr('Maximum elongation'),
            type=QgsProcessingParameterNumber.Double,
            optional=True,
            defaultValue = 6
        )
        elongation.setFlags(elongation.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(elongation)

        compactness = QgsProcessingParameterNumber(
            self.COMPACTNESS,
            self.tr('Maximum compactness'),
            type=QgsProcessingParameterNumber.Double,
            optional=True,
            defaultValue = 0.12
        )
        compactness.setFlags(compactness.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(compactness)

        area = QgsProcessingParameterNumber(
            self.AREA,
            self.tr('Area factor to detect very long motorways.'),
            type=QgsProcessingParameterNumber.Double,
            optional=True,
            defaultValue = 60000
        )
        area.setFlags(area.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(area)

        width = QgsProcessingParameterNumber(
            self.WIDTH,
            self.tr('Maximum width of the minimum rotated rectangle'),
            type=QgsProcessingParameterNumber.Double,
            optional=True,
            defaultValue = 20
        )
        width.setFlags(width.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(width)

        huber = QgsProcessingParameterNumber(
            self.HUBER,
            self.tr('Huber width for long motorways'),
            type=QgsProcessingParameterNumber.Integer,
            optional=True,
            defaultValue = 16
        )
        huber.setFlags(huber.flags() | QgsProcessingParameterDefinition.FlagAdvanced)
        self.addParameter(huber)

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Detected dual carriageways')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        # Get the QGIS source from the parameters
        source = self.parameterAsSource(parameters, self.INPUT, context)
      
        # Convert the source to GeoDataFrame, get the list of records and the number of entities
        gdf = gpd.GeoDataFrame.from_features(source.getFeatures())
        
        # Retrieve parameters
        importance = self.parameterAsString(parameters, self.IMPORTANCE, context)
        value = self.parameterAsInt(parameters, self.VALUE, context)
        concavity = self.parameterAsDouble(parameters, self.CONCAVITY, context)
        elongation = self.parameterAsDouble(parameters, self.ELONGATION, context)
        compactness = self.parameterAsDouble(parameters, self.COMPACTNESS, context)
        area = self.parameterAsDouble(parameters, self.AREA, context)
        width = self.parameterAsDouble(parameters, self.WIDTH, context)
        huber = self.parameterAsInt(parameters, self.HUBER, context)

        #Set the value of some parameters according to the inputs
        if importance == 'None':
            importance = None
        if value == 99:
            value = None

        # Actual algorithm
        dc = detect_dual_carriageways(
            gdf, importance = importance, value = value, concavity = concavity,
            elongation = elongation, compactness = compactness,
            area = area, width = width, huber = huber
        )
        
        # Convert the result to a list of dicts
        dc = dc.to_dict('records')
          
        # Manually create an empty QgsFeature() if there are no dual carriageways detected
        if len(dc) == 0:
            fields = QgsFields()
            fields.append(QgsField("area", QVariant.Double))
            fields.append(QgsField("perimeter", QVariant.Double))
            fields.append(QgsField("concavity", QVariant.Double))
            fields.append(QgsField("elongation", QVariant.Double))
            fields.append(QgsField("compactness", QVariant.Double))
            fields.append(QgsField("length", QVariant.Double))
            fields.append(QgsField("width", QVariant.Double))
            fields.append(QgsField("huber", QVariant.Double))
            fields.append(QgsField("cid",  QVariant.Int))
            
            res = [QgsFeature(fields)]

            QMessageBox.warning(None, "Warning", "No dual carriageways detected, output layer is empty.")

        else:    
            #convert the result to a gdf, then a list of dicts and finally a list of QgsFeature()
            gdf_final = gpd.GeoDataFrame(dc, crs = source.sourceCrs().authid())
            res = gdf_final.to_dict('records')
            res = list_to_qgis_feature(res)

        #Create the output sink
        (sink, dest_id) = self.parameterAsSink(
                parameters, self.OUTPUT, context,
                fields=res[0].fields(),
                geometryType=QgsWkbTypes.Polygon,
                crs=source.sourceCrs()
            )

        #Add features to the sink    
        sink.addFeatures(res, QgsFeatureSink.FastInsert)

        return {
            self.OUTPUT: dest_id
        }
        

class CollapseDualCarriageways(QgsProcessingAlgorithm):
    """
     Collapse dual carriageways using a TIN skeleton.

    This algorithm proposed by Thom :footcite:p:`thom:2005`
    collapses the network faces considered as dual carriageways
    using a skeleton calculated from a Delaunay triangulation.

    Parameters
    ----------
    roads : GeoDataFrame of LineString
        The road network.
    carriageways : GeoDataFrame of Polygon
        The polygons representing the faces of the network detected as dual carriageways.
    sigma : float, optional
        If not None, apply a gaussian smoothing to the collapsed dual carriageways to
        avoid jagged lines that can be created during the TIN skeleton creation.
    propagate_attributes : list of str, optional
        Propagate the provided list of column name to the resulting network.
        The propagated attribute is the one from the longest line.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    INPUT_ROAD = 'INPUT_ROAD'
    INPUT_CARRIAGEWAYS = 'INPUT_CARRIAGEWAYS'
    SIGMA = 'SIGMA'
    PROPAGATE_ATTRIBUTES = 'PROPAGATE_ATTRIBUTES'
    OUTPUT = 'OUTPUT'

    def tr(self, string):
        """
        Returns a translatable string with the self.tr() function.
        """
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return CollapseDualCarriageways()

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Collapse dual carriageways'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Network'

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr("Collapse dual carriageways using a TIN skeleton.\nThis algorithm proposed by Thom collapses the network faces considered as dual carriageways using a skeleton calculated from a Delaunay triangulation.\nCarriageways : the polygons representing the faces of the network detected as dual carriageways.\nSigma : if not None, apply a gaussian smoothing to the collapsed dual carriageways to avoid jagged lines that can be created during the TIN skeleton creation.\nPropagate_attributes : propagate the provided list of column name to the resulting network. The propagated attribute is the one from the longest line.")
        
    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return PLUGIN_ICON

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. 
        input_road = QgsProcessingParameterFeatureSource(
                self.INPUT_ROAD,
                self.tr('Input road network'),
                [QgsProcessing.TypeVectorLine]
            )
        self.addParameter(input_road)

        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_CARRIAGEWAYS,
                self.tr('Input dual carriageways'),
                [QgsProcessing.TypeVectorPolygon],
                optional=False
            )
        )          
                
        sigma = QgsProcessingParameterNumber(
            self.SIGMA,
                self.tr('Sigma value'),
                type=QgsProcessingParameterNumber.Double,
                optional=True,
                defaultValue= 0
            )
        self.addParameter(sigma)	 

        self.addParameter(QgsProcessingParameterField(self.PROPAGATE_ATTRIBUTES,
            self.tr('Attributes to propagate'),
            None, 'INPUT_ROAD', QgsProcessingParameterField.Any, True, optional = True))

    

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Collapsed branching crossroads')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
      
        # Get the QGIS source from the parameters
        source = self.parameterAsSource(parameters, self.INPUT_ROAD, context)

        # Create the output sink
        (sink, dest_id) = self.parameterAsSink(
                parameters, self.OUTPUT, context,
                fields=source.fields(),
                geometryType=QgsWkbTypes.LineString,
                crs=source.sourceCrs()
            )
        
        # Convert the source to GeoDataFrame, get the list of records and the number of entities
        gdf = gpd.GeoDataFrame.from_features(source.getFeatures())
        
        #retrieve the carriageways layer and transform it to a gdf
        dc = self.parameterAsSource(parameters, self.INPUT_CARRIAGEWAYS, context)
        dc = gpd.GeoDataFrame.from_features(dc.getFeatures())

        #retrieve the sigma parameters and prevent it from being 0
        sigma = self.parameterAsDouble(parameters, self.SIGMA, context)
        if sigma == 0:
            sigma = None

        # retrieve parameter
        attr =  self.parameterAsFields(parameters, self.PROPAGATE_ATTRIBUTES, context)
        
        #perform the CartAGen algorithm
        cllpsed = collapse_dual_carriageways(gdf, dc, sigma=sigma, propagate_attributes=attr)

        # try to convert the result of the algorithm to a list of dicts
        # if not possible, convert the initial gdf instead
        try:
            cllpsed = cllpsed.to_dict('records')
        except AttributeError:
            cllpsed = gdf.to_dict('records')
           
        #Convert the list to a list of QgsFeature()
        res = list_to_qgis_feature_2(cllpsed,source.fields())

        #Add features to the sink    
        sink.addFeatures(res, QgsFeatureSink.FastInsert)       

        return {
            self.OUTPUT: dest_id
        }