# -*- coding: utf-8 -*-

"""
/***************************************************************************
 CartAGen4QGIS
                                 A QGIS plugin
 Cartographic generalization
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2024-06-18
        copyright            : (C) 2024 by Guillaume Touya, Justin Berli & Paul Bourcier
        email                : guillaume.touya@ign.fr
 ***************************************************************************/
"""

__author__ = 'Guillaume Touya, Justin Berli & Paul Bourcier'
__date__ = '2024-06-18'
__copyright__ = '(C) 2024 by Guillaume Touya, Justin Berli & Paul Bourcier'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.core import (
    QgsProject,
    QgsVectorLayer,
    QgsField,
    QgsFeature,
    QgsGeometry,
    QgsFields,
)
from qgis.utils import iface
from qgis.PyQt.QtCore import QVariant


from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessing, QgsFeatureSink, QgsProcessingAlgorithm,
    QgsFeature, QgsGeometry, QgsProcessingParameterDefinition,
    QgsProcessingException, QgsWkbTypes,
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterBoolean,
    QgsProcessingParameterNumber,
    QgsProcessingParameterDistance,
)

import geopandas as gpd

from cartagen.enrichment import detect_dead_ends
from cartagen import eliminate_dead_ends

from cartagen4qgis import PLUGIN_ICON
from cartagen4qgis.src.tools import list_to_qgis_feature, list_to_qgis_feature_2


class DetectDeadEnds(QgsProcessingAlgorithm):
    """
    This function detects dead ends inside a road network. Returns the same road network layer with new attributes, including 'deadend' which indicates if the entity is a dead end or not (boolean).

    Parameters:
       - roads (geopandas GeoDataFrame) : The GeoDataFrame containing the road network as LineString geometries.
       - outside_faces (boolean, optional) : If set to true, detects dead-ends on the network faces located on the border.

    Six attributes are added:
       - face: the id of the network face it belongs to.
       - deid: the id of the dead end group inside a given face.
       - connected: set to ‘true’ if the dead end group is connected to the network.
       - root: set to true if the road section is the root of the dead end group, i.e. the section connecting the dead end group to the road network.
       - hole: set to true if the road section touches a hole inside the dead end group.
       - deadend: set to true if yhe road section is considered as a dead end.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    INPUT = 'INPUT'
    OUTPUT = 'OUTPUT'
    OUTSIDE_FACES = 'OUTSIDE_FACES'

    def tr(self, string):
        """
        Returns a translatable string with the self.tr() function.
        """
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return DetectDeadEnds()

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Detect dead ends'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Network'

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr("This function detects dead ends inside a road network. Returns the same road network layer with new attributes, including 'deadend' which indicates if the entity is a dead end or not (boolean).\nOutside faces : whether dead-ends should be calculated on the outside faces of the road network. This can induce wrong characterization on the border of the provided dataset.")
        
    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return PLUGIN_ICON

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. It can have any kind of
        # geometry.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr('Input road network'),
                [QgsProcessing.TypeVectorLine]
            )
        )
  		        
        outside_faces = QgsProcessingParameterBoolean(
            self.OUTSIDE_FACES,
            self.tr('Outside faces'),
            defaultValue=True,
            optional=True
        )    
        self.addParameter(outside_faces)	     
	
        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Detected dead ends')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        # Get the QGIS source from the parameters
        source = self.parameterAsSource(parameters, self.INPUT, context)
        
        # Convert the source to GeoDataFrame, get the list of records and the number of entities
        gdf = gpd.GeoDataFrame.from_features(source.getFeatures())
        
        # Retrieve parameters
        o_faces = self.parameterAsBoolean(parameters, self.OUTSIDE_FACES, context)
        
        # Actual algorithm
        dends = detect_dead_ends(gdf, o_faces)
    
        # Convert the result to a list of dicts
        dends = dends.to_dict('records')
      
        # Define the fields (get the source fields and a add some fields)
        fields = source.fields()
        fields.append(QgsField("deadend", QVariant.Bool))

        if all('face' in dict for dict in dends):
            fields.append(QgsField("face",  QVariant.Int))
            fields.append(QgsField("deid",  QVariant.Int))
            fields.append(QgsField("rid",  QVariant.Int))
            fields.append(QgsField("connected", QVariant.Bool))
            fields.append(QgsField("root", QVariant.Bool))
            fields.append(QgsField("hole", QVariant.Bool))

        #transform the result to a list of QgsFeature()
        res = list_to_qgis_feature_2(dends,fields)
      
        # Define the output sink
        (sink, dest_id) = self.parameterAsSink(
            parameters, self.OUTPUT, context,
            fields=res[0].fields(),
            geometryType=QgsWkbTypes.LineString,
            crs=source.sourceCrs()
        )
        
        #Add features to the sink
        sink.addFeatures(res, QgsFeatureSink.FastInsert)

        return {
            self.OUTPUT: dest_id
        }
        


class EliminateDeadEnds(QgsProcessingAlgorithm):
    """
 Eliminate dead-ends groups.

    Eliminates roads section detected as dead-ends inside a road network
    if the length of their main component is below a given threshold.
        
    Parameters
    ----------
    roads : GeoDataFrame of LineString
        The road network with dead-ends properly characterized.
    length : float, optional
        Length below which dead-ends are eliminated.
        If left to 0, shorter dead-ends are not eliminated.
    keep_longest : bool, optional
        If set to True, in case of complex dead-end, keep only the main component (*c.f.* description).
        If False, eliminate the whole dead-end group.
        This concerns also dead-ends with a length above the provided threshold.

    Returns
    -------
    GeoDataFrame of LineString

    Warning
    -------
    This function requires the detection of dead-ends beforehand.

    See Also
    --------
    detect_dead_ends :
        Characterize dead-ends groups inside a road network.

    Notes
    -----
    The **main component** of a dead-end is defined as:
    
    - If the dead-end is simple (*i.e.* just one road), the main component is the road.
    - If the dead-end contains multiple ramification of roads,
      the main component represents the path between the entry and the longest ramification.
    - If the dead-end contains inner network faces (*i.e.* enclosed roads),
      the main component represents the longest of the shortest paths
      between the entry and all the vertex of the dead ends.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    INPUT = 'INPUT'
    OUTPUT = 'OUTPUT'
    KEEP_LONGEST = 'KEEP_LONGEST'
    LENGTH = 'LENGTH'

    def tr(self, string):
        """
        Returns a translatable string with the self.tr() function.
        """
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return EliminateDeadEnds()

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Eliminate dead ends'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Network'

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr("Eliminates roads section detected as dead-ends inside a road network if the length of their main component is below a given threshold.\nLength : length below which dead-ends are eliminated. If left to 0, shorter dead-ends are not eliminated.\nKeep_longest : if set to True, in case of complex dead-end, keep only the main component. If False, eliminate the whole dead-end group. This concerns also dead-ends with a length above the provided threshold.")
        
    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return PLUGIN_ICON

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source. It can have any kind of
        # geometry.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr('Input road network'),
                [QgsProcessing.TypeVectorLine]
            )
        )
  		        
        keep_longest = QgsProcessingParameterBoolean(
            self.KEEP_LONGEST,
            self.tr('keep longest'),
            defaultValue=True,
            optional=True
        )    
        self.addParameter(keep_longest)	     
	
        length = QgsProcessingParameterNumber(
            self.LENGTH,
                self.tr('Length'),
                type=QgsProcessingParameterNumber.Double,
                optional=False
            )
        self.addParameter(length)

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Eliminated dead ends')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """
        # Get the QGIS source from the parameters
        source = self.parameterAsSource(parameters, self.INPUT, context)
 	
        # Convert the source to GeoDataFrame, get the list of records and the number of entities
        gdf = gpd.GeoDataFrame.from_features(source.getFeatures())
        
        # Retrieve parameters
        k_longest = self.parameterAsBoolean(parameters, self.KEEP_LONGEST, context)
        length = self.parameterAsDouble(parameters, self.LENGTH, context)

        # Actual algorithm
        elim = eliminate_dead_ends(gdf, length, k_longest)

        # Convert the result into a list of dictionnaries, and this list into a list of QgsFeature()
        elim = elim.to_dict('records')
        res = list_to_qgis_feature(elim)
       
        # Define the output sink
        (sink, dest_id) = self.parameterAsSink(
            parameters, self.OUTPUT, context,
            fields=res[0].fields(),
            geometryType=QgsWkbTypes.LineString,
            crs=source.sourceCrs()
        )
        
        # add features to the sink
        sink.addFeatures(res, QgsFeatureSink.FastInsert)

        return {
            self.OUTPUT: dest_id
        }
        
              
        
        
        
        
        
