# -*- coding: utf-8 -*-

"""
/***************************************************************************
 CartAGen4QGIS
                                 A QGIS plugin
 Cartographic generalization
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-05-11
        copyright            : (C) 2023 by Guillaume Touya, Justin Berli & Paul Bourcier
        email                : guillaume.touya@ign.fr
 ***************************************************************************/
"""

__author__ = 'Guillaume Touya, Justin Berli & Paul Bourcier'
__date__ = '2023-05-11'
__copyright__ = '(C) 2023 by Guillaume Touya, Justin Berli & Paul Bourcier'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessing,
    QgsFeatureSink, 
    QgsProcessingAlgorithm,
    QgsFeature,
    QgsGeometry,
    QgsProcessingParameterDefinition,
    QgsProcessingParameterFeatureSource, 
    QgsProcessingParameterFeatureSink, 
    QgsProcessingParameterNumber,
    QgsProcessingParameterDistance,
    QgsProcessingParameterBoolean,
    QgsProcessingException,
    QgsProcessingParameterMultipleLayers
)

import geopandas as gpd
from cartagen4qgis import PLUGIN_ICON
from cartagen import visvalingam_whyatt, raposo, douglas_peucker
from shapely.wkt import loads

from cartagen4qgis.src.tools import qgis_source_to_geodataframe, list_to_qgis_feature_2

class VisvalingamWhyattQGIS(QgsProcessingAlgorithm):
    """
    Simplify lines using the Visvalingam-Whyatt algorithm
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    INPUT = 'INPUT'
    TOLERANCE = 'TOLERANCE'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr('Input line'),
                [QgsProcessing.TypeVectorLine]
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.TOLERANCE,
                self.tr('Tolerance area'),
                type=QgsProcessingParameterNumber.Integer,
                defaultValue=100,
                optional=False
            )
        )

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Simplified Visvalingam-Whyatt')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.
        source = self.parameterAsSource(parameters, self.INPUT, context)
        (sink, dest_id) = self.parameterAsSink(parameters, self.OUTPUT,
                context, source.fields(), source.wkbType(), source.sourceCrs())

        # Compute the number of steps to display within the progress bar and
        # get features from source
        total = 100.0 / source.featureCount() if source.featureCount() else 0
        features = source.getFeatures()

        for current, feature in enumerate(features):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break

            wkt = feature.geometry().asWkt()
            shapely_geom = loads(wkt)

            # CartAGen's algorithm
            simplified = visvalingam_whyatt(shapely_geom, self.parameterAsInt(parameters,self.TOLERANCE,context))

            result = QgsFeature()
            result.setGeometry(QgsGeometry.fromWkt(simplified.wkt))
            result.setAttributes(feature.attributes())

            # Add a feature in the sink
            sink.addFeature(result, QgsFeatureSink.FastInsert)

            # Update the progress bar
            feedback.setProgress(int(current * total))

        # Return the results of the algorithm. In this case our only result is
        # the feature sink which contains the processed features, but some
        # algorithms may return multiple feature sinks, calculated numeric
        # statistics, etc. These should all be included in the returned
        # dictionary, with keys matching the feature corresponding parameter
        # or output names.
        return {
            self.OUTPUT: dest_id
        }

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Visvalingam-Whyatt simplification'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Lines'

    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return PLUGIN_ICON

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return VisvalingamWhyattQGIS()

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr("Area-based line simplification.\nThe principle of the algorithm is to select the vertices to delete (the less characteristic ones). To select the vertices to delete, there is an iterative process, and at each iteration, the triangles formed by three consecutive vertices are computed. If the area of the smallest triangle is smaller than a threshold, the middle vertex is deleted, and another iteration starts. The algorithm is relevant for the simplification of natural line or polygon features such as rivers, forests, or coastlines.\nArea tolerance : the minimum triangle area to keep a vertex in the line.")
        

class RaposoSimplificationQGIS(QgsProcessingAlgorithm):
    """
    Simplify lines using the Raposo hexagon-based line simplification
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    INPUT = 'INPUT'
    INITIAL_SCALE = 'INITIAL_SCALE'
    FINAL_SCALE = 'FINAL_SCALE'
    CENTROID = 'CENTROID'
    TOBLER = 'TOBLER'

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr('Input line'),
                [QgsProcessing.TypeVectorLine]
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.INITIAL_SCALE,
                self.tr('Initial scale'),
                type=QgsProcessingParameterNumber.Integer,
                optional=False
            )
        )

        self.addParameter(
            QgsProcessingParameterNumber(
                self.FINAL_SCALE,
                self.tr('Final scale'),
                type=QgsProcessingParameterNumber.Integer,
                optional=False
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.CENTROID,
                self.tr('Replace line vertices with the hexagons centroid'),
                defaultValue=True,
                optional=False
            )
        )

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.TOBLER,
                self.tr("Compute cell resolution based on Tobler's formula"),
                defaultValue=False,
                optional=False
            )
        )

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Simplified Raposo')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.
        source = self.parameterAsSource(parameters, self.INPUT, context)
        (sink, dest_id) = self.parameterAsSink(parameters, self.OUTPUT,
                context, source.fields(), source.wkbType(), source.sourceCrs())

        # Compute the number of steps to display within the progress bar and
        # get features from source
        total = 100.0 / source.featureCount() if source.featureCount() else 0
        features = source.getFeatures()

        for current, feature in enumerate(features):
            # Stop the algorithm if cancel button has been clicked
            if feedback.isCanceled():
                break

            wkt = feature.geometry().asWkt()
            shapely_geom = loads(wkt)

            simplified = raposo(
                shapely_geom,
                self.parameterAsInt(parameters,self.INITIAL_SCALE,context),
                self.parameterAsInt(parameters,self.FINAL_SCALE,context),
                centroid=self.parameterAsBoolean(parameters, self.CENTROID, context),
                tobler=self.parameterAsBoolean(parameters, self.TOBLER, context)
            )

            result = QgsFeature()
            result.setGeometry(QgsGeometry.fromWkt(simplified.wkt))
            result.setAttributes(feature.attributes())

            # Add a feature in the sink
            sink.addFeature(result, QgsFeatureSink.FastInsert)

            # Update the progress bar
            feedback.setProgress(int(current * total))

        # Return the results of the algorithm. In this case our only result is
        # the feature sink which contains the processed features, but some
        # algorithms may return multiple feature sinks, calculated numeric
        # statistics, etc. These should all be included in the returned
        # dictionary, with keys matching the feature corresponding parameter
        # or output names.
        return {
            self.OUTPUT: dest_id
        }

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Raposo hexagonal simplification'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Lines'

    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return PLUGIN_ICON

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr("Hexagon-based line simplification.\nThis algorithm proposed by Raposo simplifies lines based on a hexagonal tessellation. Only one vertex is kept inside each cell. This point can be the centroid of the removed vertices, or a projection on the initial line of this centroid. The algorithm also works for the simplification of the border of a polygon object. \nInitial_scale : initial scale of the provided line (25000.0 for 1:25000 scale)\nFinal_scale : final scale of the simplified line\nCentroid : if true, uses the center of the hexagonal cells as the new vertex, if false, the center is projected on the nearest point in the initial line.\nTobler : If True, compute cell resolution based on Tobler’s formula, else uses Raposo’s formula")   

    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return RaposoSimplificationQGIS()

class DouglasPeucker(QgsProcessingAlgorithm):
    """
    Distance-based line simplification.

    This algorithm was proposed by Ramer and by Douglas and Peucker. 
    It is a line filtering algorithm, which means that it filters the vertices of the line (or polygon)
    to only retain the most important ones to preserve the shape of the line. 
    The algorithm iteratively searches the most characteristics vertices of portions of the line and decides
    to retain or remove them given a distance threshold.

    The algorithm tends to unsmooth geographic lines, and is rarely used to simplify geographic features. 
    But it can be very useful to quickly filter the vertices of a line inside another algorithm.

    This is a simple wrapper around shapely.simplify().

    Parameters:

    line (LineString) – The line to simplify.

    threshold (float) – The distance threshold to remove the vertex from the line.

    preserve_topology (bool, optional) – If set to True, the algorithm will prevent invalid geometries from being created 
    (checking for collapses, ring-intersections, etc). The trade-off is computational expensivity.
    """

    # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    
    INPUT = 'INPUT'

    THRESHOLD = 'THRESHOLD'
    PRESERVE_TOPOLOGY = 'PRESERVE_TOPOLOGY'
 
    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Douglas-Peucker simplification'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Lines'

    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return PLUGIN_ICON

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr("Distance-based line simplification.\nThis algorithm was proposed by Ramer and by Douglas and Peucker. It is a line filtering algorithm, which means that it filters the vertices of the line (or polygon) to only retain the most important ones to preserve the shape of the line. The algorithm iteratively searches the most characteristics vertices of portions of the line and decides to retain or remove them given a distance threshold.\nThe algorithm tends to unsmooth geographic lines, and is rarely used to simplify geographic features. But it can be very useful to quickly filter the vertices of a line inside another algorithm.\nThis is a simple wrapper around shapely.simplify().\nThreshold : the distance thresholdto remove the vertex from the line.\nPreserve topology : if set to True, the algorithm will prevent invalid geometries from being created (checking for collapses, ring-intersections, etc). The trade-off is computational expensivity.")
        
    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return DouglasPeucker()

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT,
                self.tr('Input lines or polygons'),
                [QgsProcessing.TypeVectorPolygon, QgsProcessing.TypeVectorLine]
            )
        )

        threshold = QgsProcessingParameterNumber(
            self.THRESHOLD,
            self.tr('Distance threshold to remove the vertex from the line'),
            type=QgsProcessingParameterNumber.Double,
            defaultValue=10.0,
            optional=False
        )
        self.addParameter(threshold)
       
        preserve_topology = QgsProcessingParameterBoolean(
            self.PRESERVE_TOPOLOGY,
                self.tr('Preserve topology'),
                optional=False,
                defaultValue=True
            )
        self.addParameter(preserve_topology)

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Simplified Douglas-Peucker')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.
        source = self.parameterAsSource(parameters, self.INPUT, context)

        gdf = gpd.GeoDataFrame.from_features(source.getFeatures())
        
        # Retrieve the other parameter values 
        threshold = self.parameterAsDouble(parameters, self.THRESHOLD, context)
        preserve_topology = self.parameterAsBoolean(parameters, self.PRESERVE_TOPOLOGY, context)

        # Perform the CartAGen algorithm and convert the result to a list of QgsFeature()
        dp = gdf.copy()
        for i in range(len(gdf)):
            dp.loc[i,'geometry'] = douglas_peucker(list(gdf.geometry)[i],threshold=threshold, preserve_topology=preserve_topology)
            res = dp.to_dict('records')
            res = list_to_qgis_feature_2(res, source.fields())


        # Create the output sink    
        (sink, dest_id) = self.parameterAsSink(
            parameters, self.OUTPUT, context,
            res[0].fields(), source.wkbType(), source.sourceCrs()
        )
        
        # Add a feature in the sink
        sink.addFeatures(res, QgsFeatureSink.FastInsert)

        return {
            self.OUTPUT: dest_id
        }
