# -*- coding: utf-8 -*-

"""
/***************************************************************************
 CartAGen4QGIS
                                 A QGIS plugin
 Cartographic generalization
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2023-05-11
        copyright            : (C) 2023 by Guillaume Touya, Justin Berli & Paul Bourcier
        email                : guillaume.touya@ign.fr
 ***************************************************************************/
"""

__author__ = 'Guillaume Touya, Justin Berli & Paul Bourcier'
__date__ = '2023-05-11'
__copyright__ = '(C) 2023 by Guillaume Touya, Justin Berli & Paul Bourcier'

# This will get replaced with a git SHA1 when you do a git archive

__revision__ = '$Format:%H$'

from qgis.PyQt.QtCore import QCoreApplication
from qgis.core import (
    QgsProcessing, QgsFeatureSink, QgsProcessingAlgorithm,
    QgsFeature, QgsGeometry, QgsProcessingParameterDefinition
)
from qgis.core import (
    QgsProcessingParameterFeatureSource,
    QgsProcessingParameterFeatureSink,
    QgsProcessingParameterBoolean,
    QgsProcessingParameterNumber,
    QgsProcessingParameterDistance,
    QgsProcessingParameterMultipleLayers
)

import geopandas as gpd
import pandas
from cartagen4qgis import PLUGIN_ICON
from cartagen import morphological_amalgamation, partition_networks
from cartagen4qgis.src.tools import list_to_qgis_feature

from shapely import Polygon
from shapely.wkt import loads

class MorphologicalAmalgamation(QgsProcessingAlgorithm):
    """
    Amalgamate buildings using dilation and erosion.
    
    The amalgamation algorithm proposed by Damen et al. is based on morphological dilations and erosions with a square cap.
    It is particularly useful to keep the overall shape of building blocks.

    Parameters
    ----------
    buildings : list of Polygon
        Buildings to amalgamate.
    buffer : float
        Size of the buffer used for dilation (in meters).
        Buildings closer than 2 times the buffer size are amalgamated.
    edge_length : float
        Minimum length of edges in the amalgamated geometries
        (a simplification process is carried out).
    """

     # Constants used to refer to parameters and outputs. They will be
    # used when calling the algorithm from another algorithm, or when
    # calling from the QGIS console.

    OUTPUT = 'OUTPUT'
    
    INPUT_BUILDINGS = 'INPUT_BUILDINGS'

    NETWORK_PARTITIONING_TF = 'NETWORK_PARTITIONING_TF'
    INPUT_NETWORK_PART = 'INPUT_NETWORK_PART'

    BUFFER = 'BUFFER'
    EDGE_LENGTH = 'EDGE_LENGTH' 

    def name(self):
        """
        Returns the algorithm name, used for identifying the algorithm. This
        string should be fixed for the algorithm, and must not be localised.
        The name should be unique within each provider. Names should contain
        lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Morphological amalgamation'

    def displayName(self):
        """
        Returns the translated algorithm name, which should be used for any
        user-visible display of the algorithm name.
        """
        return self.tr(self.name())

    def group(self):
        """
        Returns the name of the group this algorithm belongs to. This string
        should be localised.
        """
        return self.tr(self.groupId())

    def groupId(self):
        """
        Returns the unique ID of the group this algorithm belongs to. This
        string should be fixed for the algorithm, and must not be localised.
        The group id should be unique within each provider. Group id should
        contain lowercase alphanumeric characters only and no spaces or other
        formatting characters.
        """
        return 'Buildings'

    def icon(self):
        """
        Should return a QIcon which is used for your provider inside
        the Processing toolbox.
        """
        return PLUGIN_ICON

    def shortHelpString(self):
        """
        Returns a localised short helper string for the algorithm. This string
        should provide a basic description about what the algorithm does and the
        parameters and outputs associated with it..
        """
        return self.tr("Amalgamate buildings using dilation and erosion.\nThe amalgamation algorithm proposed by Damen et al. is based on morphological dilations and erosions with a square cap. It is particularly useful to keep the overall shape of building blocks.\nBuffer : size of the buffer used for dilation (in meters). Buildings closer than 2 times the buffer size are amalgamated.\nEdge length : minimum length of edges in the amalgamated geometries (a simplification process is carried out).")
        
    def tr(self, string):
        return QCoreApplication.translate('Processing', string)

    def createInstance(self):
        return MorphologicalAmalgamation()

    def initAlgorithm(self, config):
        """
        Here we define the inputs and output of the algorithm, along
        with some other properties.
        """

        # We add the input vector features source.
        self.addParameter(
            QgsProcessingParameterFeatureSource(
                self.INPUT_BUILDINGS,
                self.tr('Input buildings'),
                [QgsProcessing.TypeVectorPolygon]
            )
        )

        buffer = QgsProcessingParameterNumber(
            self.BUFFER,
            self.tr('Size of the buffer used for dilation'),
            type=QgsProcessingParameterNumber.Double,
            defaultValue=10.0,
            optional=False
        )
        self.addParameter(buffer)
       
        edge_length = QgsProcessingParameterNumber(
            self.EDGE_LENGTH,
            self.tr('Minimum length of edges in the amalgamated geometries'),
            type=QgsProcessingParameterNumber.Double,
            defaultValue=0,
            optional=False
        )
        self.addParameter(edge_length)

        self.addParameter(
            QgsProcessingParameterBoolean(
                self.NETWORK_PARTITIONING_TF,
                self.tr('Activate network partitioning'),
                defaultValue=False,
                optional=True
            )
        )
        
        self.addParameter(
            QgsProcessingParameterMultipleLayers(
                self.INPUT_NETWORK_PART,
                self.tr('Input lines for the network partition'),
                layerType=QgsProcessing.TypeVectorLine,
                optional=True
            )
        )

        # We add a feature sink in which to store our processed features (this
        # usually takes the form of a newly created vector layer when the
        # algorithm is run in QGIS).
        self.addParameter(
            QgsProcessingParameterFeatureSink(
                self.OUTPUT,
                self.tr('Amalgamated')
            )
        )

    def processAlgorithm(self, parameters, context, feedback):
        """
        Here is where the processing itself takes place.
        """

        # Retrieve the feature source and sink. The 'dest_id' variable is used
        # to uniquely identify the feature sink, and must be included in the
        # dictionary returned by the processAlgorithm function.
        source = self.parameterAsSource(parameters, self.INPUT_BUILDINGS, context)
        gdf = gpd.GeoDataFrame.from_features(source.getFeatures())

        # Compute the number of steps to display within the progress bar and
        # get features from source
        # total = 100.0 / source.featureCount() if source.featureCount() else 0
       
        # Retrieve the network and transform it into a list of GeoDataFrame
        networks = self.parameterAsLayerList(parameters, self.INPUT_NETWORK_PART, context)
        gdf_list_networks = []
        for i in networks:
            gdf_list_networks.append(gpd.GeoDataFrame.from_features(i.getFeatures()))
        
        # Retrieve the other parameter values 
        activate_network_part = self.parameterAsBoolean(parameters, self.NETWORK_PARTITIONING_TF, context)
        network_part = self.parameterAsLayerList(parameters, self.INPUT_NETWORK_PART, context)
        buffer = self.parameterAsDouble(parameters, self.BUFFER, context)
        edge_length = self.parameterAsDouble(parameters, self.EDGE_LENGTH, context)

        # Use the CartAGen algorithm with or without network partitionning
        # Convert the result to a list of QgsFeature()
        if len(network_part) == 0 or activate_network_part == False:
            amal = morphological_amalgamation(list(gdf.geometry),buffer = buffer, edge_length = edge_length)
            amal_gdf = gpd.GeoDataFrame(geometry=gpd.GeoSeries(amal))
            res = amal_gdf.to_dict('records')     
        else:
            part = partition_networks(gdf,gdf_list_networks[0])
            list_gdf = []
            for i in range(len(part[0])):
                amal = gdf.copy()
                gdf = gdf.iloc[part[0][i]]
                try:
                    generalized = morphological_amalgamation(list(gdf.geometry), buffer = buffer, edge_length = edge_length)
                    
                except: 
                    generalized = gdf.geometry
            
                list_gdf.append(gpd.GeoDataFrame(geometry=gpd.GeoSeries(generalized)))
            combined_gdf = pandas.concat(list_gdf, ignore_index=True)
            combined_gdf = gpd.GeoDataFrame(combined_gdf, geometry='geometry')  
            res = combined_gdf.to_dict('records')

        if len(res) > 0:
            res = list_to_qgis_feature(res)
        else:
            res = []

        
        # Create the output sink
        (sink, dest_id) = self.parameterAsSink(parameters, self.OUTPUT,
                context, source.fields(), source.wkbType(), source.sourceCrs())
        
        # Add a feature in the sink
        sink.addFeatures(res, QgsFeatureSink.FastInsert)

        return {
            self.OUTPUT: dest_id
        }