# -*- coding: utf-8 -*-
"""
/***************************************************************************
 CanvasDocker
                                 A QGIS plugin
 This plugin allows the QGIS map canvas to be undocked and maximized on a secondary screen, freeing up space in the main window for other dockable panels.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-09-16
        git sha              : $Format:%H$
        copyright            : (C) 2025 by Emiel Verstegen
        email                : emiel@verstegen.email
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction
# import MessageLevel
from qgis.core import Qgis

# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .canvas_docker_dialog import CanvasDockerDialog
import os.path


class CanvasDocker:
    def log(self, message):
        from qgis.core import QgsMessageLog
        QgsMessageLog.logMessage(message, 'CanvasDocker', Qgis.MessageLevel.Info)  # 0 = INFO
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'CanvasDocker_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Canvas Docker')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('CanvasDocker', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/canvas_docker/icon.png'
        # Create a toggle (checkable) action
        self.toggle_action = QAction(QIcon(icon_path), self.tr(u'Canvas Docker'), self.iface.mainWindow())
        self.toggle_action.setCheckable(True)
        self.toggle_action.setStatusTip(self.tr(u'Toggle undock/dock QGIS canvas'))
        self.toggle_action.toggled.connect(self.toggle_canvas)
        self.iface.addToolBarIcon(self.toggle_action)
        self.iface.addPluginToMenu(self.menu, self.toggle_action)
        self.actions.append(self.toggle_action)

        self.canvas_floating_widget = None
        self.original_canvas_parent = None
        self.original_canvas_geometry = None


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&Canvas Docker'),
                action)
            self.iface.removeToolBarIcon(action)
        # Restore canvas if plugin is unloaded while undocked
        if self.toggle_action.isChecked():
            self.toggle_action.setChecked(False)
    def toggle_canvas(self, checked):
        """Undock or dock the QGIS map canvas using QDockWidget, restoring to main window on redock."""
        from qgis.PyQt.QtWidgets import QDockWidget
        canvas = self.iface.mapCanvas()
        main_window = self.iface.mainWindow()
        if checked:
            self.log("Toggle ON: Undocking canvas.")
            # Save original parent and layout
            self.original_canvas_parent = canvas.parentWidget()
            self.original_canvas_layout = self.original_canvas_parent.layout() if self.original_canvas_parent else None


            # Create a dock widget for the canvas if not already done
            if not hasattr(self, 'canvas_dock_widget') or self.canvas_dock_widget is None:
                self.log("Creating QDockWidget for canvas.")
                from qgis.PyQt.QtWidgets import QPushButton, QWidget, QHBoxLayout, QApplication
                from qgis.PyQt.QtGui import QIcon
                from qgis.PyQt.QtCore import QSize, Qt
                from qgis.PyQt.QtWidgets import QStyle
                self.canvas_dock_widget = QDockWidget("Map Canvas", main_window)
                self.canvas_dock_widget.setWidget(canvas)
                self.canvas_dock_widget.setFloating(True)
                # Compatibility: PyQt5 vs PyQt6
                try:
                    dock_area = Qt.DockWidgetArea.LeftDockWidgetArea  # PyQt6
                except AttributeError:
                    dock_area = Qt.LeftDockWidgetArea  # PyQt5
                main_window.addDockWidget(dock_area, self.canvas_dock_widget)
                # Custom title bar with maximize and close buttons
                from qgis.PyQt.QtWidgets import QLabel, QStyle
                title_bar = QWidget()
                title_bar.setStyleSheet("background: #e0e0e0; border: 1px solid #888;")
                layout = QHBoxLayout(title_bar)
                layout.setContentsMargins(6, 2, 6, 2)
                layout.setSpacing(4)
                # Add left-aligned label
                label = QLabel("Canvas Docker")
                label.setStyleSheet("background: transparent; border: none; font-weight: bold; color: #222;")
                # Compatibility: PyQt5 vs PyQt6 alignment flag
                try:
                    align_left = Qt.AlignmentFlag.AlignLeft  # PyQt6
                except AttributeError:
                    align_left = Qt.AlignLeft  # PyQt5
                layout.addWidget(label, alignment=align_left)
                layout.addStretch()
                maximize_btn = QPushButton()
                # Robust maximize button icon selection for PyQt5/PyQt6
                maximize_icon = QIcon()
                try:
                    # Try Qt6 standard pixmap
                    maximize_icon = self.canvas_dock_widget.style().standardIcon(getattr(QStyle.StandardPixmap, "SP_TitleBarMaxButton", None))
                except Exception:
                    maximize_icon = QIcon()
                if maximize_icon.isNull():
                    maximize_icon = QIcon.fromTheme("window-maximize")
                if maximize_icon.isNull():
                    maximize_icon = QIcon(":/plugins/canvas_docker/icon.png")  # fallback to your plugin icon or any custom icon
                maximize_btn.setIcon(maximize_icon)
                maximize_btn.setIconSize(QSize(16, 16))
                maximize_btn.setToolTip("Maximize to screen")
                maximize_btn.setStyleSheet("border: none; background: transparent;")
                layout.addWidget(maximize_btn)
                close_btn = QPushButton()
                # Robust close button icon selection for PyQt5/PyQt6
                close_icon = QIcon()
                try:
                    close_icon = QIcon.fromTheme("window-close")
                except Exception:
                    pass
                if close_icon.isNull():
                    try:
                        close_icon = self.canvas_dock_widget.style().standardIcon(getattr(QStyle, "SP_TitleBarCloseButton", None))
                    except Exception:
                        close_icon = QIcon()
                close_btn.setIcon(close_icon)
                close_btn.setIconSize(QSize(16, 16))
                close_btn.setToolTip("Close and redock")
                close_btn.setStyleSheet("border: none; background: transparent;")
                layout.addWidget(close_btn)
                title_bar.setLayout(layout)
                self.canvas_dock_widget.setTitleBarWidget(title_bar)

                # Store previous geometry for restore
                self._dock_prev_geom = None
                def maximize_dock():
                    # Detect which screen the dock widget is on
                    dock_geom = self.canvas_dock_widget.geometry()
                    for screen in QApplication.screens():
                        if screen.geometry().contains(dock_geom.center()):
                            target_geom = screen.availableGeometry()
                            break
                    else:
                        target_geom = QApplication.primaryScreen().availableGeometry()
                    if self._dock_prev_geom is None:
                        self._dock_prev_geom = dock_geom
                        self.canvas_dock_widget.setGeometry(target_geom)
                        maximize_btn.setToolTip("Restore size")
                    else:
                        self.canvas_dock_widget.setGeometry(self._dock_prev_geom)
                        self._dock_prev_geom = None
                        maximize_btn.setToolTip("Maximize to screen")
                maximize_btn.clicked.connect(maximize_dock)
                def close_dock():
                    # Redock the canvas by toggling the button off
                    self.toggle_action.setChecked(False)
                close_btn.clicked.connect(close_dock)


            self.log("Showing dock widget.")
            self.canvas_dock_widget.show()
            self.canvas_dock_widget.raise_()
            # Hide the welcome screen panel (last child widget)
            try:
                central_children = main_window.centralWidget().children()
                if len(central_children) > 1:
                    stacked_children = central_children[1].children()
                    if stacked_children:
                        self.log(f"Hiding welcome screen widget: {stacked_children[-1]}")
                        stacked_children[-1].hide()
            except Exception as e:
                self.log(f"Error hiding welcome screen widget: {e}")
        else:
            self.log("Toggle OFF: Redocking canvas.")
            # Dock back to main window: remove canvas from dock widget and restore to original parent/layout
            if hasattr(self, 'canvas_dock_widget') and self.canvas_dock_widget is not None:
                self.log("Removing canvas from dock widget.")
                self.canvas_dock_widget.setWidget(None)
                if self.original_canvas_parent and self.original_canvas_layout:
                    self.log("Restoring canvas to original parent.")
                    canvas.setParent(self.original_canvas_parent)
                    self.original_canvas_layout.addWidget(canvas)
                    # Ensure the canvas is the current widget in the QStackedWidget
                    if hasattr(self.original_canvas_parent, 'setCurrentWidget'):
                        self.original_canvas_parent.setCurrentWidget(canvas)
                    self.original_canvas_parent.show()
                    canvas.show()
                    canvas.repaint()
                    main_window.update()
                self.log("Closing and deleting dock widget.")
                self.canvas_dock_widget.close()
                self.canvas_dock_widget.deleteLater()
                self.canvas_dock_widget = None



    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start == True:
            self.first_start = False
            self.dlg = CanvasDockerDialog()

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            # Do something useful here - delete the line containing pass and
            # substitute with your code.
            pass
