"""
/***************************************************************************
 CAMSDataManager
                                 A QGIS plugin
 Manage and analyze CAMS air quality data in QGIS
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2025-05-02
        git sha              : $Format:%H$
        copyright            : (C) 2025 by POLIMI
        email                : zhanbin.wu@mail.polimi.it
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os
import sys
import datetime
import itertools
import webbrowser

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QMessageBox, QFileDialog

from qgis.core import QgsProject, QgsRasterLayer, QgsVectorFileWriter, QgsFeatureRequest

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the dialog
from .cams_data_manager_dialog import CAMSDataManagerDialog

# Import tool modules
from .tools.ui_handler import collect_download_parameters
from .tools.downloader import submit_cams_request
from .tools.validator import validate_params
from .tools.config import DEFAULT_DOWNLOAD_DIR, MODEL_BOUNDS
from .tools.unzipper import unzip_and_get_netcdf

# For loading layers to QGIS
from qgis.core import QgsRasterLayer

from .tools.ui_handler import NETCDF_VARIABLE_MAP

import tempfile
import pandas as pd
import geopandas as gpd
from .tools.aoi_utils import clip_netcdf_by_shapefile

from .gui.analysis_tab import AnalysisTab

import xarray as xr
import numpy as np


class CAMSDataManager:
    """
    Main QGIS Plugin Implementation.
    
    This class manages the entire CAMS Data Manager plugin, handling the UI interactions,
    data downloads, and integration with QGIS. It serves as the central controller for
    all plugin functionality.
    """

    def __init__(self, iface):
        """
        Constructor - initializes the plugin and sets up basic properties.

        Args:
            iface: The QGIS interface instance that provides the hook to manipulate
                  the QGIS application at runtime.
        """
        # Ensure sys.stdout and sys.stderr are not None to prevent logging errors
        if sys.stdout is None:
            sys.stdout = open(os.devnull, 'w')
        if sys.stderr is None:
            sys.stderr = open(os.devnull, 'w')

        print("PLUGIN PYTHON PATH:", sys.executable)

        # Save reference to the QGIS interface - allows interaction with QGIS
        self.iface = iface
        
        # Initialize plugin directory - used for loading resources and translations
        self.plugin_dir = os.path.dirname(__file__)
        
        # Initialize locale - sets up translation based on system locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'CAMSDataManager_{}.qm'.format(locale))

        # Load translation file if it exists
        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Initialize instance attributes
        self.actions = []  # List to store toolbar/menu actions
        self.menu = self.tr(u'&CAMS Europe AQ Data Manager')  # Plugin menu name

        # Flag to check if plugin was started the first time in current QGIS session
        # This flag is used to avoid rebuilding the UI every time the plugin is opened
        self.first_start = None
        
        # Reference to the plugin dialog - initialized in run()
        self.dlg = None
        
        # Current area of interest - used to store selected AOI for download
        self.current_aoi = None

        # AnalysisTab instantiation and binding
        # self.analysis_tab = AnalysisTab(parent=self.dlg)
        # analysis_tab_widget = self.dlg.mainTabWidget.findChild(QWidget, "tabAnalysisResults")
        # if analysis_tab_widget and hasattr(analysis_tab_widget, 'layout') and analysis_tab_widget.layout():
        #     analysis_tab_widget.layout().addWidget(self.analysis_tab)
        # else:
        #     print("[DEBUG] Analysis tab widget or layout not found!")

    def tr(self, message):
        """
        Get the translation for a string using Qt translation API.
        
        This custom implementation is needed since we don't inherit QObject.

        Args:
            message: String to be translated.

        Returns:
            Translated version of the message.
        """
        return QCoreApplication.translate('CAMSDataManager', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """
        Add a toolbar icon to the QGIS interface.
        
        This method creates a QAction and adds it to the QGIS toolbar and/or menu.

        Args:
            icon_path: Path to the icon for this action (resource or file system path).
            text: Text shown in menu items for this action.
            callback: Function to be called when the action is triggered.
            enabled_flag: Flag indicating if the action should be enabled by default.
            add_to_menu: Flag indicating whether the action should be added to the menu.
            add_to_toolbar: Flag indicating whether the action should be added to the toolbar.
            status_tip: Optional text to show in a popup when mouse hovers over the action.
            whats_this: Optional text to show in the status bar when mouse hovers over the action.
            parent: Parent widget for the new action.

        Returns:
            The QAction that was created.
        """
        # Create icon from the specified path
        icon = QIcon(icon_path)
        
        # Create action with specified parameters
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)  # Connect to callback function
        action.setEnabled(enabled_flag)     # Set enabled state

        # Set optional properties if provided
        if status_tip is not None:
            action.setStatusTip(status_tip)
        if whats_this is not None:
            action.setWhatsThis(whats_this)

        # Add to toolbar if requested
        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        # Add to menu if requested
        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        # Store action for later removal
        self.actions.append(action)

        return action

    def initGui(self):
        """
        Create the menu entries and toolbar icons inside the QGIS GUI.
        
        This method is called by QGIS when the plugin is loaded.
        It sets up the plugin icon and action in the QGIS interface.
        """
        # Path to the plugin icon
        icon_path = ':/plugins/cams_data_manager/icon.png'
        
        # Add plugin icon to toolbar with action connected to run method
        self.add_action(
            icon_path,
            text=self.tr(u'CAMS Europe AQ Data Manager'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # Set first_start flag to True - will be set to False after first run
        self.first_start = True

    def unload(self):
        """
        Removes the plugin menu item and icon from QGIS GUI.
        
        This method is called by QGIS when the plugin is unloaded.
        It cleans up all UI elements added by the plugin.
        """
        # Remove each action from menu and toolbar
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&CAMS Europe AQ Data Manager'),
                action)
            self.iface.removeToolBarIcon(action)

    def setup_checkbox_dictionaries(self):
        """
        Create dictionaries mapping year and month strings to their respective checkbox widgets.
        
        This method centralizes dictionary creation logic to ensure consistent access
        to checkbox widgets throughout the code. The dictionaries are stored as attributes
        of the dialog object to allow access from other methods.
        """
        # Create year checkbox dictionary (2013-2023)
        # Keys are year strings, values are the corresponding checkbox widgets
        self.dlg.yearCheckBox = {
            str(y): getattr(self.dlg, f"checkYear{y}", None) 
            for y in range(2013, 2024) 
            if hasattr(self.dlg, f"checkYear{y}")
        }
        
        # Create month checkbox dictionary (01-12)
        # Keys are month strings with leading zeros (e.g., "01"), values are checkbox widgets
        self.dlg.monthCheckBox = {
            f"{m:02}": getattr(self.dlg, f"checkMonth{m:02}", None) 
            for m in range(1, 13) 
            if hasattr(self.dlg, f"checkMonth{m:02}")
        }

    def update_current_aoi(self):
        """
        Automatically update self.current_aoi based on UI control states.
        """
        # --- Debug: Print control states ---
        print("[DEBUG] update_current_aoi called")
        print("[DEBUG] radioFullArea.isChecked():", self.dlg.radioFullArea.isChecked())
        print("[DEBUG] radioCustomAOI.isChecked():", self.dlg.radioCustomAOI.isChecked())
        print("[DEBUG] checkUseShapefile.isChecked():", self.dlg.checkUseShapefile.isChecked())
        if hasattr(self.dlg, 'checkBoxSelectedOnly'):
            print("[DEBUG] checkBoxSelectedOnly.isChecked():", self.dlg.checkBoxSelectedOnly.isChecked())
        if hasattr(self.dlg, 'comboShapeFile'):
            print("[DEBUG] comboShapeFile.currentData():", self.dlg.comboShapeFile.currentData())
        # --- End debug ---
        if self.dlg.radioFullArea.isChecked():
            self.current_aoi = MODEL_BOUNDS
        elif self.dlg.radioCustomAOI.isChecked():
            if self.dlg.checkUseShapefile.isChecked():
                layer_id = self.dlg.comboShapeFile.currentData()
                if layer_id:
                    if hasattr(self.dlg, 'checkBoxSelectedOnly') and self.dlg.checkBoxSelectedOnly.isChecked():
                        layer = QgsProject.instance().mapLayer(layer_id)
                        selected_ids = layer.selectedFeatureIds() if layer else []
                        if selected_ids:
                            self.current_aoi = {'layer_ids': [layer_id], 'selected_only': True}
                        else:
                            self.current_aoi = None
                    else:
                        self.current_aoi = {'layer_ids': [layer_id]}
                else:
                    self.current_aoi = None
            else:
                try:
                    north = float(self.dlg.lineNorth.text())
                    south = float(self.dlg.lineSouth.text())
                    east = float(self.dlg.lineEast.text())
                    west = float(self.dlg.lineWest.text())
                    if north <= south or east <= west:
                        self.current_aoi = None
                        return
                    if not (-90 <= south < north <= 90):
                        self.current_aoi = None
                        return
                    if not (-180 <= west < east <= 180):
                        self.current_aoi = None
                        return
                    self.current_aoi = {
                        'north': north,
                        'south': south,
                        'east': east,
                        'west': west
                    }
                except Exception:
                    self.current_aoi = None
        else:
            self.current_aoi = None
        # --- Debug: Print self.current_aoi ---
        print("[DEBUG] self.current_aoi set to:", self.current_aoi)

    def setup_connections(self):
        """
        Connect UI signals to their handler methods.
        
        This method sets up all the signal-slot connections for the UI elements,
        linking user interactions to the appropriate handler methods.
        It also initializes default UI states.
        """
        print("[DEBUG] setup_connections called")
        # Connect tab navigation buttons
        self.dlg.btnNext.clicked.connect(self.on_next_clicked)
        self.dlg.btnCancel.clicked.connect(self.dlg.reject)
        
        # Connect download-related buttons
        self.dlg.btnDownload.clicked.connect(self.on_download_clicked)
        self.dlg.btnCancelDownload.clicked.connect(self.dlg.reject)
        self.dlg.btnBrowse.clicked.connect(self.on_browse_folder_clicked)
        
        # Connect AOI selection radio buttons
        self.dlg.radioFullArea.toggled.connect(self.on_aoi_mode_changed)
        self.dlg.radioFullArea.toggled.connect(self.update_current_aoi)
        self.dlg.radioCustomAOI.toggled.connect(self.on_aoi_mode_changed)
        self.dlg.radioCustomAOI.toggled.connect(self.update_current_aoi)
        
        # Connect shapefile checkbox
        self.dlg.checkUseShapefile.toggled.connect(self.on_use_shapefile_toggled)
        self.dlg.checkUseShapefile.toggled.connect(self.update_current_aoi)
        
        # Connect shapefile combo box
        if hasattr(self.dlg, 'comboShapeFile'):
            self.dlg.comboShapeFile.currentIndexChanged.connect(self.update_current_aoi)
            print("[DEBUG] Connected comboShapeFile.currentIndexChanged to update_current_aoi")
        else:
            print("[DEBUG] comboShapeFile not found for signal connection!")
        
        # Connect bounding box edits
        if hasattr(self.dlg, 'lineNorth'):
            self.dlg.lineNorth.textChanged.connect(self.update_current_aoi)
        if hasattr(self.dlg, 'lineSouth'):
            self.dlg.lineSouth.textChanged.connect(self.update_current_aoi)
        if hasattr(self.dlg, 'lineEast'):
            self.dlg.lineEast.textChanged.connect(self.update_current_aoi)
        if hasattr(self.dlg, 'lineWest'):
            self.dlg.lineWest.textChanged.connect(self.update_current_aoi)
        
        # Connect tab change signal
        self.dlg.mainTabWidget.currentChanged.connect(self.on_tab_changed)

        # Connect QGIS project layer signals
        QgsProject.instance().layersAdded.connect(self.on_layers_changed)
        QgsProject.instance().layersRemoved.connect(self.on_layers_changed)
        
        # Initialize default UI states
        self.dlg.radioFullArea.setChecked(True)
        self.dlg.groupBoundingBox.setEnabled(False)
        self.dlg.groupShapeFile.setEnabled(False)
        default_download_dir = os.path.expanduser(DEFAULT_DOWNLOAD_DIR)
        if not os.path.exists(default_download_dir):
            os.makedirs(default_download_dir, exist_ok=True)
        self.dlg.lineFolder.setText(default_download_dir)
        self.dlg.progressBar.setValue(0)
        # Synchronize AOI once during initialization
        self.update_current_aoi()

        # Analysis tab signal connections
        self.dlg.btnAggregate.clicked.connect(self.on_aggregate_clicked)
        self.dlg.btnBrowseOutput.clicked.connect(self.on_browse_output)
        self.dlg.btnRunStats.clicked.connect(self.on_run_stats_clicked)
        self.dlg.btnRunBivariate.clicked.connect(self.on_run_bivariate_clicked)
        # Analysis statistics variable linkage
        self.dlg.comboStatsLayer.currentIndexChanged.connect(self.populate_bivariate_vars)
        # Connect Terms and Conditions button
        if hasattr(self.dlg, 'btnViewTerms'):
            self.dlg.btnViewTerms.clicked.connect(self.open_terms_url)

    def populate_vector_layers(self):
        """
        Populate the comboShapeFile dropdown with all polygon vector layers
        currently loaded in the QGIS project, including memory layers.
        """
        print("[DEBUG] populate_vector_layers called")
        if hasattr(self.dlg, "comboShapeFile") and self.dlg.comboShapeFile is not None:
            self.dlg.comboShapeFile.clear()
        else:
            print("[DEBUG] comboShapeFile not found or not initialized!")
            return

        polygon_layers = []
        for layer in QgsProject.instance().mapLayers().values():
            print(f"[DEBUG] Checking layer: {layer.name()} (type={getattr(layer, 'type', lambda: 'N/A')()})")
            if layer.type() == 0 and hasattr(layer, 'geometryType') and layer.geometryType() == 2:
                if layer.isValid():
                    print(f"[DEBUG] Adding polygon layer: {layer.name()} (id={layer.id()})")
                    polygon_layers.append(layer)
                else:
                    print(f"[DEBUG] Layer {layer.name()} is not valid!")
            else:
                print(f"[DEBUG] Layer {layer.name()} is not a polygon vector layer.")

        if not polygon_layers:
            self.dlg.comboShapeFile.addItem("No polygon layer available. Please load a polygon layer in QGIS.")
            self.dlg.comboShapeFile.setEnabled(False)
            print("[DEBUG] No polygon layers found.")
            return

        for layer in polygon_layers:
            name = layer.name()
            if hasattr(layer, 'isTemporary') and layer.isTemporary():
                name += " (temporary)"
            self.dlg.comboShapeFile.addItem(name, layer.id())
            print(f"[DEBUG] comboShapeFile.addItem: name={name}, id={layer.id()}")
        self.dlg.comboShapeFile.setEnabled(True)
        print("[DEBUG] comboShapeFile items count:", self.dlg.comboShapeFile.count())
        for i in range(self.dlg.comboShapeFile.count()):
            print(f"[DEBUG] combo index {i}: text={self.dlg.comboShapeFile.itemText(i)}, data={self.dlg.comboShapeFile.itemData(i)}")

    def on_aoi_mode_changed(self):
        """
        Handler for AOI mode radio button changes.
        
        This method updates the UI to show/hide appropriate controls
        based on the selected AOI mode (full area or custom).
        """
        # Check if custom AOI is selected
        use_custom = self.dlg.radioCustomAOI.isChecked()
        
        # Enable/disable bounding box and shapefile inputs based on selection
        self.dlg.groupBoundingBox.setEnabled(use_custom)
        self.dlg.groupShapeFile.setEnabled(use_custom)
        self.update_current_aoi()

    def on_use_shapefile_toggled(self, checked):
        """
        Handler for shapefile checkbox toggle.
        
        This method updates the UI to enable/disable appropriate controls
        based on whether the user wants to use a shapefile for defining the AOI.

        Args:
            checked: Boolean indicating if the checkbox is checked.
        """
        # Enable/disable shapefile dropdown based on checkbox state
        self.dlg.comboShapeFile.setEnabled(checked)
        
        # Enable/disable bounding box inputs (only enable if using custom AOI but not shapefile)
        self.dlg.groupBoundingBox.setEnabled(not checked and self.dlg.radioCustomAOI.isChecked())
        self.update_current_aoi()

    def on_next_clicked(self):
        """
        Handler for Next button on the AOI tab.
        
        This method validates the AOI settings, stores the selected AOI,
        and navigates to the Download tab if validation passes.
        """
        self.update_current_aoi()
        if not self.is_aoi_valid():
            QMessageBox.warning(self.dlg, "Invalid AOI", "Please set a valid Area of Interest (AOI) before proceeding.")
            return
        if self.dlg.radioCustomAOI.isChecked():
            if self.dlg.checkUseShapefile.isChecked():
                # New logic: Validate that there must be selected features when "Selected features only" is checked
                layer_id = self.dlg.comboShapeFile.currentData()
                if hasattr(self.dlg, 'checkBoxSelectedOnly') and self.dlg.checkBoxSelectedOnly.isChecked():
                    layer = QgsProject.instance().mapLayer(layer_id)
                    selected_ids = layer.selectedFeatureIds() if layer else []
                    if not selected_ids:
                        QMessageBox.warning(
                            self.dlg,
                            "No Features Selected",
                            "You have checked 'Selected features only' but have not selected any features in the layer. Please use the QGIS select tool to select at least one polygon."
                        )
                        return
                if self.dlg.comboShapeFile.currentIndex() < 0:
                    QMessageBox.warning(
                        self.dlg,
                        "Missing Selection",
                        "Please select a polygon vector layer for your AOI."
                    )
                    return
                if not layer_id:
                    QMessageBox.warning(
                        self.dlg,
                        "Invalid Selection",
                        "Invalid vector layer selection."
                    )
                    return
                # AOI parameters are already set in update_current_aoi
            else:
                # Using manual coordinates: validate input values
                try:
                    if not all([
                        self.dlg.lineNorth.text(), self.dlg.lineSouth.text(),
                        self.dlg.lineEast.text(), self.dlg.lineWest.text()
                    ]):
                        raise ValueError("All coordinates must be filled.")
                    north = float(self.dlg.lineNorth.text())
                    south = float(self.dlg.lineSouth.text())
                    east = float(self.dlg.lineEast.text())
                    west = float(self.dlg.lineWest.text())
                    if north <= south or east <= west:
                        raise ValueError("Please enter a valid rectangular area (North > South, East > West).")
                    if not (-90 <= south < north <= 90):
                        raise ValueError("Latitude must be between -90 and 90, and North > South.")
                    if not (-180 <= west < east <= 180):
                        raise ValueError("Longitude must be between -180 and 180, and East > West.")
                    self.current_aoi = {
                        'north': north,
                        'south': south,
                        'east': east,
                        'west': west
                    }
                except ValueError as e:
                    QMessageBox.warning(
                        self.dlg,
                        "Invalid Rectangular Area",
                        str(e)
                    )
                    return
        else:
            self.current_aoi = MODEL_BOUNDS
        self.dlg.mainTabWidget.setCurrentIndex(2)

    def on_browse_folder_clicked(self):
        """
        Handler for the Browse button to select download folder.
        
        This method opens a file dialog for the user to select the output directory
        for downloaded data.
        """
        # Open directory selection dialog
        folder = QFileDialog.getExistingDirectory(
            self.dlg,
            "Select Output Directory",
            self.dlg.lineFolder.text()
        )
        
        # Update the folder path in the UI if a directory was selected
        if folder:
            self.dlg.lineFolder.setText(folder)

    def on_download_clicked(self):
        """
        Handler for the Download button on the Download tab.
        
        This method collects all parameters from the UI, validates them,
        initiates the download process, and handles success or failure.
        """
        # Force refresh AOI to ensure we get the latest selection state
        self.update_current_aoi()
        # Check if the .cdsapirc file exists in the user's home directory
        cdsapirc_path = os.path.expanduser("~/.cdsapirc")
        if not os.path.exists(cdsapirc_path):
            QMessageBox.warning(
                self.dlg,
                "API Key Missing",
                "Please enter your API KEY in the Help tab and click Save before downloading."
            )
            # Switch to the Help tab (tab index 0, adjust if needed)
            self.dlg.mainTabWidget.setCurrentIndex(0)
            return

        print("PLUGIN PYTHON PATH (download):", sys.executable)

        # Collect parameters from the UI using the helper function
        params = collect_download_parameters(self.dlg)

        # Print AOI parameters for debugging
        print("[DEBUG] AOI parameter 'area':", params.get('area'))

        # Add AOI information to params
        print("[DEBUG] on_download_clicked: self.current_aoi:", self.current_aoi)
        if self.current_aoi:
            params['area'] = self.current_aoi
        print("[DEBUG] on_download_clicked: params['area']:", params.get('area'))

        # Validate parameters using the validator helper
        if not validate_params(params):
            return  # Stop if validation failed

        # Additional validation for selected years and months
        # CAMS API requires exactly one year and one month for each request
        if len(params['years']) != 1:
            QMessageBox.warning(
                self.dlg,
                "Invalid Year Selection",
                "Please select exactly ONE year."
            )
            return

        if len(params['months']) != 1:
            QMessageBox.warning(
                self.dlg,
                "Invalid Month Selection",
                "Please select exactly ONE month."
            )
            return

        # --- Availability check before download ---
        from .tools.ui_handler import AVAILABILITY
        var = self.dlg.comboVariable.currentText()
        model = self.dlg.comboModel.currentText()
        typ = self.dlg.comboType.currentText()
        years_available = AVAILABILITY.get((var, model, typ), [])
        if not years_available:
            QMessageBox.warning(
                self.dlg,
                "No Available Years",
                "No available years for the current selection. Please change parameters."
            )
            return
        # --- End availability check ---

        # Reset progress bar before starting
        self.dlg.progressBar.setValue(0)

        try:
            # Set progress bar to indeterminate mode during download
            self.dlg.progressBar.setRange(0, 0)

            # Submit the request to the CAMS API
            submit_cams_request(params)

            # Update progress bar to show completion
            self.dlg.progressBar.setRange(0, 100)
            self.dlg.progressBar.setValue(100)

            # Construct the output file path (using same convention as cds_api.py)
            year_str = "_".join(params['years'])
            month_str = "_".join(params['months'])
            filename = f"{params['variable']}_{params['model']}_{year_str}_{month_str}.zip"
            output_file = os.path.join(params['folder'], filename)

            # Automatically unzip and load the first NetCDF file
            nc_file = unzip_and_get_netcdf(output_file)
            clipped_nc_file = None

            if params.get('area') and 'layer_ids' in params['area']:
                print("[DEBUG] Entered AOI clipping branch")
                area = params['area']
                try:
                    gdfs = []
                    temp_dirs = []
                    for layer_id in area['layer_ids']:
                        print("[DEBUG] Processing layer_id:", layer_id)
                        layer = QgsProject.instance().mapLayer(layer_id)
                        print("[DEBUG] Layer object:", layer)
                        if layer:
                            temp_dir = tempfile.mkdtemp()
                            temp_dirs.append(temp_dir)
                            temp_shp = os.path.join(temp_dir, f"{layer.name()}.shp")
                            # Export all features if not selected_only, otherwise only selected
                            if area.get('selected_only', False):
                                selected_ids = layer.selectedFeatureIds()
                                print("[DEBUG] selected_ids:", selected_ids)
                                if selected_ids:
                                    request = QgsFeatureRequest().setFilterFids(selected_ids)
                                    _ = QgsVectorFileWriter.writeAsVectorFormat(layer, temp_shp, "utf-8", layer.crs(), "ESRI Shapefile", onlySelected=True)
                                else:
                                    print("[DEBUG] No selected features, exporting all features instead.")
                                    _ = QgsVectorFileWriter.writeAsVectorFormat(layer, temp_shp, "utf-8", layer.crs(), "ESRI Shapefile")
                            else:
                                _ = QgsVectorFileWriter.writeAsVectorFormat(layer, temp_shp, "utf-8", layer.crs(), "ESRI Shapefile")
                            print("[DEBUG] temp_shp:", temp_shp)
                            try:
                                gdf = gpd.read_file(temp_shp)
                                print("[DEBUG] gdf.shape:", gdf.shape)
                                print("[DEBUG] gdf.head():\n", gdf.head())
                            except Exception as e:
                                print("[DEBUG] gpd.read_file failed:", e)
                                gdf = None
                            if gdf is not None:
                                gdfs.append(gdf)
                    if gdfs:
                        merged_gdf = gpd.GeoDataFrame(pd.concat(gdfs, ignore_index=True))
                        merged_gdf = merged_gdf.set_crs(gdfs[0].crs)
                        temp_dir = tempfile.mkdtemp()
                        temp_shp = os.path.join(temp_dir, 'merged_aoi.shp')
                        merged_gdf.to_file(temp_shp)
                        print("[DEBUG] merged_gdf.shape:", merged_gdf.shape)
                        print("[DEBUG] merged temp_shp:", temp_shp)
                        clipped_nc_file = nc_file.replace('.nc', '_clipped.nc')
                        print("[DEBUG] Calling clip_netcdf_by_shapefile:", nc_file, clipped_nc_file, temp_shp)
                        clip_netcdf_by_shapefile(
                            nc_file,
                            clipped_nc_file,
                            temp_shp
                        )
                        file_to_load = clipped_nc_file
                    else:
                        print("[DEBUG] gdfs is empty, no valid AOI shapefile generated")
                        file_to_load = nc_file
                except Exception as e:
                    print(f"[ERROR] Clipping failed: {e}")
                    print(f"[ERROR] area: {area}")
                    print(f"[ERROR] nc_file: {nc_file}")
                    file_to_load = nc_file
            else:
                print("[DEBUG] Did not enter AOI clipping branch, loading original data")
                file_to_load = nc_file
                QMessageBox.information(
                    self.dlg,
                    "Download Complete",
                    f"Data downloaded and loaded into QGIS:\n{file_to_load}"
                )

            if file_to_load and os.path.exists(file_to_load):
                netcdf_var = NETCDF_VARIABLE_MAP.get(params["variable"], params["variable"])
                self.load_data_to_qgis(file_to_load, netcdf_var)
            else:
                QMessageBox.warning(
                    self.dlg,
                    "No NetCDF Found",
                    "No NetCDF file found in the ZIP archive."
                )

            # Log successful download
            self.log_download(params, success=True)

        except Exception as e:
            # Reset progress bar on error
            self.dlg.progressBar.setRange(0, 100)
            self.dlg.progressBar.setValue(0)

            # Log failed download
            self.log_download(params, success=False, error=str(e))

            # Show error message
            QMessageBox.critical(
                self.dlg,
                "Download Failed",
                f"An error occurred while downloading data:\n{str(e)}"
            )

    def log_download(self, params, success=True, error=None):
        """
        Log the download attempt to a file.
        
        This method records details about each download attempt to a log file
        for tracking and debugging purposes.

        Args:
            params: Dictionary of download parameters.
            success: Boolean indicating if the download was successful.
            error: Error message string if the download failed.
        """
        # Define log file path in user's home directory
        log_path = os.path.expanduser("~/cams_plugin.log")
        
        # Append entry to log file
        with open(log_path, "a", encoding="utf-8") as f:
            f.write("=" * 60 + "\n")  # Separator line
            f.write(f"Timestamp: {datetime.datetime.now()}\n")
            f.write(f"Status: {'Success' if success else 'Failed'}\n")
            f.write(f"Variable: {params.get('variable')}\n")
            f.write(f"Model: {params.get('model')}\n")
            f.write(f"Years: {params.get('years')}\n")
            f.write(f"Months: {params.get('months')}\n")
            f.write(f"Folder: {params.get('folder')}\n")
            # Include error message if download failed
            if error:
                f.write(f"Error: {error}\n")
            f.write("\n")  # End with blank line

    def check_dependencies(self):
        """
        Check if required Python packages are installed.
        
        This method verifies that the cdsapi package is available,
        showing an error message and switching to the Help tab if not.

        Returns:
            Boolean indicating if all dependencies are satisfied.
        """
        try:
            # Try to import cdsapi to check if it's installed
            import cdsapi
            return True
        except ImportError:
            # Show error message if cdsapi is missing
            QMessageBox.critical(
                self.dlg,
                "Missing Dependencies",
                "Required package 'cdsapi' is missing.\n\n"
                "Please install it using:\npip install cdsapi\n\n"
                "See the Help tab for more information."
            )
            # Switch to the Help tab (index 3)
            self.dlg.mainTabWidget.setCurrentIndex(3)
            return False

    def check_api_key(self):
        """
        Check if the CDS API key file exists.
        
        This method checks for the presence of the .cdsapirc configuration file
        that contains the API credentials needed to access the CAMS data service.

        Returns:
            Boolean indicating if the API key file exists.
        """
        # Path to the cdsapi configuration file in user's home directory
        api_rc_file = os.path.expanduser("~/.cdsapirc")
        
        # Check if the file exists
        if not os.path.exists(api_rc_file):
            # Show warning if file is missing
            QMessageBox.warning(
                self.dlg,
                "API Configuration Missing",
                "The CDS API configuration file is missing.\n\n"
                "Please create a file at: " + api_rc_file + "\n\n"
                "With content:\n"
                "url: https://ads.atmosphere.copernicus.eu/api/v2\n"
                "key: <your-api-key>\n\n"
                "See the Help tab for details."
            )
            # Switch to the Help tab (index 3)
            self.dlg.mainTabWidget.setCurrentIndex(3)
            return False
        return True

    def load_data_to_qgis(self, file_path, variable_name=None):
        """
        Load a NetCDF (.nc) raster layer into QGIS using GDAL.

        Args:
            file_path (str): Absolute path to the .nc file (NetCDF)
            variable_name (str, optional): NetCDF variable name to load (not needed for most CAMS files)
        """
        # Let QGIS/GDAL auto-detect all bands/variables, just like manual loading
        uri = f'NETCDF:"{file_path}"'
        layer_name = os.path.basename(file_path).replace(".nc", "")

        raster_layer = QgsRasterLayer(uri, layer_name, "gdal")

        if raster_layer.isValid():
            QgsProject.instance().addMapLayer(raster_layer)
            QMessageBox.information(
                self.dlg,
                "Layer Added",
                f"NetCDF file '{file_path}' has been loaded into QGIS."
            )
        else:
            QMessageBox.critical(
                self.dlg,
                "Load Failed",
                f"Failed to load NetCDF file '{file_path}'.\n"
                f"Please check GDAL NetCDF support or file integrity."
            )

    def setup_year_month_defaults(self):
        """
        Initialize year and month checkboxes without default selection.
        """
        # No default selection for years and months
        pass

    def is_aoi_valid(self):
        if self.dlg.radioFullArea.isChecked():
            return True
        if self.dlg.radioCustomAOI.isChecked():
            if self.dlg.checkUseShapefile.isChecked():
                return self.dlg.comboShapeFile.currentIndex() >= 0 and \
                       "No polygon layer" not in self.dlg.comboShapeFile.currentText()
            else:
                # Check bounding box fields
                try:
                    north = float(self.dlg.lineNorth.text())
                    south = float(self.dlg.lineSouth.text())
                    east = float(self.dlg.lineEast.text())
                    west = float(self.dlg.lineWest.text())
                    if north <= south or east <= west:
                        return False
                    if not (-90 <= south < north <= 90):
                        return False
                    if not (-180 <= west < east <= 180):
                        return False
                    return True
                except Exception:
                    return False
        return False

    def on_tab_changed(self, index):
        # Download tab index is assumed to be 2
        if index == 2 and not self.is_aoi_valid():
            QMessageBox.warning(self.dlg, "Invalid AOI", "Please set a valid Area of Interest (AOI) before downloading.")
            self.dlg.mainTabWidget.setCurrentIndex(1)  # Switch back to AOI tab
        # Analysis tab index is assumed to be 3 (please adjust according to actual order)
        if index == 3:
            print("[DEBUG] Switching to Analysis tab, refreshing NetCDF list...")
            self.refresh_analysis_tab()

    def refresh_analysis_tab(self):
        self.populate_netcdf_list()
        self.populate_stats_layer_combo()
        self.populate_bivariate_vars()

    def populate_netcdf_list(self):
        self.dlg.listNetcdfLayers.clear()
        seen_paths = set()
        # 1. NetCDF layers already loaded in QGIS
        for layer in QgsProject.instance().mapLayers().values():
            if isinstance(layer, QgsRasterLayer):
                source = layer.source()
                if source.lower().endswith('.nc') or 'NETCDF:' in source.upper():
                    if 'NETCDF:"' in source:
                        path = source.split('NETCDF:"')[-1].split('"')[0]
                    else:
                        path = source
                    if os.path.exists(path) and path not in seen_paths:
                        self.dlg.listNetcdfLayers.addItem(path)
                        seen_paths.add(path)
        # 2. ~/CAMS_Data directory
        netcdf_dir = os.path.expanduser("~/CAMS_Data")
        if not os.path.exists(netcdf_dir):
            os.makedirs(netcdf_dir, exist_ok=True)
        for fname in os.listdir(netcdf_dir):
            if fname.endswith(".nc"):
                fpath = os.path.join(netcdf_dir, fname)
                if fpath not in seen_paths:
                    self.dlg.listNetcdfLayers.addItem(fpath)
                    seen_paths.add(fpath)
        # 3. Current folder selected in Download tab
        download_folder = self.dlg.lineFolder.text().strip()
        if download_folder and os.path.exists(download_folder):
            for fname in os.listdir(download_folder):
                if fname.endswith(".nc"):
                    fpath = os.path.join(download_folder, fname)
                    if fpath not in seen_paths:
                        self.dlg.listNetcdfLayers.addItem(fpath)
                        seen_paths.add(fpath)

    def populate_stats_layer_combo(self):
        self.dlg.comboStatsLayer.clear()
        seen_paths = set()
        # 1. NetCDF layers already loaded in QGIS
        for layer in QgsProject.instance().mapLayers().values():
            if isinstance(layer, QgsRasterLayer):
                source = layer.source()
                if source.lower().endswith('.nc') or 'NETCDF:' in source.upper():
                    if 'NETCDF:"' in source:
                        path = source.split('NETCDF:"')[-1].split('"')[0]
                    else:
                        path = source
                    if os.path.exists(path) and path not in seen_paths:
                        self.dlg.comboStatsLayer.addItem(path)
                        seen_paths.add(path)
        # 2. ~/CAMS_Data directory
        netcdf_dir = os.path.expanduser("~/CAMS_Data")
        if not os.path.exists(netcdf_dir):
            os.makedirs(netcdf_dir, exist_ok=True)
        for fname in os.listdir(netcdf_dir):
            if fname.endswith(".nc"):
                fpath = os.path.join(netcdf_dir, fname)
                if fpath not in seen_paths:
                    self.dlg.comboStatsLayer.addItem(fpath)
                    seen_paths.add(fpath)
        # 3. Current folder selected in Download tab
        download_folder = self.dlg.lineFolder.text().strip()
        if download_folder and os.path.exists(download_folder):
            for fname in os.listdir(download_folder):
                if fname.endswith(".nc"):
                    fpath = os.path.join(download_folder, fname)
                    if fpath not in seen_paths:
                        self.dlg.comboStatsLayer.addItem(fpath)
                        seen_paths.add(fpath)

    def populate_bivariate_vars(self):
        """
        Populate the primary and secondary variable combo boxes for bivariate statistics.
        This function collects all NetCDF files from both loaded QGIS layers and the current download folder,
        removes duplicates, and fills the combo boxes.
        """
        self.dlg.comboPrimaryVar.clear()
        self.dlg.comboSecondaryVar.clear()
        file_paths = set()

        # 1. Collect NetCDF files from loaded QGIS raster layers
        for layer in QgsProject.instance().mapLayers().values():
            if isinstance(layer, QgsRasterLayer):
                source = layer.source()
                if source.lower().endswith('.nc') or 'NETCDF:' in source.upper():
                    if 'NETCDF:"' in source:
                        path = source.split('NETCDF:"')[-1].split('"')[0]
                    else:
                        path = source
                    if os.path.exists(path):
                        file_paths.add(path)

        # 2. Collect NetCDF files from the current download folder
        download_folder = self.dlg.lineFolder.text().strip()
        if download_folder and os.path.exists(download_folder):
            for fname in os.listdir(download_folder):
                if fname.endswith(".nc"):
                    fpath = os.path.join(download_folder, fname)
                    file_paths.add(fpath)

        # 3. Fill the combo boxes with all unique NetCDF file paths
        for fpath in sorted(file_paths):
            fname = os.path.basename(fpath)
            self.dlg.comboPrimaryVar.addItem(fname, fpath)
            self.dlg.comboSecondaryVar.addItem(fname, fpath)

    def get_selected_stats(self):
        stats = []
        if self.dlg.checkMean.isChecked():
            stats.append('mean')
        if self.dlg.checkMax.isChecked():
            stats.append('max')
        if self.dlg.checkMin.isChecked():
            stats.append('min')
        if self.dlg.checkStd.isChecked():
            stats.append('std')
        return stats

    def on_aggregate_clicked(self):
        selected_items = self.dlg.listNetcdfLayers.selectedItems()
        if not selected_items:
            QMessageBox.warning(self.dlg, "No file selected", "Please select at least one NetCDF file.")
            return
        file_paths = [item.text() for item in selected_items]
        agg_type = self.dlg.comboAggType.currentText()
        agg_map = {"Daily": "1D", "Weekly": "1W", "Monthly": "1M", "Quarterly": "1Q", "Yearly": "1Y"}
        resample_str = agg_map.get(agg_type, "1M")
        output_path = self.dlg.lineOutputPath.text().strip()
        if not output_path:
            QMessageBox.warning(self.dlg, "No output path", "Please specify an output file path.")
            return
        try:
            self.dlg.progressBarAgg.setRange(0, 0)
            import xarray as xr
            QMessageBox.information(self.dlg, "Selected Files", "\n".join(file_paths))
            ds = xr.open_mfdataset(file_paths, combine='by_coords', engine='netcdf4', chunks={})
            # Force all variables to numpy, avoid dask
            for v in ds.data_vars:
                ds[v] = ds[v].compute()
            agg_ds = ds.resample(time=resample_str).mean()
            agg_ds = agg_ds.compute()
            agg_ds.to_netcdf(output_path)
            self.dlg.progressBarAgg.setRange(0, 100)
            self.dlg.progressBarAgg.setValue(100)
            if self.dlg.checkLoadToQgis.isChecked():
                self.load_data_to_qgis(output_path)
            QMessageBox.information(self.dlg, "Aggregation Complete", f"Aggregated file saved to:\n{output_path}")
        except Exception as e:
            self.dlg.progressBarAgg.setRange(0, 100)
            self.dlg.progressBarAgg.setValue(0)
            QMessageBox.critical(self.dlg, "Aggregation Failed", f"Error: {str(e)}")

    def on_browse_output(self):
        out_path, _ = QFileDialog.getSaveFileName(self.dlg, "Select Output NetCDF File", "", "NetCDF Files (*.nc)")
        if out_path:
            self.dlg.lineOutputPath.setText(out_path)

    def on_run_stats_clicked(self):
        file_path = self.dlg.comboStatsLayer.currentText().strip()
        if not file_path or not os.path.exists(file_path):
            QMessageBox.warning(self.dlg, "No file selected", "Please select a valid NetCDF file.")
            return
        stats = self.get_selected_stats()
        if not stats:
            QMessageBox.warning(self.dlg, "No statistics selected", "Please select at least one statistic.")
            return
        try:
            ds = xr.open_dataset(file_path)
            var_names = [v for v in ds.data_vars if ds[v].ndim > 0 and v.lower() not in {"spatial_ref", "crs", "grid_mapping"}]
            if not var_names:
                QMessageBox.warning(self.dlg, "No valid variable", "No valid scientific variable found in this file.")
                return
            var_name = var_names[0]
            data = ds[var_name]
            arr = data.values.flatten()  # Merge all spatial pixels
            # Process missing_value/-999/999/.FillValue
            missing_vals = set()
            for attr in ["missing_value", "_FillValue"]:
                if attr in data.attrs:
                    missing_vals.add(data.attrs[attr])
            missing_vals.update([-999, 999])
            for mv in missing_vals:
                arr = np.where(arr == mv, np.nan, arr)
            results = []
            if 'mean' in stats:
                results.append(f"Mean: {float(np.nanmean(arr)):.4f}")
            if 'max' in stats:
                results.append(f"Max: {float(np.nanmax(arr)):.4f}")
            if 'min' in stats:
                results.append(f"Min: {float(np.nanmin(arr)):.4f}")
            if 'std' in stats:
                results.append(f"Std. Dev: {float(np.nanstd(arr)):.4f}")
            self.dlg.textStatsResult.setPlainText("\n".join(results))
        except Exception as e:
            QMessageBox.critical(self.dlg, "Statistics Failed", f"Error: {str(e)}")

    def on_run_bivariate_clicked(self):
        file1 = self.dlg.comboPrimaryVar.currentData()
        file2 = self.dlg.comboSecondaryVar.currentData()
        if not file1 or not file2:
            QMessageBox.warning(self.dlg, "No file selected", "Please select two NetCDF files for analysis.")
            return
        try:
            import xarray as xr
            import numpy as np
            ds1 = xr.open_dataset(file1)
            ds2 = xr.open_dataset(file2)
            # Automatically recognize primary variable
            var_names1 = [v for v in ds1.data_vars if ds1[v].ndim > 0 and v.lower() not in {"spatial_ref", "crs", "grid_mapping"}]
            var_names2 = [v for v in ds2.data_vars if ds2[v].ndim > 0 and v.lower() not in {"spatial_ref", "crs", "grid_mapping"}]
            if not var_names1 or not var_names2:
                QMessageBox.warning(self.dlg, "No valid variable", "No valid scientific variable found in one of the files.")
                return
            var1 = var_names1[0]
            var2 = var_names2[0]
            data1 = ds1[var1].values.flatten()
            data2 = ds2[var2].values.flatten()
            # Process invalid values
            missing_vals1 = set()
            for attr in ["missing_value", "_FillValue"]:
                if attr in ds1[var1].attrs:
                    missing_vals1.add(ds1[var1].attrs[attr])
            missing_vals1.update([-999, 999])
            for mv in missing_vals1:
                data1 = np.where(data1 == mv, np.nan, data1)
            missing_vals2 = set()
            for attr in ["missing_value", "_FillValue"]:
                if attr in ds2[var2].attrs:
                    missing_vals2.add(ds2[var2].attrs[attr])
            missing_vals2.update([-999, 999])
            for mv in missing_vals2:
                data2 = np.where(data2 == mv, np.nan, data2)
            # Align lengths
            min_len = min(len(data1), len(data2))
            data1 = data1[:min_len]
            data2 = data2[:min_len]
            mask = ~np.isnan(data1) & ~np.isnan(data2)
            data1 = data1[mask]
            data2 = data2[mask]
            method = self.dlg.comboAnalysisMethod.currentText()
            if method == "Correlation":
                from scipy.stats import pearsonr
                if len(data1) == 0 or len(data2) == 0:
                    self.dlg.textBivariateResult.setPlainText("No valid data for correlation.")
                    return
                corr, pval = pearsonr(data1, data2)
                result = f"Pearson correlation: {corr:.4f}\np-value: {pval:.4g}"
                self.dlg.textBivariateResult.setPlainText(result)
            elif method == "Linear Regression":
                from scipy.stats import linregress
                if len(data1) == 0 or len(data2) == 0:
                    self.dlg.textBivariateResult.setPlainText("No valid data for regression.")
                    return
                reg = linregress(data1, data2)
                result = (
                    f"Linear regression:\n"
                    f"y = {reg.slope:.4f} * x + {reg.intercept:.4f}\n"
                    f"R² = {reg.rvalue**2:.4f}\n"
                    f"p-value = {reg.pvalue:.4g}\n"
                    f"StdErr = {reg.stderr:.4g}"
                )
                self.dlg.textBivariateResult.setPlainText(result)
            elif method == "Classification Accuracy":
                try:
                    from sklearn.metrics import accuracy_score
                except ImportError:
                    self.dlg.textBivariateResult.setPlainText("scikit-learn is required for classification accuracy analysis.")
                    return
                if len(data1) == 0 or len(data2) == 0:
                    self.dlg.textBivariateResult.setPlainText("No valid data for classification accuracy.")
                    return
                try:
                    y_true = data1.astype(int)
                    y_pred = data2.astype(int)
                except Exception:
                    y_true = data1.astype(str)
                    y_pred = data2.astype(str)
                acc = accuracy_score(y_true, y_pred)
                result = (
                    f"Classification accuracy: {acc:.4f}\n"
                    f"Total samples: {len(y_true)}"
                )
                self.dlg.textBivariateResult.setPlainText(result)
            else:
                self.dlg.textBivariateResult.setPlainText("This analysis method is not implemented yet.")
        except Exception as e:
            QMessageBox.critical(self.dlg, "Bivariate Analysis Failed", f"Error: {str(e)}")

    def on_layers_changed(self):
        """
        Handler for QGIS project layer changes.
        Updates the shapefile combo box when layers are added or removed.
        """
        # Store current selection
        current_layer_id = self.dlg.comboShapeFile.currentData()
        
        # Update the combo box
        self.populate_vector_layers()
        
        # Try to restore previous selection
        if current_layer_id:
            index = self.dlg.comboShapeFile.findData(current_layer_id)
            if index >= 0:
                self.dlg.comboShapeFile.setCurrentIndex(index)
        
        # Update AOI if needed
        self.update_current_aoi()

    def run(self): 
        """
        Run method that performs all the real work.
        
        This is the main entry point for the plugin, called when the user
        clicks the plugin icon in QGIS. It initializes the UI, sets up connections,
        and handles the plugin execution.
        """
        
        print("PLUGIN PYTHON PATH:", sys.executable)

        # Create the dialog only once - first time the plugin is run
        if self.first_start == True:
            self.first_start = False
            self.dlg = CAMSDataManagerDialog()
            self.dlg.initialize_dropdown_menus()
            self.setup_checkbox_dictionaries()
            self.setup_year_month_defaults()
            self.setup_connections()
            
        # These operations need to be performed every time the plugin is opened
        # Populate vector layers dropdown
        self.populate_vector_layers()
            
        # Check for required dependencies
        if not self.check_dependencies():
            return
            
        # Check for API key configuration
        if not self.check_api_key():
            return
            
        # Reset to first tab
        self.dlg.mainTabWidget.setCurrentIndex(0)
        
        # Reset progress bar
        self.dlg.progressBar.setValue(0)
            
        # Show the dialog
        self.dlg.show()
        
        # Run the dialog event loop and wait for it to close
        result = self.dlg.exec_()
        
        # Result handling is done by connected UI handlers 

    def open_terms_url(self):
        """
        Open the Copernicus Terms and Conditions PDF in the default web browser.
        """
        url = "https://object-store.os-api.cci2.ecmwf.int/cci2-prod-catalogue/licences/licence-to-use-copernicus-products/licence-to-use-copernicus-products_b4b9451f54cffa16ecef5c912c9cebd6979925a956e3fa677976e0cf198c2c18.pdf"
        webbrowser.open(url)

