# -*- coding: utf-8 -*-

import os, shutil, traceback
from qgis.PyQt.QtWidgets import (QDialog, QFileDialog, QVBoxLayout, QHBoxLayout, QLabel, QLineEdit, QPushButton,
    QComboBox, QDoubleSpinBox, QSpinBox, QCheckBox, QWidget, QTextBrowser, QGridLayout, QListWidget, QListWidgetItem)
from qgis.PyQt.QtCore import Qt, QCoreApplication
from qgis.core import QgsVectorLayer, QgsProject
from .services.dwg_support import dwg_to_temp_dxf_auto
from .services import deps as _deps
from .services import augment as _augment

# OGR fallback imports
try:
    from osgeo import ogr, gdal, osr
    try:
        ogr.UseExceptions()
    except Exception:
        pass
    try:
        gdal.SetConfigOption("DXF_INLINE_BLOCKS", "YES")
        gdal.SetConfigOption("DXF_CLOSED_LINE_AS_POLYGON", "TRUE")
    except Exception:
        pass
except Exception:
    ogr = None
    gdal = None
    osr = None

def _ogr_list_layers(dxf_path: str):
    names = []
    if ogr is None or not os.path.exists(dxf_path):
        return names
    ds = ogr.Open(dxf_path, 0)
    if ds is None:
        return names
    lyr = ds.GetLayerByName("entities") or (ds.GetLayer(0) if ds.GetLayerCount() else None)
    if lyr is None:
        ds = None; return names
    try:
        res = ds.ExecuteSQL("SELECT DISTINCT Layer FROM entities")
        if res:
            for f in res:
                v = f.GetField("Layer")
                if v:
                    names.append(str(v))
            ds.ReleaseResultSet(res)
    except Exception:
        pass
    if not names:
        seen = set()
        lyr.ResetReading()
        for f in lyr:
            try:
                v = f.GetField("Layer")
                if v and v not in seen:
                    seen.add(v); names.append(str(v))
            except Exception:
                pass
    ds = None
    return sorted(set(names))

class CadToGisDialog(QDialog):
    def _parse_epsg(self, widget, default=None):
        # 相容 QSpinBox / QLineEdit
        try:
            return int(widget.value())  # QSpinBox
        except Exception:
            pass
        try:
            txt = widget.text().strip()
            return int(txt) if txt else default
        except Exception:
            return default
        
    def __init__(self, parent_or_iface):
        # Accept either QgisInterface or a QWidget (e.g., QMainWindow)
        try:
            from qgis.utils import iface as _global_iface
        except Exception:
            _global_iface = None
        parent = None
        self.iface = None
        # If we got a QgisInterface-like object
        if hasattr(parent_or_iface, 'mainWindow') and callable(getattr(parent_or_iface, 'mainWindow')):
            self.iface = parent_or_iface
            try:
                parent = parent_or_iface.mainWindow()
            except Exception:
                parent = None
        else:
            # Assume it's already a QWidget parent (e.g., QMainWindow)
            parent = parent_or_iface
            self.iface = _global_iface
        super().__init__(parent)

        self.setWindowTitle("CAD to GIS Converter")
        self.setMinimumWidth(860)

        grid = QGridLayout()

        # Row 0: Input + buttons
        grid.addWidget(QLabel("Input CAD (DXF/DWG)"), 0, 0)
        self.in_edit = QLineEdit()
        btn_in = QPushButton("Browse"); btn_in.clicked.connect(self.browse_input)
        btn_scan = QPushButton("Scan Layers"); btn_scan.clicked.connect(self.scan_layers)
        wrap = QWidget(); hb = QHBoxLayout(wrap); hb.setContentsMargins(0,0,0,0)
        hb.addWidget(self.in_edit); hb.addWidget(btn_in); hb.addWidget(btn_scan)
        grid.addWidget(wrap, 0, 1, 1, 2)

        # Row 1: Layers CSV + control buttons
        grid.addWidget(QLabel("Layer names (CSV, auto-filled from preview)"), 1, 0)
        self.layers_edit = QLineEdit()
        btn_select_all = QPushButton("Select All"); btn_select_all.clicked.connect(self.select_all_layers)
        btn_clear = QPushButton("Clear"); btn_clear.clicked.connect(self.clear_layers_selection)
        wrap2 = QWidget(); hb2 = QHBoxLayout(wrap2); hb2.setContentsMargins(0,0,0,0)
        hb2.addWidget(self.layers_edit); hb2.addWidget(btn_select_all); hb2.addWidget(btn_clear)
        grid.addWidget(wrap2, 1, 1, 1, 2)

        # Row 2: Layer preview list
        grid.addWidget(QLabel("Layer Preview"), 2, 0, Qt.AlignTop)
        self.layer_list = QListWidget()
        self.layer_list.setSelectionMode(self.layer_list.MultiSelection)
        self.layer_list.itemSelectionChanged.connect(self.sync_layers_csv_from_preview)
        self.layer_list.itemChanged.connect(self.sync_layers_csv_from_preview)
        grid.addWidget(self.layer_list, 2, 1, 1, 2)

        # Row 3: Source EPSG
        grid.addWidget(QLabel("Source EPSG"), 3, 0)
        self.src_epsg = QLineEdit()
        grid.addWidget(self.src_epsg, 3, 1, 1, 2)

        # Row 4: Target EPSG
        grid.addWidget(QLabel("Target EPSG (optional)"), 4, 0)
        self.tgt_epsg = QLineEdit()
        grid.addWidget(self.tgt_epsg, 4, 1, 1, 2)

        # Row 5: Block mode
        grid.addWidget(QLabel("Block handling"), 5, 0)
        self.block_mode = QComboBox(); self.block_mode.addItems(["keep-merge", "explode"])
        grid.addWidget(self.block_mode, 5, 1, 1, 2)

        # Row 6: Merge tolerance (default 0.0)
        grid.addWidget(QLabel("Line-merge tolerance"), 6, 0)
        self.merge_tol = QDoubleSpinBox(); self.merge_tol.setDecimals(6); self.merge_tol.setRange(0.0, 1e9); self.merge_tol.setValue(0.0)
        grid.addWidget(self.merge_tol, 6, 1, 1, 2)

        # Row 7: Spline tolerance
        grid.addWidget(QLabel("Spline tolerance"), 7, 0)
        self.spline_tol = QDoubleSpinBox(); self.spline_tol.setDecimals(6); self.spline_tol.setRange(0.0, 1e6); self.spline_tol.setValue(0.2)
        grid.addWidget(self.spline_tol, 7, 1, 1, 2)

        # Row 8: Driver
        grid.addWidget(QLabel("Output driver"), 8, 0)
        self.driver = QComboBox(); self.driver.addItems(["GPKG","ESRI Shapefile"])
        grid.addWidget(self.driver, 8, 1, 1, 2)

        # Row 9: Output path
        self.out_label = QLabel("Output GeoPackage (GPKG)")
        grid.addWidget(self.out_label, 9, 0)
        self.out_edit = QLineEdit()
        self.btn_out = QPushButton("Browse"); self.btn_out.clicked.connect(self.browse_output)
        wrap3 = QWidget(); hb3 = QHBoxLayout(wrap3); hb3.setContentsMargins(0,0,0,0)
        hb3.addWidget(self.out_edit); hb3.addWidget(self.btn_out)
        grid.addWidget(wrap3, 9, 1, 1, 2)

        # Row 10: DWG options
        grid.addWidget(QLabel("DWG converter preference"), 10, 0)
        self.dwg_pref = QComboBox(); self.dwg_pref.addItems(["auto","oda","libredwg"])
        grid.addWidget(self.dwg_pref, 10, 1, 1, 2)

        grid.addWidget(QLabel("DXF version for DWG conversion"), 11, 0)
        self.dxf_version = QLineEdit(); self.dxf_version.setText("ACAD2013")
        grid.addWidget(self.dxf_version, 11, 1, 1, 2)

        # Row 12: Options
        grid.addWidget(QLabel("Options"), 12, 0, Qt.AlignTop)
        options_wrap = QWidget(); fl = QVBoxLayout(options_wrap); fl.setContentsMargins(0,0,0,0)
        self.chk_overwrite = QCheckBox("Overwrite existing"); self.chk_overwrite.setChecked(False)
        self.chk_load = QCheckBox("Load outputs into project"); self.chk_load.setChecked(True)
        self.chk_text_attrs = QCheckBox("Extract TEXT/MTEXT to attributes (annotation points) — only selected layers"); self.chk_text_attrs.setChecked(True)
        self.chk_block_attrs = QCheckBox("Expand block attributes into fields (BLOCKS layer) — only selected layers"); self.chk_block_attrs.setChecked(True)
        self.chk_block_transform = QCheckBox("Keep block transform fields (x,y,rotation,scale) — only selected layers"); self.chk_block_transform.setChecked(True)
        fl.addWidget(self.chk_overwrite); fl.addWidget(self.chk_load)
        fl.addWidget(self.chk_text_attrs); fl.addWidget(self.chk_block_attrs); fl.addWidget(self.chk_block_transform)
        grid.addWidget(options_wrap, 12, 1, 1, 2)

        # Row 13: Run
        run_bar = QWidget(); hb4 = QHBoxLayout(run_bar); hb4.setContentsMargins(0,0,0,0)
        self.btn_run = QPushButton("Run"); self.btn_run.clicked.connect(self.run_convert)
        hb4.addStretch(1); hb4.addWidget(self.btn_run)
        grid.addWidget(run_bar, 13, 0, 1, 3)

        # Row 14: Log/HTML
        self.out_html = QTextBrowser(); self.out_html.setOpenExternalLinks(True)
        grid.addWidget(self.out_html, 14, 0, 1, 3)

        self.driver.currentIndexChanged.connect(self.on_driver_changed)
        self.on_driver_changed()

        lay = QVBoxLayout(self); lay.addLayout(grid)

    def browse_input(self):
        p, _ = QFileDialog.getOpenFileName(self, "Select CAD file", "", "CAD (*.dxf *.dwg);;DXF (*.dxf);;DWG (*.dwg)")
        if p:
            self.in_edit.setText(p)
            self.scan_layers()

    def browse_output(self):
        if self.driver.currentText() == "GPKG":
            p, _ = QFileDialog.getSaveFileName(self, "Output GeoPackage", "", "GeoPackage (*.gpkg)")
        else:
            p = QFileDialog.getExistingDirectory(self, "Output folder for Shapefiles")
        if p: self.out_edit.setText(p)

    def on_driver_changed(self):
        if self.driver.currentText() == "GPKG":
            self.out_label.setText("Output GeoPackage (GPKG)")
            self.btn_out.setText("Browse")
        else:
            self.out_label.setText("Output folder (SHP)")
            self.btn_out.setText("Select Folder")

    def log(self, msg):
        if msg:
            self.out_html.append(msg.replace("\n","<br>"))
            try:
            # 滾到最底，確保新訊息可見
                self.out_html.moveCursor(self.out_html.textCursor().End)
            except Exception:
                pass
        # ★ 關鍵：立即處理事件，讓 UI 不必等任務結束才重畫
            QCoreApplication.processEvents()

    def sync_layers_csv_from_preview(self):
        names = set()
        for i in range(self.layer_list.count()):
            it = self.layer_list.item(i)
            if it.checkState() == Qt.Checked or it.isSelected():
                names.add(it.text())
        self.layers_edit.setText(", ".join(sorted(names)))

    def select_all_layers(self):
        self.layer_list.blockSignals(True)
        for i in range(self.layer_list.count()):
            it = self.layer_list.item(i)
            it.setSelected(True)
            it.setCheckState(Qt.Checked)
        self.layer_list.blockSignals(False)
        self.sync_layers_csv_from_preview()

    def clear_layers_selection(self):
        self.layer_list.blockSignals(True)
        for i in range(self.layer_list.count()):
            it = self.layer_list.item(i)
            it.setSelected(False)
            it.setCheckState(Qt.Unchecked)
        self.layer_list.blockSignals(False)
        self.sync_layers_csv_from_preview()

    def scan_layers(self):
        self.layer_list.clear()
        cad_path = self.in_edit.text().strip()
        if not cad_path or not os.path.isfile(cad_path):
            self.log("<span style='color:#b00'>Please pick a valid DXF/DWG first.</span>")
            return
        # Try to prepare ezdxf; do not abort if unavailable
        ez_ok = _deps.ensure_ezdxf_safe(feedback=SimpleFeedback(self))

        temp_dir = None
        try:
            src_for_scan = cad_path
            if cad_path.lower().endswith(".dwg"):
                self.log("Converting DWG → temporary DXF for layer scan ...")
                src_for_scan = dwg_to_temp_dxf_auto(cad_path, prefer=self.dwg_pref.currentText(),
                                                    dxf_version=(self.dxf_version.text().strip() or "ACAD2013"))
                temp_dir = os.path.dirname(src_for_scan)

            names = []
            if ez_ok:
                try:
                    import ezdxf
                    self.log("<b>Reading layers via ezdxf ...</b>")
                    doc = ezdxf.readfile(src_for_scan)
                    names = sorted([str(t.dxf.name) for t in doc.layers])
                except Exception as e:
                    self.log(f"<i>ezdxf layer scan failed:</i> {e}")
            if not names:
                self.log("Trying OGR fallback ...")
                names = _ogr_list_layers(src_for_scan)

            if not names:
                self.log("<i>No layers found.</i>")
            else:
                for nm in names:
                    it = QListWidgetItem(nm); it.setCheckState(Qt.Unchecked)
                    self.layer_list.addItem(it)
                self.log(f"Found {len(names)} layer(s).")
        except Exception as e:
            tb = traceback.format_exc()
            self.log(f"<div style='color:#b00'><b>Layer scan failed:</b> {e}</div>")
            self.log(f"<pre>{tb}</pre>")
        finally:
            if temp_dir and os.path.isdir(temp_dir):
                shutil.rmtree(temp_dir, ignore_errors=True)

    def run_convert(self):
        
 
        try:
            self.btn_run.setEnabled(False)
            self.log("<b>Checking dependency: ezdxf</b>")
            ez_ok = _deps.ensure_ezdxf_safe(feedback=SimpleFeedback(self))

            from .services.conversion_service import precise_convert, write_outputs

            cad_path = self.in_edit.text().strip()
            if not cad_path or not os.path.isfile(cad_path):
                raise RuntimeError(f"File not found: {cad_path}")

            layers_csv = self.layers_edit.text().strip()
            target_layers = [s.strip() for s in layers_csv.split(',') if s.strip()] if layers_csv else []
            src_epsg = self._parse_epsg(self.src_epsg)
            tgt_epsg = self._parse_epsg(self.tgt_epsg)
            mode = self.block_mode.currentText()
            merge_tol = float(self.merge_tol.value())
            spline_tol = float(self.spline_tol.value())
            driver = self.driver.currentText()
            out_path = self.out_edit.text().strip()
            if not out_path:
                raise RuntimeError("Please specify output path.")
            dwg_pref = self.dwg_pref.currentText()
            dxf_version = self.dxf_version.text().strip() or "ACAD2013"
            overwrite = self.chk_overwrite.isChecked()
            do_load = self.chk_load.isChecked()

            input_for_convert = cad_path
            temp_dir = None
            if cad_path.lower().endswith(".dwg"):
                self.log("Converting DWG → temporary DXF ...")
                temp_dxf = dwg_to_temp_dxf_auto(cad_path, prefer=dwg_pref, dxf_version=dxf_version)
                input_for_convert = temp_dxf
                temp_dir = os.path.dirname(temp_dxf)

            try:
                self.log("<b>Running conversion ...</b>")
                buckets = precise_convert(
                    src_path=input_for_convert,                    # ← 傳字串，不要包 list
                    source_epsg=src_epsg,
                    target_epsg=tgt_epsg,
                    include_3d=False,
                    bbox_wgs84=None,
                    target_layers=(target_layers if target_layers else None),
                    block_mode=mode,
                    merge_tolerance=merge_tol,                     # ← 正確參數名
                    flat_dist_precise=spline_tol,
                    on_progress=lambda s: self.log(s or "")
                )

                # Optional augment (best with ezdxf; OGR fallback may skip)
                try:
                    if _augment and (self.chk_text_attrs.isChecked() or self.chk_block_attrs.isChecked()):
                        extras = _augment.collect_annotations_and_blocks(
                            input_for_convert,
                            src_epsg=src_epsg,
                            tgt_epsg=tgt_epsg,
                            include_text=self.chk_text_attrs.isChecked(),
                            include_blocks=self.chk_block_attrs.isChecked(),
                            keep_block_transform=self.chk_block_transform.isChecked(),
                            target_layers=target_layers,
                            on_progress=lambda s: self.log(s or ""),
                        )
                        if extras and isinstance(buckets, dict):
                            buckets.update(extras)
                except Exception as e_aug:
                    self.log(f"<i>augment skipped:</i> {e_aug}")

                self.log("<b>Writing outputs ...</b>")
                written = write_outputs(
                    buckets,
                    out_path=out_path,
                    driver="GPKG" if driver=="GPKG" else "ESRI Shapefile",
                    overwrite=overwrite,
                    on_progress=lambda s: self.log(s or "")
                )
            finally:
                if temp_dir and os.path.isdir(temp_dir):
                    shutil.rmtree(temp_dir, ignore_errors=True)

            # Auto load
            if do_load:
                for w in (written or []):
                    path = w.get("path"); name = w.get("layer")
                    uri = path if driver == "ESRI Shapefile" else f"{path}|layername={name}"
                    v = QgsVectorLayer(uri, name, "ogr")
                    if v and v.isValid():
                        QgsProject.instance().addMapLayer(v)

            html = "<h3>CAD to GIS Converter</h3><ul>" + "\n".join(
                [f"<li><b>{w.get('layer')}</b> ({w.get('count')}) → {w.get('path')}</li>" for w in (written or [])]
            ) + "</ul>"
            self.out_html.append(html)
            self.log("<b>Done.</b>")
        except Exception as e:
            tb = traceback.format_exc()
            self.out_html.append(f"<div style='color:#b00'><b>ERROR:</b> {e}</div><pre>{tb}</pre>")
        finally:
            self.btn_run.setEnabled(True)

class SimpleFeedback:
    def __init__(self, dlg: CadToGisDialog):
        self.dlg = dlg
    def pushInfo(self, s): self.dlg.log(s or "")
