# -*- coding: utf-8 -*-

"""
/***************************************************************************
polyg4qgis		A QGIS plugin Tools for managing polygon vector layers

                             -------------------
        begin                : 2012-08-20
        copyright            : (C) 2012 by giuliano curti
        email                : giulianc at tiscali dot it
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

# Import standard libraries
import os,math

# Import the PyQt and QGIS libraries
from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis.core import *
from qgis.gui import *

# Import the NumPy library
import numpy as np

# Initialize Qt resources from file resources.py
# import resources

# import related code 
# Set up current path.
currentPath = os.path.dirname( __file__ )

# Import specific libraries
from cad4qgis_core import *

from genericDlg import genericDlg
import matrixTransformation as trans2D
from polyg4qgisUI import polyg4qgisUI
from pLine4qgisPlineNavDlg import pLine4qgisPlineNavDlg
from polyg4qgisItemsSelectDlg import polyg4qgisItemsSelectDlg

"""
	Plugin for qGis to perform inserting/editing of polygons
	in OGR vector layers.
"""

class polyg4qgis:

	vers = '0.00'
	build_date = '2012-08-20'

	selectList = []	# lista dei selezionati
	markerList = []	# lista dei marcatori
	numPnts = 10		# alla fine gestire come in pLine4qgis
	helpFlag = False
	"""
		goFlag management:
		cLayer != '' -> goFlag=1000
		capabilities=ok -> goFlag=1100
		geometryType == 0 -> goFlag=1110
			inquiry functions allowed
		isEditable == True -> goFlag=1111;
			inserting/editing functions allowed
	"""
	goFlag = 0
	"""
		clickTool.canvasClicked.disconnect() [QgsMapToolEmitPoint()] seems raise error
		when activated without connections; we register in isClickToolActivated when
		clickTool is connected;
	"""
	isClickToolActivated = False

	def __init__(self, iface):
		# Save reference to the QGIS interface
		self.iface = iface
		# reference to map canvas
		self.canvas = self.iface.mapCanvas()
		# stabilisce la tolleranza di ricerca
		self.eps = self.canvas.mapUnitsPerPixel() * 5
		# out click tool will emit a QgsPoint on every click
		self.clickTool = QgsMapToolEmitPoint(self.canvas)
		# create our GUI dialog
		self.dlg = polyg4qgisUI()
		# controle status layer
		self.handleLayerChange()

	def initGui(self):
		# Create action that will start plugin configuration
		self.action = QAction(QIcon(":/plugins/cad4qgis/cad4qgis.png"), \
			"polyg4qgis", self.iface.mainWindow())
		# connect the action to the run method
		QObject.connect(self.action, SIGNAL("triggered()"), self.run)
		# Add toolbar button and menu item
		self.iface.addToolBarIcon(self.action)
		self.iface.addPluginToMenu("cad4qgis", self.action)

		# connect the button handler to signal   
		QObject.connect(self.dlg.sbNumPnts,SIGNAL("valueChanged(int)"),self.numPntsChanged)
		QObject.connect(self.dlg.sbToler,SIGNAL("valueChanged(int)"),self.toleranceVisualize)
		QObject.connect(self.dlg.pbInqPolyg,SIGNAL("clicked()"),self.polygInqTool)
		QObject.connect(self.dlg.pbPolygValidation,SIGNAL("clicked()"),self.polygValidationTool)
		QObject.connect(self.dlg.pbInqEdge,SIGNAL("clicked()"),self.polygEdgeInqTool)
		QObject.connect(self.dlg.pbInqVertex,SIGNAL("clicked()"),self.polygVertexInqTool)
		QObject.connect(self.dlg.pb2PntDist,SIGNAL("clicked()"),self.polyg2PntDistTool)
		QObject.connect(self.dlg.pbAngle3Pnt,SIGNAL("clicked()"),self.polygAngle3PntTool)
		QObject.connect(self.dlg.pbPolyCentroid,SIGNAL("clicked()"),self.polygCentroidInqTool)
		QObject.connect(self.dlg.pbPolyLength,SIGNAL("clicked()"),self.polyLengthTool)
		QObject.connect(self.dlg.pbPolyArea,SIGNAL("clicked()"),self.polyAreaTool)
		QObject.connect(self.dlg.pbPntsDiag,SIGNAL("clicked()"),self.pntDiagTool)
		QObject.connect(self.dlg.pbPolygCopy,SIGNAL("clicked()"),self.prova)
		QObject.connect(self.dlg.pbPolygRectSeries,SIGNAL("clicked()"),self.polygRectSeriesTool)
		QObject.connect(self.dlg.pbPolygPolarSeries,SIGNAL("clicked()"),self.polyPolarSeriesTool)
		QObject.connect(self.dlg.pbPolygAxialMirror,SIGNAL("clicked()"),self.polygAxialMirrorTool)
		QObject.connect(self.dlg.pbPolygRadialMirror,SIGNAL("clicked()"),self.polygRadialMirrorTool)
		QObject.connect(self.dlg.pbPolygRegPolyg,SIGNAL("clicked()"),self.polygRegPolygTool)
		QObject.connect(self.dlg.pbPolygCircle2P,SIGNAL("clicked()"),self.polygCircle2PTool)
		QObject.connect(self.dlg.pbPolygCircleInCbyP,SIGNAL("clicked()"),self.polygCircleInCbyPTool)
		QObject.connect(self.dlg.pbPolygCircle3P,SIGNAL("clicked()"),self.polygCircle3PTool)
		QObject.connect(self.dlg.pbPolygEllisse,SIGNAL("clicked()"),self.polygEllisseTool)
		QObject.connect(self.dlg.pbPolygEllisseByP,SIGNAL("clicked()"),self.polygEllisseByPTool)
		QObject.connect(self.dlg.pbPolygVrtsEditMouse,SIGNAL("clicked()"),self.polygVrtsEditMouse)
		QObject.connect(self.dlg.pbPolygVrtsEdit,SIGNAL("clicked()"),self.polygVrtsEditTool)
		QObject.connect(self.dlg.pbPolygFilletSingle,SIGNAL("clicked()"),self.polygFilletSingleTool)
		QObject.connect(self.dlg.pbPolygChamferSingle,SIGNAL("clicked()"),self.polygChamferSingleTool)
		QObject.connect(self.dlg.pbPolygChamfer,SIGNAL("clicked()"),self.polygChamferTool)
		QObject.connect(self.dlg.pbPolygOffset,SIGNAL("clicked()"),self.polygOffsetTool)
		QObject.connect(self.dlg.pbPolygBuffer,SIGNAL("clicked()"),self.polygBufferTool)
		QObject.connect(self.dlg.pbLineTranslation,SIGNAL("clicked()"),self.polygTranslationTool)
		QObject.connect(self.dlg.pbLineScaling,SIGNAL("clicked()"),self.polygScalingTool)
		QObject.connect(self.dlg.pbLineRotation,SIGNAL("clicked()"),self.polygRotationTool)
		QObject.connect(self.dlg.pbLineCollimation1P,SIGNAL("clicked()"),self.polygCollimation1PTool)
		QObject.connect(self.dlg.pbLineCollimation2P,SIGNAL("clicked()"),self.polygCollimation2PTool)
		QObject.connect(self.dlg.pbLineCollimationNP,SIGNAL("clicked()"),self.polygCollimationNPTool)
		QObject.connect(self.dlg.pbPolygDel,SIGNAL("clicked()"),self.polygDelTool)
		QObject.connect(self.dlg.buttonAbout,SIGNAL("clicked()"),self.about)
		QObject.connect(self.dlg.buttonInfo,SIGNAL("clicked()"),self.info)
		QObject.connect(self.dlg.buttonHelpFlag,SIGNAL("clicked()"),self.helpFlagOnOff)
		# connect the layer changed handler to a signal that the TOC layer has changed
		QObject.connect(self.iface,SIGNAL("currentLayerChanged(QgsMapLayer *)"), self.handleLayerChange)

	# run method that performs all the real work
	def run(self):
		# espone l'avviso all'utente
#		self.info()
		# make our clickTool the tool that we'll use for now
		self.canvas.setMapTool(self.clickTool)
		# show the dialog
		self.dlg.show()
		result = self.dlg.exec_()

	def unload(self):
		# try to disconnect all signals
		if self.isClickToolActivated:
			self.clickTool.canvasClicked.disconnect()
			self.isClickToolActivated = False
		# clean up
		cleanSelection(self)
		self.iface.currentLayerChanged.disconnect()
		# Remove the plugin menu item and icon
		self.iface.removePluginMenu("cad4qgis",self.action)
		self.iface.removeToolBarIcon(self.action)

	def about(self):
		about(self.iface.mainWindow())

	def info(self):
		info(self.iface.mainWindow())

	def helpFlagOnOff(self):
		"""
			(Dis)Attiva l'help contestuale
		"""
		if self.helpFlag:
			self.helpFlag = False
		else:
			self.helpFlag = True

	def handleLayerChange(self):
		"""
			Aggiorna il layer corrente, il provider e il crs e aggiorna il goFlag.
		"""
		self.goFlag = 0
		self.cLayer = ''
		layName = ''
		layEdit = ''
		prName = ''
		crsId = ''
		if self.canvas.currentLayer():
			self.cLayer = self.canvas.currentLayer()
			if self.cLayer:
				self.goFlag = 1000
				layName = self.cLayer.name()
				crs = self.cLayer.crs()
				self.canvas.mapRenderer().setDestinationCrs(crs)
				crsId = crs.authid()
				# stabilisce la tolleranza di ricerca
				self.eps = self.canvas.mapUnitsPerPixel() * 5
				# definisce il provider
				self.provider = self.cLayer.dataProvider()
				prName = self.provider.name()
				caps = self.provider.capabilities()
				if (caps & QgsVectorDataProvider.AddFeatures) and (caps & QgsVectorDataProvider.DeleteFeatures) and (caps & QgsVectorDataProvider.ChangeGeometries):
					self.goFlag = 1100
					if self.cLayer.geometryType() == QGis.Polygon:
						self.goFlag = 1110
						QObject.connect(
							self.cLayer,
							SIGNAL("editingStarted()"),
							self.handleLayerEditabilityStart
						)
						QObject.connect(
							self.cLayer,
							SIGNAL("editingStopped()"),
						self.handleLayerEditabilityStop
						)
						if self.cLayer.isEditable():
							layEdit = 'editable'
							self.goFlag = 1111
					else:
						QMessageBox.information(
							self.iface.mainWindow(),
							'handleLayerChange',
							"Sorry, I can't manipulate this type of layer (%d)" % (self.goFlag)
						)
#	qui bisognerebbe decidere, sull base del goFlag ottenuta, se connettere o
#  disconnettere, però forse andrebbe sconneso all'inizio il layer precedente
		# aggiorna il pannello di dialogo
		self.dlg.setLayer(layName,layEdit,prName,crsId,self.eps)

	def handleLayerEditabilityStart(self):
		"""
			Controlla l'inizio della editabilità del layer
		"""
		self.goFlag = 1111
		self.dlg.setLayerEdit('editable')

	def handleLayerEditabilityStop(self):
		"""
			Controlla la chiusura della editabilità del layer
		"""
		self.goFlag = 1110
		self.dlg.setLayerEdit('')
		# try to disconnect all signals
		if self.isClickToolActivated:
			self.clickTool.canvasClicked.disconnect()
			self.isClickToolActivated = False

	def warning(self,target):
		"""
			Visualizza l'errore;
			l'errore  segnalato solo nel caso < target, ad es. nel caso delle funzioni
			di inquiry il target è 110 (100 layer defined + 10 geometryType = POINT),
			pertanto la segnalazione circa la non editabilità del layer non verrà mai
			segnalata perchè goFlag == 110 eguaglia il target vnificando il test.
		"""
		if self.goFlag < target and self.goFlag < 1000:
			QMessageBox.information(
				self.iface.mainWindow(),
				"Warning",
				"The layer is not defined (%d)" % (self.goFlag)
			)
		elif self.goFlag < target and self.goFlag < 1100:
			QMessageBox.information(
				self.iface.mainWindow(),
				"Warning",
				"The provider has not this capability (%d)" % (self.goFlag)
			)
		elif self.goFlag < target and self.goFlag < 1110:
			QMessageBox.information(
				self.iface.mainWindow(),
				"Warning",
				"The layer is of incorrect type (%d)" % (self.goFlag)
			)
		elif self.goFlag < target and self.goFlag < 1111:
				QMessageBox.information(
					self.iface.mainWindow(),
					"Warning",
					"The layer is not editable (%d)" % (self.goFlag)
				)

	def numPntsChanged(self):
		self.numPnts = self.dlg.getNumPnts()

	def toleranceVisualize(self):
		"""
			Gestione del raggio di ricerca;
			il range è compreso fra 1/1000 e 1000 (i valori dello spinbox nella UI vanno
			da 1 a 1000000.
		"""
		tmp = self.dlg.getToler()
		self.eps = float(tmp/1000)
		ext = self.cLayer.extent()
		c = ext.center()
		x,y = c.x(),c.y()
		cleanSelection(self)	# cancella il precedente
		self.markerList.append(pntHighlight(self.canvas,x,y,self.eps))

# -------------------------- prova ------------------------

	def prova(self):
		QMessageBox.information(
			self.iface.mainWindow(),
			'polygInq',
			'Option not available'
		)
		return

		# try to disconnect all signals
		if self.isClickToolActivated:
			self.clickTool.canvasClicked.disconnect()
		cleanSelection(self)
		# connect to click signal
		QObject.connect(
			self.clickTool,
			SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
			self.prova2
		)
		self.isClickToolActivated = True

	def prova2(self,point):
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
 			self.markerList.append(polygHighlight(self.canvas,feat))
			msg = "Feature id=%d" % (feat.id())
			# legge la geometria
			geom = feat.geometry()
			msg = msg + " GEOSvalid=%r" % (geom.isGeosValid())
			err = geom.validateGeometry()
			msg = msg + " errori: %s" % (err)
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygInq',
				msg
			)
			cleanSelection(self)


# -------------------------- inquiry ------------------------

	def polygInqTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pLineInq',
				"Inquiry a pLine"
			)
			self.helpFlag = False
		if self.goFlag >= 1110:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.polygInq
			)
			self.isClickToolActivated = True
		else:
			self.warning(1110)

	def polygInq(self,point):
		"""
			Inquiry an pLine
		"""
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
 			self.markerList.append(polygHighlight(self.canvas,feat))
			msg = "Feature id=%d" % (feat.id())
			# legge la geometria
			geom = feat.geometry()
			msg = msg + " type=%s length=%f area=%f" % (geom.wkbType(),geom.length(),geom.area())	
			if geom.isMultipart:
				msg = msg + " multipart=SI"
			else:
				msg = msg + "multipart=NO"
			plgns = geom.asPolygon()
			msg = msg + " costituito da %d parti di" % (len(plgns))
			for p in plgns:
				msg = msg + " #vertices=%d" % (len(p))
			msg = msg + " GEOSvalid=%r" % (geom.isGeosValid())
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygInq',
				msg
			)
			cleanSelection(self)

	def polygValidationTool(self):
		pass
		# try to disconnect all signals
		if self.isClickToolActivated:
			self.clickTool.canvasClicked.disconnect()
		cleanSelection(self)
		# connect to click signal
		QObject.connect(
			self.clickTool,
			SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
			self.polygValidation
		)
		self.isClickToolActivated = True

	def polygValidation(self,point):
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
 			self.markerList.append(polygHighlight(self.canvas,feat))
			msg = "Feature id=%d" % (feat.id())
			# legge la geometria
			geom = feat.geometry()
			msg = msg + " GEOSvalid=%r" % (geom.isGeosValid())
			err = geom.validateGeometry()
			msg = msg + " errori: %s" % (err)
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygValidation',
				msg
			)
			cleanSelection(self)

# valutare un inquiry per le singole parti

	def polygEdgeInqTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygEdgeInq',
				"Inquiry an edge"
			)
			self.helpFlag = False
		if self.goFlag >= 1110:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.polygEdgeInq
			)
			self.isClickToolActivated = True
		else:
			self.warning(1110)

	def polygEdgeInq(self,point):
		searchPolygEdge(self,point)
		if len(self.selectList):
			eId,[p1,p2] = self.selectList.pop()
			d,a = edgeLengthAndInclin(p1,p2)
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygEdgeInq',
				'edge start=(%f %f) end=(%f %f) length=%f  inclination=%f' % (p1.x(),p1.y(),p2.x(),p2.y(),d,a)
			)
			cleanSelection(self)

	def polygVertexInqTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygVertexInq',
				"Inquiry a vertex of a polyg"
			)
			self.helpFlag = False
		if self.goFlag >= 1110:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.polygVertexInq
			)
			self.isClickToolActivated = True
		else:
			self.warning(1110)

	def polygVertexInq(self,point):
		"""
			Inquiry an polyg
		"""
		searchPolygVertex(self,point)
		if len(self.selectList):
			p = self.selectList.pop()
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygVertexInq',
				"X=%f Y=%f" % (p.x(),p.y())	
			)
			cleanSelection(self)

	def polyg2PntDistTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polyg2PntDist',
				"Distance between two points; press the Cancel button to finish"
			)
			self.helpFlag = False
		if self.goFlag >= 1110:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.polyg2PntDist
			)
			self.isClickToolActivated = True
		else:
			self.warning(1110)

	def polyg2PntDist(self,point):
		"""
			Calcola la distanza fra due punti
		"""
		searchPolygVertex(self,point)
		if len(self.selectList) >= 2:
			# esegue il calcolo
			v2 = self.selectList.pop()
			v1 = self.selectList.pop()
			d,a = edgeLengthAndInclin(v1,v2)
			x1,y1 = v1.x(),v1.y()
			x2,y2 = v2.x(),v2.y()
			self.markerList.append(edgeHighlight(self.canvas,v1,v2))
			QMessageBox.information(
				self.iface.mainWindow(),
				"twoPntsDist",
				"Distance=%12.3f Inclination=%f (dx=%12.3f dy=%12.3f)" % (d,a,x2-x1,y2-y1),
			)
			cleanSelection(self)

	def polygAngle3PntTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygAngle3Pnt',
				"Angle between 3 points"
			)
			self.helpFlag = False
		if self.goFlag >= 1110:
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			cleanSelection(self)
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.polygAngle3Pnt
			)
			self.isClickToolActivated = True
			# inizializza i registri
			self.rubBnd = QgsRubberBand(self.canvas)
			self.markerList.append(self.rubBnd)
		else:
			self.warning(1110)

	def polygAngle3Pnt(self,point):
		"""
			Calcola l'angolo fra tre vertici v1-v2-v3
		"""
		searchPolygVertex(self,point)
		# esegue il calcolo
		if len(self.selectList) >= 3:
			v3 = self.selectList.pop()
			self.rubBnd.addPoint(v3)
			v2 = self.selectList.pop()
			self.rubBnd.addPoint(v2)
			v1 = self.selectList.pop()
			self.rubBnd.addPoint(v1)
			a = angle3Pnt(v1,v2,v3)
			QMessageBox.information(
				self.iface.mainWindow(),
				"polygAngle3Pnt",
				"Angle %f" % (a)
			)
			cleanSelection(self)

	def polygCentroidInqTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygCentroidInq',
				"Inquiry the centroid of a polygon"
			)
			self.helpFlag = False
		if self.goFlag >= 1110:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.polygCentroidInq
			)
			self.isClickToolActivated = True
		else:
			self.warning(1110)

	def polygCentroidInq(self,point):
		"""
			Inquiry an pLine
		"""
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			c = geom.centroid().asPoint()
			x,y = c.x(),c.y()
 			m = pntHighlight(self.canvas,x,y)
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygCentroidInq',
				'X=%f Y=%f' % (x,y)
			)
			self.canvas.scene().removeItem(m)

	def polyLengthTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polyLength',
				"Partial and total length of a polyline; press the Cancel button to finish"
			)
			self.helpFlag = False
		if self.goFlag >= 1110:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.polyLength
			)
			self.isClickToolActivated = True
			# inizializza i registri
			self.lenTot = 0.0
			self.x1,self.y1 = 0.0,0.0
			self.Dx,self.Dy = 0.0,0.0
			self.counter = 0
			self.rubBnd = QgsRubberBand(self.canvas)
			self.markerList.append(self.rubBnd)
		else:
			self.warning(1110)

	def polyLength(self,point):
		"""
			Calcola la lunghezza di una polyline
		"""
		searchPolygVertex(self,point)
		if len(self.selectList):
			p = self.selectList.pop()
			self.rubBnd.addPoint(p)
			x2,y2 = p.x(),p.y()
			self.counter += 1
			if self.counter >= 2:
				dx,dy = x2-self.x1,y2-self.y1
				d = math.sqrt(dx**2+dy**2)
				self.lenTot = self.lenTot + d
				self.Dx = self.Dx + dx
				self.Dy = self.Dy + dy
				result = QMessageBox.question(
					self.iface.mainWindow(),
					"polyLength",
					"Last length:%12.3f - Total length:%12.3f (dx=%12.3f dy=%12.3f)" % (d,self.lenTot,self.Dx,self.Dy),
					1,
					2
				)
				if result == 2:
					# inizializza i registri
					self.lenTot = 0.0
					self.x1,self.y1 = 0.0,0.0
					self.Dx,self.Dy = 0.0,0.0
					self.counter = 0
					cleanSelection(self)
					self.rubBnd = QgsRubberBand(self.canvas)
					self.markerList.append(self.rubBnd)
		self.x1,self.y1 = x2,y2

	def polyAreaTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'PolyArea',
				"Partial and total area underlying a polyline (with closed contour get the polygon area"
			)
			self.helpFlag = False
		if self.goFlag >= 1110:
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			cleanSelection(self)
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.polyArea
			)
			self.isClickToolActivated = True
		else:
			self.warning(1110)

	def polyArea(self,point):
		"""
			Calcola l'area (proiettata nel piano XY) di un poligono
			con la formula dei trapezi.
		"""
		if len(self.selectList) == 0:
			self.rubBnd = self.edgeHighlightStart()
		searchFeat(self,point)
		feat = QgsFeature()
		while self.cLayer.nextFeature(feat):
			self.selectList.append(feat)
			id,x,y = pointCds(feat)
			self.markerList.append(pntHighlight(self.canvas,x,y))
			self.rubBnd.addPoint(QgsPoint(x,y))
			# esegue il calcolo
			if len(self.selectList) >= 2:
				p = self.selectList[-1]
				n1,x2,y2 = pointCds(p)
				p = self.selectList[-2]
				p1,x1,y1 = pointCds(p)
				area = (y1+y2)*(x2-x1)/2
				self.areaTot = self.areaTot + area
				result = QMessageBox.question(
					self.iface.mainWindow(),
					"polyArea",
					"Last area %f (total %f)" % (area,self.areaTot),
					1,
					2
				)
				if result == 2:
					# disconnect from click signal
					self.clickTool.canvasClicked.disconnect()
					self.isClickToolActivated = False
					cleanSelection(self)
					return
			else:
				# inizializza l'area
				self.areaTot = 0

	def pntDiagTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'pntDiag',
				"Polyline vertices diagnostic"
			)
			self.helpFlag = False
		if self.goFlag >= 1110:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.pntDiag
			)
			self.isClickToolActivated = True

	def pntDiag(self,point):
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			plgns = geom.asPolygon()
			# percorre ogni poligono
			for line in plgns:
				x0,y0 = line[0]
				# percorre il perimetro del poligono
				for k in range(1,len(line)):
					x1,y1 = line[k]			# passa in rassegna dall'1 in poi
					dx,dy = x1-x0,y1-y0
					d = math.sqrt(dx**2+dy**2)
					if d <= self.eps:
						self.markerList.append(pntHighlight(self.canvas,x1,y1))
						result = QMessageBox.question(
							self.iface.mainWindow(),
							'pntDiag',
							"feat=%d vertices %d (%f %f) and %d (%f %f) differ %f under the range %f" % (feat.id(),k-1,x0,y0,k,x1,y1,d,self.eps),
							1,
							2
						)
						if result == 2:
							cleanSelection(self)
							return
					x0,y0 = x1,y1
		cleanSelection(self)		

# ------------------ rect & polar series ------------------------

	def polygRectSeriesTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygRectSeries',
				"Creation of rectangular series of polyline"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			myDlg = genericDlg('pLineRectSeries',['# copies','Dx (m)','Dy (m)'])
			myDlg.setValues([0,0.0,0.0])
			myDlg.show()
			result = myDlg.exec_()
			if result:
				tmp = myDlg.getValues()
				self.Dy = float(tmp.pop())
				self.Dx = float(tmp.pop())
				self.numRep = int(tmp.pop())
				if self.numRep <= 0:
					QMessageBox.information(
						self.iface.mainWindow(),
						"polygRectSeries",
						"Warning: the number of copies cannot be null"
					)
					return
				if self.Dx == 0.0 and self.Dy == 0.0:
					QMessageBox.information(
					self.iface.mainWindow(),
						"polygRectSeries",
						"Warning: the vector cannot be null"
					)
					return
				# try to disconnect all signals
				if self.isClickToolActivated:
					self.clickTool.canvasClicked.disconnect()
				# connect to click signal
				QObject.connect(
					self.clickTool,
					SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
					self.polygRectSeries
				)
				self.isClickToolActivated = True
		else:
			self.warning(1111)

	def polygRectSeries(self,point):
		"""
			Rectangular series of polygons
		"""
		# legge il selezionato
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			plgns = geom.asPolygon()
			for i in range(self.numRep):
				# nuova geometria
				newPlgns = []
				for line in plgns:
					# costruisce il nuovo poligono
					newLine = []
					for v in line:
						newPoint = QgsPoint(v.x()+(i+1)*self.Dx,v.y()+(i+1)*self.Dy)
						newLine.append(newPoint)
					# aggiorn la geometria
					newPlgns.append(newLine)
				newFeat = QgsFeature()
#				newFeat.addAttribute(0,"hello")
				newFeat.setGeometry(QgsGeometry.fromPolygon(newPlgns))
				self.cLayer.addFeatures( [ newFeat ] )

	def polyPolarSeriesTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygPolarSeries',
				"Creation of polar series of polyline"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
#			selezione del centro di rotazione
			# selezione del centro di rotazione
			myDlg = polyg4qgisItemsSelectDlg(self,'vertex','Insert the symmetry center')
			myDlg.show()
			result = myDlg.exec_()
			if result:
				if len(self.selectList) >= 1:
					p = self.selectList.pop()
					x0,y0 = p.x(),p.y()
					pM = pntHighlight(self.canvas,x0,y0)
					# dialogo per parametri
					myDlg = genericDlg('polygPolarSeries',['# copies','angle (360)'])
					myDlg.setValues([0,0.0])
					myDlg.show()
					result = myDlg.exec_()
					if result:
						tmp = myDlg.getValues()
						angle = float(tmp.pop())
						self.numRep = int(tmp.pop())
						if self.numRep <= 0:
							QMessageBox.information(
								self.iface.mainWindow(),
								"polygPolarSeries",
								"Warning: the number of copies cannot be null"
							)
							return
						if angle == 0.0:
							QMessageBox.information(
								self.iface.mainWindow(),
								"polygPolarSeries",
								"Warning: the angle cannot be null"
							)
							return
						# prepara matrici di trasformazione
						da = trans2D.sessag2rad(angle/(self.numRep-1))
						mat1 = trans2D.translation(-x0,-y0)
						mat2 = trans2D.rotation(da)
						mat = np.dot(mat2,mat1)
						mat3 = trans2D.translation(x0,y0)
						self.mat = np.dot(mat3,mat)
						# try to disconnect all signals
						if self.isClickToolActivated:
							self.clickTool.canvasClicked.disconnect()
						# connect to click signal
						QObject.connect(
							self.clickTool,
							SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
							self.polygPolarSeries
						)
						self.isClickToolActivated = True
		else:
			self.warning(1111)

	def polygPolarSeries(self,point):
		"""
			Polar series of polygons
		"""
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			plgns = geom.asPolygon()
			# eseguo ciclo
			for i in range(self.numRep):
				# nuova geometria
				for pId,line in enumerate(plgns):
					# trasforma i vertici della polilinea
					for vId,v in enumerate(line):
						x,y,w = np.dot(self.mat,[v.x(),v.y(),1])
						line[vId] = QgsPoint(x,y)	# deve fare l'aggiornamento sul posto
					# aggiorn la geometria
					plgns[pId] = line		# deve fare l'aggiornamento sul posto
				newFeat = QgsFeature()
#				newFeat.addAttribute(0,"hello")
				newFeat.setGeometry(QgsGeometry.fromPolygon(plgns))
				self.cLayer.addFeatures( [ newFeat ] )

# ------------------ axial & radial mirror ------------------------

	def polygAxialMirrorTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygAxialMirror',
				"Creation of axial mirroring of polygons"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			cleanSelection(self)
			# selezione dell'asse di simmetria
			myDlg = polyg4qgisItemsSelectDlg(self,'vertex','Insert the symmetry axys')
			myDlg.show()
			result = myDlg.exec_()
			if result:
				if len(self.selectList) >= 2:
					# preleva l'asse di simmetria
					p2 = self.selectList.pop()
					p1 = self.selectList.pop()
					self.markerList.append(edgeHighlight(self.canvas,p1,p2))
					self.x1,self.y1 = p1.x(),p1.y()	# servono anche nella routine collegata
					x2,y2 = p2.x(),p2.y()
					# calcolo i coefficienti
					ux,uy = x2-self.x1,y2-self.y1
					mu = math.sqrt(ux**2+uy**2)
					# print 'ux,uy,mu',ux,uy,mu
					self.ux,self.uy = ux/mu,uy/mu
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.polygAxialMirror
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"poligAxialMirror",
						"Axys invalid"
					)
		else:
			self.warning(1111)

	def polygAxialMirror(self,point):
		"""
			Axial mirroring of polygons
		"""
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			plgns = geom.asPolygon()
			# nuova geometria
			newPlgns = []
			for line in plgns:
				# costruisce il nuovo poligono
				newLine = []
				for v in line:
					xi,yi = v.x(),v.y()
					tx,ty = xi-self.x1,yi-self.y1
					tu = tx*self.ux + ty*self.uy
					PQx,PQy = 2*(tx-tu*self.ux),2*(ty-tu*self.uy)
					Qx,Qy = xi-PQx,yi-PQy
					newPoint = QgsPoint(Qx,Qy)
					newLine.append(newPoint)
				# aggiorn la geometria
				newPlgns.append(newLine)
			newFeat = QgsFeature()
#			newFeat.addAttribute(0,"hello")
			newFeat.setGeometry(QgsGeometry.fromPolygon(newPlgns))
			self.cLayer.addFeatures( [ newFeat ] )

	def polygRadialMirrorTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygRadialMirror',
				"Creation of radial mirroring of polygons"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			cleanSelection(self)
			# selezione dell'asse di simmetria
			myDlg = polyg4qgisItemsSelectDlg(self,'vertex','Insert the symmetry center')
			myDlg.show()
			result = myDlg.exec_()
			if result:
				# preleva centro di simmetria
				if len(self.selectList) >= 1:
					# preleva l'asse di simmetria
					p = self.selectList.pop()
					self.x0,self.y0 = p.x(),p.y()	# servono anche nell routine collegata
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.polygRadialMirror
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"polygRadialMirror",
						"Center invalid"
					)
		else:
			self.warning(1111)

	def polygRadialMirror(self,point):
		"""
			Radial mirroring of polygons
		"""
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			plgns = geom.asPolygon()
			# nuova geometria
			newPlgns = []
			for line in plgns:
				# costruisce il nuovo poligono
				newLine = []
				for v in line:
					newPoint = QgsPoint(2*self.x0-v.x(),2*self.y0-v.y())
					newLine.append(newPoint)
				# aggiorn la geometria
				newPlgns.append(newLine)
			newFeat = QgsFeature()
#			newFeat.addAttribute(0,"hello")
			newFeat.setGeometry(QgsGeometry.fromPolygon(newPlgns))
			self.cLayer.addFeatures( [ newFeat ] )

# ------------ quadratic interpolation points (circle) ------------------

	def polygRegPolygTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygRegPolyg',
				"Polygon of N vertices with center C."
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			# definisce i parametri
			dlg = genericDlg('polygRegPolyg',['radius (m)'])
			dlg.setValues([0.0])
			dlg.show()
			result = dlg.exec_()
			if result:
				tmp = dlg.getValues()
				self.rad = float(tmp.pop())
				if self.rad > 0:
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					cleanSelection(self)
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.polygRegPolyg
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"polygRegPolyg",
						"The radius must be greater than zero"
					)
		else:
			self.warning(1111)

	def polygRegPolyg(self,point):
		"""
			Calcola un poligono regolare di n lati, con centro p e raggio r.
		"""
		searchPolygVertex(self,point)
		if len(self.selectList):
			p = self.selectList.pop()
			newLine = cerchio(p,self.rad,self.numPnts)	
			newPlgns = [newLine]
			feat = QgsFeature()
#			feat.addAttribute(0,"hello")
			feat.setGeometry(QgsGeometry.fromPolygon(newPlgns))
			self.cLayer.addFeatures( [ feat ] )

	def polygCircle2PTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'PntCircle2P',
				"Circle with diameter in N1 and N2"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			cleanSelection(self)
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.polygCircle2P
			)
			self.isClickToolActivated = True
		else:
			self.warning(1111)

	def polygCircle2P(self,point):
		"""
			Disegna il cerchio per 2 punti
		"""
		searchPolygVertex(self,point)
		if len(self.selectList) >= 2:
			p2 = self.selectList.pop()
			p1 = self.selectList.pop()
			newLine = cerchio2P(p1,p2,self.numPnts)
			if len(newLine) > 0:
				newPlgns = [newLine]
				feat = QgsFeature()
#				feat.addAttribute(0,"hello")
				feat.setGeometry(QgsGeometry.fromPolygon(newPlgns))
				self.cLayer.addFeatures( [ feat ] )
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					"polygCircle2P",
					"Oops, the circle cannot be calculated: let controle the diameter vertices"
				)
			cleanSelection(self)

	def polygCircleInCbyPTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygCircleInCbyP',
				"Circle with center C by P"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			cleanSelection(self)
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.polygCircleInCbyP
			)
			self.isClickToolActivated = True
		else:
			self.warning(1111)

	def polygCircleInCbyP(self,point):
		"""
			Disegna il cerchio per con centro C passante per P
		"""
		searchPolygVertex(self,point)
		if len(self.selectList) >= 2:
			p2 = self.selectList.pop()
			p1 = self.selectList.pop()
			newLine = cerchioInCperP(p1,p2,self.numPnts)	
			if len(newLine) > 0:
				newPlgns = [newLine]
				feat = QgsFeature()
#				feat.addAttribute(0,"hello")
				feat.setGeometry(QgsGeometry.fromPolygon(newPlgns))
				self.cLayer.addFeatures( [ feat ] )
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					"polygCircleInCbyP",
					"Oops, the circle cannot be calculated: let controle the points"
				)
			cleanSelection(self)

	def polygCircle3PTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygCircle3P',
				"Circle by 3 points"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			cleanSelection(self)
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.polygCircle3P
			)
			self.isClickToolActivated = True
		else:
			self.warning(1111)

	def polygCircle3P(self,point):
		"""
			Disegna il cerchio per con centro C passante per P
		"""
		searchPolygVertex(self,point)
		if len(self.selectList) >= 3:
			p3 = self.selectList.pop()
			p2 = self.selectList.pop()
			p1 = self.selectList.pop()
			newLine = cerchio3P(p1,p2,p3,self.numPnts)
			if len(newLine) > 0:
				newPlgns = [newLine]
				feat = QgsFeature()
#				feat.addAttribute(0,"hello")
				feat.setGeometry(QgsGeometry.fromPolygon(newPlgns))
				self.cLayer.addFeatures( [ feat ] )
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					"polygCircle3P",
					"Oops, the circle cannot be calculated: let controle the points"
				)
			cleanSelection(self)

# ------------ quadratic interpolation points (ellipsys) ------------------

	def polygEllisseTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygEllisse',
				"Ellipsis with foci F1 e F2 and sum of distances D"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			# definisce i parametri
			dlg = genericDlg('polygEllisse',['sum of distances'])
			dlg.setValues([0.0,10])
			dlg.show()
			result = dlg.exec_()
			if result:
				tmp = dlg.getValues()
				self.dist = float(tmp.pop())
				if self.dist > 0:
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					cleanSelection(self)
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.polygEllisse
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"polygEllisse",
						"sum of distances must be greater than zero"
					)
		else:
			self.warning(1111)

	def polygEllisse(self,point):
		"""
			Calcola l'ellisse con fuochi in n1,n2 e somma delle distanza pari a d
			per un migliore rappresentazione il dominio X è diviso in n-5 tratti
			e 4 punti sono sollocati a
				-d/2+dx/4
				-d/2+dx/3
				d/2-dx/3
				d/2-dx/4.
		"""
		searchPolygVertex(self,point)
		if len(self.selectList) >= 2:
			p2 = self.selectList.pop()
			p1 = self.selectList.pop()
			newLine = ellisse(p1,p2,self.dist,self.numPnts)
			if len(newLine) > 0:
				newPlgns = [newLine]
				feat = QgsFeature()
#				feat.addAttribute(0,"hello")
				feat.setGeometry(QgsGeometry.fromPolygon(newPlgns))
				self.cLayer.addFeatures( [ feat ] )
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					"polygEllisse",
					"Oops, the ellipsys cannot be calculated: let controle the points"
				)
			cleanSelection(self)

	def polygEllisseByPTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygEllisseByP',
				"Ellipsis with foci F1 e F2 and sum of distances D"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			cleanSelection(self)
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.polygEllisseByP
			)
			self.isClickToolActivated = True
		else:
			self.warning(1111)

	def polygEllisseByP(self,point):
		"""
			Calcola l'ellisse con fuochi in f1,f2 passante per P.
		"""
		searchPolygVertex(self,point)
		if len(self.selectList) >= 3:
			p3 = self.selectList.pop()
			p2 = self.selectList.pop()
			p1 = self.selectList.pop()
			newLine = ellisseByP(p1,p2,p3,self.numPnts)
			if len(newLine) > 0:
				newPlgns = [newLine]
				feat = QgsFeature()
#				feat.addAttribute(0,"hello")
				feat.setGeometry(QgsGeometry.fromPolygon(newPlgns))
				self.cLayer.addFeatures( [ feat ] )
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					"polygEllisseByP",
					"Oops, the ellipsys cannot be calculated: let controle the points"
				)
			cleanSelection(self)

# ------------------ editing polygons ------------------------

	def polygVrtsEditMouse(self):
		QMessageBox.information(
			self.iface.mainWindow(),
			'polygeVrtsEditMouse',
			"For analogic editing of polygon vertices please use the native command of qGis"
		)

	def polygVrtsEditTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygVrtsEdit',
				"Polyline vertices editing"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.polygVrtsEdit
			)
			self.isClickToolActivated = True
		else:
			self.warning(1111)

	def polygVrtsEdit(self,point):
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			isChanged = False
			# legge la geometria
			geom = feat.geometry()
			plgns = geom.asPolygon()
			for id,line in enumerate(plgns):
				title = "polygId %d partId %d" % (feat.id(),id)
				myDlg = pLine4qgisPlineNavDlg(self,title,line)
				myDlg.show()
				result = myDlg.exec_()	# serve a fermare la finestra di dialogo
				if result:
					plgns[id] = myDlg.line
					isChanged = True
			if isChanged:
				self.cLayer.changeGeometry(feat.id(),QgsGeometry.fromPolygon(plgns))
				self.canvas.refresh()
		cleanSelection(self)

# ------------------ fillet & chamfer ------------------------

	def polygFilletSingleTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygFilletSingle',
				"Filleting a polygon vertex"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			myDlg = genericDlg('polygFilletSingle',['radius'])
			myDlg.setValues([0.0])
			myDlg.show()
			result = myDlg.exec_()
			if result:
				tmp = myDlg.getValues()
				self.rad = float(tmp[0])
				if self.rad > 0.0:
					cleanSelection(self)
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.polygFilletSingle
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"pPolygFilletSingle",
						"Warning: the radius cannot be null"
					)
		else:
			self.warning(1111)

	def polygFilletSingle(self,point):
		"""
			Esegue il fillet di un vertice di poligono;
			meglio lasciare queste istruzioni al posto di searchPolygVerte() perchè
			occorre comunque leggete i vertici e le coordinate adiacenti;
		"""
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			# cerca il vertice
			tmp = geom.closestVertex(point)
			if len(tmp):
				p2,v2id,v1id,v3id,d = tmp
				# legge coordinate
				p1 = geom.vertexAt(v1id)
				x1,y1 = p1.x(),p1.y()
				x2,y2 = p2.x(),p2.y()
				p3 = geom.vertexAt(v3id)
				x3,y3 = p3.x(),p3.y()
				# vettore 1-2
				ux,uy = x2-x1,y2-y1
				mu = math.sqrt(ux**2+uy**2)
				a = math.atan2(uy,ux) - math.pi	# viene misurato nel verso opposto
				# vettore 2-3
				vx,vy = x3-x2,y3-y2
				mv = math.sqrt(vx**2+vy**2)
				# angolo compreso
				b = math.atan2(vy,vx)
				g = b-a
				g = g - math.floor(g/(2*math.pi))*2*math.pi	# normalizza l'angolo
				# distanza dal vertice comune
				d = abs(self.rad/math.tan(g/2))
				# calcolo vertice p21
				x21,y21 = x2-d*ux/mu,y2-d*uy/mu
				p21 = QgsPoint(x21,y21)
				# calcolo vertice p23
				x23,y23 = x2+d*vx/mv,y2+d*vy/mv
				p23 = QgsPoint(x23,y23)
				# punto medio
				mx,my = (x21+x23)/2,(y21+y23)/2
				tx,ty = mx-x2,my-y2
				mt = math.sqrt(tx**2+ty**2)
				"""
					se l'angolo differenza mod360 è inferiore a 180^ il centro del fillet
					sta a dx del primo vettore e l'arco va in senso orario dal secondo
					vettore al primo; e invece l'angolo differenza è maggiore di 180, il
					centro sta a sx e l'arco va in senso orario dal primo vettore al
					secondo; mentre il centro viene localizzato automaticamente, la partenza
					ed arrivo dell'aro vanno settati manualmente (da controllare);
				"""
				# 	centro
				s = math.sqrt(self.rad**2+d**2)
				cx,cy = x2+s*tx/mt,y2+s*ty/mt
				# creo arco p21-p23
				if g < math.pi:
					newList = arc(QgsPoint(cx,cy),p23,p21,self.numPnts) # CONTROLLARE BENE!!!!
					flagReverse = True	# per ricordare che l'arco va poi rovesciato
				else:
					newList = arc(QgsPoint(cx,cy),p21,p23,self.numPnts)
					flagReverse = False
				# sposta vertice v2 in v21
				geom.moveVertex(x21,y21,v2id)
				# inserisco il vertice 2-3
				geom.insertVertex(x23,y23,v2id+1)
				# inserisce i punti dell'arco
				newList.pop(0)	# toglie il primo e....
				newList.pop()	# l'ultimo che sono doppioni
				if flagReverse:
					newList.reverse()
				pos = v2id+1
				for p in newList:
					geom.insertVertex(p.x(),p.y(),pos)
					pos += 1
				# sistema la geometria e la feature
				self.cLayer.changeGeometry(feat.id(),geom)
				self.canvas.refresh()
			cleanSelection(self)

	def polygChamferSingleTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygChamferSingle',
				"Chamfering a polygon vertex"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			myDlg = genericDlg('polygChamferSingle',['dimension'])
			myDlg.setValues([0.0])
			myDlg.show()
			result = myDlg.exec_()
			if result:
				tmp = myDlg.getValues()
				self.cFer = float(tmp[0])
				if self.cFer > 0.0:
					cleanSelection(self)
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.polygChamferSingle
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"polygChamferSingle",
						"Warning: the chamfer cannot be null"
					)
		else:
			self.warning(1111)

	def polygChamferSingle(self,point):
		"""
			polygon (single) vertex chamfer;
			vle lo stesso discorso di polygFilletSingle()
		"""
		# legge il selezionato
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			# cerca il vertice
			tmp = geom.closestVertex(point)
			if len(tmp):
				p2,v2id,v1id,v3id,d = tmp
				# legge coordinate
				p1 = geom.vertexAt(v1id)
				x1,y1 = p1.x(),p1.y()
				x2,y2 = p2.x(),p2.y()
				p3 = geom.vertexAt(v3id)
				x3,y3 = p3.x(),p3.y()
				# calcolo vertice 2-1
				dx,dy = x2-x1,y2-y1
				d = math.sqrt(dx**2+dy**2)
				x21,y21 = x2-self.cFer*dx/d,y2-self.cFer*dy/d
				# calcolo vertice 2-3
				dx,dy = x3-x2,y3-y2
				d = math.sqrt(dx**2+dy**2)
				x23,y23 = x2+self.cFer*dx/d,y2+self.cFer*dy/d
				# aggiorno vertice 2 (-> 2-1)
				geom.moveVertex(x21,y21,v2id)
				# inserisco il vertice 2-3
				geom.insertVertex(x23,y23,v2id+1)
				# aggiorna la geometria
				self.cLayer.changeGeometry(feat.id(),geom)
				self.canvas.refresh()


# (polygFillet)


	def polygChamferTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygChamfer',
				"Chamfering a polygon"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			myDlg = genericDlg('polygChamfer',['dimension'])
			myDlg.setValues([0.0])
			myDlg.show()
			result = myDlg.exec_()
			if result:
				tmp = myDlg.getValues()
				self.cFer = float(tmp[0])
				if self.cFer > 0.0:
					cleanSelection(self)
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.polygChamfer
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"polygChamfer",
						"Warning: the chamfer cannot be null"
					)
		else:
			self.warning(1111)

	def polygChamfer(self,point):
		"""
			polygon chamfer
		"""
		# legge il selezionato
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			plgns = geom.asPolygon()
			# nuova geometria
			newPlgns = []
			for line in plgns:
				# inizializzo il nuovo poligono
				newLine = []
				p1 = line.pop(0)
				x1,y1 = p1.x(),p1.y()
				# leggo secondo vertice
				p2 = line.pop(0)
				x2,y2 = p2.x(),p2.y()
				# sposto il primo vertice
				dx,dy = x2-x1,y2-y1
				d = math.sqrt(dx**2+dy**2)
				x12,y12 = x1+self.cFer*dx/d,y1+self.cFer*dy/d
				newLine.append(QgsPoint(x12,y12))
				for p3 in line:
					# calcolo vertice p21
					x21,y21 = x2-self.cFer*dx/d,y2-self.cFer*dy/d
					# creo segmento p1-p21
					newLine.append(QgsPoint(x21,y21))
					# calcolo vertice p23
					x3,y3 = p3.x(),p3.y()
					dx,dy = x3-x2,y3-y2
					d = math.sqrt(dx**2+dy**2)
					x23,y23 = x2+self.cFer*dx/d,y2+self.cFer*dy/d
					# creo lato p21-p23
					newLine.append(QgsPoint(x23,y23))
					# aggiorno i vertici
					x1,y1 = x23,y23
					x2,y2 = x3,y3
				# completo la plinea
				x32,y32 = x3-self.cFer*dx/d,y3-self.cFer*dy/d
				newLine.append(QgsPoint(x32,y32))
				# chiude sul primo vertice
				newLine.append(QgsPoint(x12,y12))
				# aggiorn la geometria
				newPlgns.append(newLine)
			self.cLayer.changeGeometry(feat.id(),QgsGeometry.fromPolygon(newPlgns))
			self.canvas.refresh()

# ------------------ line offset & buffer ------------------------

	def polygOffsetTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygOffset',
				"Polygon offset"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			myDlg = genericDlg('polygOffset',['offset'])
			myDlg.setValues([0.0])
			myDlg.show()
			result = myDlg.exec_()
			if result:
				tmp = myDlg.getValues()
				self.offset = float(tmp.pop())
				if self.offset != 0:
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.polygOffset
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						'polygOffset',
						"The offset cannot be null"
					)
		else:
			self.warning(1111)

	def polygOffset(self,point):
		"""
			Calcola l'offset di un poligono; l'originale viene mantenuto.
			Attenzione: è indifeso rispetto al primo punto doppio.
			Questa routine è difficoltosa per la scelta del centro del fillet
			a seconda dell giacitura dei lati: VALUTARE BENE PRIMA DI APPORTARE
			MODIFICHE.
		"""
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			plgns = geom.asPolygon()
			# nuova geometria
			newPlgns = []
			for line in plgns:
				# costruisce il nuovo poligono
				newLine = []
				p1 = line.pop(0)
				x1,y1 = p1.x(),p1.y()
				p2 = line.pop(0)
				x2,y2 = p2.x(),p2.y()
				ux,uy = x2-x1,y2-y1
				mu = math.sqrt(ux**2+uy**2)
				# creo punto 12
				x12,y12 = x1-self.offset*uy/mu,y1+self.offset*ux/mu
				pnt12 = QgsPoint(x12,y12)
				newLine.append(pnt12)
				# creo punto 21
				x21,y21 = x2-self.offset*uy/mu,y2+self.offset*ux/mu
				pnt21 = QgsPoint(x21,y21)
				# eseguo ciclo
				for p3 in line:
					# leggo vertice seguente
					x3,y3 = p3.x(),p3.y()
					ux,uy = x3-x2,y3-y2
					mu = math.sqrt(ux**2+uy**2)
					if mu:
						# creo punto 23
						x23,y23 = x2-self.offset*uy/mu,y2+self.offset*ux/mu
						pnt23 = QgsPoint(x23,y23)
						# creo punto 32
						x32,y32 = x3-self.offset*uy/mu,y3+self.offset*ux/mu
						pnt32 = QgsPoint(x32,y32)
						# controlliamo se si incontrano
						tmp = intersection1(pnt12,pnt21,pnt23,pnt32)
						if tmp != -1:
							x,y = tmp	# intersezione valida
							pnt23 = QgsPoint(x,y)
							newLine.append(pnt23)
							pnt21 = pnt23
						else:
							# creo arco p21-p23
							newList = arc(QgsPoint(x2,y2),pnt21,pnt23,self.numPnts)
							newLine.extend(newList)
					# aggiorno i vertici
					x1,y1 = x2,y2
					x2,y2 = x3,y3
					pnt12 = pnt23
					pnt21 = pnt32
				# completo la plinea	(QUESTO VA LLA FINE TRASLATO)
				x32,y32 = x3-self.offset*uy/mu,y3+self.offset*ux/mu
				pnt32 = QgsPoint(x32,y32)
				newLine.append(pnt32)
				newPlgns.append(newLine)
			# aggiungo la feature
			newFeat = QgsFeature()
#			newFeat.addAttribute(0,"hello")
			newFeat.setGeometry(QgsGeometry.fromPolygon(newPlgns))
			self.cLayer.addFeatures( [ newFeat ] )

	def polygBufferTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygBuffer',
				"Polygon Buffer"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			myDlg = genericDlg('polygBuffer',['buffer'])
			myDlg.setValues([0.0])
			myDlg.show()
			result = myDlg.exec_()
			if result:
				tmp = myDlg.getValues()
				self.buff = float(tmp.pop())
				if self.buff != 0:
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.polygBuffer
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						'polygBuffer',
						"The buffer cannot be null"
					)
		else:
			self.warning(1111)

	def polygBuffer(self,point):
		"""
			Calcola il buffer di un poligono usando la funzione nativa
		"""
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			newGeom = geom.buffer(self.buff,0)
			# aggiungo la feature
			newFeat = QgsFeature()
#			newFeat.addAttribute(0,"hello")
			newFeat.setGeometry(newGeom)
			self.cLayer.addFeatures( [ newFeat ] )

# ------------------ transformations ------------------------

	def polygTranslationTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygTranslation',
				"Polygon translation by a vector"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			myDlg = genericDlg('polygTranslation',['Dx (m)','Dy (m)'])
			myDlg.setValues([0.0,0.0])
			myDlg.show()
			result = myDlg.exec_()
			if result:
				tmp = myDlg.getValues()
				self.Dy = float(tmp.pop())
				self.Dx = float(tmp.pop())
				if self.Dx == 0.0 and self.Dy == 0.0:
					QMessageBox.information(
						self.iface.mainWindow(),
						"pLineTranslation",
						"Warning: the vector cannot be null"
					)
				else:
					# disconnect from click signal
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
						self.isClickToolActivated = False
			 		cleanSelection(self)
					# attiva il MapClicktTool
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.polygTranslation
					)
					self.isClickToolActivated = True
		else:
			self.warning(1111)

	def polygTranslation(self,point):
		# legge il selezionato
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			plgns = geom.asPolygon()
			# nuova geometria
			newPlgns = []
			for line in plgns:
				# costruisce il nuovo poligono
				newLine = []
				for v in line:
					newPoint = QgsPoint(v.x()+self.Dx,v.y()+self.Dy)
					newLine.append(newPoint)
				# aggiorn la geometria
				newPlgns.append(newLine)
			self.cLayer.changeGeometry(feat.id(),QgsGeometry.fromPolygon(newPlgns))
			self.canvas.refresh()

	def polygScalingTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygScaling',
				"polyg anysotropic scaling"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			# selezione del centro di rotazione
			myDlg = polyg4qgisItemsSelectDlg(self,'vertex','Insert the symmetry center')
			myDlg.show()
			result = myDlg.exec_()
			if result:
				if len(self.selectList) >= 1:
					p = self.selectList.pop()
					self.x0,self.y0 = p.x(),p.y()
					pM = pntHighlight(self.canvas,self.x0,self.y0)
					# dialogo per paramentri
					myDlg = genericDlg('polygScaling',['Sx','Sy'])
					myDlg.setValues([0.0,0.0])
					myDlg.show()
					result = myDlg.exec_()
			 		cleanSelection(self)
					if result:
						tmp = myDlg.getValues()
						self.Sy = float(tmp.pop())
						self.Sx = float(tmp.pop())
						if self.Sx != 0.0 and self.Sy != 0.0:
							# disconnect from click signal
							if self.isClickToolActivated:
								self.clickTool.canvasClicked.disconnect()
							# attiva il MapClicktTool
							QObject.connect(
								self.clickTool,
								SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
								self.polygScaling
							)
							self.isClickToolActivated = True
						else:
							QMessageBox.information(
							self.iface.mainWindow(),
								"polygScaling",
								"Warning: the scaling factors cannot be null"
							)
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"polygScaling",
						"Please, give me a vertex"
					)
		else:
			self.warning(1111)

	def polygScaling(self,point):
		"""
			polyg scaling
		"""
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			plgns = geom.asPolygon()
			# nuova geometria
			newPlgns = []
			for line in plgns:
				# costruisce il nuovo poligono
				newLine = []
				for v in line:
					x,y = self.x0+(v.x()-self.x0)*self.Sx, self.y0+(v.y()-self.y0)*self.Sy	# uso forma semplificata
					newPoint = QgsPoint(x,y)
					newLine.append(newPoint)
				# aggiorn la geometria
				newPlgns.append(newLine)
			self.cLayer.changeGeometry(feat.id(),QgsGeometry.fromPolygon(newPlgns))
			self.canvas.refresh()

	def polygRotationTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygRotation',
				"polyg rotation"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			# selezione del centro di rotazione
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			myDlg = polyg4qgisItemsSelectDlg(self,'vertex','Insert the rotation center')
			myDlg.show()
			result = myDlg.exec_()
			if result:
				if len(self.selectList) >= 1:
					p = self.selectList.pop()
					x0,y0 = p.x(),p.y()
					pM = pntHighlight(self.canvas,x0,y0)
					# dialogo per paramentri
					myDlg = genericDlg('polygRotation',['angle'])
					myDlg.setValues([0.0])
					myDlg.show()
					result = myDlg.exec_()
			 		cleanSelection(self)
					if result:
						tmp = myDlg.getValues()
						angle = float(tmp.pop())
						if angle != 0.0:
							# prepara matrici di trasformazione
							da = trans2D.sessag2rad(angle)	# converte in radianti
							mat1 = trans2D.translation(-x0,-y0)
							mat2 = trans2D.rotation(da)
							mat = np.dot(mat2,mat1)
							mat3 = trans2D.translation(x0,y0)
							self.mat = np.dot(mat3,mat)
							# disconnect from click signal
							if self.isClickToolActivated:
								self.clickTool.canvasClicked.disconnect()
								self.isClickToolActivated = False
							# attiva il MapClicktTool
							QObject.connect(
								self.clickTool,
								SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
								self.polygRotation
							)
							self.isClickToolActivated = True
						else:
							QMessageBox.information(
							self.iface.mainWindow(),
								"polygRotation",
								"Warning: the rotation angle cannot be null"
							)
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"polygScaling",
						"Please, give me a vertex"
					)
		else:
			self.warning(1111)

	def polygRotation(self,point):
		"""
			polyg rotation
		"""
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			plgns = geom.asPolygon()
			# nuova geometria
			newPlgns = []
			for line in plgns:
				# costruisce il nuovo poligono
				newLine = []
				for v in line:
					x,y,w = np.dot(self.mat,[v.x(),v.y(),1])
					newPoint = QgsPoint(x,y)
					newLine.append(newPoint)
				# aggiorn la geometria
				newPlgns.append(newLine)
			self.cLayer.changeGeometry(feat.id(),QgsGeometry.fromPolygon(newPlgns))
			self.canvas.refresh()

# ------------------ collimations ------------------------

	def polygCollimation1PTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygCollimation1P',
				"Aligns a number of olygs by making a matching pair of points"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
#			selezione della coppia di collimazione
			myDlg = polyg4qgisItemsSelectDlg(self,'vertex','Insert the matching pair')
			myDlg.show()
			result = myDlg.exec_()
			if result:
				if len(self.selectList) >= 2:
					# preleva la coppia di punti
					p2 = self.selectList.pop()
					p1 = self.selectList.pop()
					x1,y1 = p1.x(),p1.y()
					x2,y2 = p2.x(),p2.y()
					self.markerList.append(edgeHighlight(self.canvas,QgsPoint(x1,y1),QgsPoint(x2,y2)))
					# calcola lo spostamento
					self.dx,self.dy = x2-x1,y2-y1
					# try to disconnect all signals
					if self.isClickToolActivated:
						self.clickTool.canvasClicked.disconnect()
					# connect to click signal
					QObject.connect(
						self.clickTool,
						SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
						self.polygCollimation1P
					)
					self.isClickToolActivated = True
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"polygCollimation1P",
						"Matching pair invalid"
					)
		else:
			self.warning(1111)

	def polygCollimation1P(self,point):
		"""
			Esegue la collimazione di 1 punto
		"""
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			plgns = geom.asPolygon()
			# nuova geometria
			newPlgns = []
			for line in plgns:
				# costruisce il nuovo poligono
				newLine = []
				for v in line:
					newPoint = QgsPoint(v.x()+self.dx,v.y()+self.dy)
					newLine.append(newPoint)
				# aggiorn la geometria
				newPlgns.append(newLine)
			self.cLayer.changeGeometry(feat.id(),QgsGeometry.fromPolygon(newPlgns))
			self.canvas.refresh()

	def polygCollimation2PTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygCollimation2P',
				"Aligns a number of points by making two matching pairs"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			# selezione delle coppie di collimazione
			myDlg = polyg4qgisItemsSelectDlg(self,'vertex','Insert two matching pair')
			myDlg.show()
			result = myDlg.exec_()
			if result:
				if len(self.selectList) >= 4:
					# preleva le coppie di punti
					d2 = self.selectList.pop()
					d2x,d2y = d2.x(),d2.y()
					s2 = self.selectList.pop()
					s2x,s2y = s2.x(),s2.y()
					d1 = self.selectList.pop()
					d1x,d1y = d1.x(),d1.y()
					s1 = self.selectList.pop()
					s1x,s1y = s1.x(),s1.y()
					if s1 == d1 and s2 == d2:
						QMessageBox.information(
							self.iface.mainWindow(),
							'polygCollimation2P',
							"It doesn't make any sense to have coinciding points to correspond."
						)
					else:
						self.markerList.append(edgeHighlight(self.canvas,s1,d1))	
						self.markerList.append(edgeHighlight(self.canvas,s2,d2))
						# calcola matrice di trsformazione
						# trasla n1 nell'origine
						mat = trans2D.translation(-s1x,-s1y)
						# ruota la linea n1-n2 sull'orizzontale
						a1 = math.atan2(s2y-s1y,s2x-s1x)
						# ruota per allinearla a P1-P2
						a2 = math.atan2(d2y-d1y,d2x-d1x)
						# applica la rotazione complessiva
						mat1 = trans2D.rotation(a2-a1)
						mat = np.dot(mat1,mat)
						# trasla nel punto P1
						mat1 = trans2D.translation(d1x,d1y)
						self.mat = np.dot(mat1,mat)
						# try to disconnect all signals
						if self.isClickToolActivated:
							self.clickTool.canvasClicked.disconnect()
						# connect to click signal
						QObject.connect(
							self.clickTool,
							SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
							self.polygCollimation2P
						)
						self.isClickToolActivated = True
			else:
				QMessageBox.information(
					self.iface.mainWindow(),
					'polygCollimation2P',
					"Please insert four points, two source and two destination."
				)
		else:
			self.warning(1111)

	def polygCollimation2P(self,point):
		"""
			Collimazione con coppia di punti
		"""
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			plgns = geom.asPolygon()
			# nuova geometria
			newPlgns = []
			for line in plgns:
				# costruisce il nuovo poligono
				newLine = []
				for v in line:
					x,y,w = np.dot(self.mat,[v.x(),v.y(),1])
					newLine.append(QgsPoint(x,y))
				# aggiorn la geometria
				newPlgns.append(newLine)
			self.cLayer.changeGeometry(feat.id(),QgsGeometry.fromPolygon(newPlgns))
			self.canvas.refresh()

	def polygCollimationNPTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygCollimationNP',
				"Aligns a number of points by making three or more matching pairs"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			cleanSelection(self)
			# selezione delle coppie di collimazione
			myDlg = polyg4qgisItemsSelectDlg(self,'vertex','Insert three or more matching pairs')
			myDlg.show()
			result = myDlg.exec_()
			if result:
				numCoppie = len(self.selectList)/2
				if len(self.selectList)-2*numCoppie == 0:
					if numCoppie >= 3:
						# prepara le liste dei punti di collimazione
						oldCdsList = []
						newCdsList = []
						# preleva le coppie (A RITROSO!)
						for i in range(numCoppie):
							d = self.selectList.pop()
							x,y = d.x(),d.y()
							newCdsList.append([x,y])
							s = self.selectList.pop()
							x,y = s.x(),s.y()
							oldCdsList.append([x,y,1])	# qui va aggiunta la coordinata omogenea
							self.markerList.append(edgeHighlight(self.canvas,s,d))	
# meglio controllare che siano distinti
						# calcola i parametri di trasformazione mediante LSM
						# probabilmente si potrebbe fre tutto in una passata
						# ma in questa prima implementazione con numpy preferisco essere cauto
						X = []
						Y = []
						for p in newCdsList:
							X.append(p[0])
							Y.append(p[1])
						sol1 = np.linalg.lstsq(oldCdsList,X)
						# print sol1
						sol2 = np.linalg.lstsq(oldCdsList,Y)
						# print sol2
						# costruisce la matrice di trasformazione (2x3)
						self.mat = np.matrix([[sol1[0][0],sol1[0][1],sol1[0][2]],[sol2[0][0],sol2[0][1],sol2[0][2]]])
						# try to disconnect all signals
						if self.isClickToolActivated:
							self.clickTool.canvasClicked.disconnect()
						# connect to click signal
						QObject.connect(
							self.clickTool,
							SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
							self.polygCollimationNP
						)
						self.isClickToolActivated = True
					else:
						QMessageBox.information(
							self.iface.mainWindow(),
							"polygCollimationNP",
							"Warning: are needed three or more couples of points"
						)
				else:
					QMessageBox.information(
						self.iface.mainWindow(),
						"polygCollimationNP",
						"Error: the number of points must be even"
					)
		else:
			self.warning(1111)

	def polygCollimationNP(self,point):
		"""
			Esegue la collimazione dei punti contenuti in plist
			in modo che assumano i punti dell srcList
			vdno  collimare con quelli contenuti in destList.
			Usa il LSM.
			
			ATTENZIONE: la routine è molto simile a Collimation2P, però
			la 2P opera con una matrice 3x3, la NP con matrice 2x3 e poi la prima dà
			direttamente le coordinate x,y, la seconda attraverso la lista cds (altrimenti
			dà errore); se si allineano possono esserefuse in una sola routine.
		"""
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			# legge la geometria
			geom = feat.geometry()
			plgns = geom.asPolygon()
			# nuova geometria
			newPlgns = []
			for line in plgns:
				# costruisce il nuovo poligono
				newLine = []
				for v in line:
					cds = np.dot(self.mat,[v.x(),v.y(),1])	# ttenzione: self.mat 2x3
					newLine.append(QgsPoint(cds[0,0],cds[0,1]))
				# aggiorn la geometria
				newPlgns.append(newLine)
			self.cLayer.changeGeometry(feat.id(),QgsGeometry.fromPolygon(newPlgns))
			self.canvas.refresh()

# ------------------ polygon deletion ------------------------

	def polygDelTool(self):
		if self.helpFlag:
			QMessageBox.information(
				self.iface.mainWindow(),
				'polygDel',
				"pLine deleting"
			)
			self.helpFlag = False
		if self.goFlag >= 1111:
			cleanSelection(self)
			# try to disconnect all signals
			if self.isClickToolActivated:
				self.clickTool.canvasClicked.disconnect()
			# connect to click signal
			QObject.connect(
				self.clickTool,
				SIGNAL("canvasClicked(const QgsPoint &, Qt::MouseButton)"),
				self.polygDel
			)
			self.isClickToolActivated = True
		else:
			self.warning(1111)

	def polygDel(self,point):
		"""
			delete an pLine
		"""
		searchFeat(self,point)
		feat = QgsFeature()
		self.cLayer.nextFeature(feat)
		if feat.isValid():
			pM = polygHighlight(self.canvas,feat)
			result = QMessageBox.question(
				self.iface.mainWindow(),
				'polygDel',
				"Feature id=%d" % (feat.id()),
				1,
				2
			)
			if result == 1:
				self.cLayer.deleteFeature(feat.id())
				self.canvas.refresh()
			self.canvas.scene().removeItem(pM)

