from requests import get
from .webi import dict2url
from .id import av2bv, bv2av
from pprint import pprint as ppt
from qgis.PyQt.QtCore import QThread, pyqtSignal


def get_headers(bvid: str, cookie=''):
    return {
        'origin': 'https://www.bilibili.com',
        'priority': 'u=1, i',
        'Referer': f'https://www.bilibili.com/video/{bvid}/',
        'Sec-Fetch-Dest': 'empty',
        'accept': 'application/json',
        'cookie': cookie,
        'Sec-Fetch-Mode': 'cors',
        'Sec-Fetch-Site': 'same-site',
        'referer': 'https://www.bilibili.com/video/BV1bEqrYaEJj/?spm_id_from=333.1387.upload.video_card.click&vd_source=10d0f86227f3c318f8237345caac47c8',
        'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/142.0.0.0 Safari/537.36 Edg/142.0.0.0',
    }


class CMT:
    def __init__(self, name: str, content: str):
        # 昵称
        self.name = name
        # 性别
        self.sex = ''
        # 评论内容
        self.content = content
        # 评论id
        self.rid = 0
        # 视频bvid
        self.bvid = 0
        # 评论者id
        self.mid = 0
        # 回复数
        self.rcount = 0
        # 父评论id
        self.parent = 0
        # 等级
        self.level = 0
        # 评论时间戳
        self.ctime = ''
        # 点赞数
        self.like = 0
        # 地区
        self.location = ''

    def __repr__(self):
        return f'{self.name}: {self.content}, {self.location}'


def generate_url(avid: str, offset_tr: str):
    API = 'https://api.bilibili.com/x/v2/reply/wbi/main'

    if offset_tr == '':
        fmtOffsetStr = '{"offset":""}'
    else:
        fmtOffsetStr = '{"offset":' + '"' + offset_tr + '"' + '}'

    params = {
        'oid': avid,
        'type': '1',
        'mode': '2',
        'seek_rpid': '',
        'pagination_str': fmtOffsetStr,
        'plat': '1',
        'seek_rpid': '',
        'web_location': '1315875',
    }

    query_url = dict2url(params)
    return f'{API}?{query_url}'


def generate_sub_url(avid: str, root: str, next: any):
    API = 'https://api.bilibili.com/x/v2/reply/reply'
    params = {
        'oid': avid,
        'type': '1',
        'root': root,
        'ps': '20',
        'pn': f'{next}',
        'web_location': '333.788',
    }

    query_url = dict2url(params)
    return f'{API}?{query_url}'


def fetch_comments_count(avid: str) -> int:
    url = f"https://api.bilibili.com/x/v2/reply/count?type=1&oid={avid}"
    resp = get(url, headers=get_headers(av2bv(avid)))
    if resp.status_code == 200:
        return resp.json()['data']['count']
    else:
        return -1


class Worker(QThread):
    step_signal = pyqtSignal(float)
    finished_signal = pyqtSignal(list)

    def __init__(self, bvid='', cookie=''):
        super(Worker, self).__init__()
        self.bvid = bvid
        self.cookie = cookie

    def run(self):
        avid = bv2av(self.bvid)
        cmts: list[CMT] = []
        MAX_COUNT = fetch_comments_count(avid)
        if MAX_COUNT <= 0:
            return cmts

        ppt(f"Fetching {MAX_COUNT} comments...")
        fulled = False
        offset_str = ''
        tries = 0
        while fulled is False:
            try:
                url = generate_url(avid, offset_str)
                resp = get(url, headers=get_headers(self.bvid, self.cookie))
                if resp.status_code == 200:
                    data = resp.json().get('data')
                    if not data:
                        ppt('No Data.')
                        break
                    replys = data['replies']
                    if len(replys) == 0:
                        tries += 1
                        ppt('No more replies, skiped.')
                        break

                    ppt(f'Fetched {len(replys)} comments.')
                    for p in replys:
                        uname = p['member']['uname']
                        sex = p['member']['sex']
                        level = p['member']['level_info']['current_level']

                        mid = p['mid']
                        parent = p['parent']
                        rcount = p['rcount']
                        like = p['like']
                        ctime = p['ctime']
                        rpid = p['rpid']
                        content = p['content']['message']
                        location = (
                            p.get('reply_control', {'location': ''})
                            .get('location', '未知')
                            .replace('IP属地：', '')
                        )
                        cmt = CMT(uname, content)
                        cmt.sex = sex
                        cmt.rid = rpid
                        cmt.bvid = self.bvid
                        cmt.mid = mid
                        cmt.rcount = rcount
                        cmt.parent = parent
                        cmt.level = level
                        cmt.ctime = ctime
                        cmt.like = like
                        cmt.location = location
                        cmts.append(cmt)

                    pct = len(cmts) / MAX_COUNT * 100
                    ppt(f'{pct}')
                    self.step_signal.emit(pct)
                    offset_str = data['cursor']['pagination_reply'].get(
                        'next_offset', ''
                    )
                    if len(cmts) >= MAX_COUNT:
                        fulled = True
                else:
                    ppt(f'Error: {resp.status_code}')
                    break
            except Exception as e:
                ppt(f'Error: {e}')
        self.finished_signal.emit(cmts)


if __name__ == '__main__':
    th = Worker(
        'BV1bEqrYaEJj',
        '''buvid3=0DC5714D-2D87-5F8D-A235-E39A260932D835413infoc; b_nut=1753588935; _uuid=E98EAC1C-F714-93F7-FCF10-DE7C49A7A84835673infoc; enable_web_push=DISABLE; buvid_fp=cdf6ea7e953eb5d818b0e0c63da84be6; theme-tip-show=SHOWED; theme-avatar-tip-show=SHOWED; rpdid=|(Y|llm)kuu0J'u~lJl)mmlu; LIVE_BUVID=AUTO6817535904358646; hit-dyn-v2=1; buvid4=52212837-D580-A5EA-2551-2FBCEC69329A33098-025042521-tPp6ZMhTQc+9GB4AlqWEVX8eIL6BRZTg5hGt0Mdf3IaMHtPHr0+C8n0K2Rx1XiWK; theme-switch-show=SHOWED; theme_style=dark; DedeUserID=479424003; DedeUserID__ckMd5=3095bad518dcee61; CURRENT_QUALITY=80; PVID=3; home_feed_column=5; browser_resolution=1432-713; bili_ticket=eyJhbGciOiJIUzI1NiIsImtpZCI6InMwMyIsInR5cCI6IkpXVCJ9.eyJleHAiOjE3NjU1MjI0NDEsImlhdCI6MTc2NTI2MzE4MSwicGx0IjotMX0.S6zPf8Zddu9eMJAwtF3yWYbuhfSnq8Z52Gm9FKj2pO8; bili_ticket_expires=1765522381; SESSDATA=e47ba214%2C1780815242%2Cc38c2%2Ac1CjD-fNGBxUk42pXA87jEuhDSpsvMAEPbDSO-K_7V086t3vRchQjENtBq1FSLkJzZwFUSVkVLUExNZFNIQ0pzZTZyV1g3ZHkwd1JpWGtFYVBNd1FKSk95MGxxQTVfZ0xndmFJVUF0VkxCNEgxS2ZYMWFzeEhsdWItNWVKTElPWUkyVXJrSDRZNjBRIIEC; bili_jct=edd3f2b4fc7cbe800e3427c73ecc398e; sid=89543gdc; b_lsid=D216CD61_19B0CEDE1BF; CURRENT_FNVAL=4048; bp_t_offset_479424003=1145082549341519872''',
    )
    th.start()
