# -*- coding: utf-8 -*-
"""
/***************************************************************************
 BhuvanWebServices
                                 A QGIS plugin
 This plugin allows you to load geospatial web map services from Bhuvan ISRO's Geoportal
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2019-08-28
        git sha              : $Format:%H$
        copyright            : (C) 2019 by Breny Kurien
        email                : brenykurien007@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, pyqtSignal, QEvent
from qgis.PyQt.QtGui import QIcon, QTextCursor
from qgis.PyQt.QtWidgets import QAction, QMessageBox, QTableWidgetItem, QHeaderView, QAbstractItemView, QProgressBar, QProgressDialog
from qgis._core import QgsRasterLayer, QgsProject, Qgis

# Initialize Qt resources from file resources.py
from .info_dialog import InfoDialog
from .common.mappings.ServiceUrlMap import service_url_map
from .common.enums.Service import Service
from .common.enums.ServiceType import ServiceType
from .common.mappings.ServiceTextMap import service_text_map
from .common.mappings.ServiceTypeMap import service_type_map

from .resources import *
# Import the code for the dialog
from .bhuvan_web_services_dialog import BhuvanWebServicesDialog
import os.path
from owslib.wms import WebMapService
from owslib.wmts import WebMapTileService
from collections import OrderedDict


class BhuvanWebServices:

    def __init__(self, iFace):
        # Save reference to the QGIS interface
        self.iFace = iFace
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'BhuvanWebServices_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        self.dlg = BhuvanWebServicesDialog()
        self.dlginfo = InfoDialog()
        self.generatedService = None
        self.bar = QProgressBar()
        self.bar.setRange(0, 0)
        self.bar.setGeometry(950, 500, 200, 25)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Bhuvan Web Services')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('BhuvanWebServices', message)

    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iFace.addToolBarIcon(action)

        if add_to_menu:
            self.iFace.addPluginToWebMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        self.add_all_action()
        self.dlg.table_widget.itemSelectionChanged.connect(self.updateDesc)
        self.dlg.help_button.clicked.connect(self.openDlgInfo)
        self.dlg.close_button.clicked.connect(self.closeDlg)
        self.dlg.search_box.textEdited.connect(self.search)
        self.dlg.add_button.released.connect(self.loadWebService)

        # will be set False in run()
        self.first_start = True

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iFace.removePluginWebMenu(
                self.tr(u'&Bhuvan Web Services'),
                action)
            self.iFace.removeToolBarIcon(action)

    def run(self):
        if self.generatedService.web_map_service is not None:
            self.dlg.search_box.clear()
            self.fill_table(self.generatedService.web_map_service.contents)
            self.dlg.show()
            result = self.dlg.exec_()
            if result:
                pass

    def run_0(self):
        self.loadServiceList(Service.BhuvanPanchayat.value)
        self.run()

    def run_1(self):
        self.loadServiceList(Service.LULC205KDataset.value)
        self.run()

    def run_2(self):
        self.loadServiceList(Service.BhuvanV1WMS.value)
        self.run()

    def run_3(self):
        self.loadServiceList(Service.BhuvanV2WMS.value)
        self.run()

    def run_4(self):
        self.loadServiceList(Service.BhuvanV1WMTS.value)
        self.run()

    def run_5(self):
        self.loadServiceList(Service.BhuvanV2WMTS.value)
        self.run()

    def add_all_action(self):
        icon_path = ':/plugins/bhuvan_web_services/icon.png'

        self.add_action(icon_path,
                        text=self.tr(service_text_map[Service.BhuvanPanchayat.value]),
                        callback=self.run_0,
                        whats_this=str(Service.BhuvanPanchayat.value),
                        parent=self.iFace.mainWindow())

        self.add_action(icon_path,
                        text=self.tr(service_text_map[Service.LULC205KDataset.value]),
                        callback=self.run_1,
                        whats_this=str(Service.LULC205KDataset.value),
                        parent=self.iFace.mainWindow())

        self.add_action(icon_path,
                        text=self.tr(service_text_map[Service.BhuvanV1WMS.value]),
                        callback=self.run_2,
                        whats_this=str(Service.BhuvanV1WMS.value),
                        parent=self.iFace.mainWindow())

        self.add_action(icon_path,
                        text=self.tr(service_text_map[Service.BhuvanV2WMS.value]),
                        callback=self.run_3,
                        whats_this=str(Service.BhuvanV2WMS.value),
                        parent=self.iFace.mainWindow())

        self.add_action(icon_path,
                        text=self.tr(service_text_map[Service.BhuvanV1WMTS.value]),
                        callback=self.run_4,
                        whats_this=str(Service.BhuvanV1WMTS.value),
                        parent=self.iFace.mainWindow())

        self.add_action(icon_path,
                        text=self.tr(service_text_map[Service.BhuvanV2WMTS.value]),
                        callback=self.run_5,
                        whats_this=str(Service.BhuvanV2WMTS.value),
                        parent=self.iFace.mainWindow())


    def loadServiceList(self, service_id: int):
        self.iFace.messageBar().pushMessage('Info: ', 'Please wait loading layers ... ', level=Qgis.Info)
        self.bar.show()
        self.iFace.mainWindow().repaint()
        self.generatedService = WebMapServiceClass(service_id)
        url = self.generatedService.service_url
        if self.generatedService.service_type == ServiceType.WebMapService.value:
            try:
                wms = WebMapService(url)
                self.generatedService.setWebMapService(wms)
            except Exception as e:
                QMessageBox.information(None, "ERROR:", 'Unable to load this service now.' + str(e))
        elif self.generatedService.service_type == ServiceType.WebMapTileService.value:
            try:
                wmts = WebMapTileService(url)
                self.generatedService.setWebMapService(wmts)
            except Exception as e:
                QMessageBox.information(None, "ERROR:", 'Unable to load this service now.' + str(e))
        self.bar.close()

    def openDlgInfo(self):
        self.dlginfo.show()

    def closeDlg(self):
        self.generatedService = None
        self.dlg.search_box.clear()
        self.dlg.table_widget.setRowCount(0)
        self.dlg.table_widget.setColumnCount(0)
        self.dlg.layer_name_box.clear()
        self.dlg.close()
        if self.dlginfo:
            self.dlginfo.close()

    def fill_table(self, contentOrderedDict):
        self.dlg.table_widget.setRowCount(0)
        count = self.dlg.table_widget.rowCount()
        self.dlg.table_widget.setColumnCount(4)

        for content in contentOrderedDict:
            index = count
            # id_int = int(contentOrderedDict[content].index[2:])
            name = contentOrderedDict[content].name
            title = contentOrderedDict[content].title
            abstract = contentOrderedDict[content].abstract
            self.dlg.table_widget.insertRow(index)  # inserts a blank row
            # lets fill that row:
            # self.dlg.table_widget.setItem(index, 0, QTableWidgetItem(str(id_int)))  # fills in with the ID
            self.dlg.table_widget.setItem(index, 1, QTableWidgetItem(str(name)))  # fills in with the Name
            self.dlg.table_widget.setItem(index, 2, QTableWidgetItem(str(title)))  # fills in with the Title
            self.dlg.table_widget.setItem(index, 3, QTableWidgetItem(str(abstract)))  # fills in with Abstract

        self.dlg.table_widget.setHorizontalHeaderLabels(["ID", "Name", "Title", "Abstract"])
        self.setTableWidgetBehaviour()

    def setTableWidgetBehaviour(self):
        # set row and column sizes and lock them
        self.dlg.table_widget.setColumnWidth(0, 0)
        self.dlg.table_widget.setColumnWidth(1, 200)
        self.dlg.table_widget.setColumnWidth(2, 200)
        self.dlg.table_widget.setColumnWidth(3, 200)
        self.dlg.table_widget.horizontalHeader().setSectionResizeMode(QHeaderView.Fixed)
        self.dlg.table_widget.verticalHeader().setSectionResizeMode(QHeaderView.Fixed)

        self.dlg.table_widget.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.dlg.table_widget.setEditTriggers(QAbstractItemView.NoEditTriggers)
        self.dlg.table_widget.setSelectionMode(QAbstractItemView.ExtendedSelection)

    def search(self):
        criteria = self.dlg.search_box.text()
        criteria = criteria.lower()
        wms_filtered_contents = OrderedDict()
        contents = self.generatedService.web_map_service.contents
        for content in contents:
            name = contents[content].name
            if criteria in name:
                wms_filtered_contents[content] = contents[content]
        self.fill_table(wms_filtered_contents)

    def getSelectedItemsFromTable(self):
        rowNames = []
        selected = self.dlg.table_widget.selectedItems()
        if len(selected) > 0:
            for i in range(0, len(selected), 4):
                row = self.dlg.table_widget.row(selected[i])
                name = self.dlg.table_widget.item(row, 1).text()
                rowNames.append(name)

        selectedServices = OrderedDict()
        contents = self.generatedService.web_map_service.contents
        for rowName in rowNames:
            for content in contents:
                name_itr = contents[content].name
                if name_itr == rowName:
                    selectedServices[content] = contents[content]

        return selectedServices

    def updateDesc(self):
        selectedServices = self.getSelectedItemsFromTable()
        self.dlg.layer_name_box.clear()
        names = ''
        for selectedService in selectedServices:
            name_itr = selectedServices[selectedService].name
            names += name_itr + ','
        names = names[:-1]
        self.dlg.layer_name_box.setText(names)
        self.dlg.layer_name_box.setReadOnly(True)

    def loadWebService(self):
        # get selected items and add to the map
        self.bar.show()
        EPSG_CODE_4326 = 'EPSG:4326'
        selectedServices = self.getSelectedItemsFromTable()
        web_map_service = self.generatedService.web_map_service
        for selectedService in selectedServices:
            if self.generatedService.service_url is not None:
                layer_name = selectedServices[selectedService].name
                url = 'contextualWMSLegend=0'
                if hasattr(web_map_service[layer_name], 'crsOptions'):
                    if len(web_map_service[layer_name].crsOptions) > 0:
                        if EPSG_CODE_4326 in web_map_service[layer_name].crsOptions:
                            url += '&crs=' + EPSG_CODE_4326
                            if self.generatedService.service_type == ServiceType.WebMapTileService.value:
                                    url += '&tileMatrixSet=' + EPSG_CODE_4326
                        else:
                            url += '&crs=' + web_map_service[layer_name].crsOptions[0]
                            if self.generatedService.service_type == ServiceType.WebMapTileService.value:
                                    url += '&tileMatrixSet=' + web_map_service[layer_name].crsOptions[0]
                else:
                    url += '&crs=' + EPSG_CODE_4326
                    if self.generatedService.service_type == ServiceType.WebMapTileService.value:
                        url += '&tileMatrixSet=' + EPSG_CODE_4326
                url += '&dpiMode=7&featureCount=10&format=image/png&styles' + \
                       '&layers=' + layer_name + \
                       '&url=' + str(self.generatedService.service_url)
                rlayer = QgsRasterLayer(url, selectedServices[selectedService].title, 'wms')
                if not rlayer.isValid():
                    QMessageBox.information(None, "ERROR:", 'Unable to load ' +
                                            selectedServices[selectedService].title +
                                            ' this layer now.')
                else:
                    QgsProject.instance().addMapLayer(rlayer)
            else:
                QMessageBox.information(None, "ERROR:", 'Service url is None')
        self.bar.close()


class WebMapServiceClass:

    def __init__(self, service_id):
        self.service_id = service_id
        self.service_type = service_type_map[service_id]
        self.service_text = service_text_map[service_id]
        self.service_url = service_url_map[service_id]
        self.web_map_service = None

    def setWebMapService(self, map_service):
        self.web_map_service = map_service
