# -*- coding: utf-8 -*-
"""
/***************************************************************************
 BDTopoIporter
                                 A QGIS plugin
 Import facile des couches de la BDTopo dans une base de données PostGIS
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2021-01-13
        git sha              : $Format:%H$
        copyright            : (C) 2021 by Augustin Roche
        email                : augustin.roche@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
import os.path

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QFileDialog

from qgis.core import (
    Qgis, QgsProcessingUtils, QgsMessageLog, QgsTask, QgsApplication
)
from qgis import processing


# Initialize Qt resources from file resources.py
from .resources import *
# Import the code for the dialog
from .bdtopo_importer_dialog import BDTopoImporterDialog
from . import extractors


MESSAGE_CATEGORY = "BDTopo Importer"


class LayerImportTask(QgsTask):
    """ Task for importing a single layer 
    options :
    - connection : the db connection
    - schema : the schema to use
    - import_method : download/compressed/folder
    - overwrite
    - extract_method
    """

    def __init__(self, source, layer, theme, options):
        super().__init__(f"Importing BDTopo layer {layer}", QgsTask.CanCancel)
        self.layer = layer
        self.source = source
        self.theme = theme
        self.options = options

        self.tempdir = QgsProcessingUtils.tempFolder()

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('BDTopoImporter', message)

    def process_layer(self, layerpath):
        """
        Runs the GDAL algoritm to store layer in postgis
        """
        
        db = self.options['connection']
        params = {
            'DATABASE': db,
            'INPUT': layerpath,
            'SCHEMA': self.options['schema'],
            'TABLE': self.layer.lower(),
            'APPEND': not self.options['overwrite'],
            'OVERWRITE': self.options['overwrite']
        }
        processing.run('gdal:importvectorintopostgisdatabaseavailableconnections', params)
        self.setProgress(100)

    def run(self):
        if self.options['import_method'] == 'download':
            # first download file
            # TODO (or not)
            pass
        if self.options['import_method'] in ('download', 'compressed'):
            # extracting from compressed file
            archive = self.source
            
            executable = self.options.get('executable')
            extract_method = self.options.get('extract_method')
            try:
                layerpath = extractors.extract_7zip(archive, self.theme, 
                                        self.layer, self.tempdir, 
                                        extract_method, executable)
            except extractors.ProgramNotFoundError:
                # TODO: Let user select 7zip appli from main window
                QgsMessageLog.logMessage(
                        self.tr("L'exécutable 7zip n'a pas été trouvé"), 
                        MESSAGE_CATEGORY, Qgis.Critical)
                return False
        else:
            # process already extracted files
            layerpath = extractors.get_folder(self.source, self.theme, self.layer)
        self.setProgress(50)
        if layerpath is None: # the layer was not found
            return False
        if self.isCanceled():
            return False
        self.process_layer(layerpath)
        return True

    def cancel(self):
        QgsMessageLog.logMessage(
            self.tr(f"Annulation de l'import de la couche {self.layer}."),
            MESSAGE_CATEGORY, Qgis.Info
        )
        super().cancel()

    def finished(self, result):
        # delete created files
        if self.options['import_method'] != 'folder':
            extractors.delete_files(self.tempdir, self.layer)
        if result:
            QgsMessageLog.logMessage(self.tr(f"Export de la couche {self.layer} réussi."),
            MESSAGE_CATEGORY, Qgis.Success)
        else:
            pass
            # TODO: raise exceptions


class BDTopoImporter:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'BDTopoIporter_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&BDTopo Importer')

        # Check if plugin was started the first time in current QGIS session
        # Must be set in initGui() to survive plugin reloads
        self.first_start = None

        # must keep it in scope
        self.task_manager = QgsApplication.taskManager()

    # noinspection PyMethodMayBeStatic
    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('BDTopoImporter', message)


    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar.

        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            # Adds plugin icon to Plugins toolbar
            self.iface.addToolBarIcon(action)

        if add_to_menu:
            self.iface.addPluginToMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/bdtopo_importer/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Import BDTopo'),
            callback=self.run,
            parent=self.iface.mainWindow())

        # will be set False in run()
        self.first_start = True
        
    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginMenu(
                self.tr(u'&BDTopo Importer'),
                action)
            self.iface.removeToolBarIcon(action)

    def importLayers(self):
        """ Main method to import selected layers """

        layers = self.dlg.getCheckedLayers()
        if len(layers) == 0:
            self.iface.messageBar().pushMessage(self.tr("Erreur"),
                self.tr("Sélectionnez au moins une couche."), level=Qgis.Warning)
            return

        if self.dlg.import_method() == 'compressed':
            source = self.dlg.lineEdit_file_path.text()
            if not os.path.isfile(source):
                self.iface.messageBar().pushMessage(self.tr("Erreur"),
                    self.tr("Sélectionnez un fichier valide."), level=Qgis.Critical)
                return
        else: # folder import
            source = self.dlg.lineEdit_folder_path.text()
            if not os.path.isdir(source):
                self.iface.messageBar().pushMessage(self.tr("Erreur"),
                    self.tr("Sélectionnez un dossier valide."), level=Qgis.Critical)
                return
        s = QSettings()
        executable = s.value('bdtopo_importer/extract_command', "7z")
        self.iface.messageBar().pushMessage(self.tr("BDTopo"),
                    self.tr("Lancement des tâches d'import."), level=Qgis.Info)
        
        # create background tasks
        options = {
            'executable': executable,
            'connection': self.dlg.dbselect.currentConnection(),
            'schema': self.dlg.schemaselect.currentSchema(),
            'import_method': self.dlg.import_method(),
            'overwrite': self.dlg.radioButton_overwrite.isChecked(),
            'extract_method': s.value('bdtopo_importer/extract_method')
        }
        self.dlg.accept()
        for theme, layer in layers:
            task = LayerImportTask(source, layer, theme, options)
            self.task_manager.addTask(task)


    def run(self):
        """Run method that performs all the real work"""

        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.first_start:
            self.first_start = False
            self.dlg = BDTopoImporterDialog()
            self.dlg.button_box.accepted.connect(self.importLayers)

        # show the dialog
        self.dlg.show()
        # Run the dialog event loop
        result = self.dlg.exec_()
        # See if OK was pressed
        if result:
            pass
