"""
/***************************************************************************
 BDOT10k_pluginDialog
                                 A QGIS plugin
 This plugin operates on BDOT10k.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2022-01-06
        git sha              : $Format:%H$
        copyright            : (C) 2022 by marylaGIS
        email                : maryla4gis@gmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import os

from qgis.PyQt import uic
from qgis.PyQt.QtCore import Qt, pyqtSlot
from qgis.PyQt.QtGui import QCursor
from qgis.PyQt.QtWidgets import QApplication, QDialog, QMessageBox

from qgis.core import (Qgis, QgsMessageLog, QgsApplication,
                       QgsMapLayerProxyModel, QgsTask)

from .dialog_mixin import DialogMixin
from .task_dwnl_bdot import DownloadBdotTask
from .task_select import SelectTask
from .utils import *

# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'bdot10k_dialog_by_layer.ui'))


class BDOT10kDialogByLayer(QDialog, FORM_CLASS, DialogMixin):
    """Dialog for downloading BDOT10k data by selecting counties
    based on their intersection with the selected layer."""

    def __init__(self, parent=None):
        """Constructor."""
        super().__init__(parent)
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)

        self.powiatyTerytByLayer = []
        self.taskDwnl = None
        self.taskIdDwnl = None
        self.taskIdSelect = None
        self.taskManager = QgsApplication.taskManager()

        self.btnCancelDwnl.hide()
        self.btnDwnl.setEnabled(False)
        self.btnDwnl.setToolTip("Najpierw wybierz warstwę wektorową.")
        # set filters for the map layer combo box - only vector layers
        self.mcbLayer.setFilters(QgsMapLayerProxyModel.Filter.VectorLayer)
        self.txt.clear()

    def on_mcbLayer_layerChanged(self):
        """Selects counties for data download based on their intersection
        with the layer selected in the combo box."""

        self.txt.clear()
        self.powiatyTerytByLayer.clear()
        self.btnDwnl.setEnabled(False)

        if self.mcbLayer.currentIndex() != -1:
            selectionLayer = self.mcbLayer.currentLayer()

        try:
            QApplication.setOverrideCursor(QCursor(Qt.CursorShape.BusyCursor))

            if selectionLayer.featureCount() == 0:
                QApplication.restoreOverrideCursor()
                self.txt.append("Brak powiatów do pobrania.")
                if self.isVisible():
                    QMessageBox.warning(self, "Uwaga", "Wybrana warstwa nie zawiera obiektów.")
            else:
                self.txt.append(
                    '<span style="font-style:italic;">Trwa wyszukiwanie powiatów...</span>'
                )

                task = SelectTask(
                    description="Trwa wyszukiwanie powiatów.",
                    selectionLayer=selectionLayer.clone()
                )

                task.result.connect(lambda x: self.handle_selection_result(x))

                self.taskManager.addTask(task)
                self.taskIdSelect = self.taskManager.taskId(task)
                self.taskManager.statusChanged.connect(self.refresh_dialog)

        except Exception as e:
            QApplication.restoreOverrideCursor()
            self.txt.clear()
            QgsMessageLog.logMessage(
                f"Wystąpił błąd podczas selekcji powiatów. Treść błędu: {e}",
                "BDOT10k",
                Qgis.MessageLevel.Critical
            )

    def refresh_dialog(self, taskId: int, status: int):
        """Clears the text browser and restores cursor
        after a SelectTask was canceled or terminated.

        :param taskId: The id of the SelectTask.
        :param status: The status of the SelectTask.
        """

        if taskId == self.taskIdSelect and status in (
            QgsTask.Flag.CancelWithoutPrompt,
            QgsTask.TaskStatus.Terminated,
        ):
            QApplication.restoreOverrideCursor()
            self.txt.clear()

    def handle_selection_result(self, selectionList: list) -> list:
        """Prepare a list of county TERYT codes to download BDOT10k data.

        :param selectionList: List of county names and TERYT codes
            selected by the SelectTask.
        :returns: List of TERYT codes ready for download.
        """

        QApplication.restoreOverrideCursor()
        self.txt.clear()

        if selectionList:
            self.powiatyTerytByLayer = [
                i[:4] for i in selectionList
            ]
            self.txt.append(f"Liczba powiatów do pobrania: {len(selectionList)}")
            self.txt.append("Powiaty: " + ", ".join(selectionList))
            self.btnDwnl.setEnabled(True)
            self.btnDwnl.setToolTip('')
        else:
            self.txt.append(f"Liczba powiatów do pobrania: {len(selectionList)}")
            if self.isVisible():
                QMessageBox.warning(
                    self,
                    "Uwaga",
                    "Wybrana warstwa nie znajduje się na terenie żadnego powiatu."
                )

        return self.powiatyTerytByLayer

    @pyqtSlot()
    def on_btnDwnl_clicked(self):
        """Checks parameters required for downloading data
        and creates a task."""

        if not self.powiatyTerytByLayer:
            QMessageBox.critical(self, "Błąd", "Brak powiatów do pobrania.")
        else:
            downloadPath = self.dwnlPath.filePath()

            oldSchema, bdot10kDataFormat = self.get_bdot10k_data_options()

            if check_dwnl_path(downloadPath):

                self.btnDwnl.hide()
                self.btnCancelDwnl.show()
                self.btnCancelDwnl.setEnabled(True)

                self.taskDwnl = DownloadBdotTask(
                    description="Pobieranie paczek BDOT10k",
                    downloadPath=downloadPath,
                    oldSchema=oldSchema,
                    bdot10kDataFormat=bdot10kDataFormat,
                    powiatyTerytList=self.powiatyTerytByLayer
                )

                self.taskManager.addTask(self.taskDwnl)
                self.taskIdDwnl = self.taskManager.taskId(self.taskDwnl)
                self.taskManager.statusChanged.connect(self.switch_buttons)
