#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Compile translation files from .ts to .qm format.
Run this script whenever you update translations.
"""

import os
import glob
import subprocess
import sys

def compile_translations():
    """Compile all .ts files to .qm files."""
    
    # Get the i18n directory
    script_dir = os.path.dirname(os.path.abspath(__file__))
    i18n_dir = os.path.join(script_dir, 'i18n')
    
    if not os.path.exists(i18n_dir):
        print(f"Error: i18n directory not found at {i18n_dir}")
        return False
    
    # Find all .ts files
    ts_files = glob.glob(os.path.join(i18n_dir, '*.ts'))
    
    if not ts_files:
        print(f"No .ts files found in {i18n_dir}")
        return False
    
    print(f"Found {len(ts_files)} translation file(s)")
    
    # Try to find lrelease command
    lrelease_commands = ['lrelease', 'lrelease-qt5', 'lrelease-qt6']
    lrelease_cmd = None
    
    for cmd in lrelease_commands:
        try:
            subprocess.run([cmd, '-version'], 
                         capture_output=True, 
                         check=True)
            lrelease_cmd = cmd
            print(f"Using: {cmd}")
            break
        except (subprocess.CalledProcessError, FileNotFoundError):
            continue
    
    if not lrelease_cmd:
        print("\nError: lrelease command not found!")
        print("Please install Qt Linguist tools:")
        print("  - On Debian/Ubuntu: sudo apt-get install qttools5-dev-tools")
        print("  - On Windows: Install Qt from https://www.qt.io/download")
        print("  - Or use QGIS built-in: Add QGIS bin folder to PATH")
        return False
    
    # Compile each .ts file
    success_count = 0
    for ts_file in ts_files:
        qm_file = ts_file.replace('.ts', '.qm')
        print(f"\nCompiling: {os.path.basename(ts_file)} -> {os.path.basename(qm_file)}")
        
        try:
            result = subprocess.run(
                [lrelease_cmd, ts_file, '-qm', qm_file],
                capture_output=True,
                text=True,
                check=True
            )
            if result.stdout:
                print(result.stdout.strip())
            success_count += 1
        except subprocess.CalledProcessError as e:
            print(f"Error compiling {ts_file}:")
            print(e.stderr)
    
    print(f"\n{'='*60}")
    print(f"Successfully compiled {success_count}/{len(ts_files)} translation files")
    print(f"{'='*60}")
    
    return success_count == len(ts_files)

if __name__ == '__main__':
    success = compile_translations()
    sys.exit(0 if success else 1)
