# -*- coding: utf-8 -*-
"""
/***************************************************************************
 BBoxVisualizerDialog
                                 A QGIS plugin
 Visualize bounding boxes from coordinate inputs
 ***************************************************************************/
"""

import os
import re

from qgis.PyQt import uic
from qgis.PyQt.QtCore import pyqtSlot, Qt, QCoreApplication
from qgis.PyQt.QtGui import QColor
from qgis.PyQt.QtWidgets import QDialog, QPushButton, QDialogButtonBox
from qgis.core import (
    QgsVectorLayer,
    QgsFeature,
    QgsGeometry,
    QgsRectangle,
    QgsProject,
    QgsCoordinateReferenceSystem,
    QgsCoordinateTransform
)

# This loads your .ui file so that PyQt can populate your plugin with the elements from Qt Designer
FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'bbox_visualizer_dialog_base.ui'))


class BBoxVisualizerDialog(QDialog, FORM_CLASS):
    def __init__(self, iface=None, parent=None):
        """Constructor."""
        super(BBoxVisualizerDialog, self).__init__(parent)
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)
        self.iface = iface
        
        # Force retranslation of UI elements
        if hasattr(self, 'retranslateUi') and callable(self.retranslateUi):
            self.retranslateUi(self)
        
        # Add the "Visualize BBox" button
        self.visualizeButton = QPushButton(self.tr("Visualize BBox"))
        # Qt5/Qt6 compatibility: use ActionRole enum correctly
        try:
            # Qt6 style
            button_role = QDialogButtonBox.ButtonRole.ActionRole
        except AttributeError:
            # Qt5 style
            button_role = QDialogButtonBox.ActionRole
        self.buttonBox.addButton(self.visualizeButton, button_role)
        
        # Connect signals
        self.bboxLineEdit.textChanged.connect(self.validate_bbox)
        self.visualizeButton.clicked.connect(self.visualize_bbox)
        
        # Set default CRS to project CRS
        project_crs = QgsProject.instance().crs()
        self.crsSelector.setCrs(project_crs)
        
        # Initially disable the visualize button until valid input
        self.visualizeButton.setEnabled(False)

    def validate_bbox(self):
        """Validate the bbox input format."""
        bbox_text = self.bboxLineEdit.text().strip()
        
        # Clear warning and reset style
        self.warningLabel.setStyleSheet("color: red;")
        self.warningLabel.setText("")
        
        if not bbox_text:
            self.visualizeButton.setEnabled(False)
            return False
        
        # Remove all spaces from input
        bbox_text_cleaned = bbox_text.replace(' ', '')
        
        # Check if the format matches: number,number,number,number
        # Allow integers and decimals
        pattern = r'^-?\d+\.?\d*,-?\d+\.?\d*,-?\d+\.?\d*,-?\d+\.?\d*$'
        
        if not re.match(pattern, bbox_text_cleaned):
            self.warningLabel.setText(self.tr("⚠ Invalid format! Use: minx,miny,maxx,maxy"))
            self.visualizeButton.setEnabled(False)
            return False
        
        # Parse coordinates
        try:
            coords = [float(x.strip()) for x in bbox_text_cleaned.split(',')]
            x1, y1, x2, y2 = coords
            
            # Auto-correct min/max order
            minx = min(x1, x2)
            maxx = max(x1, x2)
            miny = min(y1, y2)
            maxy = max(y1, y2)
            
            # Check if bbox has area
            if minx == maxx or miny == maxy:
                self.warningLabel.setStyleSheet("color: red;")
                self.warningLabel.setText(self.tr("⚠ BBox must have area (min cannot equal max)"))
                self.visualizeButton.setEnabled(False)
                return False
            
            # Format number intelligently (no .0 for whole numbers)
            def format_coord(num):
                return str(int(num)) if num == int(num) else str(num)
            
            # Create normalized bbox string
            normalized_bbox = f"{format_coord(minx)},{format_coord(miny)},{format_coord(maxx)},{format_coord(maxy)}"
            
            # Show normalized version if different from input
            if normalized_bbox != bbox_text_cleaned:
                self.warningLabel.setStyleSheet("color: blue;")
                self.warningLabel.setText(self.tr("ℹ Will use: {0}").format(normalized_bbox))
            else:
                self.warningLabel.setText("")
            
            # Valid!
            self.visualizeButton.setEnabled(True)
            return True
            
        except ValueError:
            self.warningLabel.setStyleSheet("color: red;")
            self.warningLabel.setText(self.tr("⚠ Invalid numbers in coordinates"))
            self.visualizeButton.setEnabled(False)
            return False

    def visualize_bbox(self):
        """Create a temporary layer to visualize the bounding box."""
        if not self.validate_bbox():
            return
        
        bbox_text = self.bboxLineEdit.text().strip()
        # Remove all spaces
        bbox_text_cleaned = bbox_text.replace(' ', '')
        
        coords = [float(x.strip()) for x in bbox_text_cleaned.split(',')]
        x1, y1, x2, y2 = coords
        
        # Auto-correct min/max order
        minx = min(x1, x2)
        maxx = max(x1, x2)
        miny = min(y1, y2)
        maxy = max(y1, y2)
        
        # Get selected CRS
        selected_crs = self.crsSelector.crs()
        
        # Create layer name with bbox coordinates
        layer_name = self.tr("BBox Visualization: {0},{1},{2},{3}").format(minx, miny, maxx, maxy)
        
        # Create a memory layer
        layer = QgsVectorLayer(
            f"Polygon?crs={selected_crs.authid()}",
            layer_name,
            "memory"
        )
        
        provider = layer.dataProvider()
        
        # Create a rectangle geometry
        rect = QgsRectangle(minx, miny, maxx, maxy)
        feature = QgsFeature()
        feature.setGeometry(QgsGeometry.fromRect(rect))
        
        # Add feature to layer
        provider.addFeatures([feature])
        layer.updateExtents()
        
        # Set layer style - transparent fill with red border
        layer.renderer().symbol().setColor(QColor(0, 0, 0, 0))  # Transparent
        layer.renderer().symbol().symbolLayer(0).setStrokeColor(QColor(255, 0, 0))  # Red
        layer.renderer().symbol().symbolLayer(0).setStrokeWidth(1.5)
        
        # Add layer to project
        QgsProject.instance().addMapLayer(layer)
        
        # Zoom to the bbox
        canvas = self.iface.mapCanvas() if hasattr(self, 'iface') else None
        if canvas:
            # Transform rectangle to map canvas CRS if needed
            canvas_crs = canvas.mapSettings().destinationCrs()
            if selected_crs != canvas_crs:
                transform = QgsCoordinateTransform(
                    selected_crs,
                    canvas_crs,
                    QgsProject.instance()
                )
                rect_transformed = transform.transformBoundingBox(rect)
                canvas.setExtent(rect_transformed)
            else:
                canvas.setExtent(rect)
            canvas.refresh()
        
        # Close the dialog
        self.accept()
