# -*- coding: utf-8 -*-
"""
/***************************************************************************
 BBoxVisualizer
                                 A QGIS plugin
 Visualize bounding boxes from coordinate inputs
 ***************************************************************************/
"""
import os.path

from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import QAction, QApplication

# Initialize Qt resources from file resources.py
from .resources import *
from .bbox_visualizer_dialog import BBoxVisualizerDialog


class BBoxVisualizer:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface
        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)
        
        # initialize locale
        locale = QSettings().value('locale/userLocale')
        if locale:
            locale = str(locale)
        else:
            locale = 'en'
        
        locale_short = locale[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'bbox_visualizer_{}.qm'.format(locale_short))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            if self.translator.load(locale_path):
                # Install translator on the QApplication instance
                app = QApplication.instance()
                if app:
                    app.installTranslator(self.translator)
        
        # Declare instance attributes
        self.actions = []
        self.menu = self.tr('&BBox Visualizer')
        self.toolbar = self.iface.addToolBar(self.tr('BBoxVisualizer'))
        self.toolbar.setObjectName('BBoxVisualizer')

        self.dlg = None
    
    def tr(self, message):
        """Get the translation for a string using Qt translation API."""
        return QCoreApplication.translate('BBoxVisualizer', message)

    def add_action(
        self,
        icon_path,
        text,
        callback,
        enabled_flag=True,
        add_to_menu=True,
        add_to_toolbar=True,
        status_tip=None,
        whats_this=None,
        parent=None):
        """Add a toolbar icon to the toolbar."""

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToVectorMenu(
                self.menu,
                action)

        self.actions.append(action)

        return action

    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = os.path.join(self.plugin_dir, 'icon.png')
        self.add_action(
            icon_path,
            text=self.tr('Visualize BBox'),
            callback=self.run,
            parent=self.iface.mainWindow())

    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""
        for action in self.actions:
            self.iface.removePluginVectorMenu(
                self.tr('&BBox Visualizer'),
                action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar
        
        # Remove translator
        if hasattr(self, 'translator'):
            app = QApplication.instance()
            if app:
                app.removeTranslator(self.translator)

    def run(self):
        """Run method that performs all the real work"""
        # Create the dialog with elements (after translation) and keep reference
        # Only create GUI ONCE in callback, so that it will only load when the plugin is started
        if self.dlg is None:
            self.dlg = BBoxVisualizerDialog(self.iface)
        
        # show the dialog
        self.dlg.show()
        # Run the dialog event loop (compatible with both QT5 and QT6)
        if hasattr(self.dlg, 'exec'):
            result = self.dlg.exec()
        else:
            result = self.dlg.exec_()
