# -*- coding: utf-8 -*-
"""
/***************************************************************************
 AWS Earth Catalog
                                 A QGIS plugin
 Plugin to access AWS Earth for show COG scenes
                             -------------------
        begin                : 2025-10-15
        copyright            : (C) 2025 by Luiz Motta
        email                : motta.luiz@gmail.com

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 This script initializes the plugin, making it known to QGIS.
"""

__author__ = "Luiz Motta"
__date__ = "2025-09-02"
__copyright__ = "(C) 2025, Luiz Motta"
__revision__ = "$Format:%H$"


import os

from typing import Callable

from qgis.PyQt.QtCore import (
    QObject,
    QDir,
    pyqtSlot
)
from qgis.PyQt.QtGui import QIcon
from qgis.PyQt.QtWidgets import  QAction

from qgis.gui import QgisInterface

from .awse.taskmanager import TaskProcessor
from .awse.awse_stacclient import AWSEStacClient
from .awse.stacprocessor import StacProcessor
from .awse.config import configCollection

from .awse.catalogwidget import CatalogWidget
from .awse.catalog import Catalog

from .awse.translate import setTranslation


def classFactory(iface:QgisInterface):
    return AWSCatalogPlugin(iface)


class AWSCatalogPlugin(QObject):
    def __init__(self, iface:QgisInterface):
        super().__init__()
        self.iface = iface

        setTranslation( type(self).__name__, os.path.dirname(__file__) )

        self.plugin_name = 'AWSCatalog'
        self.action_name = 'AWSE Catalog'
        self.action = None

        # Catalog Widget
        task_processor = TaskProcessor( self.iface, 'AWSE Catalog' )
        client = AWSEStacClient()
        self._processor = StacProcessor( iface, task_processor, client )
        self._config_collection = configCollection()

        self.catalog = None # initGui

    def initGui(self)->None:
        # Actions
        path = QDir( os.path.dirname(__file__) )
        icon = QIcon( path.filePath('resources/awse_catalog.svg'))
        self.action = QAction( icon, self.action_name, self.iface.mainWindow() )
        self.action.setToolTip( self.action_name )
        self.action.setCheckable( True )
        self.action.triggered.connect(self.on_Clicked)

        self.menu_name = f"&{self.action_name}"
        self.iface.addPluginToWebMenu( self.menu_name, self.action )
        self.iface.webToolBar().addAction(self.action)

        # Catalog Widget
        widget = CatalogWidget( self.iface, self._config_collection, 'awsecatalogwidget' )
        self.catalog = Catalog( widget, self._processor )
        self.catalog.addWidget()

    def unload(self)->None:
        self.iface.removePluginWebMenu( self.menu_name, self.action )
        self.iface.webToolBar().removeAction( self.action )
        self.iface.unregisterMainWindowAction( self.action )
        # Disconnect
        try:
            self.action.triggered.disconnect( self.on_Clicked )
        except Exception:
            pass
        self.action.deleteLater()

        del self.catalog

    @pyqtSlot(bool)
    def on_Clicked(self, enabled:bool)->None:
        self.catalog.enabled( enabled )
