"""
/***************************************************************************
 atlasDialog
                                 A QGIS plugin
    Create map series given a feature layer and a composer template

                             -------------------
        begin                : 2012-01-14
        copyright            : (C) 2012 by Vincent Picavet (Oslandia)
        email                : vincent.picavet@oslandia.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from math import floor
import re
from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis import core, gui
from qgis.core import QgsPaintEngineHack
from ui_atlas import Ui_atlas
from ui_atlashelp import Ui_atlasHelp
# Import module for about dialog
from atlasabout import aboutDialog

class atlasDialog(QDialog):
    def __init__(self, iface):
        QDialog.__init__(self)
        self.iface = iface
        # get paths
        self.user_plugin_dir = QFileInfo(core.QgsApplication.qgisUserDbFilePath()).path() + "/python/plugins"
        self.atlas_plugin_dir = self.user_plugin_dir + "/atlas"
        # initiate internal properties
        self.composer = None
        self.sLayer = None
        self.mapItem = None
        self.hideCoverage = False
        self.margin = 10
        self.filenamePattern = ""
        self.outputDir = QDir.homePath()
        # Set up the user interface from Designer.
        self.ui = Ui_atlas()
        self.ui.setupUi(self)
        # plugin's settings
        self.settings = QSettings()
        self.loadSettings()
        # fill documentation
        self.directory = QFileInfo(core.QgsApplication.qgisUserDbFilePath()).path() + "/python/plugins/atlas"
        # fill combobox values
        self.updateBoxes()

        self.enableExportButton()
        # connect signals
        # composer changed, update maps
        self.connect(self.ui.compoBox, SIGNAL('currentIndexChanged(int)'), self.updateMaps)
        # any comboBox changed
        self.connect(self.ui.sLayerBox, SIGNAL('currentIndexChanged(int)'), self.anyComboBox_currentIndexChanged)
        self.connect(self.ui.compoBox, SIGNAL('currentIndexChanged(int)'), self.anyComboBox_currentIndexChanged)
        self.connect(self.ui.mapBox, SIGNAL('currentIndexChanged(int)'), self.anyComboBox_currentIndexChanged)
        # any QLineEdit changed
        self.connect(self.ui.outputDir, SIGNAL('textChanged(QString)'), self.anyLineEdit_textChanged)
        self.connect(self.ui.filenamePattern, SIGNAL('textChanged(QString)'), self.anyLineEdit_textChanged)
        # browse button
        self.connect(self.ui.browseOutputDir, SIGNAL('clicked()'), self.updateOutputDir)
        # export button
        self.connect(self.ui.exportButton, SIGNAL('clicked()'), self.renderMap)
        # refresh template button
        self.connect(self.ui.refreshTemplateTool, SIGNAL('clicked()'), self.updateComposers)
        # show composer
        self.connect(self.ui.showComposerButton, SIGNAL('clicked()'), self.showComposer)
        # show about dialog
        self.connect(self.ui.aboutButton, SIGNAL('clicked()'), self.showAbout)
        # show help dialog
        self.connect(self.ui.helpButton, SIGNAL('clicked()'), self.showHelp)
        # save before closing
        self.connect(self.ui.buttonBox, SIGNAL('rejected()'), self.saveSettings)

    def loadSettings(self):
        """Load plugin settings from main application settings"""
        # get values from settings and set values in UI
        self.outputDir = self.settings.value("atlas/outputdir", self.outputDir).toString()
        self.ui.outputDir.setText(self.outputDir)
        self.filenamePattern = self.settings.value("atlas/filenamepattern", "output_file.pdf").toString()
        self.ui.filenamePattern.setText(self.filenamePattern)
        self.margin = self.settings.value("atlas/margin", 10).toInt()[0]
        self.ui.margin.setValue(self.margin)
        if self.settings.value("atlas/hidecoverage", False).toBool():
            self.ui.hideCoverage.setChecked(True)

    def saveSettings(self):
        """Save plugin settings to main application settings"""
        # get values from UI and set them in settings
        self.settings.setValue("atlas/outputdir", self.ui.outputDir.text())
        self.settings.setValue("atlas/filenamepattern", self.ui.filenamePattern.text())
        self.settings.setValue("atlas/margin", self.ui.margin.value())
        self.settings.setValue("atlas/hidecoverage", self.ui.hideCoverage.isChecked())

    def showAbout(self):
        """Show Atlas about dialog box"""
        about = aboutDialog()
        about.show()
        result = about.exec_()
        del about

    def showHelp(self):
        """Show help dialog box"""
        # Create a dialog and setup UI
        hdialog = QDialog()
        hdialog.ui = Ui_atlasHelp()
        hdialog.ui.setupUi(hdialog)
        # load help file
        hdialog.ui.helpContent.setUrl(QUrl("file://" + self.atlas_plugin_dir + '/help/atlas_help.html'))
        hdialog.show()
        result = hdialog.exec_()
        del hdialog

    def anyLineEdit_textChanged(self,  thisText):
        """Slot to be called when user or program changes QLineEdit"""
        self.enableExportButton()

    def anyComboBox_currentIndexChanged(self,  thisIndex):
        """Slot to be called when user or program changes current choice in comboBox"""
        self.enableExportButton()

    def enableExportButton(self):
        """enable exportButton corresponding to the user input and choices"""

        doEnable = (self.ui.sLayerBox.currentIndex() != -1)
        if doEnable:
            self.sLayer = self.ui.sLayerBox.itemData(self.ui.sLayerBox.currentIndex()).toPyObject()
        if doEnable:
            doEnable = (self.ui.compoBox.currentIndex() != -1)
        if doEnable:
            doEnable = (self.ui.mapBox.currentIndex() != -1)
        if doEnable:
            dirName = self.ui.outputDir.text()
            doEnable = (not dirName.isEmpty())
            if doEnable:
                doEnable = QFile(dirName).exists()
        if doEnable:
            doEnable = (not self.ui.filenamePattern.text().isEmpty())

        self.ui.exportButton.setEnabled(doEnable)

    def updateOutputDir(self):
        """Open dialog to choose a directory and set UI"""
        self.ui.outputDir.setText(QFileDialog.getExistingDirectory(self, "Choose output directory"))

    def updateBoxes(self):
        """Updates layers, composers and map combo boxes"""
        self.updateLayers()
        self.updateComposers()
        self.updateMaps()

    def updateLayers(self):
        """Update layers combo box"""
        # empty combo box
        self.ui.sLayerBox.clear()
        # get layers and fill combo box
        for layer in self.iface.mapCanvas().layers():
            if layer.type() == layer.VectorLayer:
                self.ui.sLayerBox.addItem(layer.name(), QVariant(layer))
        # set internal layer to current selected layer
        self.sLayer = self.ui.sLayerBox.itemData(self.ui.sLayerBox.currentIndex()).toPyObject()
        self.ui.sLayerBox.emit(SIGNAL("currentIndexChanged"))

    def updateComposers(self):
        """Update composer combo box"""
        # empty combo box
        self.ui.compoBox.clear()
        # get composers from qgis interface and add them to combo box
        compos = self.iface.activeComposers()
        for cv in compos:
            self.ui.compoBox.addItem(cv.composerWindow().windowTitle(), QVariant(cv))
        # set internal composer to current selected composer
        self.composer = self.ui.compoBox.itemData(self.ui.compoBox.currentIndex()).toPyObject()
        self.ui.compoBox.emit(SIGNAL("currentIndexChanged"))

    def updateMaps(self):
        """Update map list combo box"""
        # empty map combo box
        self.ui.mapBox.clear()
        # only look for maps if a composer is selected
        if self.ui.compoBox.currentIndex() != -1:
            # get current composer and set internal value
            self.composer = self.ui.compoBox.itemData(self.ui.compoBox.currentIndex()).toPyObject()
            # add only map items to map combo box
            for item in self.composer.composition().items():
                if item.type() == core.QgsComposerItem.ComposerMap:
                    self.ui.mapBox.addItem(QApplication.translate("atlas", "Map", None,\
                            QApplication.UnicodeUTF8) + " %s"% item.id(), item.id)
            self.ui.mapBox.emit(SIGNAL("currentIndexChanged"))

    def showComposer(self):
        """Activate selected composer window"""
        self.composer.composerWindow().show()
        self.composer.composerWindow().activate()

    def setLabelReplacementInfos(self):
        """Find labels with replacement text and create struct to deal with them"""
        # init data
        self.labelReplacementInfos = []
        # find text labels in composer
        labels = [item for item in self.composer.composition().items()\
                if item.type() == core.QgsComposerItem.ComposerLabel]
        # find labels with $FIELD() string
        for label in labels:
            fields = set(re.findall('\$FIELD\((\w*)\)', label.text()))
            if fields:
                self.labelReplacementInfos.append(\
                        {'label':label,
                            'originalText':label.text(),
                            'fields':fields})
        # get global field list for feature select method
        if self.labelReplacementInfos:
            if len(self.labelReplacementInfos) == 1:
                selectFieldNames = list(set(*[lri['fields'] for lri in self.labelReplacementInfos]))
            else:
                selectFieldNames = list(set.union(\
                        *[lri['fields'] for lri in self.labelReplacementInfos]))
        else:
            selectFieldNames = []
        selectFields = []
        # list coverage layer fields
        if self.sLayer:
            self.selectFields = filter(lambda x:x>=0,\
                    [self.sLayer.fieldNameIndex(name) for name in selectFieldNames])

    def replaceLabelText(self, fieldValues):
        """Given replacement infos and field values, replace reference to fields by field values"""
        for lri in self.labelReplacementInfos:
            pos = 0
            outText = ''
            # get match groups
            for mg in re.finditer('\$FIELD\((\w*)\)', str(lri['originalText'])):
                # text from current pos to beginnig of the match
                outText += lri['originalText'][pos:mg.start()]
                # given field name, get its index
                fieldIndex = self.sLayer.fieldNameIndex(mg.group(1))
                # if we find the field, add its value
                if fieldValues.has_key(fieldIndex):
                    outText += fieldValues[fieldIndex].toString()
                    pos = mg.end()
            # add the end of the text
            outText += lri['originalText'][pos:]
            # finally sets the label text to the replaced value
            lri['label'].setText(outText)

    def restoreLabelText(self):
        """Restore original text in labels"""
        for lri in self.labelReplacementInfos:
            lri['label'].setText(lri['originalText'])

    def renderMap(self):
        """Render the maps"""
        # deactivate export button
        self.ui.exportButton.setEnabled(False)
        # get values from user interface
        compos = self.iface.activeComposers()
        if compos:
            # FIXME : if activeComposers changed since combobox was filled, bug
            cview = compos[self.ui.compoBox.currentIndex()]
            # get map item object
            self.mapItem = cview.composition().getComposerMapById(\
                    self.ui.mapBox.itemData(self.ui.mapBox.currentIndex()).toInt()[0])
            # set internal values from UI
            self.margin = self.ui.margin.value()
            self.outputDir = self.ui.outputDir.text()
            self.filenamePattern = self.ui.filenamePattern.text()
            self.hideCoverage = self.ui.hideCoverage.isChecked()

            # validate data
            # TODO
            # get pathname
            fileInfo = QFileInfo(self.outputDir + '/' + self.filenamePattern)
            self.filenameBase = fileInfo.absolutePath() + '/' + fileInfo.baseName()
            self.filenameExt = fileInfo.suffix()
            # setup composer
            self.mapItem.setPreviewMode(core.QgsComposerMap.Render)
            # hide coverage layer
            if self.hideCoverage:
                self.iface.legendInterface().setLayerVisible(self.sLayer, False)
            # get composer label replacement info
            self.setLabelReplacementInfos()
            # get provider and start feature selection
            provider = self.sLayer.dataProvider()
            provider.select(self.selectFields)
            feat = core.QgsFeature()
            feature_list = []
            # keep geometries
            while provider.nextFeature(feat):
                feature_list.append(core.QgsFeature(feat))
            # create a progress bar dialog
            progress = QProgressDialog("Rendering maps...", "Abort", 0, len(feature_list), self)
            progress.setWindowModality(Qt.WindowModal)

            # loop on geometries to render coverages
            for i, feature in enumerate(feature_list):
                # if user pressed cancel in progress dialog, stop
                if progress.wasCanceled():
                    break
                # set bbox for map item
                self.mapItemSetBBox(feature.geometry(), self.margin)
                # replace text in labels with feature fields
                self.replaceLabelText(feature.attributeMap())
                # generate output file name
                output_name = "%s_%s.%s" % (self.filenameBase, i, self.filenameExt)
                # choose between PDF or image output and generate image
                if self.filenameExt.toUpper() == 'PDF':
                    self.exportComposerPdf(output_name)
                else:
                    self.exportComposerImage(output_name)
                self.ui.exportComment.setText("Rendering %s" % output_name)
                progress.setValue(i + 1)
            # reset coverage layer
            if self.hideCoverage:
                self.iface.legendInterface().setLayerVisible(self.sLayer, True)
            # reset labels
            self.restoreLabelText()
            # Render is finished or was canceled
            if progress.wasCanceled():
                self.ui.exportComment.setText("Rendering aborted, %s/%s done." % (i, len(feature_list)))
            else:
                self.ui.exportComment.setText("Rendering finished")
            # reenable rendering
            self.enableExportButton()

    def mapItemSetBBox(self, geom, margin = None):
        """Set new extent with optional margin (in %) for map item"""
        self.mapItem.setNewExtent(self.getNewExtent(geom, margin))

    def getNewExtent(self, geom, margin = None):
        """Compute an extent of geometry, with given margin (in %)
        to be able to show it in the selected map item
        Deal with non-square geometries to keep same ratio"""
        # compute coordinates and ratio
        new_extent = None
        x1, y1, x2, y2 = (0, 0, 0, 0)
        geom_rect = geom.boundingBox()
        geom_ratio = geom_rect.width() / geom_rect.height()
        xa1 = geom_rect.xMinimum()
        xa2 = geom_rect.xMaximum()
        ya1 = geom_rect.yMinimum()
        ya2 = geom_rect.yMaximum()
        map_rect = self.mapItem.boundingRect()
        map_ratio = map_rect.width() / map_rect.height()
        # geometry height is too big
        if geom_ratio < map_ratio:
            y1 = ya1
            y2 = ya2
            x1 = (xa1 + xa2 + map_ratio * (ya1 - ya2)) / 2.0
            x2 = x1 + map_ratio * (ya2 - ya1)
            new_extent = core.QgsRectangle(x1, y1, x2, y2)
        # geometry width is too big
        elif geom_ratio > map_ratio:
            x1 = xa1
            x2 = xa2
            y1 = (ya1 + ya2 + (xa1 - xa2) / map_ratio) / 2.0
            y2 = y1 + (xa2 - xa1) / map_ratio
            new_extent = core.QgsRectangle(x1, y1, x2, y2)
        # same ratio: send geom bounding box
        else:
            new_extent = geom_rect
        # add margin to computed extent
        if margin:
            new_extent.scale(1 + margin / 100.0)
        return new_extent

    def exportComposerImage(self, filePath):
        """Save composer image in given file"""
        composition = self.composer.composition()
        saved_plot_style = composition.plotStyle()
        composition.setPlotStyle(core.QgsComposition.Print)
        QApplication.setOverrideCursor(Qt.BusyCursor)

        targetArea, image = self.renderCompositionAsRaster(composition)

        image.save(filePath)
        QApplication.restoreOverrideCursor()
        composition.setPlotStyle(saved_plot_style)

    def exportComposerPdf(self, filePath):
        """Save composer content as pdf in given file"""
        composition = self.composer.composition()
        saved_plot_style = composition.plotStyle()
        composition.setPlotStyle(core.QgsComposition.Print)
        QApplication.setOverrideCursor(Qt.BusyCursor)

        printer = self.setupPdfExport(composition, filePath)
        QgsPaintEngineHack.fixEngineFlags(printer.paintEngine())
        p = QPainter(printer)
        if composition.printAsRaster():
            targetArea, image = self.renderCompositionAsRaster(composition)
            p.drawImage(targetArea, image, targetArea)
        else:
            paperRectMM = printer.pageRect(QPrinter.Millimeter)
            paperRectPixel = printer.pageRect(QPrinter.DevicePixel)
            self.composer.setPaintingEnabled(False)
            composition.render(p, paperRectPixel, paperRectMM)
            self.composer.setPaintingEnabled(True)
        p.end()
        QApplication.restoreOverrideCursor()
        composition.setPlotStyle(saved_plot_style)

    def renderCompositionAsRaster(self, composition):
        """Render the given composition as raster image"""
        width = floor(composition.printResolution() * composition.paperWidth() / 25.4)
        height = floor(composition.printResolution() * composition.paperHeight() / 25.4)
        image = QImage(QSize(width, height), QImage.Format_ARGB32)
        image.setDotsPerMeterX(composition.printResolution() / 25.4 * 1000)
        image.setDotsPerMeterY(composition.printResolution() / 25.4 * 1000)
        image.fill(0)
        imagePainter = QPainter(image)
        sourceArea = QRectF(0, 0, composition.paperWidth(), composition.paperHeight())
        targetArea = QRectF(0, 0, width, height)
        composition.render(imagePainter, targetArea, sourceArea)
        imagePainter.end()
        return targetArea, image

    def setupPdfExport(self, composition, filePath):
        """Setup a printer for PDF output given a composition and output file name"""
        printer = QPrinter()
        printer.setOutputFormat(QPrinter.PdfFormat)
        printer.setOutputFileName(filePath)
        printer.setPaperSize(QSizeF(\
                composition.paperWidth(), composition.paperHeight()),\
                QPrinter.Millimeter)
        printer.setFullPage(True)
        printer.setColorMode(QPrinter.Color)
        printer.setResolution(composition.printResolution())
        return printer




