"""
/***************************************************************************
 atlasDialog
                                 A QGIS plugin
    Create map series given a feature layer and a composer template
                                 
                             -------------------
        begin                : 2012-01-14
        copyright            : (C) 2012 by Vincent Picavet (Oslandia)
        email                : vincent.picavet@oslandia.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

from math import floor
import re
from PyQt4.QtCore import *
from PyQt4.QtGui import *
from qgis import core, gui
from ui_atlas import Ui_atlas
from ui_atlashelp import Ui_atlasHelp
# Import module for about dialog
from atlasabout import aboutDialog

class atlasDialog(QDialog):
    def __init__(self, iface):
        QDialog.__init__(self)
        self.iface = iface
        # get paths
        self.user_plugin_dir = QFileInfo(core.QgsApplication.qgisUserDbFilePath()).path() + "/python/plugins"
        self.atlas_plugin_dir = self.user_plugin_dir + "/atlas"
        # initiate internal properties
        self.composer = None
        self.sLayer = None
        self.mapItem = None
        self.hideCoverage = False
        self.margin = 10
        self.filenamePattern = ""
        self.outputDir = "~/"
        # Set up the user interface from Designer.
        self.ui = Ui_atlas()
        self.ui.setupUi(self)
        # fill documentation
        self.directory = QFileInfo(core.QgsApplication.qgisUserDbFilePath()).path() + "/python/plugins/atlas"
        # fill combobox values
        self.updateBoxes()
        # connect signals
        # composer changed, update maps
        self.connect(self.ui.compoBox, SIGNAL('currentIndexChanged(int)'), self.updateMaps)
        # browse button
        self.connect(self.ui.browseOutputDir, SIGNAL('clicked()'), self.updateOutputDir)
        # render button
        self.connect(self.ui.renderButton, SIGNAL('clicked()'), self.renderMap)
        # refresh template button
        self.connect(self.ui.refreshTemplateTool, SIGNAL('clicked()'), self.updateComposers)
        # show composer
        self.connect(self.ui.showComposerButton, SIGNAL('clicked()'), self.showComposer)
        # show about dialog
        self.connect(self.ui.aboutButton, SIGNAL('clicked()'), self.showAbout)
        # show help dialog
        self.connect(self.ui.helpButton, SIGNAL('clicked()'), self.showHelp)

    def showAbout(self):
        about = aboutDialog()
        about.show()
        result = about.exec_()
        del about

    def showHelp(self):
        hdialog = QDialog()
        hdialog.ui = Ui_atlasHelp()
        hdialog.ui.setupUi(hdialog)
        hdialog.ui.helpContent.setUrl(QUrl(self.atlas_plugin_dir + '/help/atlas_help.html'))
        hdialog.show()
        result = hdialog.exec_()
        del hdialog

    def updateOutputDir(self):
        self.ui.outputDir.setText(QFileDialog.getExistingDirectory(self, "Choose output directory"))
    def updateBoxes(self):
        self.updateLayers()
        self.updateComposers()
        self.updateMaps()

    def updateLayers(self):
        self.ui.sLayerBox.clear()
        for layer in self.iface.mapCanvas().layers():
            self.ui.sLayerBox.addItem(layer.name(), QVariant(layer))
        self.sLayer = self.ui.sLayerBox.itemData(self.ui.sLayerBox.currentIndex()).toPyObject()
    
    def updateComposers(self):
        self.ui.compoBox.clear()
        compos = self.iface.activeComposers()
        for cv in compos:
            self.ui.compoBox.addItem(cv.composerWindow().windowTitle(), QVariant(cv))
        self.composer = self.ui.compoBox.itemData(self.ui.compoBox.currentIndex()).toPyObject()
    
    def updateMaps(self):
        self.ui.mapBox.clear()
        if self.ui.compoBox.currentIndex() != -1:
            self.composer = self.ui.compoBox.itemData(self.ui.compoBox.currentIndex()).toPyObject()
            for item in self.composer.composition().items():
                if item.type() == core.QgsComposerItem.ComposerMap:
                    self.ui.mapBox.addItem("Map %s"% item.id(), item.id)
   
    def showComposer(self):
        self.composer.composerWindow().show()
        self.composer.composerWindow().activate()

    def setLabelReplacementInfos(self):
        # init data
        self.labelReplacementInfos = []
        # find text labels in composer
        labels = [item for item in self.composer.composition().items()\
                if item.type() == core.QgsComposerItem.ComposerLabel]
        # find labels with $FIELD() string
        for label in labels:
            fields = set(re.findall('\$FIELD\(([A-z]*)\)', label.text()))
            if fields:
                self.labelReplacementInfos.append(\
                        {'label':label,
                            'originalText':label.text(),
                            'fields':fields})
        # get global field list for feature select method
        selectFieldNames = list(set.union(\
                *[lri['fields'] for lri in self.labelReplacementInfos]))
        selectFields = []
        # list coverage layer fields
        if self.sLayer:
            self.selectFields = filter(lambda x:x>=0,\
                    [self.sLayer.fieldNameIndex(name) for name in selectFieldNames])

    def replaceLabelText(self, fieldValues):
        for lri in self.labelReplacementInfos:
            pos = 0
            outText = ''
            for mg in re.finditer('\$FIELD\(([A-z]*)\)', str(lri['originalText'])):
                outText += lri['originalText'][pos:mg.start()]
                fieldIndex = self.sLayer.fieldNameIndex(mg.group(1))
                if fieldValues.has_key(fieldIndex):
                    outText += fieldValues[fieldIndex].toString()
                    pos = mg.end()
            outText += lri['originalText'][pos:]
            lri['label'].setText(outText)

    def restoreLabelText(self):
        for lri in self.labelReplacementInfos:
            lri['label'].setText(lri['originalText'])

    def renderMap(self):
        # deactivate render button
        self.ui.renderButton.setEnabled(False)
        # activate progress bar
        # self.ui.progressBar.setEnabled(True)
        # get values from user interface
        compos = self.iface.activeComposers()
        if compos:
            cview = compos[self.ui.compoBox.currentIndex()]
            self.mapItem = cview.composition().getComposerMapById(\
                    self.ui.mapBox.itemData(self.ui.mapBox.currentIndex()).toInt()[0])
            self.margin = self.ui.margin.value()
            self.outputDir = self.ui.outputDir.text()
            self.filenamePattern = self.ui.filenamePattern.text()
            self.hideCoverage = self.ui.hideCoverage.isChecked()
            
            # validate data
            # TODO
            # get pathname
            fileInfo = QFileInfo(self.outputDir + '/' + self.filenamePattern)
            self.filenameBase = fileInfo.absolutePath() + '/' + fileInfo.baseName()
            self.filenameExt = fileInfo.suffix()
            # setup composer
            self.mapItem.setPreviewMode(core.QgsComposerMap.Render)
            # hide coverage layer
            if self.hideCoverage:
                self.iface.legendInterface().setLayerVisible(self.sLayer, False)
            # get composer label replacement info
            self.setLabelReplacementInfos()
            # get provider and start feature selection
            provider = self.sLayer.dataProvider()
            provider.select(self.selectFields)
            feat = core.QgsFeature()
            feature_list = []
            # keep geometries
            while provider.nextFeature(feat):
                feature_list.append(core.QgsFeature(feat))
            # set max for progressbar
            #self.ui.progressBar.setMaximum(len(feature_list))
            #self.ui.progressBar.setValue(0)
            #self.ui.progressBar.setTextVisible(True)
            progress = QProgressDialog("Rendering maps...", "Abort", 0, len(feature_list), self)
            progress.setWindowModality(Qt.WindowModal)
            
            # loop on geometries to render coverages
            for i, feature in enumerate(feature_list):    
                if progress.wasCanceled():
                    break
                self.mapItemSetBBox(feature.geometry(), self.margin)
                self.replaceLabelText(feature.attributeMap())
                output_name = "%s_%s.%s" % (self.filenameBase, i, self.filenameExt)
                if self.filenameExt.toUpper() == 'PDF':
                    self.exportComposerPdf(output_name)
                else:
                    self.exportComposerImage(output_name)
                self.ui.renderComment.setText("Rendering %s" % output_name)
                # self.ui.progressBar.setValue(self.ui.progressBar.value() + 1)
                progress.setValue(i + 1)
            # reset coverage layer
            if self.hideCoverage:
                self.iface.legendInterface().setLayerVisible(self.sLayer, True)
            self.restoreLabelText()
            if progress.wasCanceled():
                self.ui.renderComment.setText("Rendering aborted, %s/%s done." % (i, len(feature_list)))
            else:
                self.ui.renderComment.setText("Rendering finished")
            self.ui.renderButton.setEnabled(True)

            #self.ui.progressBar.setTextVisible(False)


    def mapItemSetBBox(self, geom, margin = None):
        self.mapItem.setNewExtent(self.getNewExtent(geom, margin))

    def getNewExtent(self, geom, margin = None):
        new_extent = None
        x1, y1, x2, y2 = (0, 0, 0, 0)
        geom_rect = geom.boundingBox()
        geom_ratio = geom_rect.width() / geom_rect.height()
        xa1 = geom_rect.xMinimum()
        xa2 = geom_rect.xMaximum()
        ya1 = geom_rect.yMinimum()
        ya2 = geom_rect.yMaximum()
        map_rect = self.mapItem.boundingRect()
        map_ratio = map_rect.width() / map_rect.height()
        # geometry height is too big
        if geom_ratio < map_ratio:
            y1 = ya1
            y2 = ya2
            x1 = (xa1 + xa2 + map_ratio * (ya1 - ya2)) / 2.0
            x2 = x1 + map_ratio * (ya2 - ya1)
            new_extent = core.QgsRectangle(x1, y1, x2, y2)
        # geometry width is too big
        elif geom_ratio > map_ratio:
            x1 = xa1
            x2 = xa2
            y1 = (ya1 + ya2 + (xa1 - xa2) / map_ratio) / 2.0
            y2 = y1 + (xa2 - xa1) / map_ratio
            new_extent = core.QgsRectangle(x1, y1, x2, y2)
        # same ratio: send geom bounding box
        else:
            new_extent = geom_rect
        if margin:
            new_extent.scale(1 + margin / 100.0)
        return new_extent

    def exportComposerImage(self, filePath):
        composition = self.composer.composition()
        saved_plot_style = composition.plotStyle()
        composition.setPlotStyle(core.QgsComposition.Print)
        QApplication.setOverrideCursor(Qt.BusyCursor)

        targetArea, image = self.renderCompositionAsRaster(composition)

        image.save(filePath)
        QApplication.restoreOverrideCursor()
        composition.setPlotStyle(saved_plot_style)

    def exportComposerPdf(self, filePath):
        composition = self.composer.composition()
        saved_plot_style = composition.plotStyle()
        composition.setPlotStyle(core.QgsComposition.Print)
        QApplication.setOverrideCursor(Qt.BusyCursor)

        printer = self.setupPdfExport(composition, filePath)
        p = QPainter(printer)
        if composition.printAsRaster():
            targetArea, image = renderCompositionAsRaster(composition)
            p.drawImage(targetArea, image, targetArea)
        else:
            paperRectMM = printer.pageRect(QPrinter.Millimeter)
            paperRectPixel = printer.pageRect(QPrinter.DevicePixel)
            composition.render(p, paperRectPixel, paperRectMM)
        p.end()
        QApplication.restoreOverrideCursor()
        composition.setPlotStyle(saved_plot_style)

    def renderCompositionAsRaster(self, composition):
        width = floor(composition.printResolution() * composition.paperWidth() / 25.4)
        height = floor(composition.printResolution() * composition.paperHeight() / 25.4)
        image = QImage(QSize(width, height), QImage.Format_ARGB32)
        image.setDotsPerMeterX(composition.printResolution() / 25.4 * 1000)
        image.setDotsPerMeterY(composition.printResolution() / 25.4 * 1000)
        image.fill(0)
        imagePainter = QPainter(image)
        sourceArea = QRectF(0, 0, composition.paperWidth(), composition.paperHeight())
        targetArea = QRectF(0, 0, width, height)
        composition.render(imagePainter, targetArea, sourceArea)
        imagePainter.end()
        return targetArea, image

    def setupPdfExport(self, composition, filePath):
        printer = QPrinter()
        printer.setOutputFormat(QPrinter.PdfFormat)
        printer.setOutputFileName(filePath)
        printer.setPaperSize(QSizeF(\
                composition.paperWidth(), composition.paperHeight()),\
                QPrinter.Millimeter)
        printer.setFullPage(True)
        printer.setColorMode(QPrinter.Color)
        printer.setResolution(composition.printResolution())
        return printer



    
