# -*- coding: utf-8 -*-
"""
/***************************************************************************
 NimboEarth
                                 A QGIS plugin
 This plugins allows you to access all Nimbo earth tile map services.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                              -------------------
        begin                : 2022-10-18
        git sha              : $Format:%H$
        copyright            : (C) 2022 by Kermap
        email                : contact@nimbo.earth
 ***************************************************************************/

 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""
from email.charset import QP
from logging import raiseExceptions
import webbrowser
from qgis.PyQt.QtCore import QSettings, QTranslator, QCoreApplication, Qt, QSize, QUrl
from qgis.PyQt.QtGui import QIcon, QPixmap, QDesktopServices
from qgis.PyQt.QtWidgets import QAction, QLineEdit, QListWidgetItem
from qgis.core import QgsRasterLayer, QgsProject
from qgis.utils import showPluginHelp

from .constants import ABOUT_URL, SERVICE_URL, ImageComposition
from .models.models import NimboUser, XYZLayerModel
from .services.checks import Checks
from .services.services import Services
from .style.style import get_style

import requests

# Initialize Qt resources from file resources.py
from .resources import *

# Import the code for the DockWidget
from .nimbo_earth_dockwidget import NimboEarthDockWidget
import os.path


class NimboEarth:
    """QGIS Plugin Implementation."""

    def __init__(self, iface):
        """Constructor.

        :param iface: An interface instance that will be passed to this class
            which provides the hook by which you can manipulate the QGIS
            application at run time.
        :type iface: QgsInterface
        """
        # Save reference to the QGIS interface
        self.iface = iface

        self.user = NimboUser()
        self.checks = Checks()
        self.services = Services()

        # initialize plugin directory
        self.plugin_dir = os.path.dirname(__file__)

        # initialize locale
        locale = QSettings().value('locale/userLocale')[0:2]
        locale_path = os.path.join(
            self.plugin_dir,
            'i18n',
            'NimboEarth_{}.qm'.format(locale))

        if os.path.exists(locale_path):
            self.translator = QTranslator()
            self.translator.load(locale_path)
            QCoreApplication.installTranslator(self.translator)

        # Declare instance attributes
        self.actions = []
        self.menu = self.tr(u'&Nimbo Earth')
        # TODO: We are going to let the user set this up in a future iteration
        self.toolbar = self.iface.addToolBar(u'NimboEarth')
        self.toolbar.setObjectName(u'NimboEarth')

        # print "** INITIALIZING NimboEarth"

        self.pluginIsActive = False
        self.dockwidget = None

    # noinspection PyMethodMayBeStatic


    def tr(self, message):
        """Get the translation for a string using Qt translation API.

        We implement this ourselves since we do not inherit QObject.

        :param message: String for translation.
        :type message: str, QString

        :returns: Translated version of message.
        :rtype: QString
        """
        # noinspection PyTypeChecker,PyArgumentList,PyCallByClass
        return QCoreApplication.translate('NimboEarth', message)


    def add_action(
            self,
            icon_path,
            text,
            callback,
            enabled_flag=True,
            add_to_menu=True,
            add_to_toolbar=True,
            status_tip=None,
            whats_this=None,
            parent=None):
        """Add a toolbar icon to the toolbar.

        :param icon_path: Path to the icon for this action. Can be a resource
            path (e.g. ':/plugins/foo/bar.png') or a normal file system path.
        :type icon_path: str

        :param text: Text that should be shown in menu items for this action.
        :type text: str

        :param callback: Function to be called when the action is triggered.
        :type callback: function

        :param enabled_flag: A flag indicating if the action should be enabled
            by default. Defaults to True.
        :type enabled_flag: bool

        :param add_to_menu: Flag indicating whether the action should also
            be added to the menu. Defaults to True.
        :type add_to_menu: bool

        :param add_to_toolbar: Flag indicating whether the action should also
            be added to the toolbar. Defaults to True.
        :type add_to_toolbar: bool

        :param status_tip: Optional text to show in a popup when mouse pointer
            hovers over the action.
        :type status_tip: str

        :param parent: Parent widget for the new action. Defaults None.
        :type parent: QWidget

        :param whats_this: Optional text to show in the status bar when the
            mouse pointer hovers over the action.

        :returns: The action that was created. Note that the action is also
            added to self.actions list.
        :rtype: QAction
        """

        icon = QIcon(icon_path)
        action = QAction(icon, text, parent)
        action.triggered.connect(callback)
        action.setEnabled(enabled_flag)

        if status_tip is not None:
            action.setStatusTip(status_tip)

        if whats_this is not None:
            action.setWhatsThis(whats_this)

        if add_to_toolbar:
            self.toolbar.addAction(action)

        if add_to_menu:
            self.iface.addPluginToWebMenu(self.menu, action)

        self.actions.append(action)

        return action


    def initGui(self):
        """Create the menu entries and toolbar icons inside the QGIS GUI."""

        icon_path = ':/plugins/nimbo_earth/assets/icon.png'
        self.add_action(
            icon_path,
            text=self.tr(u'Import XYZ Layer'),
            callback=self.run,
            parent=self.iface.mainWindow()
        )
        self.add_action(
            icon_path=':/plugins/nimbo_earth/assets/icon.png',
            text=self.tr(u'About Nimbo'),
            callback=self.go_to_about,
            parent=self.iface.mainWindow(),
            add_to_toolbar=False
        )
        self.add_action(
            icon_path=':/plugins/nimbo_earth/assets/question.png',
            text=self.tr(u'Plugin help'),
            callback=self.go_to_help,
            parent=self.iface.mainWindow(),
            add_to_toolbar=False
        )


    def go_to_about(self):
        webbrowser.open(ABOUT_URL)
    
    
    def go_to_help(self):   
        abs_path = os.path.dirname(os.path.abspath(__file__))
        help_path = os.path.join(abs_path, 'help/build/html/index.html')
        help_file = 'file:///'+help_path
        QDesktopServices.openUrl(QUrl(help_file))
    # --------------------------------------------------------------------------


    def onClosePlugin(self):
        """Cleanup necessary items here when plugin dockwidget is closed"""

        # print "** CLOSING NimboEarth"

        # disconnects
        # don't forget to disconnect push buttons otherwise it will run methods multiple times after reopening
        self.dockwidget.key_pButton.disconnect()
        self.dockwidget.login_pButton.disconnect()
        self.dockwidget.add_map_cbox_pButton.disconnect()
        self.dockwidget.add_map_list_pButton.disconnect()
        self.dockwidget.closingPlugin.disconnect(self.onClosePlugin)
        

        # remove this statement if dockwidget is to remain
        # for reuse if plugin is reopened
        # Commented next statement since it causes QGIS crashe
        # when closing the docked window:
        # self.dockwidget = None

        self.pluginIsActive = False


    def unload(self):
        """Removes the plugin menu item and icon from QGIS GUI."""

        # print "** UNLOAD NimboEarth"

        for action in self.actions:
            self.iface.removePluginWebMenu(self.tr(u'&Nimbo Earth'),action)
            self.iface.removeToolBarIcon(action)
        # remove the toolbar
        del self.toolbar

    # --------------------------------------------------------------------------


    def run(self):
        """Run method that loads and starts the plugin"""

        if not self.pluginIsActive:
            self.pluginIsActive = True

            # print "** STARTING NimboEarth"

            # dockwidget may not exist if:
            #    first run of plugin
            #    removed on close (see self.onClosePlugin method)
            if self.dockwidget == None:
                # Create the dockwidget (after translation) and keep reference
                self.dockwidget = NimboEarthDockWidget()
                get_style(self.dockwidget)

            # setting free basemap button event
            self.dockwidget.free_pButton.clicked.connect(self.addFreeLayer)
            
            # setting 3d dem basemap button event
            # self.dockwidget.layer_dem.clicked.connect(self.add3DLayer)
            
            # setting redirect button to nimbo subscription page
            self.dockwidget.subscribe_pButton.clicked.connect(lambda: webbrowser.open('https://maps.nimbo.earth/freeregister'))
            
            # setting redirect button to nimbo pricing page
            self.dockwidget.pricing_pButton.clicked.connect(lambda: webbrowser.open('https://nimbo.earth/pricing/'))
            self.dockwidget.pricing_link.clicked.connect(lambda: webbrowser.open('https://nimbo.earth/pricing/'))
            
            # setting icons for links
            self.dockwidget.nimbo_icon_label.setPixmap(QPixmap(":/plugins/nimbo_earth/assets/icon.png"))
            self.dockwidget.kermap_icon_label.setPixmap(QPixmap(":/plugins/nimbo_earth/assets/kermapLogo.png"))
            # setting text for links
            self.dockwidget.nimbo_label.setText('<a style="color: white;font-weight:bold;text-decoration: none;" href="https://nimbo.earth">{0}</a>'.format(self.tr("About Nimbo")))
            self.dockwidget.nimbo_label.setOpenExternalLinks(True)
            self.dockwidget.kermap_label.setText('<a style="color: white;font-weight:bold; text-decoration: none;" href="https://kermap.com/en/our-story/">{0}</a>'.format(self.tr("About Kermap")))
            self.dockwidget.kermap_label.setOpenExternalLinks(True)
            # disabling second tab widget
            self.dockwidget.tabWidget.setTabEnabled(1, False)
            # hiding geocredit label until login
            self.dockwidget.gc_label.setVisible(False)

            # login or entering API Key
            self.dockwidget.tabWidget.setCurrentWidget(self.dockwidget.key_tab)
            self.dockwidget.password_lineEdit.setEchoMode(QLineEdit.Password)
            self.dockwidget.eye_pButton.setIcon(QIcon(QPixmap(":/plugins/nimbo_earth/assets/eye-off.png")))
            self.dockwidget.eye_pButton.pressed.connect(self.show_password)
            self.dockwidget.eye_pButton.released.connect(self.hide_password)
            self.dockwidget.login_pButton.clicked.connect(self.login)
            self.dockwidget.key_pButton.clicked.connect(self.update_api_key)

            # selecting layer to add
            self.dockwidget.composition_selector_comBox.currentTextChanged.connect(self.clear_list_selection)
            self.dockwidget.month_comBox.currentTextChanged.connect(self.clear_list_selection)
            self.dockwidget.year_comBox.currentTextChanged.connect(self.clear_list_selection)
            self.dockwidget.add_map_cbox_pButton.clicked.connect(self.get_layer)
            self.dockwidget.add_map_list_pButton.clicked.connect(self.get_layer)

            # connect to provide cleanup on closing of dockwidget
            self.dockwidget.closingPlugin.connect(self.onClosePlugin)

            # show the dockwidget
            # TODO: fix to allow choice of dock location
            self.iface.addDockWidget(Qt.RightDockWidgetArea, self.dockwidget)
            self.dockwidget.show()
    
    
    def show_password(self):
        self.dockwidget.password_lineEdit.setEchoMode(QLineEdit.Normal)
        self.dockwidget.eye_pButton.setIcon(QIcon(QPixmap(":/plugins/nimbo_earth/assets/eye.png")))
        self.dockwidget.eye_pButton.setIconSize(QSize(16, 16))


    def hide_password(self):
        self.dockwidget.password_lineEdit.setEchoMode(QLineEdit.Password)
        self.dockwidget.eye_pButton.setIcon(QIcon(QPixmap(":/plugins/nimbo_earth/assets/eye-off.png")))
        self.dockwidget.eye_pButton.setIconSize(QSize(16, 16))


    def login(self):
        # getting email and password from line edit
        self.user.email = self.dockwidget.email_lineEdit.text().strip()
        self.user.password = self.dockwidget.password_lineEdit.text().strip()

        # getting api_key and access_token
        self.user.api_key = ''
        access_token = self.services.get_access_token(self.user.email, self.user.password)
        if access_token is not None:
            self.user.api_key = self.services.get_kermap_token(access_token)
            # get user subscription type (FREE/PRO)
            self.user.subscription_type = self.services.get_user_subscription_type(access_token)
            # print(self.user.subscription_type)
        else:
            self.user.api_key = None
            self.user.subscription_type = None

        # setting token to api key line edit
        if self.user.api_key is not None:
            # self.dockwidget.key_input_lineEdit.setText(self.user.api_key)
            self.validate_api_key(self.user.api_key)
        else:
            self.iface.messageBar().pushMessage(self.tr("Error"), self.tr("Incorrect email or password "), level=2, duration=5)


    def update_api_key(self):
        """update the api key value from the input of the line edit"""
        # resetting api key
        self.user.api_key = ''
        # getting the api key copied in the line edit
        self.user.api_key = self.dockwidget.key_input_lineEdit.text().strip()
        # checking api key validity
        self.validate_api_key(self.user.api_key)


    def validate_api_key(self, api_key):
        """checks if the api key is valid"""
        # getting the base service url and adding the api key to it
        request_url = SERVICE_URL + api_key

        headers = {'User-Agent': "QGIS Nimbo plugin"}
        # making a request with the url to see if the response status is 200 or not
        # if status is 200 then update the layer selection with the list of urls availables
        # else raising exceptions
        response = requests.get(request_url, headers=headers) 
        if response.status_code == 200:
            self.get_geocredits(response)
            # Set user subscription type from API key
            self.user.subscription_type = self.services.get_user_subscription_type_from_kermap_token(api_key)
            # print(f"User subscription type: {self.user.subscription_type}")
            xml_file = response.content
            self.iface.messageBar().pushMessage(self.tr("Success"), self.tr("Your API key is valid"), level=3, duration=5)
            self.update_layer_selection(xml_file, api_key)
        else:
            self.iface.messageBar().pushMessage(self.tr("Error"), self.tr("Your API Key is not valid"), level=2, duration=5)


    def get_geocredits(self, response):
        geocredit = int(response.headers['X-Kermap-Available-Credits'])
        if geocredit >= 1000:
            geocredit = format(geocredit/1000, ".2f")
            label = str(geocredit)+"K"
            self.dockwidget.geocredit_label.setStyleSheet('QLabel[objectName^="geocredit_label"]{color: #00b400;}')
        else: 
            label = str(geocredit)
            self.dockwidget.geocredit_label.setStyleSheet('QLabel[objectName^="geocredit_label"]{color: red;}')
        self.dockwidget.gc_label.setVisible(True)    
        self.dockwidget.geocredit_label.setText(label)
        
        
    def update_layer_selection(self, xml_file, api_key):
        # getting the tile maps list
        self.tile_maps = self.services.get_tile_maps(xml_file, self.user.subscription_type)

        # --- Patch: ensure year is always valid for FREE users ---
        if hasattr(self.user, 'subscription_type') and self.user.subscription_type == 'FREE':
            for layer in self.tile_maps.layers:
                # If year is not a valid integer, try to re-parse from title or href
                try:
                    int(layer.year)
                except Exception:
                    # Try to parse from title (prefer title for watermark)
                    parts = layer.title.split('_')
                    if len(parts) == 4 and parts[0].lower() == 'watermark':
                        layer.year = parts[1]
                    else:
                        href_parts = layer.href.split('/')[-1].split('@')[0].split('_')
                        if len(href_parts) == 4 and href_parts[0].lower() == 'watermark':
                            layer.year = href_parts[1]
        # --- End patch ---

        # populating the combo boxes and the list widget
        self.populating_cbboxes_and_listwidget()
        # changing the tab for the user to select the layer
        self.dockwidget.tabWidget.setTabEnabled(1, True)
        self.dockwidget.tabWidget.setCurrentWidget(self.dockwidget.tms_tab)


    def populating_cbboxes_and_listwidget(self):
        # populating image composition combo box if empty
        if self.dockwidget.composition_selector_comBox.count() == 0:
            # Only allow RGB for FREE/watermark users
            if hasattr(self.user, 'subscription_type') and self.user.subscription_type == 'FREE':
                self.dockwidget.composition_selector_comBox.clear()
                self.dockwidget.composition_selector_comBox.addItem(ImageComposition(1).describe())
                self.dockwidget.composition_selector_comBox.setCurrentText(ImageComposition.NATURAL.describe())
            else:
                compositions = self.services.get_composition_from_layers(self.tile_maps)
                for composition in compositions:
                    self.dockwidget.composition_selector_comBox.addItem(ImageComposition(composition).describe())
                self.dockwidget.composition_selector_comBox.setCurrentText(ImageComposition.NATURAL.describe())

        # populating year combo box if empty
        if self.dockwidget.year_comBox.count() == 0:
            for year in range(self.services.get_min_year(self.tile_maps), self.services.get_max_year(self.tile_maps)+1):
                self.dockwidget.year_comBox.addItem(str(year))
                self.dockwidget.year_comBox.setCurrentText(str(self.services.get_max_year(self.tile_maps)))

        # populating month combo box depending on year selected
        self.month_selection()
        print('month_selection =',self.month_selection())
        self.dockwidget.year_comBox.currentTextChanged.connect(self.month_selection)

        # adding layers to the listWidget
        self.dockwidget.layer_listWidget.clear()

        # Heuristic: if there are no dated (year+month) non-RGB layers in the dataset, we consider the user FREE for dated PRO
        has_dated_pro_layers = False
        for l in self.tile_maps.layers:
            try:
                comp_name = self.services.get_composition_name(l.composition)
            except Exception:
                comp_name = ''
            if comp_name != ImageComposition.NATURAL.__str__() and getattr(l, 'year', '') and getattr(l, 'month', ''):
                has_dated_pro_layers = True
                break

        # Build list, interleaving placeholders right after each dated RGB when FREE
        for layer in self.tile_maps.layers:
            # Base item text
            if "no title set" in layer.title:
                month_name = self.services.get_month_name(layer.month)
                comp_name = self.services.get_composition_name(layer.composition)
                base_text = f"{month_name} {layer.year} {comp_name}"
            else:
                base_text = layer.title
                # Also compute comp_name for condition below
                try:
                    comp_name = self.services.get_composition_name(layer.composition)
                except Exception:
                    comp_name = ''

            # If user is FREE, ignore real non-RGB compositions (NIR/NDVI/RADAR)
            if (getattr(self.user, 'subscription_type', None) == 'FREE') and comp_name != ImageComposition.NATURAL.__str__():
                # Skip adding the actual item for non-RGB
                pass
            else:
                self.dockwidget.layer_listWidget.addItem(base_text)

            # Interleave placeholders if user is FREE and this is a dated RGB
            if (getattr(self.user, 'subscription_type', None) == 'FREE') and comp_name == ImageComposition.NATURAL.__str__() and getattr(layer, 'year', '') and getattr(layer, 'month', ''):
                month_name = self.services.get_month_name(str(layer.month))
                year = str(layer.year)
                for comp_label in [
                    ImageComposition.INFRARED.__str__(),
                    ImageComposition.VEGETATION.__str__(),
                    ImageComposition.RADAR.__str__(),
                ]:
                    text = f"{month_name} {year} {comp_label} [PAID PLAN ONLY]"
                    item = QListWidgetItem(text)
                    item.setFlags(item.flags() & ~Qt.ItemIsEnabled & ~Qt.ItemIsSelectable)
                    item.setToolTip(self.tr("This layer is available with PAID plans only."))
                    self.dockwidget.layer_listWidget.addItem(item)
        self.reverse_sort_layers()
            

    def month_selection(self):
        # clear the combobox
        self.dockwidget.month_comBox.clear()

        if self.dockwidget.month_comBox.count() == 0:
            # getting all months for a semected year
            months = self.services.get_month_for_this_year(self.dockwidget.year_comBox.currentText(), self.tile_maps)
            # getting month name et adding it to the combo box
            for month in months:
                self.dockwidget.month_comBox.addItem(self.services.get_month_name(str(month)))


    def clear_list_selection(self):
        self.dockwidget.layer_listWidget.clearSelection()

    def reverse_sort_layers(self):
        # Collect item properties to preserve flags/tooltips
        items_data = []
        for i in range(self.dockwidget.layer_listWidget.count()):
            it = self.dockwidget.layer_listWidget.item(i)
            items_data.append((it.text(), it.flags(), it.toolTip()))

        # If no placeholders, keep simple reverse for backward compatibility
        has_placeholders = any('[PRO ONLY]' in text for (text, _, _) in items_data)
        if not has_placeholders:
            items_data = list(reversed(items_data))
            self.dockwidget.layer_listWidget.clear()
            for text, flags, tooltip in items_data:
                it = QListWidgetItem(text)
                it.setFlags(flags)
                if tooltip:
                    it.setToolTip(tooltip)
                self.dockwidget.layer_listWidget.addItem(it)
            return

        # Group by (year, month) so placeholders stay next to their RGB after reversing
        month_to_num = {
            'January': 1, 'February': 2, 'March': 3, 'April': 4, 'May': 5, 'June': 6,
            'July': 7, 'August': 8, 'September': 9, 'October': 10, 'November': 11, 'December': 12,
        }
        groups = {}
        others = []
        for text, flags, tooltip in items_data:
            parts = text.split(' ')
            if len(parts) >= 3 and parts[0] in month_to_num and parts[1].isdigit():
                key = (int(parts[1]), month_to_num[parts[0]])
                groups.setdefault(key, []).append((text, flags, tooltip))
            else:
                others.append((text, flags, tooltip))

        # Sort groups newest first
        sorted_keys = sorted(groups.keys(), key=lambda k: (k[0], k[1]), reverse=True)

        # Rebuild list: within each group, put RGB first then placeholders (NIR, NDVI, RADAR)
        ordered_items = []
        for key in sorted_keys:
            items = groups[key]
            rgb = [it for it in items if ('[PRO ONLY]' not in it[0]) and (' RGB' in it[0])]
            placeholders_nir = [it for it in items if ('[PRO ONLY]' in it[0]) and (' NIR ' in it[0])]
            placeholders_ndvi = [it for it in items if ('[PRO ONLY]' in it[0]) and (' NDVI ' in it[0])]
            placeholders_radar = [it for it in items if ('[PRO ONLY]' in it[0]) and (' RADAR ' in it[0])]
            others_real = [it for it in items if ('[PRO ONLY]' not in it[0]) and (' RGB' not in it[0])]
            ordered_items.extend(rgb + placeholders_nir + placeholders_ndvi + placeholders_radar + others_real)

        # Append non-dated items reversed to keep prior overall reverse behavior
        ordered_items.extend(list(reversed(others)))

        # Rebuild list preserving flags and tooltips
        self.dockwidget.layer_listWidget.clear()
        for text, flags, tooltip in ordered_items:
            it = QListWidgetItem(text)
            it.setFlags(flags)
            if tooltip:
                it.setToolTip(tooltip)
            self.dockwidget.layer_listWidget.addItem(it)

    def get_layer(self):
        # print("DEBUG: get_layer called, button click registered")  # Debug log
        # re-initializing the layer
        layer = XYZLayerModel()

        # if an item is selected on the list
        if len(self.dockwidget.layer_listWidget.selectedItems()) != 0:
            value = self.dockwidget.layer_listWidget.currentItem()
            # getting year, month and composition from the item selected
            data_retrieved = value.text().split(" ")
            layer = self.services.filtering_layers(data_retrieved)
            if layer is not None:
                self.add_layer(layer)

        else:
            # getting the composition value form the combobox
            composition_value = self.dockwidget.composition_selector_comBox.currentText()

            # getting the enum value from composition_value
            composition_string = ''
            for composition in ImageComposition:
                if composition_value.upper() == composition.name:
                    composition_string = composition.__str__()

            # getting the month value from the combo box
            month = self.dockwidget.month_comBox.currentText()

            # getting the year value from the combo box
            year = self.dockwidget.year_comBox.currentText()
            # assembling data in list
            data_retrieved = [month, year, composition_string]
            self.layer = self.services.filtering_layers(data_retrieved)
            # print(f"DEBUG: get_layer called, layer={self.layer}")  # Debug log
            if self.layer:
                self.add_layer(self.layer)
            else:
                self.iface.messageBar().pushMessage(
                    self.tr("Warning"),
                    self.tr("No Layer with those parameters: please change date or composition"), level=1, duration=5
                )


    def add_layer(self, layer):
        # Construct the correct layer identifier in the URL based on user type
        year = str(layer.year)
        month_raw = str(layer.month)
        compo = str(layer.composition)
        if hasattr(self.user, 'subscription_type') and self.user.subscription_type == 'FREE':
            month = month_raw  # Do NOT zero-pad
            if int(month_raw) < 10:
                month = month[1:]
            layer_id = f"watermark_{year}_{month}_1"
        else:
            # PRO: do not zero-pad month
            month = str(int(month_raw)) if month_raw.isdigit() else month_raw
            layer_id = f"{year}_{month}_{compo}"
        # Build the full href using the correct layer_id
        base_url = layer.href.split('/tms/1.0.0/')[0] + '/tms/1.0.0/'
        layer.href = f"type=xyz&url={base_url}{layer_id}@kermap/{{z}}/{{x}}/{{-y}}.png?kermap_token={self.user.api_key}"
        # print(f"DEBUG: add_layer called, href={layer.href}")  # Debug log
        if "no title set" in layer.title:
            title = self.services.get_month_name(str(layer.month)) + ' ' + layer.year + ' ' + self.services.get_composition_name(layer.composition)
            rlayer = QgsRasterLayer(layer.href, title, "wms")
        else:
            rlayer = QgsRasterLayer(layer.href, layer.title, "wms")

        if rlayer.isValid():
            QgsProject().instance().addMapLayer(rlayer)
            self.iface.messageBar().pushMessage(self.tr("Success"), self.tr("Layer added - wait until loading is complete"), level=3, duration=5)
        else:
            self.iface.messageBar().pushMessage(self.tr("Warning"), self.tr("Invalid layer: unable to add it to the project"), level=1, duration=5)


    def addFreeLayer(self):
        # re-initializing the layer
        layer = XYZLayerModel()
        layer.href = "type=xyz&url=https://prod-data.nimbo.earth/mapcache-free/tms/1.0.0/latest@kermap/{z}/{x}/{-y}.png"
        layer.title = "June 2023 RGB"
        flayer = QgsRasterLayer(layer.href, layer.title, "wms")
        if flayer.isValid():
            QgsProject().instance().addMapLayer(flayer)
            self.iface.messageBar().pushMessage(self.tr("Success"), self.tr("Layer added - wait until loading is complete"), level=3, duration=5)
        else:
            self.iface.messageBar().pushMessage(self.tr("Warning"), self.tr("Invalid layer: unable to add it to the project"), level=1, duration=5)
            
            
    def add3DLayer(self):
        # re-initializing the layer
        layer = XYZLayerModel()
        layer.href = "type=xyz&url=https://prod-data.nimbo.earth/tiles/terrain-dem/{z}/{x}/{y}?kermap_token=" + self.user.api_key
        layer.title = "Nimbo Dem Layer"
        flayer = QgsRasterLayer(layer.href, layer.title, "wms")
        if flayer.isValid():
            QgsProject().instance().addMapLayer(flayer)
            self.iface.messageBar().pushMessage(self.tr("Success"), self.tr("Layer added - wait until loading is complete"), level=3, duration=5)
        else:
            self.iface.messageBar().pushMessage(self.tr("Warning"), self.tr("Invalid layer: unable to add it to the project"), level=1, duration=5)
        